require 'delegate'

module TapKit

	class SnapshotStore
		attr_reader :snapshots, :current_time

		def initialize
			@current_time = Time.new

			@snapshots = {} # GlobalID -> snapshot
			@count     = {} # GlobalID -> count
			@times     = {} # GlobalID -> time
			@gids_src  = {} # GlobalID -> {'name' -> GlobalIDs (array)}
			@times_src = {} # GlobalID -> {'name' -> time}
		end

		def snapshot( gid, time = nil )
			ss = @snapshots[gid]
			if time then
				ss = nil unless ss.time >= time
			end
			ss
		end

		def snapshot_for_source( gid, name, time = nil )
			ss = @gids_src[gid][name]
			if time then
				ss = nil unless ss.time >= time
			end
			ss
		end

		def record_snapshot( gid, snapshot )
			set_timestamp_to_now
			@snapshots[gid] = snapshot
			@times[gid]     = @current_time
			@count[gid]     = 0
			@gids_src[gid]  = {}
			@times_src[gid] = {}
		end

		def record_snapshot_for_source( gids, gid, name )
			set_timestamp_to_now

			# GlobalID -> {'name' -> GlobalIDs (list)}
			gids_src       = @gids_src[gid] || {}
			gids_src[name] = gids
			@gids_src[gid] = gids_src

			# GlobalID -> {'name' -> timeStamp}
			times_src       = @times_src[gid] || {}
			times_src[name] = @current_time
			@times_src[gid] = times_src
		end

		def record_snapshots( snapshots )
			snapshots.each_key do |gid|
				record_snapshot(gid, snapshots[gid])
			end
		end

		# name => [gid, ...]
		def record_to_many_snapshots( snapshots )
			snapshots.each do |name, array|
				array.each do |gid|
					record_snapshot_for_source(array, gid, name)
				end
			end
		end

		def forget_all_snapshots
			@snapshots.clear
			@count.clear
			@times.clear
			@gids_src.clear
			@times_src.clear
		end

		def forget_snapshot( gid )
			@snapshots.delete gid
		end

		def forget_snapshots( gids )
			gids.each do |gid|
				@snapshots.delete gid
			end
		end

		# Increments (reference) count of database objects. Do we need this method?
		def increment( gid )
			if @count[gid] then
				@count[gid] += 1
			else
				@count[gid] = 1
			end
		end

		def decriment( gid )
			unless @count[gid] then
				raise RuntimeError, "The snapshot is already released."
			end
			@count[gid] -= 1

			if @count[gid] == 0 then
				forget_snapshot gid
			end
		end

		def time( gid )
			@times[gid]
		end

		def time_for_source( gid, name )
			@times_src[gid][name]
		end

		def set_timestamp_to_now
			@current_time = Time.new
		end

		def to_many_snapshots
			snapshots = {}
			@gids_src.each do |gid, hash|
				hash.each do |name, gids|
					snapshots[name] = gids
				end
			end
			snapshots
		end
	end


	# There are 2 snapshots.
	#
	# * EditingContext
	# * Database
	class Snapshot < DelegateClass(Hash)
		attr_accessor :values, :primary_key, :property_keys, :foreign_key, :locking_attributes, :time

		def initialize
			@values = {}
			@property_keys = []
			@locking_attributes = []
			super @values
		end

		# * Set stored value for each attributes
		# * Set a global ID or nil for each to-one relationships
		# * Set an array of global IDs or a fault object for each to-many relationships
		def update_from_object( editing_context, object )
			object.attribute_keys.each do |key|
				@values[key] = object.retrieve_stored_value key
			end

			object.to_one_relationship_keys.each do |key|
				dest = object.retrieve_stored_value key

				if dest.nil? then
					@values[key] = nil
				elsif dest.fault? then
					@values[key] = dest.fault_handler.gid
				else
					@values[key] = editing_context.gid dest
				end
			end

			object.to_many_relationship_keys.each do |key|
				array = object.retrieve_stored_value key
				unless FaultingDelayEvaluationArray === array then
					gids = []
					array.each do |dest|
						gids << editing_context.gid(dest)
					end
					@values[key] = gids
				else
					@values[key] = array
				end
			end
		end

		def update( changes )
			@values.update changes
		end

		def each
			@values.each do |key, value|
				yield key, value
			end
		end

		def []( key )
			@values[key]
		end

		def []=( key, value )
			@values[key] = value
		end
	end

end
