require 'csv'

module TapKit

	class CSVAdapter < Adapter
		class << self
			def expression_class
				CSVExpression
			end

			def internal_type( external_type )
				String
			end
		end

		def expression_factory
			CSVExpressionFactory.new self
		end
	end

	class CSVChannel < AdapterChannel
		attr_reader :csvdb

		def initialize( adapter_context )
			super
			@pk_cache = Hash.new(0)
		end

		def fetch_progress?
			false
		end

		def open?
			if @csvdb then
				true
			else
				false
			end
		end

		def open
			unless open? then
				connection = @adapter_context.adapter.connection
				path  = connection['path']
				@csvdb = CSVDatabase.new(path)
			end
		end

		def close
			if @csvdb then
				@csvdb = nil
			end
		end

		def select_attributes( attributes, lock, fetch_spec, entity )
			@attributes_to_fetch = attributes
		end

		def evaluate( expression )
			@entity = expression.entity
			open

			bindings = []
			row = {}
			expression.bind_variables.each do |binding|
				value = binding[expression.class::VALUE_KEY]
				attr = binding[expression.class::ATTRIBUTE_KEY]
				formatted = expression.format_value(value, attr)
				bindings << formatted
				row[attr.column_name.to_i] = formatted
			end
			expression.row = row

			statement = expression.statement.dup
			index = 0
			statement.gsub!('?') do |char|
				index += 1
				bindings[index-1]
			end

			if application then
				if application.log_options[:sql] then
					application.log_options[:out].puts expression.statement
				end
			end

			@pk_cache.clear
			@result = @csvdb.evaluate expression
		end

		def fetch_all
			names = {}
			@entity.attributes.each do |attr|
				names[attr.column_name.to_i] = attr.name
			end

			rows = []
			@result.each do |raw_row|
				row = {}
				raw_row.each_with_index do |value, index|
					column = names[index]
					attr = nil
					@attributes_to_fetch.each do |_attr|
						if _attr.name == column then
							attr = _attr
						end
					end

					if attr then
						encoding = @adapter_context.adapter.connection['encoding']
						value = _convert_value(attr, value)
						row[attr.name] = attr.new_value(value, encoding)
					else
						row[column] = value
					end
				end
				rows << row
			end
			rows
		end

		private

		def _convert_value( attr, value )
			if value == '' then
				return nil
			end

			begin
				case attr.class_name
				when 'Date'
					value = Date value
				when 'Time'
					value = Time value
				when 'Timestamp'
					value = Timestamp value
				end
			rescue
				raise ArgumentError,
					"can't cast to #{attr.class_name} - #{value.class}:#{value}"
			end

			value
		end

		public

		def evaluate_for_aggregate_spec( agg_spec, expression )
			state = self.evaluate expression
			rows = []
			state.fetch_all.each do |row|
				new_row = {}
				row.each_with_index do |value, index|
					key = state.column_names[index]
					new_row[key] = value
				end
				rows << new_row
			end
			rows
		end

		def primary_keys_for_new_row( count, entity )
			attrs = entity.primary_key_attributes
			if attrs.size > 1 then return nil end

			open
			attr = attrs.first
			column_name = attr.column_name

			if (maxnum = @pk_cache[entity]) > 0 then
				@pk_cache[entity] = maxnum + count
				return _primary_keys(attr, maxnum, count)
			end

			maxnum = @csvdb.max_id(entity.external_name, column_name.to_i)
			@pk_cache[entity] = maxnum + count
			_primary_keys(attr, maxnum, count)
		end

		private

		def _primary_keys( attribute, maxnum, count )
			keys = []
			for x in 1..count
				maxnum += 1
				keys << {attribute.name => maxnum}
			end
			keys
		end

		public

		def insert_row( row, entity )
			check_transaction
			factory = @adapter_context.adapter.expression_factory
			expr = factory.insert_statement(row, entity)
			evaluate expr
			@csvdb.rows_affected
		end

		def update_rows( row, qualifier, entity )
			check_transaction
			factory = @adapter_context.adapter.expression_factory
			expr = factory.update_statement(row, qualifier, entity)
			evaluate expr
			@csvdb.rows_affected
		end

		def delete_rows( qualifier, entity )
			check_transaction
			factory = @adapter_context.adapter.expression_factory
			expr = factory.delete_statement(qualifier, entity)
			evaluate expr
			@csvdb.rows_affected
		end
	end

	class CSVContext < AdapterContext
		def create_channel
			channel = CSVChannel.new self
			@channels << channel
			channel
		end

		def begin_transaction
			if open_transaction? then
				raise "already begun"
			else
				@channel = nil
				@channels.each do |channel|
					unless channel.fetch_progress? then
						unless channel.open? then
							channel.open
						end
					end

					@channel = channel
					break
				end

				unless @channel then
					@channel = create_channel
					@channel.open
				end
			end

			@csvdb = @channel.csvdb
			@csvdb.begin
			@open_transaction     = true
			@commit_transaction   = false
			@rollback_transaction = false
			transaction_did_begin
		end

		def commit_transaction
			@csvdb.commit
			clear
			transaction_did_commit
		end

		def rollback_transaction
			@csvdb.rollback
			clear
			transaction_did_rollback
		end

		def clear
			@csvdb = nil
			@channel = nil
			@open_transaction   = false
			@commit_transaction = true
		end
	end


	class CSVExpression < SQLExpression
		SELECT_TYPE = 0
		INSERT_TYPE = 1
		UPDATE_TYPE = 2
		DELETE_TYPE = 3

		class << self
			def adapter_class
				CSVAdapter
			end
		end

		attr_accessor :fetch_spec, :row
		attr_reader :type, :key_values

		def must_use_bind_variable?( attribute ); true; end
		def should_use_bind_variable?( attribute ); true; end
		def use_bind_variables?; true; end

		def format_value( value, attribute )
			value = attribute.adapter_value value
			case attribute.adapter_value_type
			when Attribute::ADAPTER_CHARACTERS_TYPE then sql_for_string value
			when Attribute::ADAPTER_DATE_TYPE       then sql_for_date value
			when Attribute::ADAPTER_NUMBER_TYPE     then sql_for_number value
			when Attribute::ADAPTER_BYTES_TYPE      then sql_for_data value
			else
				raise "unsupported type: '#{attribute.external_type}'"
			end
		end

		def sql_for_string( string )
			if string.nil? then
				nil
			else
				if @encoding then
					string = Utilities.encode(string, @encoding)
				end
				string.to_s
			end
		end

		def sql_for_date( date )
			if date.nil? then
				nil
			else
				date.to_s
			end
		end

		def sql_for_number( number )
			if number.nil? then
				nil
			elsif number == true then
				'1'
			elsif number == false then
				'0'
			else
				number.to_s
			end
		end

		def prepare_select( attributes, lock, fetch_spec )
			@type = SELECT_TYPE
			@fetch_spec = fetch_spec
			names = []
			attributes.each do |attr|
				names << attr.name
			end
			sql = "select #{names.join(', ')} from #{@entity.name}"
			sql << " where #{fetch_spec.qualifier}" if fetch_spec.qualifier
			@statement = sql
		end

		def prepare_insert( row )
			@type = INSERT_TYPE
			@row = []
			value_str = ''
			row.each do |key, value|
				attr = @entity.attribute key
				value_str << '?, '
				add_insert_list(attr, value)
			end
			value_str.chop!
			value_str.chop!
			@statement = "insert into #{@entity.name} (#{row.keys.join(', ')}) " +
				"values (#{value_str})"
		end

		def prepare_update( row, qualifier )
			@type = UPDATE_TYPE
			@key_values = key_values_for_qualifier(@entity, qualifier)
			value_str = ''
			row.each do |key, value|
				attr = @entity.attribute key
				value_str << "#{attr.column_name} = '#{value}', "
				add_update_list(attr, value)
			end
			value_str.chop!
			value_str.chop!
			sql = "update #{@entity.name} set #{value_str}"
			sql << " where #{qualifier}" if qualifier
			@statement = sql
		end

		def key_values_for_qualifier( entity, qualifier )
			key_values = {}
			case qualifier
			when AndQualifier
				key_values_for_and_qualifier(key_values, entity, qualifier)
			when KeyValueQualifier
				key_values_for_key_value_qualifier(key_values, entity, qualifier)
			end
			key_values
		end

		def key_values_for_and_qualifier( key_values, entity, qualifier )
			qualifier.qualifiers.each do |qual|
				key_values_for_key_value_qualifier(
					key_values, entity, qual)
			end
		end

		def key_values_for_key_value_qualifier( key_values, entity, qualifier )
			key   = qualifier.key
			value = qualifier.value
			entity.primary_key_attributes.each do |attr|
				if attr.name == key then
					key_values[attr.column_name.to_i] = value.to_s
				end
			end
		end

		def prepare_delete( qualifier )
			@type = DELETE_TYPE
			@key_values = key_values_for_qualifier(@entity, qualifier)
			@statement = "delete from #{@entity.name} where #{qualifier}"
		end
	end


	class CSVExpressionFactory < SQLExpressionFactory
		def expression_class
			CSVExpression
		end
	end


	class CSVDatabase
		attr_reader :rows_affected

		def initialize( path, lock = FileLock::FLOCK )
			@path = path
			@filelock = FileLock.filelock lock
			@inserts = []
			@updates = []
			@deletes = []
			@rows = {}
			@transaction = false
			@rows_affected = 0
		end

		def path( file )
			File.join(@path, file)
		end

		def evaluate( expr )
			case expr.type
			when CSVExpression::SELECT_TYPE
				select(expr.entity.external_name)
			when CSVExpression::INSERT_TYPE
				insert(expr)
			when CSVExpression::UPDATE_TYPE
				update(expr)
			when CSVExpression::DELETE_TYPE
				delete(expr)
			else
				raise "unsupported process: #{expr.type}"
			end
		end

		def read( entity )
			rows = []
			path = path entity
			if FileTest.exist? path then
				str = nil
				@filelock.shared_lock(path, 'r') do |f|
					str = f.read
				end
				reader = CSV::Reader.create(str)
				reader.each do |row|
					rows << row.to_a
				end
			end
			rows
		end

		def write( entity, rows )
			path = path entity
			@filelock.exclusive_lock(path, 'w') do |f|
				CSV::Writer.generate(f) do |csv|
					rows.each do |row|
						csv << row
					end
				end
			end
		end

		# Returns all rows of the entity.
		# The rows are selected with qualifier as object in upper layer.
		def select( entity )
			@rows_affected = 0
			read entity
		end

		def insert( expr )
			@inserts << expr
		end

		def update( expr )
			@updates << expr
		end

		def delete( expr )
			@deletes << expr
		end

		def max_id( entity, column )
			max = 1
			select(entity).each do |row|
				if id = row[column] then
					id = id.to_i
					if id > max then
						max = id	
					end
				end
			end
			max
		end

		def begin
			@transaction = true
			@rows_affected = 0
		end

		def commit
			new_row = []
			@inserts.each do |expr|
				expr.row.each do |key, value|
					new_row[key] = value
				end
				rows(expr.entity.external_name) << new_row
				@rows_affected += 1
			end

			@updates.each do |expr|
				rows = rows expr.entity.external_name
				to_update = rows_hash_for_expr expr
				to_update.each do |old_row, diff_row|
					index = rows.index old_row
					new_row = old_row.dup
					diff_row.each do |key, value|
						new_row[key] = value
					end
					rows[index] = new_row
				end
				@rows_affected += 1
			end

			@deletes.each do |expr|
				rows = rows expr.entity.external_name
				to_delete = rows_hash_for_expr expr
				to_delete.each_key do |row|
					rows.delete row
				end
				@rows_affected += 1
			end

			@rows.each do |entity, new_rows|
				write(entity, new_rows)
			end

			clear
		end

		def rows( entity )
			if @rows[entity].nil? then
				@rows[entity] = select(entity)
			end
			@rows[entity]
		end

		def rows_hash_for_expr( expr )
			rows_hash = {}
			existed = rows(expr.entity.external_name)
			existed.each do |row|
				flag = true
				expr.key_values.each do |key, value|
					unless row[key] == value then
						flag = false
					end
				end
				if flag == true then
					rows_hash[row] = expr.row
				end
			end
			rows_hash
		end

		def rollback
			clear
		end

		def clear
			@writer = nil
			@inserts.clear
			@updates.clear
			@deletes.clear			
			@transaction = false
		end

	end


	class FileLock
		FLOCK  = 'flock'
		ATOMIC = 'atomic'

		def self.filelock( method = FLOCK )
			case method
			when FLOCK  then self
			when ATOMIC then AtomicFileLock
			else
				self
			end
		end

		# Creates a shared file lock on a file.
		def self.shared_lock( filename, mode = 'r' )
			File.open( filename, mode ) do | io |
				io.flock File::LOCK_SH
				yield io
				io.flock File::LOCK_UN
			end
		end

		# Creates a exclusive file lock on a file.
		def self.exclusive_lock( filename, mode = 'w' )
			File.open( filename, mode ) do | io |
				io.flock File::LOCK_EX
				yield io
				io.flock File::LOCK_UN
			end
		end
	end


	# write to other file and rename.
	class AtomicFileLock < FileLock
		def self.shared_lock( filename, mode = 'r' )
		end

		def self.exclusive_lock( filename, mode = 'w' )
		end
	end
end
