#!/usr/local/bin/ruby

#
# booklist.cgi
#

require 'cgi'
require 'tapkit'
require 'database'

# path of a model for Book and Publisher
model = '../model/relationship_entity.yaml'

# URL for this application
url   = '/cgi-bin/booklist.cgi'


################################################################################

class BookList
	def initialize( model, url )
		@db = BookListDatabase.new model
		@cgi = CGI.new
		@command = @cgi['command']
		@url = url
	end

	def run
		show_header

		case @command
		when 'register_form'
			register_form
		when 'register'
			register
		when 'update_form'
			update_form
		when 'update'
			update
		when 'delete_form'
			delete_form
		when 'delete'
			delete
		when 'search'
			search_form
			search
		else
			search_form
		end

		show_footer
	end

	def show_header
		html =  "Content-Type: text/html\n\n"
		html << "<html><head><title>Book List</title></head><body><p>\n"
		html << "<a href=\"#@url?command=search\">Search</a> |"
		html << "<a href=\"#@url?command=register_form\">Register</a> |"
		html << "<a href=\"#@url?command=update_form\">Update</a> |"
		html << "<a href=\"#@url?command=delete_form\">Delete</a>"
		html << "</p>\n"
		puts html
	end

	def show_footer
		html = "<hr>\n"
		html << "<address>booklist.cgi</address>"
		puts html
	end

	def register_form
		html =  "<p><b>Register a new record for Book.</b></p>\n"
		html << "<form action=\"#@url\" method=\"POST\">\n"
		html << "<p>Title: <input type=\"text\" name=\"title\"></p>\n"
		html << _publisher_list
		html << _new_publisher_form
		html << "<input type=\"hidden\" name=\"command\" value=\"register\">\n"
		html << "<input type=\"submit\" value=\"Register\">\n"
		html << "</form>\n"
		puts html
	end

	def _publisher_list
		publishers = @db.all_publishers

		html = "<p>Publisher: <select name=\"publisher\">\n"
		publishers.each do |publisher|
			name = publisher['name']
			html << "<option>#{name}\n"
		end
		html << "</select></p>\n"

		html
	end

	def _new_publisher_form
		html = "<p>Input a new publisher (if you need): "
		html << "<input type=\"text\" name=\"new_publisher\"></p>\n"
		html
	end

	def register
		title = @cgi['title']
		publisher = @cgi['publisher']
		new_publisher = @cgi['new_publisher']

		if new_publisher.empty? then
			@db.register(title, publisher)
			registered_publisher = publisher
		else
			@db.register_with_new_publisher(title, new_publisher)
			registered_publisher = "#{new_publisher}(new)"
		end

		html =  "<p>Register a new record.</p>\n"
		html << "<p>Title:<b>#{title}</b>, Publisher:<b>#{registered_publisher}</b></p>\n"
		puts html
	end

	def update_form
		html =  "<p><b>Update a record for Book.</b></p>\n"
		html << "<form action=\"#@url\" method=\"POST\">\n"
		html << _book_list
		html << "<p>Title: <input type=\"text\" name=\"title\"></p>\n"
		html << _publisher_list
		html << _new_publisher_form
		html << "<input type=\"hidden\" name=\"command\" value=\"update\">\n"
		html << "<input type=\"submit\" value=\"Update\">\n"
		html << "</form>\n"
		puts html
	end

	def _book_list
		books = @db.all_books

		html = "<p>Select: <select name=\"book\">\n"
		books.each do |book|
			title = book['title']
			name = book['publisher']['name']
			html << "<option value=\"#{title}\">#{title} (#{name})\n"
		end
		html << "</select></p>\n"

		html
	end

	def update
		book          = @cgi['book']
		title         = @cgi['title']
		publisher     = @cgi['publisher']
		new_publisher = @cgi['new_publisher']

		if new_publisher.empty? then
			@db.update(book, title, publisher)
			updated_publisher = publisher
		else
			@db.update_with_new_publisher(book, title, new_publisher)
			updated_publisher = "#{new_publisher}(new)"
		end

		puts "<p>Update a record <b>#{book}</b> to <b>#{title}(#{updated_publisher})</b></p>\n"
	end

	def search_form
		html =  "<form action=\"#@url\" method=\"POST\">"
		html << "Search: <input type=\"text\" name=\"keywords\">\n"
		html << "<select name=\"sort\">"
		html << "<option value=\"1\">title asc</option>"
		html << "<option value=\"2\">title desc</option></select>\n"
		html << "<select name=\"like\">"
		html << "<option value=\"1\">case sensitive</option>"
		html << "<option value=\"2\">case insensitive</option></select>\n"
		html << "<input type=\"hidden\" name=\"command\" value=\"search\">"
		html << "<input type=\"submit\" value=\"Go\">\n"
		html << "</form>\n"
		puts html
	end

	def delete_form
		html =  "<p><b>Delete records.</b></p>\n"
		html << "<form action=\"#@url\" method=\"POST\">\n"
		html << _delete_list
		html << "<input type=\"hidden\" name=\"command\" value=\"delete\">\n"
		html << "<p><input type=\"submit\" value=\"Delete\"></p>\n"
		html << "</form>\n"
		puts html
	end

	def _delete_list
		books = @db.all_books

		html = '<p>'
		books.each_with_index do |book, index|
			title = book['title']
			name = book['publisher']['name']
			html << "<input type=\"checkbox\" name=\"books#{index}\"> #{title} (#{name})<br>\n"
		end
		html << '</p>'

		html
	end

	def delete
		books = []
		@cgi.params.each do |key, value|
			if key =~ /books(.*)/ then
				index = $1.to_i
				books << index
			end
		end
		@db.delete books

		puts "#{books.size} records deleted."
	end

	def search
		keywords = @cgi['keywords']
		sort     = @cgi['sort'].to_i
		like     = @cgi['like'].to_i

		unless keywords.empty? then
			books = @db.search(keywords, sort, like)
		else
			books = []
		end

		html = "<hr><p>Keyword: #{keywords}</p>"
		html << "<table border=1>\n"
		html << "<tr><th>Title</th><th>Publisher</th></tr>\n"
		books.each do |book|
			title = book['title']
			publisher = book['publisher']

			if publisher then
				# set publisher to nil if faulting is failed
				begin
					publisher = publisher['name']
				rescue
					publisher = nil
				end
			end

			html << "<tr><td>#{title}</td><td>#{publisher}</td></tr>\n"
		end
		html << "</table>\n"

		puts html
	end
end


bl = BookList.new(model, url)
bl.run
