module TapKit

	class ObjectStore
		OBJECTS_CHANGED_IN_STORE_NOTIFICATION = :objects_changed_in_store_notification

		def initialize_object( object, gid, editing_context ); end
	end

	class ObjectStoreCoordinator < ObjectStore
		attr_reader :cooperating_object_stores, :application
		attr_accessor :userinfo

		COS_NEEDED_NOTIFICATION = :cos_needed_notification

		def initialize( application )
			@application = application
			@notification_center = @application.notification_center
			@cooperating_object_stores = []
			@userinfo = {:application => @application}
		end

		def each
			@cooperating_object_stores.each do |store|
				yield store
			end
		end

		def add_cooperating_object_store( object_store )
			@cooperating_object_stores << object_store
			# post(COS_ADDED_NOTIFICATION)
		end

		def object_store( object )
			if store = _object_store(object) then
				return store
			else
				@userinfo[:object] = object
				@notification_center.post(COS_NEEDED_NOTIFICATION, self, @userinfo)
				_object_store object
			end
		end

		private

		def _object_store( object )
			each do |store|
				if store.own? object then
					return store
				end
			end
			nil
		end

		public

		def fetch( fetch_spec, editing_context )
			store = object_store_for_fetch_spec fetch_spec
			store.fetch(fetch_spec, editing_context)
		end

		def object_store_for_fetch_spec( fetch_spec )
			each do |_store|
				if _store.handle_fetch_spec? fetch_spec then
					return _store
				end
			end

			@userinfo[:object] = fetch_spec
			@notification_center.post(COS_NEEDED_NOTIFICATION, self, @userinfo)
			object_store_for_fetch_spec fetch_spec
		end


		def save( editing_context )
			if @cooperating_object_stores.empty? then
				objects = editing_context.inserted_objects
				objects.concat editing_context.deleted_objects
				objects.concat editing_context.updated_objects
				@userinfo[:object] = objects.first
				@notification_center.post(COS_NEEDED_NOTIFICATION, self, @userinfo)
			end

			each do |store|
				store.prepare_for_save(self, editing_context)
				store.record_changes
			end

			failed = false
			begin
				each { |store| store.perform_changes }
			rescue Exception => e
				failed = true
				each { |store| store.rollback }
			else
				begin
					each { |store| store.commit }
				rescue Exception => e
					failed = true
					each { |store| store.rollback }
				else
					each { |store| store.finalize_commit_changes }
				end
			end

			if failed then
				raise RuntimeError, "commit failed - #{e.message}"
			end
		end

		public

		def forward_update( object, changes )
			store = object_store object
			store.record_update(object, changes)
		end

		def initialize_object( object, gid, editing_context )
			store = object_store gid
			store.initialize_object(object, gid, editing_context)
			object.application = @application
		end

		##
		## faulting
		##

		def fault( gid, editing_context )
			store = object_store gid
			store.fault(gid, editing_context)
		end

		def array_fault( gid, relationship_name, editing_context )
			store = object_store gid
			store.array_fault(gid, relationship_name, editing_context)
		end

		def fault_for_raw_row( row, entity_name, editing_context )
			store = object_store gid
			store.fault_for_raw_row(row, entity_name, editing_context)
		end

		def refault( object, gid, editing_context = self )
			store = object_store gid
			store.refault(object, gid, editing_context)
		end

		def objects_for_source_gid( gid, name, editing_context )
			store = object_store gid
			store.objects_for_source_gid(gid, name, editing_context)
		end
	end

	# Abstract class. The default implementation is DatabaseContext.
	class CooperatingObjectStore < ObjectStore
		def commit; end
		def handles_fetch_spec( fetch_spec ); end
		def lock; end
		def own?( object ); end
		def perform_changes; end
		def prepare_for_save( coordinator, editing_context ); end
		def record_changes; end
		def record_update( object, changes ); end
		def rollback; end
		def unlock; end
		def retrieve_values( keys, object ); end
	end

end
