module TapKit

	module Faulting
		attr_reader :fault_handler

		def clear_fault
			@fault_handler = nil
		end

		def will_read
			if fault? then
				@fault_handler.complete_init self
				clear_fault
			end
		end

		def fault?
			if @fault_handler then
				true
			else
				false
			end
		end

		def turn_into_fault( handler )
			@fault_handler = handler
		end
	end


	class FaultHandler
		# abstract
		def complete_init( object ); end
		def fault_will_fire( object ); end
	end


	class AccessGenericFaultHandler < FaultHandler
		attr_reader :database_context, :editing_context

		# abstract
		def complete_init( object ); end
		def fault_will_fire( object ); end
	end


	class AccessFaultHandler < AccessGenericFaultHandler
		attr_reader :gid

		def initialize( gid, database_context, editing_context )
			@gid = gid
			@database_context = database_context
			@editing_context = editing_context
		end

		def complete_init( object )
			fault_will_fire object
			entity     = @database_context.database.entity gid.entity_name
			qualifier  = entity.qualifier_for_primary_key gid.key_values
			fetch_spec = FetchSpec.new(entity.name, qualifier)
			objects    = @database_context.fetch(fetch_spec, @editing_context)

			# if destination object is not existed, the object is not managed.
			unless objects.size == 1 then
				@editing_context.forget object
			end
		end

		def to_s
			"<AccessFaultHandler #{gid.hash}>"
		end

		def method_missing( name, *args )
			if respond_to? name then
				will_read
				__send__(name, *args)
			else
				raise NoMethodError, "undefined method '#{name.to_s}' for #{self.class}"
			end
		end
	end


	class AccessArrayFaultHandler < AccessGenericFaultHandler
		attr_reader :relationship_name, :source_gid

		def initialize( source_gid, relationship_name, database_context, editing_context )
			@source_gid        = source_gid
			@relationship_name =	relationship_name
			@database_context  = database_context
			@editing_context   = editing_context
		end

		def complete_init( object )
			fault_will_fire object

			objects = @editing_context.objects_for_source_gid(@source_gid,
			                                                  @relationship_name)
			object.update_from_objects objects
		end

		def to_s
			"<AccessArrayFaultHandler #{gid.hash}>"
		end
	end


	class FaultingDelayEvaluationArray
		include Faulting

		attr_reader :source_gid, :relationship_name

		def initialize( source_gid, relationship_name, database_context, editing_context )
			handler = AccessArrayFaultHandler.new(source_gid,
				relationship_name, database_context, editing_context)
			@array = []
			@source_gid = source_gid
			@relationship_name = relationship_name
			turn_into_fault handler
		end

		def update_from_objects( objects )
			@array.replace objects
		end

		def to_s
			if fault? then
				"<FaultingDelayEvaluationArray #{@fault_handler.source_gid.hash}>"
			else
				str = '['
				@array.each do |object|
					str << object.to_s
					if @array.last != object then
						str << ', '
					end
				end
				str << ']'
				str
			end
		end

		alias inspect to_s

		def method_missing( name, *args, &block )
			if @array.respond_to? name then
				will_read
				@array.__send__(name, *args, &block)
			else
				raise NoMethodError, "undefined method '#{name.to_s}' for #{self.class}"
			end
		end
	end


end