module TapKit

	class Model
		CACHE_SUFFIX = '.cache'

		attr_reader :model_list, :cache_path
		attr_accessor :path, :adapter_name, :connection, :entities, :model_group, :name

		def initialize( path = nil, cache_path = nil )
			if path then
				@path = path
				@name = File.basename @path
			end

			@cache_path = cache_path
			@loaded     = false
			@connection = {}
			@entities   = []

			if @path and @cache_path then
				require 'pstore'
				@cache_path = File.join(@cache_path, (@name + CACHE_SUFFIX))
				if FileTest.exist?(@cache_path) and \
					(File.mtime(@path) <= File.mtime(@cache_path)) then
					load_from_cache @cache_path
				else
					load_from_yaml @path
					create_cache @cache_path
				end
			elsif @path then
				load_from_yaml @path
			end
		end

		def create_cache( path )
			db = PStore.new path
			db.transaction do
				db[:model_list] = @model_list
			end
		end

		def load_from_cache( path )
			db = PStore.new path
			db.transaction do
				@model_list = db[:model_list]
			end
			load_from_list @model_list
		end

		def load_from_yaml( path )
			require 'yaml'
			f = File.new path
			@model_list = YAML::load f
			f.close
			load_from_list @model_list
		end

		def load_from_list( list )
			@adapter_name = list['adapter_name'] || 'DBI'
			@connection   = list['connection'] || {}
			list['entities'] ||= []
			list['entities'].each do |entity|
				@entities << Entity.new(entity, self)
			end
			validate_required_attributes
		end

		def load_all_model_objects
			if (@loaded == false) and @model_list then
				@model_list['entities'].each do |entity_list|
					entity_name = entity_list['name']
					entity = self.entity entity_name
					if rel_lists = entity_list['relationships'] then
						rel_lists.each do |rel_list|
							rel = Relationship.new(rel_list, entity)
							entity.add_relationship rel
						end
					end
				end
				@loaded = true
			end
		end

		def add_entity( entity )
			@entities << entity
		end

		def remove_entity( entity )
			@entities.delete entity
		end

		def entity( object )
			if GenericRecord === object then
				name = object.entity_name
			else
				name = object
			end

			@entities.each do |e|
				if e.name == name then
					return e
				end
			end
			nil
		end

		def entity_names
			names = []
			@entities.each do |entity|
				names << entity.name
			end
			names
		end

		def validate_required_attributes
			msg = "Model requires attributes: 'adapter_name', 'connection'"

			if @adapter_name.nil? or @connection.empty? then
				key = @apapetr_name || :model
				error = ValidationError.new(msg, key)
				raise error
			end
		end

		def to_yaml
			entities = []
			@entities.each { |entity| entities << entity.to_h }

			{
				'adapter_name' => @adapter_name,
				'connection'   => @connection,
				'entities'     => entities
			}.to_yaml(:SortKeys=>true)
		end


		#
		# Parsing relationships from entities
		#

		# name, source, destination
		def parse_relationships
			@entities.each do |src_entity|
				src_entity.primary_key_attributes.each do |src_attr|
					entities.each do |dest_entity|
						dest_entity.attributes.each do |dest_attr|
							if (src_entity != dest_entity) and \
							   (src_attr.name == dest_attr.name) then
								parse_relationship(src_entity, dest_entity, src_attr, dest_attr)
								parse_relationship(dest_entity, src_entity, dest_attr, src_attr)
							end
						end
					end
				end
			end
		end

		def parse_relationship( src_entity, dest_entity, src_attr, dest_attr )
			# check to-many
			to_many = false
			dest_entity.primary_key_attributes.each do |attr|
				unless src_entity.attribute(attr.name) then
					to_many = true
				end
			end

			rel = Relationship.new
			rel.to_many = to_many
			rel.name = dest_entity.external_name
			rel.beautify_name
			rel.destination_entity = dest_entity
			join = Join.new(src_attr, dest_attr)
			rel.add join

			if to_many then
				rel.name = plural_name rel.name
			end

			unless src_entity.class_property? rel.name then
				src_entity.class_property_names << rel.name
				src_entity.add_relationship rel
			end
		end

		def plural_name( name )
			if (name =~ /s$/i) or (name =~ /sh$/i) or \
				(name =~ /ch$/i) or (name =~ /x$/i) then
				"#{name}es"
			elsif name =~ /(?!(a|i|u|e|o))y$/i then
				"#{name.chop}ies"
			elsif name =~ /(?!(a|i|u|e|o))o$/i then
				"#{name}es"
			elsif name =~ /f$/i then
				"#{name.chop}ves"
			elsif name =~ /fe$/i then
				"#{name.chop.chop}ves"
			else
				"#{name}s"
			end
		end

	end


	class ModelGroup
		attr_reader :models

		def initialize( app )
			@app = app
			@models = []
		end

		def load_all_model_objects
			@models.each do |model|
				model.load_all_model_objects
			end
		end

		def model( name )
			@models.each do |model|
				if (model.name == name) or (model.path == name) then
					return model
				end
			end
			nil
		end

		def model_names
			names = []
			@models.each do |model|
				names << model.name
			end
			names
		end

		def add( model, cache_path = nil )
			if String === model then
				model = Model.new(model, cache_path)
			end
			@models << model
			model.model_group = self
			model.load_all_model_objects
			db = Database.new(@app, model)
			@app.add_database(model, db)
		end

		def remove( model )
			@models.delete model
		end

		def entities
			entities = []
			@models.each do |model|
				entities.concat model.entities
			end
			entities
		end

		def entity( object )
			@models.each do |model|
				if result = model.entity(object) then
					return result
				end
			end
			nil
		end
	end

end

