/*
 *  canna.c
 *  Copyright(C) 2003- Masahito Omote <omote@utyuuzin.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <stdlib.h>
#include <dlfcn.h>
#include <pwd.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include "canna.h"

static struct _canna_api RkFunc;
static void   *canna_lib;
static int     context_num;
static void    parse_line(unsigned char *, word **);
static void    canna_get_privatedicname(unsigned char **);

int canna_init(char *cannahost) {
    if(canna_lib) {
	dlclose(canna_lib);
    }

    canna_lib = dlopen("libcanna.so.1.0", RTLD_LAZY);

    if(canna_lib != NULL) {
	/* $BA4It;H$&$H$O;W$($J$$$N$GI,MW$J$b$N$@$1BP1~$E$1$k(B */
	RkFunc.Initialize = dlsym(canna_lib, "RkInitialize");
	RkFunc.Finalize = dlsym(canna_lib, "RkFinalize");
	RkFunc.GetWordTextDic = dlsym(canna_lib, "RkGetWordTextDic");
	RkFunc.ListDic = dlsym(canna_lib, "RkListDic");
	RkFunc.GetMountList = dlsym(canna_lib, "RkGetMountList");
	RkFunc.MountDic = dlsym(canna_lib, "RkMountDic");
	RkFunc.UnmountDic = dlsym(canna_lib, "RkUnmountDic");
	RkFunc.DefineDic = dlsym(canna_lib, "RkDefineDic");
	RkFunc.DeleteDic = dlsym(canna_lib, "RkDeleteDic");
	RkFunc.SyncDic = dlsym(canna_lib, "RkSync");

	if(RkFunc.Initialize == NULL || RkFunc.Finalize == NULL ||
	   RkFunc.MountDic == NULL || RkFunc.UnmountDic == NULL ||
	   RkFunc.DefineDic == NULL || RkFunc.DeleteDic == NULL ||
	   RkFunc.GetWordTextDic == NULL || RkFunc.ListDic == NULL ||
	   RkFunc.GetMountList == NULL || RkFunc.SyncDic == NULL)
	{
	    dlclose(canna_lib);
	    return -1;
	}

	context_num = RkFunc.Initialize(cannahost);

	if(context_num >= 0) {
	    return 0;
	}
	else {
	    return -1;
	}
    } else {
	return -1;
    }
}

void canna_close(void) {
    if(RkFunc.Finalize != NULL) {
	RkFunc.Finalize();
    }

    if(canna_lib) {
	dlclose(canna_lib);
    }
}

int canna_get_word_text_dic(unsigned char *dirname,
			    unsigned char *dicname,
			    word **head)
{
    unsigned char buf[1024];
    unsigned char dicname_bk[256];
    int ret = 0;

    if(RkFunc.GetWordTextDic != NULL) {
#ifndef HAVE_STRLCPY
	strncpy(dicname_bk, dicname, 256);
	dicname_bk[255] = '\0';
#else
	strlcpy(dicname_bk, dicname, 256);
#endif
	do {
	    ret = RkFunc.GetWordTextDic(context_num, dirname,
					dicname_bk, buf, 1024);
	    if(ret <= 0)
		break;

	    dicname_bk[0] = '\0';
	    parse_line(buf, head);
	} while(ret >= 0);
    }
    if(ret < 0)
	return -1;
    else
	return 0;
}

word *canna_get_word_text_priv_dic(void) {
    struct passwd *pw;
    char *username, *dirname;
    char dir[] = ":user/";
    word *list = NULL;

    pw = getpwuid(getuid());
    if(pw != NULL) {
	username = strdup(pw->pw_name);

	if(username != NULL) {
	    int len;
	    /* dirname := ":user/username" */
	    len = strlen(dir) + strlen(username) + 1;
	    dirname = malloc(sizeof(char) * len);
	    if(dirname != NULL) {
		snprintf(dirname, len, "%s%s", dir, username);
		canna_get_word_text_dic(dirname, "user", &list);
		free(dirname);
	    }
	    free(username);
	}
    }

    return list;
}

int canna_define_dic(unsigned char *dicname, unsigned char *buf) {
    /* buf := phon cclass desc */
    int ret;

    if(RkFunc.MountDic(context_num, (char *)dicname, 0) < 0 )
	return -1;

    ret = RkFunc.DefineDic(context_num, (char *)dicname, buf);

    RkFunc.UnmountDic(context_num, (char *)dicname);

    return ret;
}

int canna_delete_dic(unsigned char *dicname, unsigned char *buf) {
    /* buf := phon cclass desc */
    int ret;

    if(RkFunc.MountDic(context_num, (char *)dicname, 0) < 0 )
	return -1;

    ret = RkFunc.DeleteDic(context_num, (char *)dicname, buf);

    RkFunc.UnmountDic(context_num, (char *)dicname);

    return ret;
}

static void parse_line(unsigned char *buf, word **head) {
    /*
     * cannadic format:
     *
     * phon cclass desc (desc desc ... ) (cclass desc (desc ...) ....)
     *
     * phon  : must
     * cclass: at least 1 entry
     * desc  : at least 1 entry
     *
     * cclass and desc must be associated with each other.
     *
     */
    unsigned char phon[1024], desc[1024], cclass_code[1024];
    int buflen;
    int desc_flag = 0, cclass_flag = 1;
    int i, j;

    if(buf[0] == '\0')
	return;

    buflen = strlen(buf);

    /* $BFI$_$r8!=P(B */
    for(i = 0; i < buflen; i++) {
	if(buf[i] == ' ') {
	    phon[i] = '\0';
	    i++;
	    break;
	}
	phon[i] = buf[i];
    }

    for(j = 0 ; i <= buflen; i++, j++) {
	if(buf[i] == '#') {
	    cclass_flag = 1; desc_flag = 0;
	}

	if(buf[i] == ' '|| buf[i] == '\0') {
	    if(cclass_flag == 0 && desc_flag == 1) {
		desc[j] = '\0';
		if(buf[i+1] == '#') {
		    desc_flag = 0;
		    cclass_flag = 1;
		}
		word_append(head, WORD_TYPE_CANNA,
			    phon, desc, cclass_code,
			    0, 0, NULL);
	    } else if(cclass_flag == 1 && desc_flag == 0) {
		desc_flag = 1;
		cclass_flag = 0;
		cclass_code[j] = '\0';
	    }
	    j = -1;
	} else {
	    if(cclass_flag == 1 && desc_flag == 0) {
		cclass_code[j] = buf[i];
	    } else if (cclass_flag == 0 && desc_flag == 1) {
		desc[j] = buf[i];
	    }
	}
    }
}
