/* negotiat.c       Copyright (c) 2000-2002 Nagy Daniel
 *
 * $Date: 2002/06/19 18:03:39 $
 * $Revision: 1.14 $
 *
 * This module is the SSH negotiation part:
 *  - open TCP connection
 *  - version check
 *  - key generation
 *  - user authorization
 *  - TTY allocation
 *  - interactive shell request
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

#include<io.h>
#include<stdio.h>
#include<conio.h>
#include<string.h>

#if defined (__DJGPP__)
 #include<errno.h>
 #include"tcp_djgp.h"
#elif defined (__TURBOC__)
 #include"tcp.h"
#endif

#include"ssh.h"
#include"protocol.h"
#include"macros.h"
#include"md5.h"
#include"rsa.h"
#include"cipher.h"
#include"version.h"
#include"xmalloc.h"
#include"config.h"

void fatal(const char *, ...);
SessKeyInitPtr SessKeyInit;

/* external variables */
extern Config GlobalConfig;		/* Global configuration variables */
extern unsigned short Configuration;

extern struct Packet pktin;
extern struct Packet pktout;

char *RemoteClosed = "Remote host closed connection\r\n";
char *ConnectionClosed = "Connection closed\r\n";

static unsigned char session_id[16];
static unsigned char auth_mask[4];

/*
 * SSH version string exchange: get server's SSH protocol
 * version, examine it, and send ours if it seems that we
 * can communicate
 */
static void ssh_exchange_identification(void)
{
char buf[256], remote_version[256];
int remote_major, remote_minor, i;
int len;

   if(Configuration & VERBOSE_MODE)
        cputs("Identification Exchange\n\r");

   /* Read other side's version identification. */
   i = sock_gets(&GlobalConfig.s, buf, sizeof(buf));
   if(!i)
	fatal("read: %s", strerror(errno));
   buf[i] = '\n';

   if(Configuration & VERBOSE_MODE)
	cprintf("Remote version: %s\r",buf);

   if (sscanf(buf, "SSH-%d.%d-%[^\n]\n", &remote_major, &remote_minor,
							remote_version) != 3)
	fatal("Bad remote protocol version identification\n");

   sprintf(buf, "SSH-%d.%d-%s\n", PROTOCOL_MAJOR, PROTOCOL_MINOR, SSH_VERSION);
   if(Configuration & VERBOSE_MODE)
        cprintf("Local version: %s\r",buf);

   if(remote_major != 1 && remote_minor == 0)
	fatal("Unsupported remote protocol version\n");

   len = sock_write(&GlobalConfig.s, buf, strlen(buf));
   if(len != strlen(buf))
	fatal("write: %s", strerror(errno));
}

/*
 * Extract host keys, get authentication masks,
 * create session key and ID, send them back and begin
 * encryption
 */
static void create_ssh_keys(void)
{
unsigned short i, j;
char *RSAblock, *keystr1, *keystr2;
unsigned char cookie[8];
unsigned char session_key[32];
struct RSAKey servkey, hostkey;

   /* Wait for a public key packet from the server. */
   if(Configuration & VERBOSE_MODE)
	cputs("Wait for public key\n\r");
   packet_read_expect(SSH_SMSG_PUBLIC_KEY);

   /* get the server key */
   memcpy(cookie, pktin.body, 8);
   if(Configuration & VERBOSE_MODE)
	cputs("Extracting server keys\r\n");
   i = makekey(pktin.body + 8, &servkey, &keystr1, 0);
   j = makekey(pktin.body + 8 + i, &hostkey, &keystr2, 0);
   memcpy(auth_mask, pktin.body + 16 + i + j, sizeof(auth_mask));
   xfree(pktin.whole); /* no more bytes needed */

   /* generate session ID */
   if(Configuration & VERBOSE_MODE)
	cputs("Generating session ID\r\n");
   MD5Init();
   MD5Update(keystr2, hostkey.bytes);
   MD5Update(keystr1, servkey.bytes);
   MD5Update(cookie, 8);
   MD5Final(session_id);

   for(i = 0; i < 32; i++)
	session_key[i] = rand() % 256;

   j = (hostkey.bytes > servkey.bytes ? hostkey.bytes : servkey.bytes);
   RSAblock = (char *)xmalloc(j);
   memset(RSAblock, 0, j);

   for (i = 0; i < 32; i++) {
	RSAblock[i] = session_key[i];
	if (i < 16)
	   RSAblock[i] ^= session_id[i];
   }

   if(Configuration & VERBOSE_MODE)
	cputs("Encrypting session key\r\n");
   if (hostkey.bytes > servkey.bytes) {
	rsaencrypt(RSAblock, 32, &servkey);
	rsaencrypt(RSAblock, servkey.bytes, &hostkey);
   }
   else{
	rsaencrypt(RSAblock, 32, &hostkey);
	rsaencrypt(RSAblock, hostkey.bytes, &servkey);
   }

   s_wrpkt_start(SSH_CMSG_SESSION_KEY, j + 15);
   pktout.body[0] = GlobalConfig.CipherType;
   memcpy(pktout.body + 1, cookie, 8);
   pktout.body[9] = (j * 8) >> 8;
   pktout.body[10] = (j * 8) & 0xFF;
   memcpy(pktout.body+11, RSAblock, j);
   pktout.body[j+11] = pktout.body[j+12] = pktout.body[j+13] = pktout.body[j+14] = 0;
   if(Configuration & VERBOSE_MODE)
	cputs("Sending encrypted session key\r\n");
   s_wrpkt();
   xfree(RSAblock);

   SessKeyInit(session_key);
   Configuration |= CIPHER_ENABLED; /* enable cipher */
}

/*
 * Allocate memory for password and ask for password.
 * Don't forget to free it later.
 */
static unsigned short AskPassword(void)
{
unsigned short len = 0;
unsigned char ch;

   GlobalConfig.password = xmalloc(MAX_PASSWORD_LENGTH*sizeof(char));
   while((ch = getch()) != 0x0d && len < MAX_PASSWORD_LENGTH){
	if(ch == 8 && len > 0)
		GlobalConfig.password[--len] = 0;
	else if(ch >= 32)
		GlobalConfig.password[len++] = ch;
   } /* while */
   cprintf("\n\r");
   return(len);
}

/*
 * Send our user name to remote host
 */
static int Send_User_Name(void){
unsigned short len;

   if(Configuration & VERBOSE_MODE) cputs("Sending username\r\n");
   len = strlen(GlobalConfig.username);
   s_wrpkt_start(SSH_CMSG_USER, len + 4);
   pktout.body[0] = pktout.body[1] = pktout.body[2] = 0;
   pktout.body[3] = len;
   memcpy(pktout.body+4, GlobalConfig.username, len);
   s_wrpkt();

   packet_read_type();
   switch(pktin.type){
	case SSH_SMSG_SUCCESS:		/* no authentication needed */
	   break;

	case SSH_SMSG_FAILURE:  	/* must authenticate */
	   return(1);

	default:
	   fatal("Received invalid packet");
   } /* switch */
   return(0);
}

/*
 * Try password authentication
 */
static void Password_Auth(void)
{
char *pwd = "Password: ";
unsigned short len, n = 1;

   if(GlobalConfig.password)
	len = strlen(GlobalConfig.password); /* if password is specified */
   else{
	cputs(pwd);
	if(GlobalConfig.brailab)
	   fputs(pwd, GlobalConfig.brailab);
	len = AskPassword();
	n++;
   }

nextpass:
   s_wrpkt_start(SSH_CMSG_AUTH_PASSWORD, len + 4);
   pktout.body[0] = pktout.body[1] = pktout.body[2] = 0;
   pktout.body[3] = len;
   memcpy(pktout.body + 4, GlobalConfig.password, len);
   s_wrpkt();
   memset(GlobalConfig.password, 0, strlen(GlobalConfig.password));
   xfree(GlobalConfig.password);

   packet_read_type();
   switch(pktin.type){
	case SSH_SMSG_SUCCESS:		/* no authentication needed */
	   break;

	case SSH_SMSG_FAILURE:  	/* get password */
	   if(n-- == 0)
		fatal("Invalid password");
	   cputs(pwd);
	   if(GlobalConfig.brailab)
		fputs(pwd, GlobalConfig.brailab);
	   len = AskPassword();
	   goto nextpass;

	default:
		fatal("Received invalid packet");
   } /* switch */
}

/*
 * Try RSA public key authentication
 */
static int RSA_Key_auth(void)
{
char *comment = NULL;
int type;
struct RSAKey pubkey;
Bignum challenge, response;
int i;
unsigned char buffer[32];

   if(GlobalConfig.identity == NULL) /* is there akey file? */
	return(1);       /* go and try password authentication */

   if(Configuration & VERBOSE_MODE)
	cputs("Begin RSA key authentication\r\n");

   /* Check key */
   type = key_type(GlobalConfig.identity);
   if(type != SSH_KEYTYPE_SSH1){
	if(Configuration & VERBOSE_MODE)
	   cprintf("Key is of wrong type (%s)\r\n", key_type_to_str(type));
	return(1);       /* go and try password authentication */
   }

   if(rsakey_encrypted(GlobalConfig.identity, &comment)){
	cprintf("Passphrase for key \"%.100s\": ", comment);
	xfree(comment);
	AskPassword();
   } /* if */
   else
	if(Configuration & VERBOSE_MODE)
	   cputs("No passphrase required\r\n");

   i = loadrsakey(GlobalConfig.identity, &pubkey, GlobalConfig.password);
   if(i == 0){
	cprintf("Couldn't load private key from %s\r\n",
				 GlobalConfig.identity);
	return(1);       /* go and try password authentication */
   }
   if(i == -1){
	cputs("Wrong passphrase\r\n");
	return(1);       /* go and try password authentication */
   }

   /* Send a public key attempt */
   i = ssh1_bignum_length(pubkey.modulus);
   s_wrpkt_start(SSH_CMSG_AUTH_RSA, i);
   ssh1_write_bignum(pktout.body, pubkey.modulus);
   s_wrpkt();

   packet_read_block();
   if(pktin.type == SSH_SMSG_FAILURE){
	if(Configuration & VERBOSE_MODE)
	   cputs("Server refused RSA key\r\n");
	xfree(pktin.whole);
	return(1);
   }
   if(pktin.type != SSH_SMSG_AUTH_RSA_CHALLENGE)
	fatal("Received invalid packet");

   if(Configuration & VERBOSE_MODE)
	cputs("Got RSA challenge, calculating response\r\n");
   ssh1_read_bignum(pktin.body, &challenge);
   xfree(pktin.whole);

   response = rsadecrypt(challenge, &pubkey);
   freebn(pubkey.private_exponent);	/* burn the evidence */

   for (i = 0; i < 32; i++)
	buffer[i] = bignum_byte(response, 31 - i);

   MD5Init();
   MD5Update(buffer, 32);
   MD5Update(session_id, 16);
   MD5Final(buffer);

   if(Configuration & VERBOSE_MODE)
	cputs("Sending RSA response\r\n");

   s_wrpkt_start(SSH_CMSG_AUTH_RSA_RESPONSE, 16);
   memcpy(pktout.body, buffer, 16);
   s_wrpkt();

   packet_read_type();
   if(pktin.type == SSH_SMSG_FAILURE){
	if(Configuration & VERBOSE_MODE)
	   cputs("Failed to authenticate with RSA key\r\n");
	return(1);	       /* go and try password */
   }
   if(pktin.type != SSH_SMSG_SUCCESS)
	fatal("Received invalid packet");

   return(0);
}

/*
 * Set maximum SSH packet size
 */
static void packetsize(void)
{
unsigned short size = MAX_PACKET_SIZE;

   if(Configuration & VERBOSE_MODE)
	cputs("Setting maximum packet size...");
   s_wrpkt_start(SSH_CMSG_MAX_PACKET_SIZE, 4);
   pktout.body[0] = pktout.body[1] = 0;
   pktout.body[2] = size >> 8;
   pktout.body[3] = size & 0xFF;
   s_wrpkt();

   packet_read_type();
   if(Configuration & VERBOSE_MODE){
	if(pktin.type==SSH_SMSG_SUCCESS)
		cputs("success\r\n");
	else
		fatal("failed\r\n");
   } /* if */
}

/*
 * Connect to remote host via TCP
 */
short TCPConnect(char *remotehost)
{
unsigned short localport;
longword remoteip;
int status;

   /* Allocate local port */
   localport = (rand() % 512) + 512;
   if(Configuration & NONPRIVILEGED_PORT)
	localport = localport + 512;

   sock_init(); /* Initialize socket */

   /* Resolve hostname */
   if((remoteip = resolve(remotehost)) == 0){
	cprintf("\n\rUnable to resolve `%s'\n\r", remotehost);
	return(1);
   }

   /* Open TCP */
   if(!tcp_open(&GlobalConfig.s, localport, remoteip,
		 GlobalConfig.RemotePort, NULL)){
	cputs("Unable to open connection\n\r");
	return(1);
   }

   /* Negotiate TCP connection */
   cputs("waiting for remote host to connect\n\r");
   fflush(stdout);
   sock_wait_established(&GlobalConfig.s, sock_delay, NULL, &status);

   return(0);

sock_err:
   switch(status){
	case 1:
	   cputs (ConnectionClosed);
	   break;
	case -1:
	   cputs (RemoteClosed);
	   break;
   }
   return(1);
}

/*
 * Run SSH negotiation process
 */
short SSHConnect(void)
{
int status;

   sock_wait_input(&GlobalConfig.s, sock_delay, NULL, &status);

/* Begin SSH negotiation protocol */

   /* Start negotiation on network */
   ssh_exchange_identification();

   /* Create SSH keys */
   create_ssh_keys();

   /* Check that our authentication method is allowed */
   if((auth_mask[3] & (1 << SSH_AUTH_RSA)) == 0)
	cputs("Server does not allow RSA key authentication\r\n");
   if((auth_mask[3] & (1 << SSH_AUTH_PASSWORD)) == 0)
	cputs("Server does not allow password authentication\r\n");

   /* Wait for encrypted ACK */
   if(Configuration & VERBOSE_MODE)
	cputs("Waiting for first encrypted ACK\n\r");
   packet_read_expect(SSH_SMSG_SUCCESS);
   xfree(pktin.whole);

   if(Send_User_Name())	     /* Send user name */
	if(RSA_Key_auth())   /* 1st: Public key authentication */
	   Password_Auth();  /* 2nd: Password authentication */

   /* Set maximum packet size */
   packetsize();

   /* Start compression if configured */
   if(Configuration & COMPRESSION_REQUESTED)
	Request_Compression(6);

   return(0);

sock_err:
   switch (status){
	case 1:
	   cputs (ConnectionClosed);
	   break;

	case -1:
	   cputs (RemoteClosed);
	   break;
   } /* switch */
   fatal("Socket error");
   return(0);	/* for BorlandC */
}
