#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
# === This file is part of Calamares - <http://github.com/calamares> ===
#
#   Copyright 2014, Pier Luigi Fiorini <pierluigi.fiorini@gmail.com>
#   Copyright 2015-2017, Teo Mrnjavac <teo@kde.org>
#   Copyright 2016-2017, Kyle Robbertze <kyle@aims.ac.za>
#
#   Calamares is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   Calamares is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with Calamares. If not, see <http://www.gnu.org/licenses/>.

import subprocess
import libcalamares
from libcalamares.utils import check_target_env_call, target_env_call
from string import Template

class PackageManager:
    """ Package manager class.

    :param backend:
    """
    def __init__(self, backend):
        self.backend = backend
        if libcalamares.globalstorage.value("hasInternet"):
            check_target_env_call(["pacman-key", "--init"])
            check_target_env_call(["pacman-key", "--populate"])

    def create_pkglist(self):
        machine = subprocess.check_output(("uname", "-m")).strip()
        arch = machine.rstrip().decode("ascii")

        root_mount_point = libcalamares.globalstorage.value("rootMountPoint")
        if arch == "i686": 
            ps = subprocess.Popen(("chroot", root_mount_point, "pacman", "-Slq", "core", "extra", "community", "bluestar"), stdout=subprocess.PIPE)
        elif arch == "x86_64":
            ps = subprocess.Popen(("chroot", root_mount_point, "pacman", "-Slq", "core", "extra", "community", "multilib", "bluestar"), stdout=subprocess.PIPE)

        with open("/tmp/pkglist", "w") as outfile:
            subprocess.check_call(("sort"), stdin=ps.stdout, stdout=outfile)
            ps.wait()

    def install(self, pkgs, from_local=False):
        """ Installs packages.

        :param pkgs:
        :param from_local:
        """
        if from_local:
            pacman_flags = "-U"
        else:
            pacman_flags = "-Sy"

        root_mount_point = libcalamares.globalstorage.value("rootMountPoint")
        try:
            subprocess.check_call(["java", "-jar", "/usr/share/java/POutput.jar", "-t", "Package Installation",  "chroot", root_mount_point, "pacman", "--force", pacman_flags, "--noconfirm", "--needed"] + pkgs)
        except subprocess.CalledProcessError as e:
            return "Cannot install packages.", "pacman terminated with exit code {}.".format(e.returncode)

    def remove(self, pkgs):
        """ Removes packages.

        :param pkgs:
        """
        check_target_env_call(["pacman", "-Rs", "--noconfirm"] + pkgs)

    def update_db(self):
        check_target_env_call(["pacman", "-Syy"])

    def update_system(self):
        root_mount_point = libcalamares.globalstorage.value("rootMountPoint")
        try:
            subprocess.check_call(["java", "-jar", "/usr/share/java/POutput.jar", "-t", "System Update", "chroot", root_mount_point, "pacman", "--force", "-Syu", "--noconfirm", "--needed"])
        except subprocess.CalledProcessError as e:
            return "Cannot update system.", "pacman terminated with exit code {}.".format(e.returncode)

    def run(self, script):
        if script != "":
            check_target_env_call(script.split(" "))


def subst_locale(list):
    ret = []
    locale = libcalamares.globalstorage.value("locale")
    if locale:
        for e in list:
            if  locale != "en":
                entry = Template(e)
                ret.append(entry.safe_substitute(LOCALE=locale))
            elif 'LOCALE' not in e:
                ret.append(e)
    else:
        ret = list
    return ret

def pkg_exists(pkg_name):
    """ Checks if package name exists in repos

    :param pkg_name:
    """
    with open("/tmp/pkglist", "r") as f:
        for line_terminated in f:
            line = line_terminated.rstrip('\n')
            if line == pkg_name:
                return pkg_name

    return ""

def run_operations(pkgman, entry):
    """ Call package manager with given parameters.

    :param pkgman:
    :param entry:
    """
    pkgs = []
    for key in entry.keys():
        entry[key] = subst_locale(entry[key])
        if key == "install":
            if isinstance(entry[key], list):
                for package in entry[key]:
                    if isinstance(package, str):
                        checked_pkg = pkg_exists(package)
                        if checked_pkg != "":
                            try:
                                pkgs.index(checked_pkg)
                            except ValueError:
                                pkgs.append(checked_pkg)
                if len(pkgs) > 0:
                    libcalamares.utils.debug("INFO: about to perform installation from pkgs")
                    pkgman.install(pkgs)
            else:
                pkgman.install(entry[key])
        elif key == "try_install":
            # we make a separate package manager call for each package so a single
            # failing package won't stop all of them
            for package in entry[key]:
                if isinstance(package, str):
                    try:
                        pkgman.install([package])
                    except subprocess.CalledProcessError:
                        libcalamares.utils.debug("WARNING: could not install package {}".format(package))
        elif key == "remove":
            pkgman.remove(entry[key])
        elif key == "try_remove":
            for package in entry[key]:
                try:
                    pkgman.remove([package])
                except subprocess.CalledProcessError:
                    libcalamares.utils.debug("WARNING: could not remove package {}".format(package))
        elif key == "localInstall":
            pkgman.install(entry[key], from_local=True)


def run():
    """ Calls routine with detected package manager to install local packages
    or remove drivers not needed on the installed system.

    :return:
    """
    backend = libcalamares.job.configuration.get("backend")
    root_mount_point = libcalamares.globalstorage.value("rootMountPoint")

    pkgman = PackageManager(backend)

    version = libcalamares.globalstorage.value("bslxInstallation")
    installation = libcalamares.job.configuration.get(version, [])
    update_db = libcalamares.job.configuration.get("update_db", False)

    libcalamares.utils.debug("INFO: version selected: {}".format(version))

    if update_db and libcalamares.globalstorage.value("hasInternet"):
        pkgman.update_db()

    if libcalamares.globalstorage.value("hasInternet"):

        libcalamares.utils.debug("INFO: about to perform update")

        pkgman.update_system()
        pkgman.create_pkglist()

        for verpkgs in installation:
            run_operations(pkgman, verpkgs)

    operations = libcalamares.job.configuration.get("operations", [])
    for entry in operations:
        run_operations(pkgman, entry)

    if libcalamares.globalstorage.contains("packageOperations"):
        run_operations(pkgman, libcalamares.globalstorage.value("packageOperations"))

    return None

