/**
 *   Copyright 2007 Y.Murakamin
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package net.murakamin.sticker.commands;

import net.murakamin.sticker.CommandRunner;

import org.apache.commons.lang.StringUtils;

/**
 * It is tag that shows the definition of the column of the table in an internal
 * database that uses HSQLDB. It uses it with the <table> tag and the <tables>
 * tag.
 * 
 * @author Y.Murakamin
 * @see net.murakamin.sticker.commands.TablesCommand
 * @see net.murakamin.sticker.commands.TableCommand
 */
public class ColumnCommand implements Command
{

	private String defaultValue;
	private String name;
	private String optionValue;
	private boolean nullable;
	private String type;

	/**
	 * do nothing. It is a method that calls from
	 * {@link net.murakamin.sticker.CommandRunner}.
	 * 
	 * @param pool
	 * @param context
	 */
	public void execute(final CommandRunner runner) throws Exception
	{
		runner.run(this);
	}

	/**
	 * The definition that shows the default value in the definition of the
	 * column of the CreateTable statement is acquired.
	 * 
	 * @return default value definition
	 */
	public String getDefault()
	{
		return this.defaultValue;
	}

	/**
	 * The column name is acquired in the definition of the column of the
	 * CreateTable sentence.
	 * 
	 * @return column name definition
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * The addition definition of the column (restriction and column attribute
	 * etc. such as primary keys) is acquired in the definition of the column of
	 * the CreateTable sentence.
	 * 
	 * @return The addition definition of the column
	 */
	public String getOption()
	{
		return this.optionValue;
	}

	/**
	 * The type of the column is acquired in the definition of the column of the
	 * CreateTable sentence.
	 * 
	 * @return
	 */
	public String getType()
	{
		return type;
	}

	/**
	 * @see net.murakamin.sticker.commands.Command#getVersionTerm()
	 */
	public VersionTerm getVersionTerm()
	{
		return VersionTerm.PERMANENT_COMMAND;
	}

	/**
	 * @see net.murakamin.sticker.commands.Command#isDebugPrint()
	 */
	public boolean isDebugPrint()
	{
		return false;
	}

	/**
	 * The state of the Null constraint of the column is acquired in the
	 * definition of the column of the CreateTable sentence.
	 * 
	 * @return If it is true, Null is permitted.
	 * 
	 */
	public boolean isNullable()
	{
		return nullable;
	}

	/**
	 * set the default value
	 * 
	 * @param value
	 *            default value
	 */
	public void setDefault(final String value)
	{
		this.defaultValue = value;
	}

	/**
	 * set the column name
	 * 
	 * @param name
	 *            column name
	 */
	public void setName(final String name)
	{
		this.name = name;
	}

	/**
	 * set the nullable
	 * 
	 * @param nullable
	 *            set the true, nullable column
	 */
	public void setNullable(final boolean nullable)
	{
		this.nullable = nullable;
	}

	/**
	 * set the additional value string
	 * 
	 * @param o
	 *            additional value string
	 */
	public void setOption(final String o)
	{
		this.optionValue = o;
	}

	/**
	 * set the column type
	 * 
	 * @param type
	 *            column type
	 */
	public void setType(final String type)
	{
		this.type = type;
	}

	@Override
	public String toString()
	{
		StringBuffer buffer = new StringBuffer();
		buffer.append(this.getName());
		buffer.append(' ');
		buffer.append(this.getType());

		if (!this.isNullable())
		{
			buffer.append(' ');
			buffer.append("not null");
		}

		if (this.defaultValue != null)
		{
			buffer.append(' ');
			buffer.append("DEFAULT");
			buffer.append(' ');
			if ("NULL".equalsIgnoreCase(this.defaultValue.trim()))
			{
				buffer.append("null");
			} else
			{
				buffer.append('\'');
				buffer.append(this.defaultValue);
				buffer.append('\'');
			}
		}

		if (!StringUtils.isEmpty(this.optionValue))
		{
			buffer.append(' ');
			buffer.append(this.getOption());
		}
		return buffer.toString();
	}
}
