/**
 *   Copyright 2007 Y.Murakamin
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package net.murakamin.sticker;

import java.sql.Connection;
import java.sql.SQLException;

import net.murakamin.sticker.commands.exception.CommandExecutionException;

/**
 * connection pool that manages data base connection
 * 
 * @author Y.Murakamin
 */
public interface ConnectionPool
{

	/**
	 * The table name made for HSQLDB is set. Sticker judges making the table by
	 * using the name defined here.
	 * 
	 * @param localTableName
	 *            HSQLDB table name
	 * @see #existsLocalTable(String)
	 */
	void addLocalTableName(final String localTableName);

	/**
	 * The all remote data base connection and HSQLDB connection is closed.
	 * 
	 * @throws SQLException
	 */
	void disConnectAllConnections() throws SQLException;

	/**
	 * When the table of the specified name exists in HSQLDB, true is returned.
	 * There is a possibility that CREATE TABLE is not actually done for making
	 * to use and to judge the name set with {@link #addLocalTableName(String)}
	 * method.
	 * 
	 * @param localTableName
	 *            Table name to be checked
	 * @return When the table of the specified name exists in HSQLDB, true is
	 *         returned.
	 * @see #addLocalTableName(String)
	 */
	boolean existsLocalTable(final String localTableName);

	/**
	 * The connection of HSQLDB is acquired.
	 * 
	 * @return The connection of HSQLDB
	 * @throws CommandExecutionException
	 */
	Connection getLocalConnection() throws CommandExecutionException;

	/**
	 * The connection to the remote data base that should be connected now is
	 * acquired. The remote data base that should be connected now is set
	 * depending on the {@link #setCurrentConnectionName(String)} method.
	 * 
	 * @return It is a remote data base connection in the specified name for
	 *         being able the string it.
	 * 
	 * @see #setCurrentConnectionName(String)
	 * @throws CommandExecutionException
	 */
	Connection getRemoteConnection() throws CommandExecutionException;

	/**
	 * The remote database connectivility name that should be connected now is
	 * set.
	 * 
	 * @param name
	 *            The remote database connectivility name
	 * @see #getRemoteConnection()
	 */
	void setCurrentConnectionName(final String name);

	/**
	 * The connection of a remote data base is set to the connection pool. At
	 * this time, the connection name to identify the connection is specified.
	 * 
	 * @param name
	 *            connection name
	 * @param con
	 *            remote data base connection
	 * @see #getRemoteConnection()
	 * @see #setCurrentConnectionName(String)
	 * @throws CommandExecutionException
	 * 
	 */
	void setRemoteConnection(final String name, final Connection con)
	        throws CommandExecutionException;
}
