/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static java.awt.event.InputEvent.*;
import static java.awt.event.KeyEvent.*;
import static javax.swing.KeyStroke.getKeyStroke;
import static net.argius.stew.Iteration.join;
import static net.argius.stew.ui.window.Resource.*;
import static net.argius.stew.ui.window.WindowLaunchHelper3.*;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.dnd.*;
import java.awt.event.*;
import java.io.*;
import java.nio.channels.*;
import java.sql.*;
import java.util.*;
import java.util.Map.Entry;
import java.util.List;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;

import net.argius.stew.*;
import net.argius.stew.io.*;

/**
 * WindowLauncher̕R[h(1)B
 * Cxg̐ݒB
 */
final class WindowLaunchHelper1 {

    private static final String COPY = "copy";
    private static final String SELECT_ALL = "selectAll";
    private static final String DELETE_ROWS = "deleteRows";
    private static final String LINK_ROWS_TO_DATABASE = "linkRowsToDatabase";
    private static final String DUPLICATE_ROWS = "duplicateRows";
    private static final String INSERT_FROM_CLIPBOARD = "insertFromClipboard";
    private static final String ADD_EMPTY_ROW = "addEmptyRow";
    private static final String PASTE = "paste";
    private static final String CLEAR_CELLS = "clearCells";
    private static final String COPY_COLUMN_NAMES = "copyColumnNames";
    private static final String COPY_WITH_ESCAPE = "copyWithEscape";
    private static final String GENERATE_INSERT_STATEMENT = "generate-insert-statement";
    private static final String GENERATE_UPDATE_STATEMENT = "generate-update-statement";
    private static final String GENERATE_SELECT_PHRASE = "generate-select-phrase";
    private static final String GENERATE_WHERE_PHRASE = "generate-where-phrase";
    private static final String REFRESH = "refresh";
    private static final String COPY_SIMPLE_NAME = "copy-simple-name";
    private static final String COPY_FULL_NAME = "copy-full-name";
    private static final String SORT = "sort";
    private static final String JUMP_TO_COLUMN_BY_NAME = "jump-to-column-by-name";
    private static final String TOGGLE_SHOW_COLUMN_NUMBER = "toggle-show-column-number";
    private static final String FIND_COLUMN_NAME = "findColumnName";
    private static final String SET_CURRENT_TIME_VALUE = "set-current-time-value";
    private static final String RECEIVER_ResultSetTable = "ResultSetTable";
    private static final String RECEIVER_ConsoleTextArea = "ConsoleTextArea";

    private WindowLaunchHelper1() {
        // empty
    }

    /**
     * DatabaseInfoTreeɃCxg蓖ĂB
     * @param tree DatabaseInfoTree
     * @param rstable ResultSetTable
     * @param textArea ConsoleTextArea
     */
    static void bindEvents(DatabaseInfoTree tree,
                           final ResultSetTable rstable,
                           final ConsoleTextArea textArea) {
        final ActionUtility actionUtility = ActionUtility.getInstance(tree);
        // [ANVCxg]
        tree.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                Parameter p = new Parameter(e.getActionCommand());
                if (!p.has(2)) {
                    return;
                }
                final String receiver = p.at(0);
                final String action = p.at(1);
                final String argsString = p.after(2);
                if (receiver.equals(RECEIVER_ResultSetTable)) {
                    controlResultSetTable(action, argsString);
                } else if (receiver.equals(RECEIVER_ConsoleTextArea)) {
                    controlConsoleTextArea(action, argsString);
                }
            }
            private void controlResultSetTable(String action, String parameter) {
                if (action.equals(JUMP_TO_COLUMN_BY_NAME)) {
                    rstable.jumpToColumn(parameter);
                }
            }
            private void controlConsoleTextArea(String action, String parameter) {
                if (action.equals("paste")) {
                    if (!textArea.isEditablePosition(textArea.getCaretPosition())) {
                        textArea.setCaretPosition(textArea.getEndPosition());
                    }
                    textArea.replaceSelection(parameter);
                    textArea.requestFocus();
                }
            }
        });
        // [ReLXgj[]
        actionUtility.setContextMenu(new String[]{"copy", COPY_SIMPLE_NAME, COPY_FULL_NAME, "",
                                                  REFRESH, "", GENERATE_WHERE_PHRASE,
                                                  GENERATE_SELECT_PHRASE,
                                                  GENERATE_UPDATE_STATEMENT,
                                                  GENERATE_INSERT_STATEMENT, "",
                                                  JUMP_TO_COLUMN_BY_NAME,
                                                  TOGGLE_SHOW_COLUMN_NUMBER,},
                                     "CNF R WSUI JT".toCharArray());
    }

    /**
     * ResultSetTableɃCxg蓖ĂB
     * @param rstable
     * @param op
     */
    static void bindEvents(final ResultSetTable rstable, final WindowOutputProcessor op) {
        // [ANVCxg]
        final ActionUtility actionUtility = ActionUtility.getInstance(rstable);
        final int shortcutKey = Resource.getMenuShortcutKeyMask();
        final Action cancelTableEditAction = new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                rstable.editingCanceled(new ChangeEvent(e.getSource()));
            }

        };
        // GXP[vtŃRs[
        actionUtility.bindAction(new AbstractAction(COPY_WITH_ESCAPE) {

            public void actionPerformed(ActionEvent e) {
                List<String> rows = new ArrayList<String>();
                for (int rowIndex : rstable.getSelectedRows()) {
                    List<Object> row = new ArrayList<Object>();
                    for (int columnIndex : rstable.getSelectedColumns()) {
                        final Object o = rstable.getValueAt(rowIndex, columnIndex);
                        row.add(CsvFormatter.AUTO.format(o == null ? "" : String.valueOf(o)));
                    }
                    rows.add(join(row, TAB));
                }
                setClipboard(join(rows, EOL));
            }

        }, getKeyStroke(VK_C, shortcutKey | ALT_DOWN_MASK));
        // \t
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                try {
                    InputStream is = new ByteArrayInputStream(getClipboard().getBytes());
                    Importer importer = new SmartImporter(is, TAB);
                    try {
                        int[] selectedColumns = rstable.getSelectedColumns();
                        for (int rowIndex : rstable.getSelectedRows()) {
                            Object[] values = importer.nextRow();
                            final int limit = Math.min(selectedColumns.length, values.length);
                            for (int x = 0; x < limit; x++) {
                                rstable.setValueAt(values[x], rowIndex, selectedColumns[x]);
                            }
                        }
                    } finally {
                        importer.close();
                    }
                    rstable.repaint();
                } catch (Exception ex) {
                    handleError(ex, op);
                } finally {
                    cancelTableEditAction.actionPerformed(e);
                }
            }

        }, PASTE);
        // 񖼂Rs[
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                List<String> a = new ArrayList<String>();
                ResultSetTableModel m = rstable.getResultSetTableModel();
                for (int i = 0, n = m.getColumnCount(); i < n; i++) {
                    a.add(m.getColumnName(i));
                }
                setClipboard(join(a, TAB));
            }

        }, COPY_COLUMN_NAMES);
        // IZ̒l폜(NULLݒ)
        actionUtility.bindAction(new AbstractAction(CLEAR_CELLS) {
            public void actionPerformed(ActionEvent e) {
                try {
                    setValueAtSelectedCells(rstable, null);
                    rstable.repaint();
                } catch (Exception ex) {
                    handleError(ex, op);
                } finally {
                    cancelTableEditAction.actionPerformed(e);
                }
            }
        }, getKeyStroke(VK_DELETE, 0));
        // Vs(񃊃N)ǉ
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                ResultSetTableModel m = rstable.getResultSetTableModel();
                final int selectedRow = rstable.getSelectedRow();
                if (selectedRow >= 0) {
                    m.insertUnlinkedRow(selectedRow + 1, new Object[m.getColumnCount()]);
                } else {
                    m.addUnlinkedRow(new Object[m.getColumnCount()]);
                }
            }

        }, ADD_EMPTY_ROW);
        // Nbv{[h}
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                try {
                    Importer importer = new SmartImporter(getClipboardAsReader(), TAB);
                    try {
                        ResultSetTableModel m = rstable.getResultSetTableModel();
                        while (true) {
                            Object[] row = importer.nextRow();
                            if (row.length == 0) {
                                break;
                            }
                            m.addUnlinkedRow(row);
                            m.linkRow(m.getRowCount() - 1);
                        }
                        rstable.repaintRowHeader("model");
                    } finally {
                        importer.close();
                    }
                } catch (Exception ex) {
                    handleError(ex, op);
                } finally {
                    cancelTableEditAction.actionPerformed(e);
                }
            }

        }, INSERT_FROM_CLIPBOARD);
        // Is𕡐Ēǉ
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                ResultSetTableModel m = rstable.getResultSetTableModel();
                List<?> rows = m.getDataVector();
                int[] selectedRows = rstable.getSelectedRows();
                int index = selectedRows[selectedRows.length - 1];
                for (int rowIndex : selectedRows) {
                    m.insertUnlinkedRow(++index, (Vector<?>)((Vector<?>)rows.get(rowIndex)).clone());
                }
                rstable.repaint();
                rstable.repaintRowHeader("model");
            }

        }, DUPLICATE_ROWS);
        // 񃊃NsDBƃN
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                ResultSetTableModel m = rstable.getResultSetTableModel();
                try {
                    for (int rowIndex : rstable.getSelectedRows()) {
                        m.linkRow(rowIndex);
                    }
                } catch (SQLException ex) {
                    handleError(ex, op);
                } finally {
                    rstable.repaintRowHeader("unlinkedRowStatus");
                }
            }

        }, LINK_ROWS_TO_DATABASE);
        // Is폜
        actionUtility.bindAction(new AbstractAction(DELETE_ROWS) {

            public void actionPerformed(ActionEvent e) {
                try {
                    ResultSetTableModel m = rstable.getResultSetTableModel();
                    while (true) {
                        final int selectedRow = rstable.getSelectedRow();
                        if (selectedRow < 0) {
                            break;
                        }
                        if (m.isLinkedRow(selectedRow)) {
                            final boolean removed = m.removeLinkedRow(selectedRow);
                            assert removed;
                        } else {
                            m.removeRow(selectedRow);
                        }
                    }
                } catch (SQLException ex) {
                    handleError(ex, op);
                } finally {
                    rstable.repaintRowHeader("model");
                }
            }

        }, getKeyStroke(VK_MINUS, shortcutKey | SHIFT_DOWN_MASK));
        // 񕝂̎
        actionUtility.bindAction(new AbstractAction("adjustColumnWidth") {
            public void actionPerformed(ActionEvent e) {
                op.adjustTableColumnWidth();
            }
        }, getKeyStroke(VK_SLASH, shortcutKey));
        // בւ
        final Point mousePositionForColumnHeader = new Point();
        actionUtility.bindAction(new AbstractAction(SORT) {
            public void actionPerformed(ActionEvent e) {
                op.sortTable(rstable.getTableHeader().columnAtPoint(mousePositionForColumnHeader));
            }
        }, getKeyStroke(VK_S, ALT_DOWN_MASK));
        // ݎ̓\t
        actionUtility.bindAction(new AbstractAction(SET_CURRENT_TIME_VALUE) {
            public void actionPerformed(ActionEvent e) {
                try {
                    setValueAtSelectedCells(rstable, new Timestamp(System.currentTimeMillis()));
                    rstable.repaint();
                } catch (Exception ex) {
                    handleError(ex, op);
                } finally {
                    cancelTableEditAction.actionPerformed(e);
                }
            }
        });
        // 񖼌
        actionUtility.bindAction(new AbstractAction(FIND_COLUMN_NAME) {
            public void actionPerformed(ActionEvent e) {
                TextSearchPanel textSearchPanel = getTextSearchPanel(op);
                if (textSearchPanel != null) {
                    rstable.getTableHeader().requestFocus();
                    textSearchPanel.setVisible(true);
                }
            }
            private TextSearchPanel getTextSearchPanel(Component c) {
                if (c instanceof TextSearchPanel) {
                    return (TextSearchPanel)c;
                }
                if (c instanceof Container) {
                    Container container = (Container)c;
                    for (Component child : container.getComponents()) {
                        TextSearchPanel result = getTextSearchPanel(child);
                        if (result != null) {
                            return result;
                        }
                    }
                }
                return null;
            }
        });
        // [ReLXgj[]
        String[] actionNames = {COPY, COPY_WITH_ESCAPE, PASTE, SELECT_ALL, "", CLEAR_CELLS,
                                SET_CURRENT_TIME_VALUE, "", COPY_COLUMN_NAMES, FIND_COLUMN_NAME,
                                "", ADD_EMPTY_ROW, INSERT_FROM_CLIPBOARD, DUPLICATE_ROWS,
                                LINK_ROWS_TO_DATABASE, DELETE_ROWS,};
        final Map<String, JMenuItem> m = new LinkedHashMap<String, JMenuItem>();
        final JPopupMenu pmenu = actionUtility.createPopupMenu(actionNames,
                                                               "CWPA RT NF EISLD".toCharArray(),
                                                               m);
        final List<String> linkActions = Arrays.asList(new String[]{PASTE, CLEAR_CELLS,
                                                                    ADD_EMPTY_ROW, DUPLICATE_ROWS,
                                                                    LINK_ROWS_TO_DATABASE,
                                                                    DELETE_ROWS,});
        pmenu.addPopupMenuListener(new PopupMenuListener() {
            public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
                ResultSetTableModel resultSetTableModel = rstable.getResultSetTableModel();
                final boolean linkable = resultSetTableModel.isLinkable();
                for (Entry<String, JMenuItem> entry : m.entrySet()) {
                    if (linkActions.contains(entry.getKey())) {
                        entry.getValue().setEnabled(linkable);
                    }
                }
                final boolean updatable = resultSetTableModel.isUpdatable();
                m.get(INSERT_FROM_CLIPBOARD).setEnabled(updatable);
            }
            public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {
                // empty
            }
            public void popupMenuCanceled(PopupMenuEvent e) {
                // empty
            }
        });
        rstable.setComponentPopupMenu(pmenu);
        // [wb_]
        final JTableHeader columnHeader = rstable.getTableHeader();
        // ReLXgj[
        final JPopupMenu pmenuCH = actionUtility.createPopupMenu(new String[]{SORT, "", COPY,
                                                                              COPY_WITH_ESCAPE,
                                                                              PASTE, SELECT_ALL,
                                                                              "",
                                                                              COPY_COLUMN_NAMES,
                                                                              FIND_COLUMN_NAME, "",
                                                                              ADD_EMPTY_ROW,
                                                                              INSERT_FROM_CLIPBOARD},
                                                                 "S CWPA NF EI".toCharArray());
        columnHeader.setComponentPopupMenu(pmenuCH);
        // ̑Cxg
        columnHeader.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(MouseEvent e) {
                mousePositionForColumnHeader.setLocation(e.getPoint());
            }
        });
        // [swb_]
        final JTable rowHeader = rstable.getRowHeader();
        // ReLXgj[
        rowHeader.setComponentPopupMenu(pmenu);
    }

    /**
     * ConsoleTextAreaɃCxg蓖ĂB
     * @param textArea
     * @param launcher
     */
    static void bindEvents(final ConsoleTextArea textArea, final WindowLauncher launcher) {
        // [ANVCxg]
        final ActionUtility actionUtility = ActionUtility.getInstance(textArea);
        textArea.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                launcher.executeCommand(e.getActionCommand());
            }

        });
        // [hbvCxg]
        textArea.setDropTarget(new DropTarget(textArea, new DropTargetAdapter() {

            public void drop(DropTargetDropEvent dtde) {
                final List<File> fileList;
                Transferable t = dtde.getTransferable();
                if (t.isDataFlavorSupported(DataFlavor.javaFileListFlavor)) {
                    dtde.acceptDrop(DnDConstants.ACTION_COPY_OR_MOVE);
                    try {
                        @SuppressWarnings("unchecked")
                        List<File> list = (List<File>)t.getTransferData(DataFlavor.javaFileListFlavor);
                        fileList = list;
                    } catch (Exception ex) {
                        handleError(ex, null);
                        return;
                    }
                } else {
                    throw new IllegalStateException("Transferable=" + t);
                }
                JPopupMenu menu = new JPopupMenu();
                JMenuItem item1 = new JMenuItem(new AbstractAction(getString("Action.paste-path",
                                                                             'P')) {

                    public void actionPerformed(ActionEvent e) {
                        final String s = join(fileList, " ");
                        final int p = textArea.getCaretPosition();
                        if (textArea.isEditablePosition(p)) {
                            textArea.insert(s, p);
                        } else {
                            textArea.append(s, false);
                        }
                    }

                });
                item1.setMnemonic('P');
                menu.add(item1);
                JMenuItem item2 = new JMenuItem(new AbstractAction(getString("Action.paste-file-content",
                                                                             'F')) {

                    public void actionPerformed(ActionEvent e) {
                        try {
                            for (File file : fileList) {
                                if (file.length() > 0) {
                                    final String s = convertContentToString(file);
                                    final int p = textArea.getCaretPosition();
                                    if (textArea.isEditablePosition(p)) {
                                        textArea.insert(s, p);
                                    } else {
                                        textArea.append(s, false);
                                    }
                                }
                            }
                        } catch (IOException ex) {
                            handleError(ex, null);
                        }
                    }

                    private String convertContentToString(File file) throws IOException {
                        ByteArrayOutputStream bos = new ByteArrayOutputStream();
                        FileInputStream fis = new FileInputStream(file);
                        try {
                            fis.getChannel().transferTo(0, file.length(), Channels.newChannel(bos));
                        } finally {
                            fis.close();
                        }
                        return bos.toString();
                    }

                });
                item2.setMnemonic('F');
                menu.add(item2);
                Point p = dtde.getLocation();
                menu.show(textArea, p.x, p.y);
            }

        }));
        // [̑]
        // ReLXgj[+AhD
        actionUtility.setActionForTextComponent();
    }

    /**
     * IꂽZɒl\tB
     * @param table Table
     * @param value Cӂ̒l
     */
    static void setValueAtSelectedCells(JTable table, Object value) {
        int[] selectedColumns = table.getSelectedColumns();
        for (int rowIndex : table.getSelectedRows()) {
            final int limit = selectedColumns.length;
            for (int x = 0; x < limit; x++) {
                table.setValueAt(value, rowIndex, selectedColumns[x]);
            }
        }
    }

}
