/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static net.argius.stew.Iteration.*;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.event.*;
import java.sql.*;
import java.util.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;

import net.argius.logging.*;
import net.argius.stew.*;

/**
 * f[^x[X\c[B
 */
final class DatabaseInfoTree extends JTree {

    private static final Logger log = LoggerFactory.getLogger(DatabaseInfoTree.class);

    private Connector currentConnector;

    /**
     * RXgN^B
     */
    DatabaseInfoTree() {
        setRootVisible(false);
        setVisible(false);
        setShowsRootHandles(true);
        setScrollsOnExpand(true);
        setCellRenderer(new Renderer());
        DefaultTreeSelectionModel m = new DefaultTreeSelectionModel();
        m.setSelectionMode(TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);
        setSelectionModel(m);
        // FIXME ȂColumnNode͑IłȂH
    }

    void addActionListener(ActionListener listener) {
        listenerList.add(ActionListener.class, listener);
    }

    void removeActionListener(ActionListener listener) {
        listenerList.remove(ActionListener.class, listener);
    }

    void fireActionPerformed(String cmd) {
        ActionEvent event = new ActionEvent(this, ActionEvent.ACTION_PERFORMED, cmd);
        for (ActionListener listener : listenerList.getListeners(ActionListener.class)) {
            listener.actionPerformed(event);
        }
    }

    /**
     * RootvfĐݒ肷B
     * @param env Environment
     * @throws SQLException
     */
    void refreshRoot(Environment env) throws SQLException {
        Connector c = env.getCurrentConnector();
        if (c == null) {
            if (log.isDebugEnabled()) {
                log.debug("not connected");
            }
            currentConnector = null;
            return;
        }
        if (c == currentConnector) {
            if (log.isDebugEnabled()) {
                log.debug("not changed");
            }
            return;
        }
        if (log.isDebugEnabled()) {
            log.debug("updating");
        }
        final DatabaseMetaData dbmeta = env.getCurrentConnection().getMetaData();
        // m[h
        ConnectorNode connectorNode = new ConnectorNode(c.getName());
        connectorNode.initialize(dbmeta);
        final DefaultTreeModel model = (DefaultTreeModel)getModel();
        model.setRoot(connectorNode);
        // TODO m[hWJɑ̃m[h[Ȃ悤ɂɂ́H
        addTreeWillExpandListener(new TreeWillExpandListener() {

            public void treeWillExpand(TreeExpansionEvent event) throws ExpandVetoException {
                // TODO [h̃vOX
                TreePath path = event.getPath();
                final Object lastPathComponent = path.getLastPathComponent();
                if (lastPathComponent instanceof BaseNode) {
                    BaseNode node = (BaseNode)lastPathComponent;
                    try {
                        node.initialize(dbmeta);
                    } catch (SQLException ex) {
                        throw new RuntimeException(ex);
                    }
                }
                model.reload();
            }

            public void treeWillCollapse(TreeExpansionEvent event) throws ExpandVetoException {
                // ignore
            }

        });
        ActionUtility actionUtility = new ActionUtility(this);
        actionUtility.bindAction(new AbstractAction("copy") {

            public void actionPerformed(ActionEvent e) {
                TreePath[] paths = getSelectionPaths();
                if (paths == null) {
                    return;
                }
                String s = join(map(Arrays.asList(paths),
                                    new Iteration.Correspondence<TreePath, Object>() {

                                        @Override
                                        public Object f(TreePath preimage) {
                                            return preimage.getLastPathComponent();
                                        }

                                    }), System.getProperty("line.separator"));
                Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
                StringSelection sselection = new StringSelection(s);
                clipboard.setContents(sselection, sselection);
            }

        }, "copy");
        actionUtility.setContextMenu(new String[]{"copy"});
        model.reload();
        setRootVisible(true);
        setVisible(true);
        this.currentConnector = c;
    }

    /**
     * vf̕`揈(Renderer)B
     */
    private static class Renderer extends DefaultTreeCellRenderer {

        Renderer() {
            // empty
        }

        @Override
        public Component getTreeCellRendererComponent(JTree tree,
                                                      Object value,
                                                      boolean sel,
                                                      boolean expanded,
                                                      boolean leaf,
                                                      int row,
                                                      boolean hasFocus) {
            super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf, row, hasFocus);
            if (value instanceof BaseNode) {
                setIcon(Resource.getImageIcon(((BaseNode)value).getIconName()));
            }
            return this;
        }

    }

    /**
     * bm[hB
     */
    private static class BaseNode extends DefaultMutableTreeNode {

        protected boolean added;

        BaseNode(Object userObject) {
            super(userObject, true);
        }

        @Override
        public boolean isLeaf() {
            return false;
        }

        final void reset() {
            removeAllChildren();
            added = false;
        }

        final void initialize(DatabaseMetaData dbmeta) throws SQLException {
            if (added) {
                return;
            }
            setChildren(dbmeta);
            added = true;
        }

        final String getIconName() {
            final String className = getClass().getName();
            final String nodeType = className.replaceFirst(".+?([^\\$]+)Node$", "$1");
            return "node-" + nodeType.toLowerCase() + ".png";
        }

        protected void setChildren(DatabaseMetaData dbmeta) throws SQLException {
            // empty
        }

        protected String getNodeFullName() {
            return "";
        }

    }

    /**
     * RlN^̃m[hB
     */
    private static class ConnectorNode extends BaseNode {

        ConnectorNode(String name) {
            super(name);
        }

        @Override
        protected void setChildren(DatabaseMetaData dbmeta) throws SQLException {
            if (dbmeta.supportsCatalogsInDataManipulation()) {
                ResultSet rs = dbmeta.getCatalogs();
                try {
                    while (rs.next()) {
                        add(new CatalogNode(rs.getString(1)));
                    }
                } finally {
                    rs.close();
                }
            } else if (dbmeta.supportsSchemasInDataManipulation()) {
                ResultSet rs = dbmeta.getSchemas();
                try {
                    while (rs.next()) {
                        add(new SchemaNode(null, rs.getString(1)));
                    }
                } finally {
                    rs.close();
                }
            } else {
                ResultSet rs = dbmeta.getTableTypes();
                try {
                    while (rs.next()) {
                        add(new TableTypeNode(null, null, rs.getString(1)));
                    }
                } finally {
                    rs.close();
                }
            }
        }

    }

    /**
     * J^Õm[hB
     */
    private static final class CatalogNode extends BaseNode {

        private final String name;

        CatalogNode(String name) {
            super(name);
            this.name = name;
        }

        @Override
        protected void setChildren(DatabaseMetaData dbmeta) throws SQLException {
            if (dbmeta.supportsSchemasInDataManipulation()) {
                ResultSet rs = dbmeta.getSchemas();
                try {
                    while (rs.next()) {
                        add(new SchemaNode(name, rs.getString(1)));
                    }
                } finally {
                    rs.close();
                }
            } else {
                ResultSet rs = dbmeta.getTableTypes();
                try {
                    while (rs.next()) {
                        add(new TableTypeNode(name, null, rs.getString(1)));
                    }
                } finally {
                    rs.close();
                }
            }
        }

    }

    /**
     * XL[}̃m[hB
     */
    private static final class SchemaNode extends BaseNode {

        private final String catalog;
        private final String schema;

        SchemaNode(String catalog, String schema) {
            super(schema);
            this.catalog = catalog;
            this.schema = schema;
        }

        @Override
        protected void setChildren(DatabaseMetaData dbmeta) throws SQLException {
            ResultSet rs = dbmeta.getTableTypes();
            try {
                while (rs.next()) {
                    add(new TableTypeNode(catalog, schema, rs.getString(1)));
                }
            } finally {
                rs.close();
            }
        }

    }

    /**
     * e[uʂ̃m[hB
     */
    private static final class TableTypeNode extends BaseNode {

        private final String catalog;
        private final String schema;
        private final String tableType;

        TableTypeNode(String catalog, String schema, String tableType) {
            super(tableType);
            this.catalog = catalog;
            this.schema = schema;
            this.tableType = tableType;
        }

        @Override
        protected void setChildren(DatabaseMetaData dbmeta) throws SQLException {
            ResultSet rs = dbmeta.getTables(catalog, schema, null, new String[]{tableType});
            try {
                while (rs.next()) {
                    add(new TableNode(catalog, schema, rs.getString(3)));
                }
            } finally {
                rs.close();
            }
        }

    }

    /**
     * e[ũm[hB
     */
    private static final class TableNode extends BaseNode {

        private final String catalog;
        private final String schema;
        private final String name;

        TableNode(String catalog, String schema, String name) {
            super(name);
            this.catalog = catalog;
            this.schema = schema;
            this.name = name;
        }

        @Override
        protected void setChildren(DatabaseMetaData dbmeta) throws SQLException {
            ResultSet rs = dbmeta.getColumns(catalog, schema, name, null);
            try {
                while (rs.next()) {
                    final String nonNull = "NO".equals(rs.getString(18)) ? "NOT NULL" : "";
                    final String s = String.format("%s [%s(%d) %s]",
                                                   rs.getString(4),
                                                   rs.getString(6),
                                                   rs.getInt(7),
                                                   nonNull);
                    add(new ColumnNode(s));
                }
            } finally {
                rs.close();
            }
        }

        @Override
        public boolean isLeaf() {
            return false;
        }

    }

    /**
     * ̃m[hB
     */
    private static final class ColumnNode extends BaseNode {

        ColumnNode(String s) {
            super(s);
        }

        @Override
        public boolean isLeaf() {
            return false;
        }

    }

}
