package com.ozacc.blog.trackback.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpRecoverableException;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.HttpConnection.ConnectionTimeoutException;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.ozacc.blog.trackback.ConnectionException;
import com.ozacc.blog.trackback.TrackBackAutoDiscovery;
import com.ozacc.blog.trackback.TrackBackException;
import com.ozacc.blog.util.CommonsHttpClientUtils;

/**
 * TrackBackAutoDiscoveryインターフェースの実装クラス。
 * 
 * @since 1.0
 * @author Tomohiro Otsuka
 * @version $Id: TrackBackAutoDiscoveryImpl.java 180 2005-07-22 09:26:25Z otsuka $
 */
public class TrackBackAutoDiscoveryImpl implements TrackBackAutoDiscovery {

	private static Log log = LogFactory.getLog(TrackBackAutoDiscoveryImpl.class);

	public static final int DEFAULT_CONNECTION_TIMEOUT = 5000;

	public static final int DEFAULT_READ_TIMEOUT = 5000;

	private int connectionTimeout = DEFAULT_CONNECTION_TIMEOUT;

	private int readTimeout = DEFAULT_READ_TIMEOUT;

	private boolean followRedirect = true;

	private Pattern rdfPattern = Pattern.compile("<rdf:RDF(.*?)</rdf:RDF>", Pattern.DOTALL
			| Pattern.CASE_INSENSITIVE);

	private Pattern trackBackPingPattern = Pattern.compile("trackback:ping=\"(.+?)\"",
			Pattern.DOTALL);

	/**
	 * @see com.ozacc.blog.trackback.TrackBackAutoDiscovery#discoverTrackBackUrls(java.net.URL)
	 */
	public URL[] discoverTrackBackUrls(URL url) throws TrackBackException {
		return discoverTrackBackUrls(url.toString());
	}

	/**
	 * @param targetPageUrl
	 * @return
	 * @throws IOException 
	 */
	private String getPageContent(URL targetPageUrl) throws IOException {
		HttpURLConnection conn = (HttpURLConnection)targetPageUrl.openConnection();
		conn.connect();
		InputStream is = conn.getInputStream();
		InputStreamReader reader = new InputStreamReader(is);

		StringBuffer buf = new StringBuffer();
		char[] cbuf = new char[8192];
		while (reader.read(cbuf) != -1) {
			buf.append(cbuf);
		}
		String content = buf.toString();

		reader.close();
		conn.disconnect();
		return content;
	}

	/**
	 * 指定されたURLにGETリクエストを送り、そのレスポンスを文字列で返します。
	 * 
	 * @param url URL
	 * @return URLからのレスポンス
	 * @throws TrackBackException
	 */
	private String getPageContent(String url) throws TrackBackException {
		HttpClient client = new HttpClient();
		log.debug("接続タイムアウトを設定します。[" + connectionTimeout + "]");
		client.setConnectionTimeout(connectionTimeout);
		log.debug("読込タイムアウトを設定します。[" + readTimeout + "]");
		client.setTimeout(readTimeout);

		HttpMethod method = new GetMethod(url);
		method.setFollowRedirects(followRedirect);
		try {
			log.debug("HTTP接続を行います。[" + url + "]");
			int statusCode = client.executeMethod(method);
			if (!CommonsHttpClientUtils.isSuccessfulResponse(statusCode, followRedirect)) {
				throw new ConnectionException("指定されたURLへのHTTP接続ができませんでした。[HTTP_STATUS='"
						+ HttpStatus.getStatusText(statusCode) + "', url='" + url + "']");
			}
			return method.getResponseBodyAsString();
		} catch (ConnectionTimeoutException e) {
			throw new com.ozacc.blog.trackback.ConnectionTimeoutException("HTTP接続タイムアウト。[url='"
					+ url + "']", e);
		} catch (HttpRecoverableException e) {
			throw new com.ozacc.blog.trackback.ConnectionTimeoutException("HTTP読込タイムアウト。[url='"
					+ url + "']", e);
		} catch (IOException e) {
			throw new ConnectionException("指定されたURLへの接続に失敗しました。[url='" + url + "']", e);
		} finally {
			method.releaseConnection();
		}
	}

	

	/**
	 * @see com.ozacc.blog.trackback.TrackBackAutoDiscovery#discoverTrackBackUrls(java.lang.String)
	 */
	public URL[] discoverTrackBackUrls(String url) throws TrackBackException {
		String content = getPageContent(url);

		List result = new ArrayList();
		Matcher rdfMatcher = rdfPattern.matcher(content);
		while (rdfMatcher.find()) {
			Matcher tbMatcher = trackBackPingPattern.matcher(rdfMatcher.group(1));
			if (tbMatcher.find()) {
				try {
					result.add(new URL(tbMatcher.group(1)));
				} catch (MalformedURLException e) {
					// ignore
				}
			}
		}
		return (URL[])result.toArray(new URL[result.size()]);
	}

	/**
	 * 接続タイムアウト時間をセットします。単位はミリ秒。
	 * デフォルトは5,000ミリ秒(5秒)です。
	 * 
	 * @param connectionTimeout 接続タイムアウト (ms)
	 */
	public void setConnectionTimeout(int connectionTimeout) {
		this.connectionTimeout = connectionTimeout;
	}

	/**
	 * 接続後の読込タイムアウト時間をセットします。単位はミリ秒。
	 * デフォルトは5,000ミリ秒(5秒)です。
	 * 
	 * @param timeout 読込タイムアウト (ms)
	 */
	public void setReadTimeout(int timeout) {
		this.readTimeout = timeout;
	}

	/**
	 * アクセスしたURLがリダイレクトレスポンス(HTTP Status Code 3xx)を返してきた場合に、
	 * リダイレクト先にアクセスするかどうかを設定します。デフォルトでは、リダイレクト先にアクセスします。
	 * 
	 * @since 1.2.4
	 * @param followRedirect リダイレクト先にアクセスする場合 true。デフォルトはtrue。
	 */
	public void setFollowRedirect(boolean followRedirect) {
		this.followRedirect = followRedirect;
	}

}