package com.ozacc.blog.ping.impl;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.Hashtable;
import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xmlrpc.XmlRpc;
import org.apache.xmlrpc.XmlRpcClient;
import org.apache.xmlrpc.XmlRpcException;

import com.ozacc.blog.ping.ConnectionException;
import com.ozacc.blog.ping.FailedUpdatePingException;
import com.ozacc.blog.ping.UpdatePing;
import com.ozacc.blog.ping.UpdatePingClient;
import com.ozacc.blog.ping.UpdatePingException;
import com.ozacc.blog.ping.UpdatePingResponseException;

/**
 * XML-RPCを用いて更新Pingを送信するUpdatePingClientインターフェースの実装クラス。
 * 
 * @since 1.0
 * @author Tomohiro Otsuka
 * @version $Id: XmlRpcUpdatePingClient.java 180 2005-07-22 09:26:25Z otsuka $
 */
class XmlRpcUpdatePingClient implements UpdatePingClient {

	private static Log log = LogFactory.getLog(XmlRpcUpdatePingClient.class);

	/** デフォルトの文字コード「UTF-8」。 */
	public static final String DEFAULT_CHARSET = "utf-8";

	/**  更新PingのRPCメソッド名。<code>weblogUpdates.ping</code> */
	public static String PING_METHOD_NAME = "weblogUpdates.ping";

	private boolean debug = false;

	private String charset = DEFAULT_CHARSET;

	/**
	 * コンストラクタ。
	 */
	public XmlRpcUpdatePingClient() {}

	/**
	 * デバッグが有効になっているかどうか返します。
	 * 
	 * @return デバッグが有効になっている場合 true
	 */
	public boolean isDebug() {
		return debug;
	}

	/**
	 * XmlRpcClientのデバッグを有効にします。<br>
	 * デフォルトでは無効(false)です。
	 * 
	 * @param debug デバッグを有効にするときは true
	 */
	public void setDebug(boolean debug) {
		this.debug = debug;
	}

	/**
	 * @return Returns the encoding.
	 */
	public String getCharset() {
		return charset;
	}

	/**
	 * Pingデータのエンコードに使用する文字コードをセットします。
	 * デフォルトはUTF-8。
	 * 
	 * @param charset Pingデータのエンコードに使用する文字コード
	 */
	public void setCharset(String charset) {
		this.charset = charset;
	}

	/**
	 * XmlRpcClient#execute() から返される結果はHashtableと想定しています。
	 * それ以外の場合、例外をスローします。
	 * 
	 * @see com.ozacc.blog.ping.UpdatePingClient#ping(java.lang.String, java.lang.String, java.lang.String)
	 */
	public String ping(String pingUrl, String blogName, String blogUrl) throws UpdatePingException {
		UpdatePing ping = new UpdatePing(blogName, blogUrl);
		return ping(pingUrl, ping);
	}

	/**
	 * @see com.ozacc.blog.ping.UpdatePingClient#ping(java.lang.String, com.ozacc.blog.ping.Ping)
	 */
	public String ping(String pingUrl, UpdatePing ping) throws UpdatePingException {
		XmlRpcClient client = getRpcClient(pingUrl);
		Object result = execute(client, ping.getParameters());
		return processResult(result);
	}

	/**
	 * 指定されたRPC実行結果オブジェクトからレスポンスメッセージを取得します。
	 * 
	 * @param result RPC実行結果オブジェクト
	 * @return 結果メッセージ
	 * @throws UpdatePingResponseException
	 * @throws FailedUpdatePingException
	 */
	protected String processResult(Object result) throws UpdatePingResponseException,
													FailedUpdatePingException {
		Hashtable table;
		if (result instanceof java.util.Hashtable) {
			table = (Hashtable)result;
		} else {
			throw new UpdatePingResponseException("対応していないフォーマットのレスポンスが返されました。");
		}

		String message = (String)table.get("message");
		Boolean flerror = (Boolean)table.get("flerror");

		// Ping 成功
		if (!flerror.booleanValue()) {
			return message;
		}
		// Ping 失敗
		throw new FailedUpdatePingException(message);
	}

	/**
	 * @param client
	 * @param params
	 * @return RPC 実行結果オブジェクト
	 * @throws UpdatePingException 
	 */
	private Object execute(XmlRpcClient client, Vector params) throws UpdatePingException {
		try {
			return client.execute(PING_METHOD_NAME, params);
		} catch (XmlRpcException e) {
			throw new UpdatePingException("サーバがエラーを返しました。", e);
		} catch (IOException e) {
			throw new ConnectionException(
					"サーバとの接続に失敗しました。サーバのURLが正しいか、またサーバがRPCをサポートしているか確認してください。", e);
		}
	}

	/**
	 * @param serverUrl
	 * @return XmlRpcClient
	 * @throws ConnectionException 
	 */
	protected XmlRpcClient getRpcClient(String serverUrl) throws ConnectionException {
		XmlRpc.setDebug(debug);
		XmlRpc.setEncoding(charset);
		try {
			return new XmlRpcClient(serverUrl);
		} catch (MalformedURLException e) {
			throw new ConnectionException("指定されたサーバのURLが正しくありません。", e);
		}
	}

}