#ifndef CREATE_SYMBOLICKLINK_H
#define CREATE_SYMBOLICKLINK_H

#define _GNU_SOURCE
#define UNLINK_ERR \
	print_error("unlink", cpinfo->dst, __FILE__, __LINE__, cpdd);\
	cpinfo->dst_para = INITIAL;

#include "ch_time_and_mod_link.h"
#include "gettext.h"
#include "struct_CPDD.h"
#include "struct_OPArg.h"

#include <stdbool.h>

#include <linux/limits.h>

#include <glib.h>

/* 関数プロトタイプ */
static void create_symboliclink(const OPArg *oparg, CPInfo *cpinfo, CPDD *cpdd);
static _Bool c_s(const OPArg *oparg, const CPInfo *cpinfo, CPDD *cpdd);

/*******************************************************************************
シンボリックリンクを作成
*******************************************************************************/
static void create_symboliclink(const OPArg *oparg, CPInfo *cpinfo, CPDD *cpdd)
{
	_Bool is_success = false;

	gboolean is_file_found = g_file_test(cpinfo->dst, G_FILE_TEST_EXISTS);
	gboolean is_link = g_file_test(cpinfo->dst, G_FILE_TEST_IS_SYMLINK);

	if((is_file_found == FALSE) && (is_link == FALSE))
	{
		is_success = c_s(oparg, cpinfo, cpdd);
	}
	else if((is_link == TRUE) && (oparg->WRITE_MODE == OVERWRITE))
	{
		errno = 0;

		if(unlink(cpinfo->dst) == -1)
		{
			UNLINK_ERR
		}
		else
		{
			is_success = c_s(oparg, cpinfo, cpdd);
		}
	}
	else if(is_link == TRUE)
	{
		struct stat stat_src;
		struct stat stat_dst;

		errno = 0;

		if(lstat(cpinfo->src, &stat_src) == -1)
		{
			print_error("lstat", cpinfo->src, __FILE__, __LINE__, cpdd);
		}
		else if(lstat(cpinfo->dst, &stat_dst) == -1)
		{
			if(errno == ENOENT)
			{
				is_success = c_s(oparg, cpinfo, cpdd);
			}
			else
			{
				print_error("lstat", cpinfo->dst, __FILE__, __LINE__, cpdd);
			}
		}
		else
		{
			switch(oparg->WRITE_MODE)
			{
			case SIZE_OR_TIME:
				if(stat_src.st_mtime != stat_dst.st_mtime)
				{
					if(unlink(cpinfo->dst) == -1)
					{
						UNLINK_ERR
					}
					else
					{
						is_success = c_s(oparg, cpinfo, cpdd);
					}
				}
				break;

			case NEW_TIME:
				if(stat_src.st_mtime > stat_dst.st_mtime)
				{
					if(unlink(cpinfo->dst) == -1)
					{
						UNLINK_ERR
					}
					else
					{
						is_success = c_s(oparg, cpinfo, cpdd);
					}
				}
				break;

			case NO_OVERWRITE:
				if(oparg->V == VERBOS)
				{
					printf(_("skip : %s\n"), cpinfo->dst);
					fflush(stdout);
				}
				break;

			default:
				print_error("NULL", cpinfo->src, __FILE__, __LINE__, cpdd);
				break;
			}
		}
	}
	else
	{
		if(oparg->V == VERBOS)
		{
			printf(_("skip : %s\n"), cpinfo->dst);
			fflush(stdout);
		}
	}

	if(is_success == true)
	{
		cpinfo->dst_para = WRITE;
	}
}

/*******************************************************************************
*******************************************************************************/
static _Bool c_s(const OPArg *oparg, const CPInfo *cpinfo, CPDD *cpdd)
{
	_Bool is_success = false;
	int readlink_len;
	char link_buf[PATH_MAX];

	if((readlink_len = readlink(cpinfo->src, link_buf, PATH_MAX)) == -1)
	{
		print_error("readlink", cpinfo->src, __FILE__, __LINE__, cpdd);
	}
	else
	{
		/*
		 * readlinkは終端にヌル文字を付加しない、
		 * という仕様の所為でlink_bufの文字列がおかしくなるバグに悩まされた。
		 * ローカル変数はスコープから抜けると解放されるが、
		 * 即座に解放されるわけではなく（パフォーマンスが落ちるし）、
		 * 領域を使い回すこともしばしばある。
		 * この関数が呼び出されるとlink_bufにシンボリックリンクの内容がコピーされるが、
		 * 関数から抜けてもlink_bufの内容は保持されたままになる。
		 * 次回呼び出されたときにlink_bufの内容が上書きされ、
		 * それが前回よりも短い文字列だった場合、ヌル文字がないので、
		 * 文字列が混ざり合っておかしな事になっていた。
		*/
		link_buf[readlink_len] = '\0';

		if(symlink(link_buf, cpinfo->dst) == -1)
		{
			print_error("symlink", cpinfo->dst, __FILE__, __LINE__, cpdd);
		}
		else
		{
			is_success = true;

			if(cpdd->mk_link_count <= ULLONG_MAX)
			{
				cpdd->mk_link_count++;
			}

			if(oparg->V == VERBOS)
			{
				printf(_("%s -> %s\n"), cpinfo->src, cpinfo->dst);
				fflush(stdout);
			}

			struct stat stat_buf;

			if(lstat(cpinfo->src, &stat_buf) == -1)
			{
				print_error("lstat", cpinfo->src, __FILE__, __LINE__, cpdd);
			}
			else
			{
				ch_time_and_mod_link(&stat_buf, cpinfo, cpdd);
			}
		}
	}

	return is_success;
}

#endif
