/* # skkinput (Simple Kana-Kanji Input)
 * Kinput.c --- Kinput Protocol
 * This file is part of skkinput.
 * Copyright (C) 1997
 * Takashi SAKAMOTO (sakamoto@yajima.kuis.kyoto-u.ac.jp)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with skkinput; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include "commondef.h"
#include "KinputP.h"
#include "protocol.h"
#include "SeparateWin.h"
#include "OverWin.h"
#include "OffWin.h"
#include "attrs.h"
#include "MyDispatch.h"
#include "MyError.h"
#include "resrcs.h"
#include "HistMgr.h"
#include "FontMgr.h"

struct proprec {
  Atom prop ;
  struct proprec *prev ;
} ;

enum {
  PROTOCOL_TYPE_UNKNOWN = 0,
  PROTOCOL_TYPE_KINPUT1,
  PROTOCOL_TYPE_KINPUT2,
  PROTOCOL_TYPE_XIM,
} ;

/*
 * ץȥ
 */
/* KinputWidget Υ٥Ƚؿ*/
static void KIP_ConversionRequestEventHandler
( Widget gw, XEvent *xevent, String *params, Cardinal *num_params ) ;
static void KIP_ConversionEndRequestEventHandler
( Widget gw, XEvent *xevent, String *params, Cardinal *num_params ) ;
static void KIP_ConversionAttributeNotifyEventHandler
( Widget gw, XEvent *xevent, String *params, Cardinal *num_params ) ;
static void KIP_SelectionRequestEventHandler
( Widget gw, XEvent *xevent, String *params, Cardinal *num_params ) ;
static void KIP_SelectionClearEventHandler
( Widget gw, XEvent *xevent, String *params, Cardinal *num_params ) ;

/* KinputWidget ιȤߤȤʤؿ*/
static void KIP_Initialize 
( Widget greq, Widget gnew, ArgList args, Cardinal *num_args ) ;
static void KIP_Realize
( Widget gw, XtValueMask *valueMask, XSetWindowAttributes *attrs ) ;
static Boolean KIP_SetValues
( Widget current, Widget request, Widget new, 
  ArgList args, Cardinal *num_args );
static void KIP_Destroy( Widget gw ) ;

/* ʳδؿ(ޤȤ¡ȤΤ) */
static void KIP_GetAtoms( Widget gw ) ;
static void KIP_SetKinput2Property( Widget gw ) ;
static void KIP_AcceptConversion
( Widget gw, struct KinputClient *client ) ;
static void KIP_RejectConversion
( Widget gw, Window requestor, Atom conversion_atom ) ;
static void KIP_SendEndOfConversion
( struct KinputClient *client ) ;
static int KIP_PrepareConversion
( Widget gw, struct KinputClient *client, Window requestor,
  Atom selection, Atom property ) ;
static int KIP_GetAttributeFromPropertySub2
( struct KinputClient *client, Atom property,
  unsigned long **attributes ) ;
static int KIP_GetAttributeFromPropertySub
( struct KinputClient *client,   Atom property,
  struct proprec *prevprop, int fromevent, int is_widget_ok ) ;
static int KIP_GetAttributeFromProperty
( struct KinputClient *client, Atom property, int is_widget_ok ) ;
static int KIP_GetAttributeFromEvent
( struct KinputClient *client, XClientMessageEvent *xcev ) ;

static struct KinputClient *KIP_PrepareNewClient
( Widget gw, Window req_window, Atom conversion_atom ) ;
static struct KinputClient *KIP_FindKinputClient
( Widget gw, Window req_window ) ;
static void KIP_RemoveKinputClient
( Widget gw, struct KinputClient *client ) ;

/* kinputWidget ¾ͤѰդ륳Хåؿ*/
static void KIP_SendMessageToRequestorCallback
( Widget gw, caddr_t client, caddr_t caller ) ;
static void KIP_EndMessageToRequestorCallback
( Widget gw, caddr_t client, caddr_t caller ) ;
static void KIP_KeyEventToRequestorCallback
( Widget gw, caddr_t client, caddr_t caller ) ;

/*
 * kinput.c ˤ̵¾ͤäƤؿΥץȥ
 */
/* ctext.c */
extern int string2ctext( struct myChar *string, unsigned char *cstr ) ;

/* FontMgr.c */
extern int prepare_font_from_atom
( Display *disp, XFontStruct *skkinput_fonts[ 2 ],
  Atom *fontatoms, Cardinal num_of_fonts ) ;
extern int close_fonts_of_skkinput
( Display *disp, XFontStruct *skkinput_fonts[ 2 ] ) ;
extern int close_one_font_of_skkinput
( Display *disp, XFontStruct *font ) ;

/*
 * Хѿ
 */

#define offset(field)  XtOffsetOf(KinputRec, kinput.field)
#define goffset(field) XtOffsetOf(WidgetRec, core.field)

/* ꥽*/
static XtResource kip_resources[] = {
  /*  kinput protocol ̤ɤ*/
  { XtNenableKinput1, XtCEnableKinput1, XtRBoolean, sizeof(Boolean),
    offset(enable_kinput1), XtRString, (XtPointer)"True" },
  /* SeparateWindow, Over-The-Spot-Window Ĥ륳Хå *
   *  KinputWidget ɤɤΤΤʤȤʤΤǡ*
   * ХåηǿƤ˽򤪴ꤤƤ롣*/
  { XtNsetupInputWindowNotify, XtCCallback, XtRCallback,
    sizeof( caddr_t ), offset( setupInputWindowCallback ), XtRCallback,
    ( caddr_t )NULL },
  /* Kinput Protocol Widget ĤθƽФХå*/
  { XtNserverCloseNotify, XtCCallback, XtRCallback, sizeof( caddr_t ),
    offset( serverCloseCallback ), XtRCallback, ( caddr_t )NULL },
  /* 饤ȤïȤ褦ȤƤ롩 */
  { XtNdestroyWindowEvent, XtCDestroyWindowEvent, XtRImmediate,
    sizeof( XDestroyWindowEvent * ),
    offset( destroyWindowEvent ), XtRImmediate, ( XtPointer )NULL },
  /* ι*/
  { XtNjisyoDirty, XtCJisyoDirty, XtRImmediate, sizeof (int),
    offset(jisyo_dirty), XtRImmediate, (XtPointer) FALSE },
} ;
#undef offset
#undef goffset

/* ٥ȤФ륢Υơ֥롣*/
static XtActionsRec kip_actions_table[] = {
  /* Ѵ׵λ˸ƤФؿ*/
  { "kip-conversion-request",	
    KIP_ConversionRequestEventHandler },
  /* Ѵλ׵λ˸ƤФؿ*/
  { "kip-conversion-end-request",
    KIP_ConversionEndRequestEventHandler },
  /* äϡ°鸫Ƥ͡פȤ׵롣*/
  { "kip-conversion-attribute-notify",
    KIP_ConversionAttributeNotifyEventHandler },
  /* _JAPANESE_CONVERSION ߤȸ줿νԤ*/
  { "kip-selection-request",
    KIP_SelectionRequestEventHandler },
  /* _JAPANESE_CONVERSION 򼺤äνԤ*/
  { "kip-selection-clear",
    KIP_SelectionClearEventHandler },
};

/*- default translation -*/
static char default_kip_translations[] =
"<Message>CONVERSION_REQUEST:          kip-conversion-request()\n\
 <Message>CONVERSION_END_REQUEST:      kip-conversion-end-request()\n\
 <Message>CONVERSION_ATTRIBUTE_NOTIFY: kip-conversion-attribute-notify()\n\
 <SelReq>:                             kip-selection-request()\n\
 <SelClr>:                             kip-selection-clear()";

KinputClassRec kinputClassRec = {
    { /* core fields */
    /* superclass		*/	&widgetClassRec,
    /* class_name		*/	"Kinput",
    /* size			*/	sizeof( KinputRec ),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	KIP_Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	KIP_Realize,
    /* actions			*/	kip_actions_table,
    /* num_actions		*/	XtNumber( kip_actions_table ),
    /* resources		*/	kip_resources,
    /* num_resources		*/	XtNumber( kip_resources ),
    /* xrm_class		*/	NULLQUARK,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	KIP_Destroy,
    /* resize			*/	NULL,
    /* expose			*/	NULL,
    /* set_values		*/	KIP_SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	default_kip_translations,
    /* query_geometry		*/	XtInheritQueryGeometry,
    }
};

WidgetClass kinputWidgetClass = (WidgetClass)&kinputClassRec;

/*
 * KinputWidgetClass νؿ
 */
static void KIP_Initialize 
( Widget greq, Widget gnew, ArgList args, Cardinal *num_args )
{
  KinputWidget w = ( KinputWidget )gnew ;
#if 0
  /* ǥեȤΥեȤ礭ƤɬפʤΤ */
  KIP_GetDefaultFontHeight( gnew ) ;
#endif
  /* Ƥ륯饤Ȥϲʤ*/
  w->kinput.client_list = NULL ;
  /* ѴѤ륢ȥƤ*/
  KIP_GetAtoms( gnew ) ;
  return ;
}

/*
 * Kinput Protocol  Message 򴹤Ѥ Atom ؿ
 */
static void KIP_GetAtoms( Widget gw )
{
  Display *disp  = XtDisplay( gw ) ;
  KinputWidget w = ( KinputWidget )gw ;

  /* Kinput Protocol ɬפȤ Atom 롣*/
#define MakeAtom(s)	XInternAtom(disp,s,False)
  w->kinput.convAtom       = MakeAtom( "_JAPANESE_CONVERSION" ) ;
  w->kinput.oldConvAtom    = MakeAtom( "JAPANESE_CONVERSION" ) ;
  w->kinput.ctextAtom      = MakeAtom( "COMPOUND_TEXT" ) ;
  w->kinput.convStringAtom = MakeAtom( "CONVERSION_STRING" ) ;
  w->kinput.convNotifyAtom = MakeAtom( "CONVERSION_NOTIFY" ) ;
  w->kinput.convEndAtom    = MakeAtom( "CONVERSION_END" ) ;
  w->kinput.convAttrAtom   = MakeAtom( "CONVERSION_ATTRIBUTE" ) ;
#undef MakeAtom
  return ;
}

/*
 * KinputWidget β˸ƤӽФؿ
 */
static void KIP_Realize
( Widget gw, XtValueMask *valueMask, XSetWindowAttributes *attrs )
{
  KinputWidget w        = ( KinputWidget )gw ;
  CoreWidgetClass super =
    ( CoreWidgetClass )XtClass( gw )->core_class.superclass ;

  /* ŬڤʥɥѰդƤʤС *
   * XChangeProperty ưʤȤˤʤ롣*/
  ( *super->core_class.realize )( gw, valueMask, attrs ) ;

  /*  CONVERSION_REQUEST  OWNER ˤʤ롣*/
  XSetSelectionOwner
    ( XtDisplay( gw ), w->kinput.convAtom, XtWindow( gw ), CurrentTime ) ;
  /* ̵ Owner ˤʤ줿ɤȽꤹ롣*/
  if( XGetSelectionOwner( XtDisplay( gw ), w->kinput.convAtom )
      != XtWindow( gw ) ){
    /* Owner ˤʤʤäˤϡѴ׵Ȥ뤳ȤϤä *
     * ǤʤΤǡ褦ʤ餹롩 */
    XtDestroyWidget( gw ) ;
    return ;
  }
  /* kinput1 protocol Υʤˤʤ٤ɤ*/
  if( w->kinput.enable_kinput1 != False ){
    /*  CONVERSION_REQUEST  OWNER ˤʤ롣*/
    XSetSelectionOwner
      ( XtDisplay( gw ), w->kinput.oldConvAtom,
	XtWindow( gw ), CurrentTime ) ;
  }
  /* Property ԤäƤ*/
  KIP_SetKinput2Property( gw ) ;
  return ;
}

/*
 *  KinputWidget λѿͤѹ򤷤Ȥ˸ƤФ
 * 
 * ------
 * ϡWidget سåäˤɤΤ褦ʽ򤹤
 * ΤȤȤȤߤʤȤǤ롣졹ꤹ
 * ݤȻפäƤޤХѿϤʤäƤޤΤĤ
 * Ǥϰܿ㤯ʤ롣
 */
static Boolean KIP_SetValues
( Widget current, Widget request, Widget new,
  ArgList args, Cardinal *num_args )
{
  KinputWidget curw = ( KinputWidget )current ;
  KinputWidget reqw = ( KinputWidget )request ;
  KinputWidget neww = ( KinputWidget )new ;
  struct KinputClient *client ;

  /* ޤï饤ȤȤ褦ȤƤ롩 */
  if( reqw->kinput.destroyWindowEvent != NULL ){
    XDestroyWindowEvent *xdwe   = reqw->kinput.destroyWindowEvent ;
    client = reqw->kinput.client_list ;
    while( client != NULL ){
      if( xdwe->window == client->req_window ||
	  xdwe->window == client->focus_window ){
	/* Ȥ褦ȤƤ뤫ϤϤͤ褦ˤ뤱ɡ */
	/* ̣Τɤʬʤ*/
	remove_allmyeventhandler
	  ( XtDisplay( new ), client->req_window ) ;
	XSafeSelectInput
	  ( XtDisplay( new ), client->req_window, NoEventMask ) ;
	remove_allmyeventhandler
	  ( XtDisplay( new ), client->focus_window ) ;
	XSafeSelectInput
	  ( XtDisplay( new ), client->focus_window, NoEventMask ) ;
	client->req_window = client->focus_window = None ;
	/* 㤢Ĥޤ礦*/
	XtDestroyWidget( client->skkinput ) ;
	break ;
      }
      client = client->next ;
    }
    curw->kinput.destroyWindowEvent =
      neww->kinput.destroyWindowEvent =
      reqw->kinput.destroyWindowEvent = NULL ;
    return( FALSE ) ;
  }
  if( reqw->kinput.jisyo_dirty != curw->kinput.jisyo_dirty ){
    client = neww->kinput.client_list ;
    while( client != NULL ){
      XtVaSetValues
	( client->skkinput, XtNjisyoDirty, reqw->kinput.jisyo_dirty, NULL ) ;
      client = client->next ;
    }
    curw->kinput.jisyo_dirty = neww->kinput.jisyo_dirty = 
      reqw->kinput.jisyo_dirty ;
    return ( FALSE ) ;
  }
  /* Ǥϲ⤹뤳Ȥʤ*/
  return( FALSE ) ;
}

/*
 * KinputWidget ˴˸ƤФؿ
 */
static void KIP_Destroy( Widget gw )
{
  KinputWidget w = ( KinputWidget )gw ;
  struct KinputClient *node, *nextNode ;

  node = w->kinput.client_list ;
  while( node != NULL ){
    nextNode = node->next ;
    /* Ximp ǽƤҥȥפˤʤä*/
    history_destroy( node->req_window ) ;
    /* Υ饤ȤϽߤ롣*/
    XtDestroyWidget( node->skkinput ) ;
    /* Υߥ󥰤ǥΡɤե꡼ƤޤäƤǽ */
    /* ǡnextNode 򤱤ƤΤǤ롣*/
    node = nextNode ;
  }
  /* Ǥˤʤޤ*/
  w->kinput.client_list = NULL ;
  
  /* ʬȤ˴٤ƥåȤ˽׵᤹롣*/
  XtCallCallbacks( gw, XtNserverCloseNotify, NULL ) ;
  /* Kinput Protocol ΥФȤƤܤߤ롣*/
  return ;
}

/*
 * Kinput2 Protocol ΥФˤʤäѰդʤФʤʤǡ 
 * X ΥץѥƥȤϿƤΤ˻Ȥؿ饤ȤϤΥ
 * ѥƥ򸫤ƤȻפ
 */ 
static void KIP_SetKinput2Property( Widget gw )
{
  Display *disp  = XtDisplay( gw ) ;
  Atom server_property, server_type ;
  unsigned long server_profile[ 10 ] ;
  
  /* ץե롩 */
  server_property = XInternAtom( disp, CONVERSION_PROFILE, False ) ;
  /* ѴФ°η */
  server_type     = XInternAtom
    ( disp, CONVERSION_ATTRIBUTE_TYPE, False ) ;
  
  /* ޤץȥΥСꤹġ*/
  server_profile[ 0 ] = MAKE_CONVERSION_ATTRIBUTE
    ( CONVERSION_PROFILE_PROTOCOL_VERSION, 1 ) ;
  server_profile[ 1 ] = XInternAtom
    ( disp, PROTOCOL_VERSION, False ) ;
  /* ˲ǽɽˡˤĤΤ餷롣*/
  server_profile[ 2 ] = MAKE_CONVERSION_ATTRIBUTE
    ( CONVERSION_PROFILE_SUPPORTED_STYLES, 1 ) ;
  server_profile[ 3 ] =
    CONVERSION_NYUURYOKU_HOUSHIKI_SEPARATEMADO |
    CONVERSION_NYUURYOKU_HOUSHIKI_OFFTHESPOT |
    CONVERSION_NYUURYOKU_HOUSHIKI_OVERTHESPOT ;

  /* ϼΣĤݡȤʤФʤʤɡϥץ *
   * ư¸򤷤ΤǡĤˤƤ롣*/
  /* CONVERSION_NYUURYOKU_HOUSHIKI_SEPARATEMADO |
     CONVERSION_NYUURYOKU_HOUSHIKI_OFFTHESPOT |
     CONVERSION_NYUURYOKU_HOUSHIKI_OVERTHESPOT */
  
  /* ξ򥵡ФξȤƥ饤Ȥǽʤ褦 *
   * ϿƤ*/
  XChangeProperty
    ( disp, XtWindow( gw ), server_property, server_type, 32,
      PropModeReplace, ( unsigned char *)server_profile, 4 ) ;
  return ;
}

/*
 * Kinput Protocol Ȥå褿ΤɤȽ
 * ꤹؿ
 */
static int KIP_CorrectClientMessageP( Widget gw, XEvent *xevent )
{
  XClientMessageEvent *ev = &xevent->xclient ;
  KinputWidget         w  = ( KinputWidget )gw ;

#ifdef KIP_DEBUG
  fprintf( stderr, "Get Client Message. I check whether it is correct.\n" ) ;
  fflush( stderr ) ;
#endif
  /* åƤΤɤȽꤹ롣*/
  if( ev->window != XtWindow( gw ) ||
      ev->format != 32 ||
      ( ev->data.l[ 0 ] != w->kinput.convAtom &&
	( w->kinput.enable_kinput1 &&
	  ev->data.l[ 0 ] != w->kinput.oldConvAtom ) ) ){
    /* ʥåƤߤʤΤǡ̵뤷ޤ*/
    return  False ;
  }
  return True ;
}

/*
 * ѴϤ׵᤬褿νԤؿ
 */
static void KIP_ConversionRequestEventHandler
( Widget gw, XEvent *xevent, String *params, Cardinal *num_params )
{
  KinputWidget w           = ( KinputWidget )gw ;
  XClientMessageEvent *xev = &( xevent->xclient ) ;
  Window requestor ;
  Atom conversion_atom, property ;
  struct KinputClient *client ;

#ifdef DEBUG
  fprintf( stderr, "Conversion Request from Window(%ld)\n",
	   ( unsigned long )xevent->xany.window ) ;
#endif
  /* ٥ȤʤΤɤåפޤ*/
  if( !KIP_CorrectClientMessageP( gw, xevent ) ){
#ifdef KIP_DEBUG
    fprintf( stderr, "Illegal conversion-request message(Kinput2)\n" ) ;
#endif
   return ;
  }
  conversion_atom = ( Atom   )xev->data.l[ 0 ] ;
  requestor       = ( Window )xev->data.l[ 1 ] ;
  property        = ( Atom   )xev->data.l[ 4 ] ;

  /*
   * ߡΥ饤ȤϽƤǤϤʤΤ
   */
  if( ( client = KIP_FindKinputClient( gw, requestor ) ) != NULL ){
    KIP_RejectConversion( gw, requestor, conversion_atom ) ;
    return ;
  }
  /* 饤Ȥ뤿Υǡ()뤿ΥΡ */
  /* ɤѰդ롣*/
  client = KIP_PrepareNewClient( gw, requestor, conversion_atom ) ;
  if( client == NULL ){
    /* ꤬­Ƥ硣exit ƤɤȤϻפɡ*/
    KIP_RejectConversion( gw, requestor, conversion_atom ) ;
    return ;
  }
  /*
   *  Ѵ饤ȤФ򳫻Ϥ롣
   */
  if( property != None ){
    client->protocol     = PROTOCOL_TYPE_KINPUT2 ;
    client->window_style = WINDOW_TYPE_SEPARATE ;
    client->event_select = EVENT_HA_INPUTONLY ;
    /* Kinput2 ʤǤϤ°꤬뤫顢ä򸫤˹Ԥ*/
    KIP_GetAttributeFromProperty( client, property, False ) ;
  } else {
    client->protocol     = PROTOCOL_TYPE_KINPUT1 ;
    client->window_style = WINDOW_TYPE_SEPARATE ;
    client->event_select = EVENT_HA_INPUTONLY ;
  }
  /* SKKINPUT WINDOW 롣Ԥ¨¤ȴ롣*/
  if( !KIP_PrepareConversion
      ( gw, client, requestor, conversion_atom, property ) ){
    KIP_RejectConversion( gw, requestor, conversion_atom ) ;
    return ;
  }
  /* INPUT WINDOW ν SKKINPUT WINDOW  *
   * CALLBACK ꤽ¾Ǥ롣*/
  client->target   = w->kinput.ctextAtom ;
  client->property = xev->data.l[ 3 ] ;
  if( client->property == None ){
    client->property = w->kinput.convStringAtom ;
  }
  /* Ѵ饤Ȥ˥ФȤ³ݤ롣*/
  KIP_AcceptConversion( gw, client ) ;
  return ;
}

/*
 * client ׵˽äơѴѤ Window 򵯤ؿ
 */
static int KIP_PrepareConversion
( Widget gw, struct KinputClient *client, Window requestor,
  Atom selection, Atom property )
{
  Window probe_window ;
  unsigned char *class_name ;
  WidgetClass class ;
  HistoryListNode *history ;

  /* ɤ뤫׵᤬ƤΤ򵭲Ƥ*/
  client->req_window   = requestor ;
  if( client->focus_window == None )
    client->focus_window = requestor ;
  /* 뤬Ǥ뤳ȤΤ餻롣*/
  client->probe        = True ;
  client->reserve      = True ;
  client->selection    = selection ;

  /* ҥȥν˺ȡ ɤʤΤ*/
  history = history_setup( client->skkinput, client->req_window ) ;

  /* Ѵ Widget 롣*/
  switch( client->window_style ){
  case WINDOW_TYPE_OVERTHESPOT :
    class_name = "overthespot" ;
    class = overthespotWinWidgetClass ;
    break ;
  case WINDOW_TYPE_OFFTHESPOT :
    class_name = "offthespot" ;
    class = offthespotWinWidgetClass ;
    break ;
  case WINDOW_TYPE_SEPARATE :
  default :
    class_name = "separate" ;
    class = separateWinWidgetClass ;
    break ;
  }
  client->skkinput = XtVaCreateWidget
    ( class_name, class, gw, 
      XtNmappedWhenManaged, False,
      XtNclientWindow, client->req_window,
      XtNconversionHistory, history, NULL ) ;
  if( client->skkinput == NULL ){
    /* ҥȥ롣˺ȡͭäѤʤ *
     * ʤ뤾ȡ*/
    if( history != NULL )
      free( history ) ;
    return False ;
  }
  /* ƤΤʤåȥå׽Ԥ*/
  XtCallCallbacks
    ( gw, XtNsetupInputWindowNotify, client->skkinput ) ;

  /* Хåꡣrealize ƤǤǤʤΤǡ *
   * 뤳Ȥˤʤ롣*/
  XtAddCallback
    ( client->skkinput, XtNfixNotify,
      ( XtCallbackProc )KIP_SendMessageToRequestorCallback,
      ( XtPointer )client ) ;
  XtAddCallback
    ( client->skkinput, XtNendNotify,
      ( XtCallbackProc )KIP_EndMessageToRequestorCallback,
      ( XtPointer )client ) ;
  XtAddCallback
    ( client->skkinput, XtNkeybackNotify,
      ( XtCallbackProc )KIP_KeyEventToRequestorCallback,
      ( XtPointer )client ) ;
  /* Widget  realize Ƥޤ°λƤ*/
  if( client->protocol == PROTOCOL_TYPE_KINPUT2 ){
#if 0
    KIP_SetFocusWindowAttribute( client ) ;
#endif
    /* Kinput2 ʤǤϤ°꤬뤫顢ä򸫤˹Ԥ*/
    KIP_GetAttributeFromProperty( client, property, True ) ;
  }
  if( client->event_select == EVENT_HA_INPUTONLY ){
    /* ׵ᤵ줿ξƩʥɥơ饭 */
    /* ȤäƤ롣*/
    probe_window = client->probe_window = XCreateWindow
      ( XtDisplay( gw ), client->focus_window, 0, 0, 9999, 9999, 0, 0,
	InputOnly, (Visual *)CopyFromParent,
	0L, (XSetWindowAttributes *)NULL);
    XMapWindow( XtDisplay( gw ),  probe_window ) ;
    /* 饤Ȥ destory event ˤդƤʤФʤʤ */
    /* ˾äƤޤƤޤ顣*/
  } else if( client->event_select == EVENT_HA_FOCUS_KARA ){
    probe_window = client->focus_window ;
  } else {
    probe_window = None ;
  }
  /* Probe Window ꤹ롣*/
  XtVaSetValues
    ( client->skkinput, XtNprobeWindow, probe_window, NULL ) ;

  /* Window  Pop Up 롣Exclusive ˤȡ¾ Popup   *
   * ˥٥ȤȤʤʤäƤޤΤǡNonexclusive  *
   * 뤳ȡ*/
  XtRealizeWidget( client->skkinput ) ;
  XFlush( XtDisplay( gw ) ) ;

  /* Event Capture Method νԤ*/
  if( client->event_select == EVENT_HA_INPUTONLY || 
      client->event_select == EVENT_HA_FOCUS_KARA ){
    add_myeventhandler
      ( XtDisplay( gw ), probe_window,
        XtWindow( client->skkinput ), KeyPress, KeyPressMask ) ;
    add_myeventhandler
      ( XtDisplay( gw ), probe_window,
        XtWindow( client->skkinput ), KeyRelease, KeyReleaseMask ) ;
  }
  add_myeventhandler
    ( XtDisplay( gw ), client->req_window, XtWindow( gw ),
      NO_EVENT_HANDLE, StructureNotifyMask ) ;
  return True ;
}

/*
 * Kinput Protocol Ѵ׵ԤؤʸԤδؿ
 *-----
 * ʸ SkkInputWidget ƽФ뤿ᡢcallbackηˤʤ
 * Ƥ롣Ͻ;ϤäѤ뤱ɡĤޤ괺ư
 * ˤʤäƤľޤ
 */
static void KIP_SendMessageToRequestorCallback
( Widget gw, caddr_t client, caddr_t caller )
{
  struct KinputClient *kclient = ( struct KinputClient * )client ;
  struct myChar *text = ( struct myChar * )caller ;

  Atom	proptype ;
  int	propsize ;
  int	propformat ;
  caddr_t propvalue ;
  struct myChar *wptr ;

  /*
   * EUC ɤǻäƤƥȤ Compound Text ؤѴ롣
   */
  for( wptr = text ; !IS_END_OF_STRING( *wptr ) ; wptr ++ ){
    if( IS_ASCII_EQUAL( *wptr, '\r' ) )
      wptr->chara = '\n' ;
  }
  proptype   = kclient->target ;
  propformat = 8 ;
  propsize   = string2ctext( text, NULL ) ;
  if ( propsize <= 0 )
    return ;
  propvalue = ( caddr_t )malloc
    ( propsize * sizeof( unsigned char ) * 4 ) ;
  string2ctext( text, ( unsigned char * )propvalue ) ;
  
  /* Property ˷̤򥻥åȤ */
  XChangeProperty
    ( XtDisplay( gw ), kclient->req_window,
      kclient->property, proptype, propformat,
      PropModeAppend, propvalue, propsize ) ;
  free( propvalue ) ;
  return ;
}					

/*
 *
 */
static void KIP_RemoveKinputClient
( Widget gw, struct KinputClient *client )
{
  KinputWidget w = ( KinputWidget )gw ;
  struct KinputClient *node, *pNode ;
  int i ;

  /* Ρɤ¸ߤʤꥹȤõ롣*/
  pNode = node = w->kinput.client_list ;
  while( node != NULL ){
    if( node == client )
      break ;
    pNode = node ;
    node  = node->next ;
  }
  /* եȤ롣*/
  for( i = 0 ; i < NUMBER_OF_CHARSET ; i ++ ){
    fontMgr_FreeFont( XtDisplay( gw ), client->fontset[ i ] ) ;
    client->fontset[ i ] = NULL ;
  }
  /* ⤽ꥹȤ̵ȤȤͿ줿ΡɤϤʾ */
  /* ؤƤΩʤ*/
  if( node == NULL ){
    free( client ) ;
    return ;
  }
  if( pNode == node ){
    /* ꥹȤƬäǤ*/
    w->kinput.client_list = node->next ;
  } else {
    /* ΥΡɤμΥΡɤϺäȤƤΡɤμΥΡɡ*/
    pNode->next = node->next ;
  }
  free( node ) ;
  return ;
}

static int KIP_EndOfConversionMain( struct KinputClient *client )
{
  /* ѤʤʤäȤΤ餻롣*/
  client->probe        = False ;
  client->reserve      = False ;

  /* λΤ롣Ƥ DestroyNotify ˤä˲Ƥ *
   * ˤƤϤθ¤ǤϤʤξˤ req_window ==    *
   * None ȤʤäƤ롣*/
  if( client->req_window != None ){
    /* 饤Ȥ˽λΤ롣*/
    KIP_SendEndOfConversion( client ) ;
  }
  /* եɥΥ⸫ƤǽΤǾäƤ*/
  if( client->focus_window != None ){
    remove_myeventhandler
      ( XtDisplay( client->protocol_widget ),
        client->focus_window, KeyPress, KeyPressMask ) ;
    remove_myeventhandler
      ( XtDisplay( client->protocol_widget ),
        client->focus_window, KeyRelease, KeyReleaseMask ) ;
  }
  /* InputOnly äˤϡƩ뤬ƤȦʤΤǡ */
  /* äƤ*/
  if( client->probe_window != None ){
    remove_allmyeventhandler
      ( XtDisplay( client->protocol_widget ),
        client->probe_window ) ;
    XSafeDestroyWindow
      ( XtDisplay( client->protocol_widget ), client->probe_window ) ;
    client->probe_window = None ;
  }
  /* ʸѴ׵ԤĤcallback˴*/
  XtRemoveAllCallbacks( client->skkinput, XtNfixNotify ) ;
  /* Ѵ׵ԤѴνλΤcallback˴*/
  XtRemoveAllCallbacks( client->skkinput, XtNendNotify ) ;
  /* Ϥ򥯥饤ȥɥؤ callback ˴*/
  XtRemoveAllCallbacks( client->skkinput, XtNkeybackNotify ) ;

  /* ǽ˥塼եå夷Ƥޤ*/
  XSafeFlush( XtDisplay( client->protocol_widget ) ) ;

  /* Ƥ륯饤ȥꥹȤ*/
  KIP_RemoveKinputClient( client->protocol_widget, client ) ;
  return 0 ;
}

/*
 * Kinput Protocol ǤѴ׵ԤؤѴλãؿ
 *-----
 * դ뤳Ȥϡ Widget  KinputWidget ǤϤʤȤ
 * ȤǤ롣äơ gw 򻲾Ȥ convEndAtom 򸫤ʤƤȤ
 * ޤϤޤޤ(;_;)
 */
static void KIP_EndMessageToRequestorCallback
( Widget gw, caddr_t client, caddr_t caller )
{
  struct KinputClient *kclient = ( struct KinputClient * )client ;
  /* history 򵭲٤ݤ*/
  if( caller != NULL )
    history_close( XtDisplay( kclient->protocol_widget ), caller ) ;
  KIP_EndOfConversionMain( ( struct KinputClient * )client ) ;
  return ;
}

/*
 * Kinput Protocol Ѵλ׵˽äƥ饤ȤȤ³λ
 * ؿ
 */
static void KIP_SendEndOfConversion( struct KinputClient *client )
{
  XEvent event ;
  KinputWidget w = ( KinputWidget )client->protocol_widget ;

  event.xclient.type          = ClientMessage ;
  event.xclient.window        = client->req_window ;
  event.xclient.message_type  = w->kinput.convEndAtom ;
  event.xclient.format        = 32 ;
  event.xclient.data.l[ 0 ]   = ( long )client->selection ;
  event.xclient.data.l[ 1 ]   = ( long )XtWindow
    ( client->protocol_widget ) ;
  event.xclient.data.l[ 2 ]   = 0L ;
  event.xclient.data.l[ 3 ]   = 0L ;
  event.xclient.data.l[ 4 ]   = 0L ;
  
  XSafeSendEvent
    ( XtDisplay( client->protocol_widget ), client->req_window,
     False, NoEventMask, &event ) ;
  return ;
}

/*
 * Kinput Protocl Ѵ׵Ф饤ȤѴ׵μ
 * Τؿ
 */
static void KIP_AcceptConversion
( Widget gw, struct KinputClient *client )
{
  XEvent event ;
  KinputWidget w = ( KinputWidget )gw ;

  event.xclient.type         = ClientMessage ;
  event.xclient.window       = client->req_window ;
  event.xclient.message_type = w->kinput.convNotifyAtom ;
  event.xclient.format       = 32 ;
  event.xclient.data.l[ 0 ]  = ( long )client->selection ;
  event.xclient.data.l[ 1 ]  = ( long )client->target ;
  event.xclient.data.l[ 2 ]  = ( long )client->property ;
  event.xclient.data.l[ 3 ]  = ( long )XtWindow( client->skkinput ) ;
  event.xclient.data.l[ 4 ]  = 0L ;

  XSendEvent
    ( XtDisplay( gw ), client->req_window, False, NoEventMask, &event ) ;
  return ;
}

/*
 * Kinput Protocl Ѵ׵Ф饤ȤѴ׵μ
 * Τؿ
 */
static void KIP_RejectConversion
( Widget gw, Window requestor, Atom conversion_atom )
{
  XEvent event ;
  KinputWidget w = ( KinputWidget )gw ;

  event.xclient.type          = ClientMessage ;
  event.xclient.window        = requestor ;
  event.xclient.message_type  = w->kinput.convNotifyAtom ;
  event.xclient.format        = 32 ;
  event.xclient.data.l[ 0 ]   = conversion_atom ;
  event.xclient.data.l[ 1 ]   = None ;
  event.xclient.data.l[ 2 ]   = 0L ;
  event.xclient.data.l[ 3 ]   = 0L ;
  event.xclient.data.l[ 4 ]   = 0L ;

  XSendEvent
    ( XtDisplay( gw ), requestor, False, NoEventMask, &event ) ;
  return ;
}

static struct KinputClient *KIP_PrepareNewClient
( Widget gw, Window req_window, Atom conversion_atom )
{
  KinputWidget w = ( KinputWidget )gw ;
  struct KinputClient *client ;
  int i ;

  client = ( struct KinputClient *)malloc
    ( sizeof( struct KinputClient ) ) ;
  if( client == NULL ){
    return NULL ;
  }
  client->probe = False ;
  client->reserve = False ;
  client->chat_adapter = False ;
  client->eggnl = False ;
  client->req_window = req_window ;
  client->target = client->property = client->selection = None ;
  client->probe_window = None ;
  client->focus_window = None ;
  client->protocol_widget = gw ;
  client->attribute_mask = 0L ;

  /* եȥåȤνԤ*/
  for( i = 0 ; i < NUMBER_OF_CHARSET ; i ++ )
    client->fontset[ i ] = NULL ;

  /* ꥹȤˤĤʤǤ*/
  client->next = w->kinput.client_list ;
  w->kinput.client_list = client ;
  return client ;
}

/*
 * ׵ᤷƤ kinput protocol ǽƤΤǤ礦ȥ
 * ؿ
 */
static struct KinputClient *KIP_FindKinputClient
( Widget gw, Window req_window )
{
  KinputWidget w = ( KinputWidget )gw ;
  struct KinputClient *client ;

  client = w->kinput.client_list ;
  while( client != NULL ){
    if( client->req_window == req_window )
      return client ;
    client = client->next ;
  }
  return client ;
}

/*
 * ѴλåνԤؿ
 */
static void KIP_ConversionEndRequestEventHandler
( Widget gw, XEvent *xevent, String *params, Cardinal *num_params )
{
  XClientMessageEvent *cev = &( xevent->xclient ) ;
  struct KinputClient *client ;

  /* ʥåŤ̵뤷ƺ夲롣*/
  if( !KIP_CorrectClientMessageP( gw, xevent ) )
    return ;

  client = KIP_FindKinputClient( gw, ( Window )cev->data.l[ 1 ] ) ;
  if( client == NULL ){
    /* ΤʤͤΤä㤤ʤΤǡŤ̵뤷ƺ *
     * 夲ޤ*/
    return ;
  }
  /* ѴĤƤλƤ*/
  XtDestroyWidget( client->skkinput ) ;
  return ;
}

/*
 * Ѵ饤Ȥ°ѹΤ褿νԤؿ
 */
static void KIP_ConversionAttributeNotifyEventHandler
( Widget gw, XEvent *xevent, String *params, Cardinal *num_params )
{
  XClientMessageEvent *xcev = &( xevent->xclient ) ;
  struct KinputClient *client ;

  /* ʥåŤ̵뤷ƺ夲롣*/
  if( !KIP_CorrectClientMessageP( gw, xevent ) )
    return ;

  client = KIP_FindKinputClient( gw, ( Window )xcev->data.l[ 1 ] ) ;
  if( client == NULL ){
    /* ΤʤͤΤä㤤ʤΤǡŤ̵뤷ƺ *
     * 夲ޤ*/
    return ;
  }
  /* Kinput2 Protocol ʳǤΤ褦׵ФƤͤ顢 *
   * ϥߡʤΤ̵뤷ޤ*/
  if( client->protocol != PROTOCOL_TYPE_KINPUT2 || !client->probe )
    return ;
  /* Ȥߥ󥰤ˤäƤϡ褿 Window 뤫顢
     ;Ȥ XErrorEvent ȯ skkinput Τ򤱤롣*/
  KIP_GetAttributeFromEvent( client, xcev ) ;
  return ;
}

/*
 * ¾ѴФ selection ׵򤷤ƤνԤؿ
 */
static void KIP_SelectionRequestEventHandler
( Widget gw, XEvent *xevent, String *params, Cardinal *num_params )
{
  XSelectionRequestEvent *xsrev = &( xevent->xselectionrequest ) ;
  XEvent xev ;

  xev.xselection.type      = SelectionNotify ;
  xev.xselection.requestor = xsrev->requestor ;
  xev.xselection.selection = xsrev->selection ;
  xev.xselection.target    = xsrev->target ;
  xev.xselection.property  = None ;
  xev.xselection.time      = xsrev->time ;

  XSendEvent
    ( xsrev->display, xsrev->requestor, False, NoEventMask, &xev ) ;
  return ;
}

/*
 * ѴФ븢¤򼺤äνԤؿ
 */
static void KIP_SelectionClearEventHandler
( Widget gw, XEvent *xevent, String *params, Cardinal *num_params )
{
  /* ʬȤ˲ƽλ롣λʬȤäƤ  * 
   * Хå˴ʤФʤʤɡġ */
#ifdef DEBUG
  fprintf( stderr, "(Kinput2)Selection Lost....\n" ) ;
#endif
  XtDestroyWidget( gw ) ;
  return ;
}

/*
 *  XtSetValues 򤷤Ѵ饤Ȥѹ줿ʬؼ
 * ٤ǤȻϻפ
 */
static struct ConvAttrsMesg KIP_ReadAttribute
( struct KinputClient *client, unsigned long *attributes, 
  struct proprec *precp, int fromevent, int is_widget_ok )
{
	int attribute_code   = CODE_OF_ATTRIBUTE  ( *attributes ) ;
	int attribute_length = LENGTH_OF_ATTRIBUTE( *attributes ) ;
	struct ConvAttrsMesg camsg ;
	
	camsg.mask = 0L ;
#ifdef DEBUG
	fprintf( stderr, "Attribute Code : %d\n", attribute_code ) ;
#endif
	attributes ++ ;
	switch( attribute_code ){
		/* °ѹϼ¤ϤʤƤʤä*/
	case CONVERSION_ATTRIBUTE_NASHI :
		break ;
	case CONVERSION_ATTRIBUTE_HA_INDIRECT :
		if( attribute_length != 1 )
			break ;
		/* ѹ줿°ǡ礭Ȥѹ줿°ǡ *
		 * ʣȤ (ξˤϤΥ٥ȤʣäƤ⤤ *
		 * ) ˤ°ǡϥץѥƥ졢Indirect Attribute ε*
		 * ǽȤäƤΥץѥƥꤹФ褤*/
		KIP_GetAttributeFromPropertySub
			( client, attributes[ 0 ], precp, fromevent, is_widget_ok ) ;
		break ;
	case CONVERSION_ATTRIBUTE_FOCUS_WINDOW :
		if( attribute_length != 1 )
			break ;
		camsg.mask |= CAFocusWindow ;
		camsg.value.focus_window = attributes[ 0 ] ;
		client->focus_window = camsg.value.focus_window ;
		break ;
	case CONVERSION_ATTRIBUTE_BASHO_SHITEI :
		if( attribute_length != 1 )
			break ;
		camsg.mask |= CASpotLocation ;
		camsg.value.spot_x = USHORT_UPPER16BIT( attributes[ 0 ] ) ;
		camsg.value.spot_y = USHORT_LOWER16BIT( attributes[ 0 ] ) ;
#ifdef DEBUG
		fprintf
			( stderr, "Basho Shitei : (x,y) = (%d, %d)\n", 
			  camsg.value.spot_x, camsg.value.spot_y ) ;
#endif
		break ;
	case CONVERSION_ATTRIBUTE_CLIENT_AREA :
		if( attribute_length != 2 )
			break ;
		camsg.mask |= CAClientArea;
		camsg.value.client_area.x = 
			( unsigned short )ULONG_UPPER16BIT( attributes[ 0 ] ) ;
		camsg.value.client_area.y =
			( unsigned short )ULONG_LOWER16BIT( attributes[ 0 ] ) ;
		camsg.value.client_area.width  = ULONG_UPPER16BIT( attributes[ 1 ] ) ;
		camsg.value.client_area.height = ULONG_LOWER16BIT( attributes[ 1 ] ) ;
#ifdef DEBUG
		fprintf
			( stderr, "Client no ichi to ookisa : (x,y) = (%d, %d), %d x %d\n", 
			  camsg.value.client_area.x, camsg.value.client_area.y,
			  camsg.value.client_area.width, camsg.value.client_area.height ) ;
#endif
		break ;
	case CONVERSION_ATTRIBUTE_NYUURYOKU_HOUSHIKI :
		if( fromevent )
			break ;
		if( attribute_length != 1 )
			break ;
		if( attributes[ 0 ] ==
			CONVERSION_NYUURYOKU_HOUSHIKI_OVERTHESPOT ){
			client->window_style = WINDOW_TYPE_OVERTHESPOT ;
		} else if( attributes[ 0 ] ==
				   CONVERSION_NYUURYOKU_HOUSHIKI_OFFTHESPOT ){
			client->window_style = WINDOW_TYPE_SEPARATE ;
		} else {
			client->window_style = WINDOW_TYPE_SEPARATE ;
		}
		break ;
	case CONVERSION_ATTRIBUTE_EVENT_NO_TOBASHIKATA :
		if( fromevent )
			break ;
		if( attribute_length != 1 )
			break ;
		switch( attributes[ 0 ] ){
		case CONVERSION_EVENT_TOBASHI_NASHI :
			client->event_select = EVENT_NO_SHORINASHI ;
			break ;
		case CONVERSION_EVENT_TOBASHI_FOCUS_NO_HOUHE :
			client->event_select = EVENT_HA_FOCUS_KARA ;
			break ;
		case CONVERSION_EVENT_TOBASHI_ZENBU_SKKINPUT_HE :
		default :
			client->event_select = EVENT_HA_INPUTONLY ;
			break ;
		}
		break ;
		/* Ѵ饤ȤѤƤ륫顼ޥåס*/
	case CONVERSION_ATTRIBUTE_COLORMAP :
		if( attribute_length != 1 )
			break ;
		camsg.mask |= CAColormap ;
		camsg.value.colormap = attributes[ 0 ] ;
		break ;
		/* Ѵ饤Ȥ fore/background 顼ξ*/
	case CONVERSION_ATTRIBUTE_COLOR :
		if( attribute_length != 2 )
			break ;
		camsg.mask |= CAColor ;
		camsg.value.foreground = attributes[ 0 ] ;
		camsg.value.background = attributes[ 1 ] ;
		break ;
		/* Ѵ饤ȤΥեȤξ*/
	case CONVERSION_ATTRIBUTE_FONT_ATOMS :
		/* ꤹ٤ΤμΤ¸ߤʤˤȴ롣*/
		if( attribute_length < 1 )
			break ;
		fontMgr_PrepareFontByAtom
			( XtDisplay( client->protocol_widget ), client->fontset,
			  ( Atom *)attributes, ( Cardinal )attribute_length ) ;
		camsg.mask |= CAFonts ;
		camsg.value.fontset = client->fontset ;
		break ;
		/* off-the-spot ڤ over-the-spot Ѥ뤳Ȥˤʤ륹ơ
		   ΰλĤʤΤover-the-spot Ǥϼ¤ϻȤäƤʤä
		   롣(^^;;; */
	case CONVERSION_ATTRIBUTE_STATUS_AREA :
		/* ꤹ٤ΤμΤ¸ߤʤˤȴ롣*/
		if( attribute_length != 2 )
			break ;
		camsg.mask |= CAStatusArea;
		camsg.value.status_area.x = USHORT_UPPER16BIT( attributes[ 0 ] ) ;
		camsg.value.status_area.y = USHORT_LOWER16BIT( attributes[ 0 ] ) ;
		camsg.value.status_area.width  = ULONG_UPPER16BIT( attributes[ 1 ] ) ;
		camsg.value.status_area.height = ULONG_LOWER16BIT( attributes[ 1 ] ) ;
		break ;
		/* եȤι⤵ꤹΤȻפΤեȤѹ
		   Ƥ⽤뵤̵ۤΤǡ¤̵뤵Ƥޤѥ᡼
		   ʤäƤ롣*/
	case CONVERSION_ATTRIBUTE_LINE_SPACING :
		/* ꤹ٤ΤμΤ¸ߤʤˤȴ롣*/
		if( attribute_length < 1 )
			break ;
		camsg.mask |= CALineSpacing  ;
		camsg.value.linespacing = attributes[ 0 ] ;
		break ;
		/* 鲼Υȥӥ塼Ȥɬפʶ̤ǤʤΤǡ */
		/* 뤳Ȥˤ롣*/
	case CONVERSION_ATTRIBUTE_BACKGROUND_PIXMAP :
	case CONVERSION_ATTRIBUTE_CURSOR :
	default :
		break ;
	}
	return camsg ;
}

/*
 *  XtSetValues 򤷤Ѵ饤Ȥѹ줿ʬؼ
 * ٤ǤȻϻפ
 */
static int KIP_SetAttribute
( struct KinputClient *client, unsigned long *attributes, 
  struct proprec *precp, int fromevent, int really_set_p )
{
  KinputWidget w = ( KinputWidget )client->protocol_widget ;
  struct ConvAttrsMesg camsg ;
  Arg arg[ 2 ] ;

  camsg = KIP_ReadAttribute
    ( client, attributes, precp, fromevent, really_set_p ) ;
  /* ꤵ줿νԤ*/
  if( camsg.mask && really_set_p ){
    w->kinput.camsg = camsg ;
    XtSetArg( arg[ 0 ], XtNconversionAttribute, &w->kinput.camsg ) ;
    XtSetValues( client->skkinput, arg, 1 ) ;
  }
  return True ;
}

static int KIP_GetAttributeFromPropertySub2
( struct KinputClient *client, Atom property, unsigned long **attributes )
{
  KinputWidget w = ( KinputWidget )client->protocol_widget ;
  Atom type ;
  int format ;
  unsigned long nums, bytesafter ;

  /*
   * kinput2 Τޤޤ˸ġǤ⡢ǥ顼Ф줿
   * н褷褦̵ǤͤɤΤʡ
   */
  XSafeGetWindowProperty
    ( XtDisplay( client->protocol_widget ), client->req_window, property,
      0L, 1000L, False, w->kinput.convAttrAtom, 
      &type, &format, &nums, &bytesafter,
      ( unsigned char **)attributes ) ;

  /* ϿƤ򸫤Ƥ롣*/
  if( format != 32 || type != w->kinput.convAttrAtom ){
    /* free ɤΤ XFree ʤܤʵ뤬ġ*/
    if( *attributes != NULL )
      XFree( ( char *)*attributes ) ;
    return 0 ;
  }
  return nums ;
}

static int KIP_GetAttributeFromPropertySub
( struct KinputClient *client,   Atom property,
  struct proprec *prevprop, int fromevent, int really_set_p )
{
  unsigned long *attributes, *attr_ptr ;
  int total_length_of_attributes ;
  struct proprec node ;

  node.prop = property ;
  node.prev = prevprop ;
  while( prevprop != NULL ){
    if( prevprop->prop == property ){
      return False ;
    }
    prevprop = prevprop->prev ;
  }

  attributes = NULL ;
  if( ( total_length_of_attributes = 
	KIP_GetAttributeFromPropertySub2
	( client, property, &attributes ) ) <= 0 )
    return False ;

  attr_ptr = attributes ;
  while( total_length_of_attributes > 0 ){
    int length_of_attributes =
      LENGTH_OF_ATTRIBUTE( attr_ptr[ 0 ] ) + 1 ;
    if( length_of_attributes > total_length_of_attributes )
      break ;
    KIP_SetAttribute( client, attr_ptr, &node, fromevent, really_set_p ) ;
    total_length_of_attributes -= length_of_attributes ;
    attr_ptr += length_of_attributes ;
  }
  /* XGetWindowProperty η̤ʤΤ XFree 롣free ǤϤʤ*/
  XFree( attributes ) ;

  return True ;
}

/*
 * Property ˤäꤵƤѴ饤Ȥ°Фؿ
 *-----
 * ࡣå XtSetValues ꤿΤԹˤ꤬¸
 * ߤʤȤΤ롣ɤ褦ġϰֺǽ 
 * Widget ˸ƤФȡĤɤ褦ʤΤǤ롣ܤ̵
 * 뤷ƣܤפȤ̵Ǥ⤷ޤ礦
 */
static int KIP_GetAttributeFromProperty
( struct KinputClient *client, Atom property, int really_set_p )
{
  KIP_GetAttributeFromPropertySub
    ( client, property, ( struct proprec *)NULL, False, really_set_p ) ;
  return 0 ;
}

static int KIP_GetAttributeFromEvent
( struct KinputClient *client, XClientMessageEvent *xcev )
{
  unsigned long *data ;
  int            length_of_attribute ;

  /* ٥ȤΤդƤ°ȴФ*/
  data = ( unsigned long * )( &( xcev->data.l[ 2 ] ) ) ;
  length_of_attribute = LENGTH_OF_ATTRIBUTE( data[ 0 ] ) ;
  /* ɤ٤ˣİʾꤹȥץȥȿ餷*/
  if( length_of_attribute >= 3 )
    return False ;
  /* °Ԥ餷*/
  KIP_SetAttribute( client, data, NULL, True, True ) ;
  return True ;
}

/*
 * 򲡤Ȥ٥Ȥ̤ؤžΤѤؿ
 */
static void KIP_KeyEventToRequestorCallback
( Widget gw, caddr_t client, caddr_t caller )
{
  struct KinputClient *kclient = ( struct KinputClient *)client ;
  XEvent *xevent = ( XEvent *)caller ;

  /* եƤؤȥå롣*/
#if 0
  xevent->xkey.window    = kclient->focus_window ;
#else
  xevent->xkey.window    = kclient->req_window ;
#endif
  xevent->xkey.subwindow = None ;

  XSendEvent
    ( XtDisplay( gw ), xevent->xkey.window,
      False, NoEventMask, xevent ) ; 
  return ;
}
