/* Copyright (c) 2020-2023 The Creators of Simphone

   See the file COPYING.LESSER.txt for copying permission.
*/

#ifndef QTFIX_H
#define QTFIX_H

#include <QCoreApplication>
#include <QStatusBar>
#include <QTableWidget>
#include <QCheckBox>
#include <QRadioButton>
#include <QComboBox>
#include <QSpinBox>
#include <QPushButton>
#include <QSplitter>
#include <QInputDialog>
#include <QMainWindow>
#include <QMessageBox>
#include <QProxyStyle>
#include <QAction>
#include <QElapsedTimer>

class fixMessageBox : public QMessageBox
{
  Q_OBJECT

  typedef QMessageBox Parent;

public:
  explicit fixMessageBox(Icon icon, const QString & title, const QString & text, StandardButtons buttons = NoButton,
                         QWidget * parent = 0, Qt::WindowFlags f = Qt::Dialog | Qt::MSWindowsFixedSizeDialogHint);

  QPushButton * addNocloseButton(const QString & text, ButtonRole role);

public slots:
  void done(int r) Q_DECL_OVERRIDE;

private:
  QList<QAbstractButton *> m_buttons;
};

class fixProxyStyle : public QProxyStyle
{
  int pixelMetric(PixelMetric metric, const QStyleOption * option, const QWidget * widget) const Q_DECL_OVERRIDE;
  QIcon standardIcon(StandardPixmap icon, const QStyleOption * option, const QWidget * widget) const Q_DECL_OVERRIDE;
  void drawPrimitive(PrimitiveElement element, const QStyleOption * option,
                     QPainter * painter, const QWidget * widget) const Q_DECL_OVERRIDE;
  void drawControl(ControlElement element, const QStyleOption * option,
                   QPainter * painter, const QWidget * widget) const Q_DECL_OVERRIDE;
  QSize sizeFromContents(ContentsType type, const QStyleOption * option,
                         const QSize & size, const QWidget * widget) const Q_DECL_OVERRIDE;
};

class qtfix
{
  Q_DECLARE_TR_FUNCTIONS(Main)

public:
  static bool hasAltModifier(QKeyEvent * keyEvent, Qt::KeyboardModifiers modifiers);
  static int hasSelectedRows(const QItemSelectionModel * model);

  static QString getStrings(const char * nick) { return QString(nick).replace('\t', ' '); }
  static QString getString(const char * nick) { return getStrings(nick).replace('\n', ' '); }
  static QString getString(QString text, const char * regexp) { return text.replace(QRegExp(regexp), " "); }

  static QString getVersion();
  static char ** getArguments() { return mc_arguments; }
  static const char * getStyleArgument() { return mc_styleArgument; }
  static const char * getStyleName() { return mc_styleName.data(); }
  static QString getStyleSetting() { return mc_styleArgument ? mc_styleArgument : mc_setStyle; }
  static QString getStyleOverride();
  static QString getStyleSheetArgument() { return mc_styleSheetArgument; }
  static QString getStyleSheet() { return mc_styleSheetArgument.isEmpty() ? mc_setStyleSheet : mc_styleSheetArgument; }
  static char getStyle() { return mc_style; }
  static bool setStyle(const QString & style, const QString & styleSheet);

  static const char * getLanguage() { return mc_language; }
  static bool setLanguage(const QString & language);

  static qint64 getOverrideCursor() { return mc_elapsed.elapsed(); }
  static qint64 setOverrideCursor(bool start);

  static QColor getColor() { return mc_color; }
  static void setColor(const QColor & color) { mc_color = color; }

  static void setPixmapColors(const QStringList & pixmapColors) { mc_pixmapColors = pixmapColors; }
  static void setMenuIndicators(const QStringList & checkmarks) { mc_indicators = checkmarks; }

  static bool setGeometry(QWidget * window, QRect geometry, bool proper);
  static void setCheckBox(QMessageBox * messageBox, QCheckBox * checkBox);

  static int getFontSize() { return mc_pointSize; };
  static void setFontSize(int pointSize, int maxPointSize) { mc_pointSize = pointSize, mc_maxPointSize = maxPointSize; }
  static void setFont(char mode);

  static QStringList fixArguments(int & argc, char **& argv);
  static const char * fixApplication(const QString & language);

  static QString fixStyleSheet(QString styleSheet);
  static void fixStyleSheet(QWidget * widget, const QString & styleSheet = QString());

  static QString fixColorString(QString string, const QByteArray & color);
  static QColor fixDefaultColor(const QColor & color, const QColor & defaultColor);

  static void fixStatusBar(QStatusBar * statusBar);
  static void fixTabWidget(QTabWidget * tabWidget, const QFont & font);

  static void fixTableView(QTableView * tableView);
  static void fixTableWidgetHorizontalHeader(QTableWidget * tableWidget, QStringList & labels);

  static QFont * createPasswordFont(int pointSize /* = 0*/, int * fontId);
  static void removePasswordFont(int fontId);

  static QFont fixFontBold(QFont font) { return font.setBold(true), font; }
  static QFont fixFontSize(QFont oldFont, const char * newFontClass, const char * styles);
  static void fixPointSize(QWidget * widget, int pointSize);

  static QPixmap fixPixmapSize(const QPixmap & pixmap, int fontH, bool nodownscale);
  static QPixmap fixPixmapColor(const QPixmap & pixmap, const QColor & color);

  static void fixCheckBoxIcon(QCheckBox * checkBox, const QFont & font);
  static void fixRadioButtonIcon(QRadioButton * radioButton, const QFont & font);
  static void fixComboBoxIcon(QComboBox * comboBox, const QFont & font);
  static void fixSpinBoxIcon(QSpinBox * spinBox, const QFont & font);
  static void fixPushButtonIcon(QPushButton * pushButton, const QFont & font);

  static void fixMenuBar(QMenuBar * menuBar, int width, QLayout * layout);
  static void fixMenuIndicator(QAction * item, QMenu * menu, const QFont & font);

  static QAction * addMenuItem(QMenu * menu, const QString & text, bool checked);
  static QAction * addMenuItem(QMenu * menu, const QString & text, QActionGroup * group = 0);
  static QAction * addMenuItem(QMenu * menu, const QString & text, const QPixmap & icon);

  static void fixIconSize(QWidget * widget, const QFont & font);
  static void fixSplitterHandle(QSplitter * splitter, const QFont & font, char enable);

  static QObject * findChild(QObject * parent, const char * type);
  static QString execInputDialog(QWidget * parent, const QString & title, const QString & label,
                                 const QString & text, bool * ok);
  static int execInputDialog(QInputDialog * input);

  static bool execMessageBox(QMessageBox::Icon icon, const char * title, const QString & message,
                             QWidget * parent, const QString & checkBox); // no title on Mac OS
  static bool execMessageBox(QCheckBox * checkBox, const QString & title, const QString & message,
                             QWidget * parent, bool yes);
  static bool execMessageBox(bool critical, const QString & title, const QString & message,
                             QWidget * parent, const QString & checkBox = QString());
  static void updateMessageBox(QMessageBox * messageBox, const QString & text);

  static void resizeMaximizedWindow(QMainWindow * window, int width = -1);
  static bool showActivateWindow(QWidget * window, bool spontaneous);
  static void showMaximizedWindow(QWidget * window);
  static void showMinimizedWindow(QWidget * window, int delay);
  static Qt::WindowStates hideMinimizedWindow(QWidget * window);
  static void fixWindowPosition(QWidget * window);

private:
  static char ** mc_arguments;
  static const char * mc_styleArgument;
  static char mc_style;
  static QByteArray mc_styleName;
  static QString mc_styleSheetArgument;
  static QString mc_setStyle;
  static QString mc_setStyleSheet;
  static int mc_pointSize;
  static int mc_maxPointSize;
  static char mc_language[64];
  static QLocale mc_locale;
  static QElapsedTimer mc_elapsed;
  static QColor mc_color;
  static QStringList mc_pixmapColors;
  static QStringList mc_indicators;
};

#endif
