﻿Option Explicit On
Option Strict On
Option Infer Off

Imports System.Net

''' <summary>
''' OMRON PLC, FinsUDP protocol, Binary, UDP socket communication
''' </summary>
''' <remarks></remarks>
Public Class FinsUdp
    Inherits PlcProtocol

    Private conn As UdpTermSync
    Private Header As String = "@00"

    Private Const CommandReadBits1 As String = "RR"     ' Internal relay (bit)
    Private Const CommandReadBits2 As String = "RL"
    Private Const CommandReadWords As String = "RD"     ' Data memory (bit)
    Private Const CommandWriteBits1 As String = "WR"    ' Internal relay (word)
    Private Const CommandWriteBits2 As String = "WL"
    Private Const CommandWriteWords As String = "WD"    ' Data memory (word)

    Private Const ICF As Integer = &H80
    Private Const RSV As Integer = &H0
    Private Const GCT As Integer = &H2
    Private Const DNA As Integer = &H0
    Private Const DA1 As Integer = &H3
    Private Const DA2 As Integer = &H0
    Private Const SNA As Integer = &H0
    Private Const SA1 As Integer = &HC
    Private Const SA2 As Integer = &H0
    Private Const SID As Integer = &H0

    Private ClientNodeNo As Integer = 66
    Private ServerNodeNo As Integer = 62

    ''' <summary>
    ''' Constructor
    ''' </summary>
    ''' <param name="Q_Protocol"></param>
    ''' <remarks></remarks>
    Public Sub New(ByVal Q_Protocol As IEnumerable(Of XElement))
        If Q_Protocol.<DestIp>.Value Is Nothing Then
            MessageBox.Show("[FinsUdp] DestIp in <DeviceArray><Device><Protocol>.<DestIp> not found!")
            End
        ElseIf Q_Protocol.<DestPort>.Value Is Nothing Then
            MessageBox.Show("[FinsUdp] DestPort in <DeviceArray><Device><Protocol>.<DestPort> not found!")
            End
        ElseIf Q_Protocol.<RecvPort>.Value Is Nothing Then
            MessageBox.Show("[FinsUdp] RecvPort in <DeviceArray><Device><Protocol>.<RecvPort> not found!")
            End
        ElseIf Q_Protocol.<ClientNodeNo>.Value Is Nothing Then
            MessageBox.Show("[FinsUdp] ClientNodeNo in <DeviceArray><Device><Protocol>.<ClientNodeNo> not found!")
            End
        ElseIf Q_Protocol.<ServerNodeNo>.Value Is Nothing Then
            MessageBox.Show("[FinsUdp] ServerNodeNo in <DeviceArray><Device><Protocol>.<ServerNodeNo> not found!")
            End
        End If

        conn = New UdpTermSync(IPAddress.Parse(Q_Protocol.<DestIp>.Value), CInt(Q_Protocol.<DestPort>.Value), CInt(Q_Protocol.<RecvPort>.Value))
        Me.ClientNodeNo = CInt(Q_Protocol.<ClientNodeNo>.Value)
        Me.ServerNodeNo = CInt(Q_Protocol.<ServerNodeNo>.Value)
    End Sub

    ''' <summary>
    ''' Connect to PLC
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function Connect() As Integer
        Dim result As Integer = conn.Connect()
        If result = -1 Then
            err = "Connection to PLC is failed."
        End If

        Connect = result
    End Function

    ''' <summary>
    ''' Disconnect from PLC
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function Disconnect() As Integer
        Dim result As Integer = conn.CloseSocket()
        Return result
    End Function

    ''' <summary>
    ''' Make header
    ''' </summary>
    ''' <param name="BitOrWord"></param>
    ''' <param name="WriteCount"></param>
    ''' <param name="SendData"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Private Function MakeHeader(ByVal BitOrWord As String, ByVal WriteCount As Integer, ByRef SendData As Byte()) As Integer
        Dim MsgLen As Integer

        If BitOrWord Is Nothing Then
            MsgLen = 10 + 2
        Else
            If WriteCount = 0 Then
                MsgLen = 10 + 8
            Else
                If BitOrWord.Equals("BIT") Then
                    MsgLen = 10 + 8 + WriteCount
                ElseIf BitOrWord.Equals("WORD") Then
                    MsgLen = 10 + 8 + WriteCount * 2
                Else
                    err = "UnknownDeviceType"
                    Return -1
                End If
            End If
        End If

        ReDim SendData(MsgLen - 1)

        SendData(0) = ICF
        SendData(1) = RSV
        SendData(2) = GCT
        SendData(3) = DNA
        SendData(4) = Convert.ToByte(ServerNodeNo) 'DA1
        SendData(5) = DA2
        SendData(6) = SNA
        SendData(7) = Convert.ToByte(ClientNodeNo) 'SA1
        SendData(8) = SA2
        SendData(9) = SID

        Return 0
    End Function

    ''' <summary>
    ''' Change PLC running mode
    ''' </summary>
    ''' <param name="Mode"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function ChangeMode(ByVal Mode As Integer) As Integer
        Try
            Dim SendData() As Byte = New Byte() {}
            Dim result As Integer = MakeHeader(Nothing, Nothing, SendData)
            Dim BytesPerData As Integer = 1

            SendData(10) = &H4          'Command code 1/2

            If Mode = 0 Then            'Stop PLC
                SendData(11) = &H2      'Command Code 2/2
            Else                        'Run PLC
                SendData(11) = &H1      'Command code 2/2
            End If

            conn.Send(SendData)


            Dim RecvLengthWish As Integer = 14
            Dim RecvDataByte() As Byte = New Byte() {}
            result = conn.RecvBinary(RecvDataByte)                              ' Receive reply message

            If result <> 0 Then
                err = conn.GetError()
                Return -1
            Else
                If RecvDataByte.GetLength(0) <> RecvLengthWish Then             ' Check data length error
                    err = "RecvLengthError"
                    Return -1
                Else
                    If RecvDataByte(12) <> &H0 Or RecvDataByte(13) <> &H0 Then  ' Check termination code
                        err = "ResultCodeError"
                        Return -1
                    Else
                        Return 0
                    End If
                End If
            End If
        Catch ex As Exception
            err = ex.Message
            Return -1
        End Try
    End Function

    ''' <summary>
    ''' Read bit data from PLC
    ''' </summary>
    ''' <param name="DeviceName"></param>
    ''' <param name="StartDevice"></param>
    ''' <param name="StartBitNo"></param>
    ''' <param name="DeviceCount"></param>
    ''' <param name="RecvDataRet"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function ReadBits(ByVal DeviceName As String, ByVal StartDevice As UInteger, ByVal StartBitNo As Integer, ByVal DeviceCount As Integer, ByRef RecvDataRet As UInt16()) As Integer
        Try
            Dim SendData() As Byte = New Byte() {}
            Dim strValue As String
            Dim result As Integer = MakeHeader("BIT", 0, SendData)
            Dim BytesPerData As Integer = 1
            ReDim RecvDataRet(DeviceCount - 1)

            SendData(10) = &H1     ' Command code 1/2
            SendData(11) = &H1     ' Command code 2/2

            If DeviceName.Equals("IR") OrElse DeviceName.Equals("RR") Then
                SendData(12) = &H30                     ' I/O Type: Channel I/O
            ElseIf DeviceName.Equals("WR") Then
                SendData(12) = &H31                     ' I/O Type: Internal Assistant Relay
            ElseIf DeviceName.Equals("DM") Then
                SendData(12) = &H2                      ' I/O Type: Data Memory
            Else
                MessageBox.Show("[FinsUdp.ReadBits] I/O Type invalid or not registered")
                End
            End If
            strValue = "0000" + Hex(StartDevice)
            SendData(13) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 3, 2))                                  'Start address 1/3
            SendData(14) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 1, 2))                                  'Start address 2/3
            SendData(15) = CByte("&h" + Strings.Mid("00" + Hex(StartBitNo), ("00" + Hex(StartBitNo)).Length - 1, 2))    'Start Address3/3
            strValue = "0000" + Hex(DeviceCount)
            SendData(16) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 3, 2))    'Data count 1/2
            SendData(17) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 1, 2))    'Data count 2/2

            conn.Send(SendData)


            Dim RecvLengthWish As Integer = 14 + DeviceCount * BytesPerData
            Dim RecvDataByte() As Byte = New Byte() {}
            result = conn.RecvBinary(RecvDataByte)  ' Receive reply message

            If result <> 0 Then
                err = conn.GetError()
                Return -1
            Else
                If RecvDataByte.GetLength(0) <> RecvLengthWish Then ' Check data length
                    err = "RecvLengthError"
                    Return -1
                Else
                    If RecvDataByte(12) <> &H0 Or RecvDataByte(13) <> &H0 Then    ' Check termination code
                        err = "ResultCodeError"
                        Return -1
                    Else
                        For count As Integer = 0 To DeviceCount - 1
                            RecvDataRet(count) = 0
                            For count2 As Integer = 0 To BytesPerData - 1
                                RecvDataRet(count) = RecvDataRet(count) + Convert.ToUInt16(RecvDataByte(14 + count * BytesPerData + count2))
                            Next
                        Next

                        Return 0
                    End If
                End If
            End If
        Catch ex As Exception
            err = ex.Message
            Return -1
        End Try
    End Function

    ''' <summary>
    ''' Read word data from PLC
    ''' </summary>
    ''' <param name="DeviceName"></param>
    ''' <param name="StartDevice"></param>
    ''' <param name="DeviceCount"></param>
    ''' <param name="RecvDataRet"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function ReadWords(ByVal DeviceName As String, ByVal StartDevice As UInteger, ByVal DeviceCount As Integer, ByRef RecvDataRet As UShort()) As Integer
        Try
            Dim SendData() As Byte = New Byte() {}
            Dim strValue As String
            Dim result As Integer = MakeHeader("WORD", 0, SendData)
            Dim BytesPerData As Integer = 2
            ReDim RecvDataRet(DeviceCount - 1)

            SendData(10) = &H1     'Command code 1/2
            SendData(11) = &H1     'Command code 2/2

            If DeviceName.Equals("IR") OrElse DeviceName.Equals("RR") Then
                SendData(12) = &HB0                                                     ' I/O Type: Channel I/O
            ElseIf DeviceName.Equals("WR") Then
                SendData(12) = &HB1                                                     ' I/O Type: Internal Assistant Relay
            ElseIf DeviceName.Equals("DM") Then
                SendData(12) = &H82                                                      ' I/O Type: Data Memory
            Else
                MessageBox.Show("[FinsUdp.ReadWords] I/O Type invalid or not registered")
                End
            End If
            strValue = "0000" + Hex(StartDevice)
            SendData(13) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 3, 2))  'Start address 1/3
            SendData(14) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 1, 2))  'Start address 2/3
            SendData(15) = &H0                                                          'Start address 3/3
            strValue = "0000" + Hex(DeviceCount)
            SendData(16) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 3, 2))  'Data count 1/2
            SendData(17) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 1, 2))  'Data count 2/2

            conn.Send(SendData)

            Dim RecvLengthWish As Integer = 14 + DeviceCount * BytesPerData
            Dim RecvDataByte() As Byte = New Byte() {}
            result = conn.RecvBinary(RecvDataByte)

            If result <> 0 Then
                err = conn.GetError()
                Return -1
            Else
                If RecvDataByte.GetLength(0) <> RecvLengthWish Then             ' Check data length
                    err = "RecvLengthError"
                    Return -1
                Else
                    If RecvDataByte(12) <> &H0 Or RecvDataByte(13) <> &H0 Then  ' Check termination code
                        err = "ResultCodeError"
                        Return -1
                    Else
                        For count As Integer = 0 To DeviceCount - 1
                            RecvDataRet(count) = 0
                            For count2 As Integer = 0 To BytesPerData - 1
                                RecvDataRet(count) = Convert.ToUInt16(RecvDataRet(count) + RecvDataByte(14 + count * BytesPerData + count2) * 256 ^ (BytesPerData - 1 - count2))
                            Next
                        Next

                        Return 0
                    End If
                End If
            End If
        Catch ex As Exception
            err = ex.Message
            Return -1
        End Try
    End Function

    ''' <summary>
    ''' Write bit data to PLC
    ''' </summary>
    ''' <param name="DeviceName">a</param>
    ''' <param name="StartDevice">b</param>
    ''' <param name="StartBitNo">c</param>
    ''' <param name="intData"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function WriteBits(ByVal DeviceName As String, ByVal StartDevice As UInteger, ByVal StartBitNo As Integer, ByVal intData As Integer) As Integer
        Try
            Dim SendData() As Byte = New Byte() {}
            Dim strValue As String
            Dim result As Integer = MakeHeader("BIT", 1, SendData)
            Dim BytesPerData As Integer = 1

            SendData(10) = &H1     ' Command code 1/2
            SendData(11) = &H2     ' Command code 2/2

            If DeviceName.Equals("IR") OrElse DeviceName.Equals("RR") Then
                SendData(12) = &H30                     ' I/O Type: Channel I/O
            ElseIf DeviceName.Equals("WR") Then
                SendData(12) = &H31                     ' I/O Type: Internal Assistant Relay
            ElseIf DeviceName.Equals("DM") Then
                SendData(12) = &H2                      ' I/O Type: Data Memory
            Else
                MessageBox.Show("[FinsUdp.WriteBits] I/O Type invalid or not registered")
                End
            End If
            strValue = "0000" + Hex(StartDevice)
            SendData(13) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 3, 2))                                  ' Start address 1/3
            SendData(14) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 1, 2))                                  ' Start address 2/3
            SendData(15) = CByte("&h" + Strings.Mid("00" + Hex(StartBitNo), ("00" + Hex(StartBitNo)).Length - 1, 2))    ' Start address 3/3
            SendData(16) = &H0              ' Data count 1/2
            SendData(17) = &H1              ' Data count 2/2
            SendData(18) = CByte(intData)   ' Write data

            conn.Send(SendData)


            Dim RecvLengthWish As Integer = 14
            Dim RecvDataByte() As Byte = New Byte() {}
            result = conn.RecvBinary(RecvDataByte)

            If result <> 0 Then
                err = conn.GetError()
                Return -1
            Else
                If RecvDataByte.GetLength(0) <> RecvLengthWish Then             ' Check data length
                    err = "RecvLengthError"
                    Return -1
                Else
                    If RecvDataByte(12) <> &H0 Or RecvDataByte(13) <> &H0 Then  ' Check termination code
                        err = "ResultCodeError"
                        Return -1
                    Else
                        Return 0
                    End If
                End If
            End If
        Catch ex As Exception
            err = ex.Message
            Return -1
        End Try
    End Function

    ''' <summary>
    ''' Write word data
    ''' </summary>
    ''' <param name="DeviceName"></param>
    ''' <param name="StartDevice"></param>
    ''' <param name="DeviceCount"></param>
    ''' <param name="UInt16Data"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function WriteWords(ByVal DeviceName As String, ByVal StartDevice As UInteger, ByVal DeviceCount As Integer, ByVal UInt16Data() As UInt16) As Integer
        Try
            Dim SendData() As Byte = New Byte() {}
            Dim strValue As String
            Dim result As Integer = MakeHeader("WORD", 1, SendData)
            Dim BytesPerData As Integer = 1

            SendData(10) = &H1     ' Command code 1/2
            SendData(11) = &H2     ' Command code 2/2

            If DeviceName.Equals("IR") OrElse DeviceName.Equals("RR") Then
                SendData(12) = &HB0                                                         ' I/O Type: Channel I/O
            ElseIf DeviceName.Equals("WR") Then
                SendData(12) = &HB1                                                         ' I/O Type: Internal Assistant Relay
            ElseIf DeviceName.Equals("DM") Then
                SendData(12) = &H82                                                          ' I/O Type: Data Memory
            Else
                MessageBox.Show("[FinsUdp.WriteWords] I/O Type invalid or not registered")
                End
            End If
            strValue = "0000" + Hex(StartDevice)
            SendData(13) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 3, 2))  ' Start address 1/3
            SendData(14) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 1, 2))  ' Start address 2/3
            SendData(15) = &H0                                                          ' Start address 3/3

            strValue = "0000" + Hex(DeviceCount)
            SendData(16) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 3, 2))  ' Data count 1/2
            SendData(17) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 1, 2))  ' Data count 2/2

            For count As Integer = 1 To DeviceCount
                SendData(17 + count * 2 - 1) = Convert.ToByte(UInt16Data(count - 1) \ 256)
                SendData(17 + count * 2) = Convert.ToByte(UInt16Data(count - 1) Mod 256)
            Next

            conn.Send(SendData)


            Dim RecvLengthWish As Integer = 14
            Dim RecvDataByte() As Byte = New Byte() {}
            result = conn.RecvBinary(RecvDataByte)

            If result <> 0 Then
                err = conn.GetError()
                Return -1
            Else
                If RecvDataByte.GetLength(0) <> RecvLengthWish Then             ' Check data length
                    err = "RecvLengthError"
                    Return -1
                Else
                    If RecvDataByte(12) <> &H0 Or RecvDataByte(13) <> &H0 Then  ' Check termination code
                        err = "ResultCodeError"
                        Return -1
                    Else
                        Return 0
                    End If
                End If
            End If
        Catch ex As Exception
            err = ex.Message
            Return -1
        End Try
    End Function
End Class
