﻿Option Explicit On
Option Strict On
Option Infer Off

Imports System.Net

''' <summary>
''' OMRON PLC, FinsTCP protocol, Binary, TCP socket communication
''' </summary>
''' <remarks></remarks>
Public Class FinsTcp
    Inherits PlcProtocol

    Private TcpClientSync As TcpClientSync
    Private ServerIp As IPAddress
    Private ServerPort As Integer

    Private Const ICF As Integer = &H80
    Private Const RSV As Integer = &H0
    Private Const GCT As Integer = &H2
    Private Const DNA As Integer = &H0
    Private Const DA1 As Integer = &H3
    Private Const DA2 As Integer = &H0
    Private Const SNA As Integer = &H0
    Private Const SA1 As Integer = &HC
    Private Const SA2 As Integer = &H0
    Private Const SID As Integer = &H0

    Private Const CommandReadBits1 As String = "RR"   ' Internal relay (bit)
    Private Const CommandReadBits2 As String = "RL"   ' Link relay (bit)
    Private Const CommandReadWords As String = "RD"
    Private Const CommandWriteBits1 As String = "WR"  ' Internal relay (word)
    Private Const CommandWriteBits2 As String = "WL"  ' Link relay (word)
    Private Const CommandWriteWords As String = "WD"

    Private Header As String = "@00"

    Private ClientNodeNo As Integer = 0
    Private ServerNodeNo As Integer = 1


    Private HeaderBytes As Integer = 8
    Private CountNumberStart As Integer = 5
    Private CountNumberBytes As Integer = 4
    Private CountStart As Integer = 9
    Private CountOrder As Integer = 0

    ''' <summary>
    ''' Constructor
    ''' </summary>
    ''' <param name="Q_Protocol"></param>
    ''' <remarks></remarks>
    Public Sub New(ByVal Q_Protocol As IEnumerable(Of XElement))
        If Q_Protocol.<ServerIp>.Value Is Nothing Then
            MessageBox.Show("[FinsTcp] ServerIp in <DeviceArray><Device><Protocol>.<ServerIp> not found!")
            End
        ElseIf Q_Protocol.<ServerPort>.Value Is Nothing Then
            MessageBox.Show("FinsTcp] ServerPort in <DeviceArray><Device><Protocol>.<ServerPort> not found!")
            End
        End If

        ServerIp = IPAddress.Parse(Q_Protocol.<ServerIp>.Value)
        ServerPort = CInt(Q_Protocol.<ServerPort>.Value)

        TcpClientSync = New TcpClientSync()
    End Sub

    ''' <summary>
    ''' Send FINS node address request to PLC
    ''' </summary>
    ''' <param name="intNodeNumber"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Function InitialCommand(ByVal intNodeNumber As Integer) As Integer
        Dim sendBytes(19) As Byte

        Try
            sendBytes(0) = &H46
            sendBytes(1) = &H49
            sendBytes(2) = &H4E
            sendBytes(3) = &H53
            sendBytes(4) = &H0
            sendBytes(5) = &H0
            sendBytes(6) = &H0
            sendBytes(7) = &HC

            sendBytes(8) = &H0
            sendBytes(9) = &H0
            sendBytes(10) = &H0
            sendBytes(11) = &H0
            sendBytes(12) = &H0
            sendBytes(13) = &H0
            sendBytes(14) = &H0
            sendBytes(15) = &H0

            sendBytes(16) = &H0
            sendBytes(17) = &H0
            sendBytes(18) = &H0
            sendBytes(19) = CByte(intNodeNumber)

            TcpClientSync.Send(sendBytes)



            Dim RecvLengthWish As Integer = 24
            Dim RecvDataByte() As Byte = New Byte() {}
            'Dim result As Integer = TcpClientSync.RecvBinary(RecvDataByte, 4, 4, 0)
            Dim result As Integer = TcpClientSync.RecvBinary(RecvDataByte, HeaderBytes, CountNumberStart, CountNumberBytes, CountStart, CountOrder)

            If result <> 0 Then
                err = TcpClientSync.GetError()
                Return -1
            Else
                If RecvDataByte.GetLength(0) <> RecvLengthWish Then ' Check data length
                    err = "RecvLengthError"
                    Return -1
                Else
                    If RecvDataByte(0) <> &H46 Or RecvDataByte(1) <> &H49 Or RecvDataByte(2) <> &H4E Or RecvDataByte(3) <> &H53 Then    ' Header 
                        err = "HeaderError"
                        Return -1
                    Else
                        If RecvDataByte(8) <> &H0 Or RecvDataByte(9) <> &H0 Or RecvDataByte(10) <> &H0 Or RecvDataByte(11) <> &H1 Then  ' Termination code
                            err = "ResultCodeError"
                            Return -1
                        Else
                            ClientNodeNo = RecvDataByte(19)
                            ServerNodeNo = RecvDataByte(23)
                            Return 0
                        End If
                    End If
                End If
            End If
        Catch
            err = "InitialCommandSendErr"
            Return -1
        End Try
    End Function

    ''' <summary>
    ''' Connect to PLC
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function Connect() As Integer
        Dim result As Integer = TcpClientSync.Connect(ServerIp, ServerPort)  ' Connect
        If result = -1 Then
            err = "Connection failed"
        Else
            InitialCommand(ClientNodeNo)
        End If
        Connect = result
    End Function

    ''' <summary>
    ''' Disconnect from PLC
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function Disconnect() As Integer
        Dim result As Integer = TcpClientSync.Close()
        Return result
    End Function


    ''' <summary>
    ''' Make header
    ''' </summary>
    ''' <param name="BitOrWord"></param>
    ''' <param name="WriteCount"></param>
    ''' <param name="SendData"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Private Function MakeHeader(ByVal BitOrWord As String, ByVal WriteCount As Integer, ByRef SendData As Byte()) As Integer
        Dim strValue As String
        Dim MsgLen As Integer

        If WriteCount = 0 Then
            MsgLen = 8 + 18
        Else
            If BitOrWord.Equals("BIT") Then
                MsgLen = 8 + 18 + WriteCount
            ElseIf BitOrWord.Equals("WORD") Then
                MsgLen = 8 + 18 + WriteCount * 2
            Else
                err = "UnknownDeviceType"
                Return -1
            End If
        End If

        ReDim SendData(MsgLen + 8 - 1)

        SendData(0) = &H46
        SendData(1) = &H49
        SendData(2) = &H4E
        SendData(3) = &H53

        strValue = "00000000" + Hex(MsgLen)
        SendData(4) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 7, 2))
        SendData(5) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 5, 2))
        SendData(6) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 3, 2))
        SendData(7) = CByte("&h" + Strings.Right(strValue, 2))

        SendData(8) = &H0
        SendData(9) = &H0
        SendData(10) = &H0
        SendData(11) = &H2
        SendData(12) = &H0
        SendData(13) = &H0
        SendData(14) = &H0
        SendData(15) = &H0

        SendData(16) = ICF
        SendData(17) = RSV
        SendData(18) = GCT
        SendData(19) = DNA
        SendData(20) = Convert.ToByte(ServerNodeNo) 'DA1
        SendData(21) = DA2
        SendData(22) = SNA
        SendData(23) = Convert.ToByte(ClientNodeNo) 'SA1
        SendData(24) = SA2
        SendData(25) = SID

        Return 0
    End Function

    ''' <summary>
    ''' Read bit data from PLC 
    ''' </summary>
    ''' <param name="DeviceName"></param>
    ''' <param name="StartDevice"></param>
    ''' <param name="StartBitNo"></param>
    ''' <param name="DeviceCount"></param>
    ''' <param name="RecvDataRet"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function ReadBits(ByVal DeviceName As String, ByVal StartDevice As UInteger, ByVal StartBitNo As Integer, ByVal DeviceCount As Integer, ByRef RecvDataRet As UInt16()) As Integer
        Try
            Dim SendData() As Byte = New Byte() {}
            Dim strValue As String
            Dim result As Integer = MakeHeader("BIT", 0, SendData)
            Dim BytesPerData As Integer = 1
            ReDim RecvDataRet(DeviceCount - 1)

            SendData(26) = &H1     ' Command code 1/2
            SendData(27) = &H1     ' Command code 2/2

            If DeviceName.Equals("IR") OrElse DeviceName.Equals("RR") Then
                SendData(28) = &H30                     ' I/O Type: Internal Relay
            ElseIf DeviceName.Equals("WR") Then
                SendData(28) = &H31                     ' I/O Type: Internal Assistant Relay
            ElseIf DeviceName.Equals("DM") Then
                SendData(28) = &H2                      ' I/O Type: Data Memory
            Else
                MessageBox.Show("[FinsTcp.ReadBits] I/O Type invalid or not registered")
                End
            End If
            strValue = "0000" + Hex(StartDevice)
            SendData(29) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 3, 2))                                  ' Start address 1/3
            SendData(30) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 1, 2))                                  ' Start address 2/3
            SendData(31) = CByte("&h" + Strings.Mid("00" + Hex(StartBitNo), ("00" + Hex(StartBitNo)).Length - 1, 2))    ' Start address 3/3
            strValue = "0000" + Hex(DeviceCount)
            SendData(32) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 3, 2))                                  ' Data count 1/2
            SendData(33) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 1, 2))                                  ' Data count 2/2

            TcpClientSync.Send(SendData)

            Dim RecvLengthWish As Integer = 30 + DeviceCount * BytesPerData
            Dim RecvDataByte() As Byte = New Byte() {}
            'result = TcpClientSync.RecvBinary(RecvDataByte, 4, 4, 0)
            result = TcpClientSync.RecvBinary(RecvDataByte, HeaderBytes, CountNumberStart, CountNumberBytes, CountStart, CountOrder)

            If result <> 0 Then
                err = TcpClientSync.GetError()
                Return -1
            Else
                If RecvDataByte.GetLength(0) <> RecvLengthWish Then ' Check data length
                    err = "RecvLengthError"
                    Return -1
                Else
                    If RecvDataByte(0) <> &H46 Or RecvDataByte(1) <> &H49 Or RecvDataByte(2) <> &H4E Or RecvDataByte(3) <> &H53 Then    ' Check header
                        err = "HeaderError"
                        Return -1
                    Else
                        If RecvDataByte(28) <> &H0 Or RecvDataByte(29) <> &H0 Then    ' Check termination code
                            err = "ResultCodeError"
                            Return -1
                        Else
                            For count As Integer = 0 To DeviceCount - 1
                                RecvDataRet(count) = 0
                                For count2 As Integer = 0 To BytesPerData - 1
                                    RecvDataRet(count) = RecvDataRet(count) + Convert.ToUInt16(RecvDataByte(30 + count * BytesPerData + count2))
                                Next
                            Next

                            Return 0
                        End If
                    End If
                End If
            End If
        Catch ex As Exception
            err = ex.Message
            Return -1
        End Try
    End Function

    ''' <summary>
    ''' Send word data to PLC
    ''' </summary>
    ''' <param name="DeviceName"></param>
    ''' <param name="StartDevice"></param>
    ''' <param name="DeviceCount"></param>
    ''' <param name="RecvDataRet"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function ReadWords(ByVal DeviceName As String, ByVal StartDevice As UInteger, ByVal DeviceCount As Integer, ByRef RecvDataRet As UInt16()) As Integer
        Try
            Dim SendData() As Byte = New Byte() {}
            Dim strValue As String
            Dim result As Integer = MakeHeader("WORD", 0, SendData)
            Dim BytesPerData As Integer = 2
            ReDim RecvDataRet(DeviceCount - 1)

            SendData(26) = &H1                                                          ' Command code 1/2
            SendData(27) = &H1                                                          ' Command code 2/2

            If DeviceName.Equals("IR") OrElse DeviceName.Equals("RR") Then
                SendData(28) = &HB0                                                     ' I/O Type: Internal Relay
            ElseIf DeviceName.Equals("WR") Then
                SendData(28) = &HB1                                                     ' I/O Type: Internal Assistant Relay
            ElseIf DeviceName.Equals("DM") Then
                SendData(28) = &H82                                                     ' I/O Type: Data Memory
            Else
                MessageBox.Show("[FinsTcp.ReadWords] I/O Type invalid or not registered")
                End
            End If
            strValue = "0000" + Hex(StartDevice)
            SendData(29) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 3, 2))  ' Start address 1/3
            SendData(30) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 1, 2))  ' Start address 2/3
            SendData(31) = &H0                                                          ' Start address 3/3
            strValue = "0000" + Hex(DeviceCount)
            SendData(32) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 3, 2))  ' Data count 1/2
            SendData(33) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 1, 2))  ' Data count 2/2

            TcpClientSync.Send(SendData)

            Dim RecvLengthWish As Integer = 30 + DeviceCount * BytesPerData
            Dim RecvDataByte() As Byte = New Byte() {}
            'result = TcpClientSync.RecvBinary(RecvDataByte, 4, 4, 0)
            result = TcpClientSync.RecvBinary(RecvDataByte, HeaderBytes, CountNumberStart, CountNumberBytes, CountStart, CountOrder)
            If result <> 0 Then
                err = TcpClientSync.GetError()
                Return -1
            Else
                If RecvDataByte.GetLength(0) <> RecvLengthWish Then ' Check data length
                    err = "RecvLengthError"
                    Return -1
                Else
                    If RecvDataByte(0) <> &H46 Or RecvDataByte(1) <> &H49 Or RecvDataByte(2) <> &H4E Or RecvDataByte(3) <> &H53 Then    ' Check header
                        err = "HeaderError"
                        Return -1
                    Else
                        If RecvDataByte(28) <> &H0 Or RecvDataByte(29) <> &H0 Then    ' Check termination code
                            err = "ResultCodeError"
                            Return -1
                        Else
                            For count As Integer = 0 To DeviceCount - 1
                                RecvDataRet(count) = 0
                                For count2 As Integer = 0 To BytesPerData - 1
                                    RecvDataRet(count) = Convert.ToUInt16(RecvDataRet(count) + RecvDataByte(30 + count * BytesPerData + count2) * 256 ^ (BytesPerData - 1 - count2))
                                Next
                            Next

                            Return 0
                        End If
                    End If
                End If
            End If
        Catch ex As Exception
            err = ex.Message
            Return -1
        End Try
    End Function

    ''' <summary>
    ''' Write bit data to PLC
    ''' </summary>
    ''' <param name="DeviceName"></param>
    ''' <param name="StartDevice"></param>
    ''' <param name="StartBitNo"></param>
    ''' <param name="intData"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function WriteBits(ByVal DeviceName As String, ByVal StartDevice As UInteger, ByVal StartBitNo As Integer, ByVal intData As Integer) As Integer
        Try
            Dim SendData() As Byte = New Byte() {}
            Dim strValue As String
            Dim result As Integer = MakeHeader("BIT", 1, SendData)
            Dim BytesPerData As Integer = 1

            SendData(26) = &H1                          ' Command code 1/2
            SendData(27) = &H2                          ' Command code 2/2

            If DeviceName.Equals("IR") OrElse DeviceName.Equals("RR") Then
                SendData(28) = &H30                     ' I/O Type: Internal Relay
            ElseIf DeviceName.Equals("WR") Then
                SendData(28) = &H31                     ' I/O Type: Internal Assistant Relay
            ElseIf DeviceName.Equals("DM") Then
                SendData(28) = &H2                      ' I/O Type: Data Memory
            Else
                MessageBox.Show("[FinsTcp.WriteBits] I/O Type invalid or not registered")
                End
            End If
            strValue = "0000" + Hex(StartDevice)
            SendData(29) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 3, 2))                                  ' Start address 1/3
            SendData(30) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 1, 2))                                  ' Start address 2/3
            SendData(31) = CByte("&h" + Strings.Mid("00" + Hex(StartBitNo), ("00" + Hex(StartBitNo)).Length - 1, 2))    ' Start address 3/3
            SendData(32) = &H0                                                                                          ' Data count 1/2
            SendData(33) = &H1                                                                                          ' Data count 2/2
            SendData(34) = CByte(intData)                                                                               ' Write data

            TcpClientSync.Send(SendData)


            Dim RecvLengthWish As Integer = 30
            Dim RecvDataByte() As Byte = New Byte() {}
            'result = TcpClientSync.RecvBinary(RecvDataByte, 4, 4, 0)
            result = TcpClientSync.RecvBinary(RecvDataByte, HeaderBytes, CountNumberStart, CountNumberBytes, CountStart, CountOrder)

            If result <> 0 Then
                err = TcpClientSync.GetError()
                Return -1
            Else
                If RecvDataByte.GetLength(0) <> RecvLengthWish Then ' Check data length
                    err = "RecvLengthError"
                    Return -1
                Else
                    If RecvDataByte(0) <> &H46 Or RecvDataByte(1) <> &H49 Or RecvDataByte(2) <> &H4E Or RecvDataByte(3) <> &H53 Then    ' Check header
                        err = "HeaderError"
                        Return -1
                    Else
                        If RecvDataByte(28) <> &H0 Or RecvDataByte(29) <> &H0 Then    ' Check termination code
                            err = "ResultCodeError"
                            Return -1
                        Else
                            Return 0
                        End If
                    End If
                End If
            End If
        Catch ex As Exception
            err = ex.Message
            Return -1
        End Try
    End Function

    ''' <summary>
    ''' Write multiple word data to PLC
    ''' </summary>
    ''' <param name="DeviceName"></param>
    ''' <param name="StartDevice"></param>
    ''' <param name="DeviceCount"></param>
    ''' <param name="UInt16Data"></param>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Overrides Function WriteWords(ByVal DeviceName As String, ByVal StartDevice As UInteger, ByVal DeviceCount As Integer, ByVal UInt16Data() As UInt16) As Integer
        Try
            Dim SendData() As Byte = New Byte() {}
            Dim strValue As String
            Dim result As Integer = MakeHeader("WORD", DeviceCount, SendData)
            Dim BytesPerData As Integer = 1

            SendData(26) = &H1                                                              ' Command code 1/2
            SendData(27) = &H2                                                              ' Command code 2/2

            If DeviceName.Equals("IR") OrElse DeviceName.Equals("RR") Then
                SendData(28) = &HB0                                                         ' I/O Type: Internal Relay
            ElseIf DeviceName.Equals("WR") Then
                SendData(28) = &HB1                                                         ' I/O Type: Internal Assistant Relay
            ElseIf DeviceName.Equals("DM") Then
                SendData(28) = &H82                                                         ' I/O Type: Data Memory
            Else
                MessageBox.Show("[FinsTcp.WriteWords] I/O Type invalid or not registered")
                End
            End If
            strValue = "0000" + Hex(StartDevice)
            SendData(29) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 3, 2))      ' Start address 1/3
            SendData(30) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 1, 2))      ' Start address 2/3
            SendData(31) = &H0                                                              ' Start address 3/3

            strValue = "0000" + Hex(DeviceCount)
            SendData(32) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 3, 2))      ' Data count 1/2
            SendData(33) = CByte("&h" + Strings.Mid(strValue, strValue.Length - 1, 2))      ' Data count 2/2

            For count As Integer = 1 To DeviceCount
                SendData(33 + count * 2 - 1) = Convert.ToByte(UInt16Data(count - 1) \ 256)
                SendData(33 + count * 2) = Convert.ToByte(UInt16Data(count - 1) Mod 256)
            Next

            TcpClientSync.Send(SendData)


            Dim RecvLengthWish As Integer = 30
            Dim RecvDataByte() As Byte = New Byte() {}
            'result = TcpClientSync.RecvBinary(RecvDataByte, 4, 4, 0)
            result = TcpClientSync.RecvBinary(RecvDataByte, HeaderBytes, CountNumberStart, CountNumberBytes, CountStart, CountOrder)

            If result <> 0 Then
                err = TcpClientSync.GetError()
                Return -1
            Else
                If RecvDataByte.GetLength(0) <> RecvLengthWish Then ' Check data length
                    err = "RecvLengthError"
                    Return -1
                Else
                    If RecvDataByte(0) <> &H46 Or RecvDataByte(1) <> &H49 Or RecvDataByte(2) <> &H4E Or RecvDataByte(3) <> &H53 Then    ' Check header
                        err = "HeaderError"
                        Return -1
                    Else
                        If RecvDataByte(28) <> &H0 Or RecvDataByte(29) <> &H0 Then    ' Check termination code
                            err = "ResultCodeError"
                            Return -1
                        Else
                            Return 0
                        End If
                    End If
                End If
            End If
        Catch ex As Exception
            err = ex.Message
            Return -1
        End Try
    End Function
End Class
