package sharin.unlinq;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Map.Entry;

import sharin.unlinq.iterable.CastIterable;
import sharin.unlinq.iterable.ConcatIterable;
import sharin.unlinq.iterable.DefaultIfEmptyIterable;
import sharin.unlinq.iterable.DistinctIterable;
import sharin.unlinq.iterable.ExceptIterable;
import sharin.unlinq.iterable.GroupJoinIterable;
import sharin.unlinq.iterable.IntersectIterable;
import sharin.unlinq.iterable.JoinIterable;
import sharin.unlinq.iterable.OfTypeIterable;
import sharin.unlinq.iterable.RangeIterable;
import sharin.unlinq.iterable.RepeatIterable;
import sharin.unlinq.iterable.SelectIterable;
import sharin.unlinq.iterable.SelectManyIterable;
import sharin.unlinq.iterable.SkipIterable;
import sharin.unlinq.iterable.SkipWhileIterable;
import sharin.unlinq.iterable.TakeIterable;
import sharin.unlinq.iterable.TakeWhileIterable;
import sharin.unlinq.iterable.UnionIterable;
import sharin.unlinq.iterable.WhereIterable;

public class BasicEnumerable<TSource> implements Enumerable<TSource> {

    private final Iterable<TSource> iterable;

    public BasicEnumerable(Iterable<TSource> iterable) {

        if (iterable == null) {
            iterable = Collections.<TSource> emptyList();
        }

        this.iterable = iterable;
    }

    public Iterator<TSource> iterator() {
        return iterable.iterator();
    }

    public static <E> Enumerable<E> from(Iterable<E> iterable) {
        return new BasicEnumerable<E>(iterable);
    }

    public static <E> Enumerable<E> from(E... objects) {
        Iterable<E> iterable = null;

        if (objects != null) {
            iterable = Arrays.asList(objects);
        }

        return from(iterable);
    }

    public static Enumerable<Boolean> fromBoolean(boolean... values) {
        Boolean[] wrappers = null;

        if (values != null) {
            wrappers = new Boolean[values.length];

            for (int i = 0; i < values.length; i++) {
                wrappers[i] = values[i];
            }
        }

        return from(wrappers);
    }

    public static Enumerable<Byte> fromByte(byte... values) {
        Byte[] wrappers = null;

        if (values != null) {
            wrappers = new Byte[values.length];

            for (int i = 0; i < values.length; i++) {
                wrappers[i] = values[i];
            }
        }

        return from(wrappers);
    }

    public static Enumerable<Character> fromChar(char... values) {
        Character[] wrappers = null;

        if (values != null) {
            wrappers = new Character[values.length];

            for (int i = 0; i < values.length; i++) {
                wrappers[i] = values[i];
            }
        }

        return from(wrappers);
    }

    public static Enumerable<Short> fromShort(short... values) {
        Short[] wrappers = null;

        if (values != null) {
            wrappers = new Short[values.length];

            for (int i = 0; i < values.length; i++) {
                wrappers[i] = values[i];
            }
        }

        return from(wrappers);
    }

    public static Enumerable<Integer> from(int... values) {
        Integer[] wrappers = null;

        if (values != null) {
            wrappers = new Integer[values.length];

            for (int i = 0; i < values.length; i++) {
                wrappers[i] = values[i];
            }
        }

        return from(wrappers);
    }

    public static Enumerable<Long> fromLong(long... values) {
        Long[] wrappers = null;

        if (values != null) {
            wrappers = new Long[values.length];

            for (int i = 0; i < values.length; i++) {
                wrappers[i] = values[i];
            }
        }

        return from(wrappers);
    }

    public static Enumerable<Float> fromFloat(float... values) {
        Float[] wrappers = null;

        if (values != null) {
            wrappers = new Float[values.length];

            for (int i = 0; i < values.length; i++) {
                wrappers[i] = values[i];
            }
        }

        return from(wrappers);
    }

    public static Enumerable<Double> fromDouble(double... values) {
        Double[] wrappers = null;

        if (values != null) {
            wrappers = new Double[values.length];

            for (int i = 0; i < values.length; i++) {
                wrappers[i] = values[i];
            }
        }

        return from(wrappers);
    }

    public TSource aggregate(Func2<TSource, TSource, TSource> func) {
        Iterator<TSource> iterator = iterable.iterator();
        TSource seed = iterator.next();
        return doAggregate(seed, func, iterator);
    }

    public <A> A aggregate(A seed, Func2<A, TSource, A> func) {
        return doAggregate(seed, func, iterable.iterator());
    }

    public <A, R> R aggregate(A seed, Func2<A, TSource, A> func,
            Func<A, R> resultSelector) {

        A a = doAggregate(seed, func, iterable.iterator());
        return resultSelector.call(a);
    }

    private <A> A doAggregate(A seed, Func2<A, TSource, A> func,
            Iterator<TSource> iterator) {

        A a = seed;

        while (iterator.hasNext()) {
            a = func.call(a, iterator.next());
        }

        return a;
    }

    public Boolean all(Func<TSource, Boolean> predicate) {

        for (TSource source : iterable) {

            if (!predicate.call(source)) {
                return false;
            }
        }

        return true;
    }

    public Boolean any() {
        return iterable.iterator().hasNext();
    }

    public Boolean any(Func<TSource, Boolean> predicate) {

        for (TSource source : iterable) {

            if (predicate.call(source)) {
                return true;
            }
        }

        return false;
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public Enumerable<TSource> asEnumerable() {
        return new BasicEnumerable<TSource>(iterable);
    }

    public double average() {
        Iterator<TSource> iterator = iterable.iterator();
        double sum = (Integer) iterator.next();
        int count = 1;

        while (iterator.hasNext()) {
            sum += (Integer) iterator.next();
            count++;
        }

        return sum / count;
    }

    public double averageLong() {
        Iterator<TSource> iterator = iterable.iterator();
        double sum = (Long) iterator.next();
        int count = 1;

        while (iterator.hasNext()) {
            sum += (Long) iterator.next();
            count++;
        }

        return sum / count;
    }

    public float averageFloat() {
        Iterator<TSource> iterator = iterable.iterator();
        float sum = (Float) iterator.next();
        int count = 1;

        while (iterator.hasNext()) {
            sum += (Float) iterator.next();
            count++;
        }

        return sum / count;
    }

    public double averageDouble() {
        Iterator<TSource> iterator = iterable.iterator();
        double sum = (Double) iterator.next();
        int count = 1;

        while (iterator.hasNext()) {
            sum += (Double) iterator.next();
            count++;
        }

        return sum / count;
    }

    public double average(Func<TSource, Integer> selector) {
        Iterator<TSource> iterator = iterable.iterator();
        double sum = selector.call(iterator.next());
        int count = 1;

        while (iterator.hasNext()) {
            sum += selector.call(iterator.next());
            count++;
        }

        return sum / count;
    }

    public double averageLong(Func<TSource, Long> selector) {
        Iterator<TSource> iterator = iterable.iterator();
        double sum = selector.call(iterator.next());
        int count = 1;

        while (iterator.hasNext()) {
            sum += selector.call(iterator.next());
            count++;
        }

        return sum / count;
    }

    public float averageFloat(Func<TSource, Float> selector) {
        Iterator<TSource> iterator = iterable.iterator();
        float sum = selector.call(iterator.next());
        int count = 1;

        while (iterator.hasNext()) {
            sum += selector.call(iterator.next());
            count++;
        }

        return sum / count;
    }

    public double averageDouble(Func<TSource, Double> selector) {
        Iterator<TSource> iterator = iterable.iterator();
        double sum = selector.call(iterator.next());
        int count = 1;

        while (iterator.hasNext()) {
            sum += selector.call(iterator.next());
            count++;
        }

        return sum / count;
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public <R> Enumerable<R> cast(final Class<R> resultClass) {
        return new BasicEnumerable<R>(new CastIterable<TSource, R>(iterable,
                resultClass));
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public Enumerable<TSource> concat(final Enumerable<TSource> second) {
        return new BasicEnumerable<TSource>(new ConcatIterable<TSource>(
                iterable, second));
    }

    public boolean contains(Object o) {

        for (TSource source : iterable) {

            if (equals(o, source)) {
                return true;
            }
        }

        return false;
    }

    public int count() {
        int c = 0;

        for (Iterator<TSource> it = iterable.iterator(); it.hasNext(); it
                .next()) {
            c++;
        }

        return c;
    }

    public int count(Func<TSource, Boolean> predicate) {
        int c = 0;

        for (TSource source : iterable) {

            if (predicate.call(source)) {
                c++;
            }
        }

        return c;
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public Enumerable<TSource> defaultIfEmpty() {
        return defaultIfEmpty(null);
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public Enumerable<TSource> defaultIfEmpty(final TSource defaultValue) {
        return new BasicEnumerable<TSource>(
                new DefaultIfEmptyIterable<TSource>(iterable, defaultValue));
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public Enumerable<TSource> distinct() {
        return new BasicEnumerable<TSource>(new DistinctIterable<TSource>(
                iterable));
    }

    public TSource elementAt(int index) {
        Iterator<TSource> iterator = iterable.iterator();

        for (int i = 0; i < index; i++) {
            iterator.next();
        }

        return iterator.next();
    }

    public TSource elementAtOrDefault(int index) {
        return elementAtOrDefault(index, null);
    }

    public TSource elementAtOrDefault(int index, TSource defaultValue) {
        Iterator<TSource> iterator = iterable.iterator();

        for (int i = 0; i < index; i++) {

            if (!iterator.hasNext()) {
                return defaultValue;
            }

            iterator.next();
        }

        if (!iterator.hasNext()) {
            return defaultValue;
        }

        return iterator.next();
    }

    public static <R> Enumerable<R> empty() {
        return new BasicEnumerable<R>((Iterable<R>) null);
    }

    /**
     * This method is Deferred Streaming Execution and Deferred Non-Streaming
     * Execution.
     */
    public Enumerable<TSource> except(final Enumerable<TSource> second) {
        return new BasicEnumerable<TSource>(new ExceptIterable<TSource>(
                iterable, second));
    }

    public TSource first() {
        return iterable.iterator().next();
    }

    public TSource first(Func<TSource, Boolean> predicate) {

        for (TSource source : iterable) {

            if (predicate.call(source)) {
                return source;
            }
        }

        throw new NoSuchElementException();
    }

    public TSource firstOrDefault() {
        return firstOrDefault((TSource) null);
    }

    public TSource firstOrDefault(TSource defaultValue) {
        Iterator<TSource> iterator = iterable.iterator();

        if (!iterator.hasNext()) {
            return defaultValue;
        }

        return iterator.next();
    }

    public TSource firstOrDefault(Func<TSource, Boolean> predicate) {
        return firstOrDefault(predicate, null);
    }

    public TSource firstOrDefault(Func<TSource, Boolean> predicate,
            TSource defaultValue) {

        for (TSource source : iterable) {

            if (predicate.call(source)) {
                return source;
            }
        }

        return defaultValue;
    }

    /**
     * This method is Deferred Non-Streaming Execution.
     */
    public <K> Enumerable<Grouping<K, TSource>> groupBy(
            Func<TSource, K> keySelector) {

        return groupBy(keySelector, getPassThroughFunc());
    }

    /**
     * This method is Deferred Non-Streaming Execution.
     */
    public <K, E> Enumerable<Grouping<K, E>> groupBy(
            Func<TSource, K> keySelector, Func<TSource, E> elementSelector) {

        return groupBy(keySelector, elementSelector,
                new Func2<K, Enumerable<E>, Grouping<K, E>>() {

                    public Grouping<K, E> call(K arg1, Enumerable<E> arg2) {
                        return new BasicGrouping<K, E>(arg1, arg2);
                    }
                });
    }

    /**
     * This method is Deferred Non-Streaming Execution.
     */
    public <K, R> Enumerable<R> groupBy(Func<TSource, K> keySelector,
            Func2<K, Enumerable<TSource>, R> resultSelector) {

        return groupBy(keySelector, getPassThroughFunc(), resultSelector);
    }

    /**
     * This method is Deferred Non-Streaming Execution.
     */
    public <K, E, R> Enumerable<R> groupBy(final Func<TSource, K> keySelector,
            final Func<TSource, E> elementSelector,
            final Func2<K, Enumerable<E>, R> resultSelector) {

        return new BasicEnumerable<R>(new Iterable<R>() {

            public Iterator<R> iterator() {
                LinkedHashMap<K, List<E>> map = new LinkedHashMap<K, List<E>>();

                for (TSource source : iterable) {
                    K key = keySelector.call(source);
                    List<E> list = map.get(key);

                    if (list == null) {
                        list = new ArrayList<E>();
                        map.put(key, list);
                    }

                    list.add(elementSelector.call(source));
                }

                List<R> resultList = new ArrayList<R>();

                for (Entry<K, List<E>> entry : map.entrySet()) {
                    resultList.add(resultSelector.call(entry.getKey(),
                            new BasicEnumerable<E>(entry.getValue())));
                }

                return resultList.iterator();
            }
        });
    }

    /**
     * This method is Deferred Streaming Execution and Deferred Non-Streaming
     * Execution.
     */
    public <I, K, R> Enumerable<R> groupJoin(final Enumerable<I> inner,
            final Func<TSource, K> outerKeySelector,
            final Func<I, K> innerKeySelector,
            final Func2<TSource, Enumerable<I>, R> resultSelector) {

        return new BasicEnumerable<R>(new GroupJoinIterable<TSource, I, K, R>(
                iterable, inner, outerKeySelector, innerKeySelector,
                resultSelector));
    }

    /**
     * This method is Deferred Streaming Execution and Deferred Non-Streaming
     * Execution.
     */
    public Enumerable<TSource> intersect(final Enumerable<TSource> second) {
        return new BasicEnumerable<TSource>(new IntersectIterable<TSource>(
                iterable, second));
    }

    /**
     * This method is Deferred Streaming Execution and Deferred Non-Streaming
     * Execution.
     */
    public <I, K, R> Enumerable<R> join(final Enumerable<I> inner,
            final Func<TSource, K> outerKeySelector,
            final Func<I, K> innerKeySelector,
            final Func2<TSource, I, R> resultSelector) {

        return new BasicEnumerable<R>(new JoinIterable<TSource, I, K, R>(
                iterable, inner, outerKeySelector, innerKeySelector,
                resultSelector));
    }

    public TSource last() {
        Iterator<TSource> iterator = iterable.iterator();

        if (!iterator.hasNext()) {
            throw new NoSuchElementException();
        }

        TSource last = null;

        while (iterator.hasNext()) {
            last = iterator.next();
        }

        return last;
    }

    public TSource last(Func<TSource, Boolean> predicate) {
        TSource last = null;
        boolean found = false;

        for (TSource source : iterable) {

            if (predicate.call(source)) {
                last = source;

                if (!found) {
                    found = true;
                }
            }
        }

        if (!found) {
            throw new NoSuchElementException();
        }

        return last;
    }

    public TSource lastOrDefault() {
        return lastOrDefault((TSource) null);
    }

    public TSource lastOrDefault(TSource defaultValue) {
        Iterator<TSource> iterator = iterable.iterator();

        if (!iterator.hasNext()) {
            return defaultValue;
        }

        TSource last = null;

        while (iterator.hasNext()) {
            last = iterator.next();
        }

        return last;
    }

    public TSource lastOrDefault(Func<TSource, Boolean> predicate) {
        return lastOrDefault(predicate, null);
    }

    public TSource lastOrDefault(Func<TSource, Boolean> predicate,
            TSource defaultValue) {

        TSource last = null;
        boolean found = false;

        for (TSource source : iterable) {

            if (predicate.call(source)) {
                last = source;

                if (!found) {
                    found = true;
                }
            }
        }

        if (!found) {
            return defaultValue;
        }

        return last;
    }

    public long longCount() {
        long c = 0;

        for (Iterator<TSource> it = iterable.iterator(); it.hasNext(); it
                .next()) {
            c++;
        }

        return c;
    }

    public long longCount(Func<TSource, Boolean> predicate) {
        long c = 0;

        for (TSource source : iterable) {

            if (predicate.call(source)) {
                c++;
            }
        }

        return c;
    }

    @SuppressWarnings("unchecked")
    public TSource max() {
        Iterator<TSource> iterator = iterable.iterator();
        TSource max = iterator.next();

        while (iterator.hasNext()) {
            TSource source = iterator.next();

            if (((Comparable<TSource>) source).compareTo(max) > 0) {
                max = source;
            }
        }

        return max;
    }

    public <R extends Comparable<R>> R max(Func<TSource, R> selector) {
        Iterator<TSource> iterator = iterable.iterator();
        R max = selector.call(iterator.next());

        while (iterator.hasNext()) {
            R r = selector.call(iterator.next());

            if (r.compareTo(max) > 0) {
                max = r;
            }
        }

        return max;
    }

    @SuppressWarnings("unchecked")
    public TSource min() {
        Iterator<TSource> iterator = iterable.iterator();
        TSource min = iterator.next();

        while (iterator.hasNext()) {
            TSource source = iterator.next();

            if (((Comparable<TSource>) source).compareTo(min) < 0) {
                min = source;
            }
        }

        return min;
    }

    public <R extends Comparable<R>> R min(Func<TSource, R> selector) {
        Iterator<TSource> iterator = iterable.iterator();
        R min = selector.call(iterator.next());

        while (iterator.hasNext()) {
            R r = selector.call(iterator.next());

            if (r.compareTo(min) < 0) {
                min = r;
            }
        }

        return min;
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public <R> Enumerable<R> ofType(final Class<R> resultClass) {
        return new BasicEnumerable<R>(new OfTypeIterable<TSource, R>(iterable,
                resultClass));
    }

    /**
     * This method is Deferred Non-Streaming Execution.
     */
    public <K> OrderedEnumerable<TSource> orderBy(Func<TSource, K> keySelector) {
        return orderBy(keySelector, null);
    }

    /**
     * This method is Deferred Non-Streaming Execution.
     */
    public <K> OrderedEnumerable<TSource> orderBy(
            final Func<TSource, K> keySelector, final Comparator<K> comparator) {

        return new BasicOrderedEnumerable<TSource>(new Iterable<TSource>() {

            public Iterator<TSource> iterator() {
                List<List<TSource>> groupList = new ArrayList<List<TSource>>();
                groupList.add(toList());
                return new OrderedIterator<TSource>(groupList, keySelector,
                        comparator);
            }
        });
    }

    /**
     * This method is Deferred Non-Streaming Execution.
     */
    public <K> OrderedEnumerable<TSource> orderByDescending(
            Func<TSource, K> keySelector) {

        return orderBy(keySelector, Collections
                .reverseOrder((Comparator<K>) null));
    }

    /**
     * This method is Deferred Non-Streaming Execution.
     */
    public <K> OrderedEnumerable<TSource> orderByDescending(
            Func<TSource, K> keySelector, Comparator<K> comparator) {

        return orderBy(keySelector, Collections.reverseOrder(comparator));
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public static Enumerable<Integer> range(final int start, final int count) {
        return new BasicEnumerable<Integer>(new RangeIterable(start, count));
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public static <R> Enumerable<R> repeat(final R element, final int count) {
        return new BasicEnumerable<R>(new RepeatIterable<R>(count, element));
    }

    /**
     * This method is Deferred Non-Streaming Execution.
     */
    public Enumerable<TSource> reverse() {
        return new BasicEnumerable<TSource>(new Iterable<TSource>() {

            public Iterator<TSource> iterator() {
                List<TSource> list = toList();
                Collections.reverse(list);
                return list.iterator();
            }
        });
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public <R> Enumerable<R> select(final Func<TSource, R> selector) {
        return new BasicEnumerable<R>(new SelectIterable<TSource, R>(iterable,
                selector));
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public <R> Enumerable<R> select(final Func2<TSource, Integer, R> selector) {
        return select(new Func<TSource, R>() {

            private int index;

            public R call(TSource arg1) {
                return selector.call(arg1, index++);
            }
        });
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public <R> Enumerable<R> selectMany(Func<TSource, Enumerable<R>> selector) {
        return selectMany(selector, new Func2<TSource, R, R>() {

            public R call(TSource arg1, R arg2) {
                return arg2;
            }
        });
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public <R> Enumerable<R> selectMany(
            final Func2<TSource, Integer, Enumerable<R>> selector) {

        return selectMany(new Func<TSource, Enumerable<R>>() {

            private int index;

            public Enumerable<R> call(TSource arg1) {
                return selector.call(arg1, index++);
            }
        });
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public <C, R> Enumerable<R> selectMany(
            final Func<TSource, Enumerable<C>> collectionSelector,
            final Func2<TSource, C, R> resultSelector) {

        return new BasicEnumerable<R>(new SelectManyIterable<TSource, C, R>(
                iterable, resultSelector, collectionSelector));
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public <C, R> Enumerable<R> selectMany(
            final Func2<TSource, Integer, Enumerable<C>> collectionSelector,
            Func2<TSource, C, R> resultSelector) {

        return selectMany(new Func<TSource, Enumerable<C>>() {

            private int index;

            public Enumerable<C> call(TSource arg1) {
                return collectionSelector.call(arg1, index++);
            }
        }, resultSelector);
    }

    public Boolean sequenceEqual(Enumerable<TSource> second) {
        Iterator<TSource> iterator = second.iterator();

        for (TSource f : iterable) {

            if (!iterator.hasNext()) {
                return false;
            }

            if (!equals(f, iterator.next())) {
                return false;
            }
        }

        if (iterator.hasNext()) {
            return false;
        }

        return true;
    }

    public TSource single() {
        Iterator<TSource> iterator = iterable.iterator();
        TSource source = iterator.next();

        if (iterator.hasNext()) {
            throw new NoSuchElementException();
        }

        return source;
    }

    public TSource single(Func<TSource, Boolean> predicate) {
        TSource single = null;
        boolean found = false;

        for (TSource source : iterable) {

            if (predicate.call(source)) {

                if (found) {
                    found = false;
                    break;
                }

                single = source;
                found = true;
            }
        }

        if (!found) {
            throw new NoSuchElementException();
        }

        return single;
    }

    public TSource singleOrDefault() {
        return singleOrDefault((TSource) null);
    }

    public TSource singleOrDefault(TSource defaultValue) {
        Iterator<TSource> iterator = iterable.iterator();

        if (!iterator.hasNext()) {
            return defaultValue;
        }

        TSource source = iterator.next();

        if (iterator.hasNext()) {
            return defaultValue;
        }

        return source;
    }

    public TSource singleOrDefault(Func<TSource, Boolean> predicate) {
        return singleOrDefault(predicate, (TSource) null);
    }

    public TSource singleOrDefault(Func<TSource, Boolean> predicate,
            TSource defaultValue) {

        TSource single = null;
        boolean found = false;

        for (TSource source : iterable) {

            if (predicate.call(source)) {

                if (found) {
                    found = false;
                    break;
                }

                single = source;
                found = true;
            }
        }

        if (!found) {
            return defaultValue;
        }

        return single;
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public Enumerable<TSource> skip(final int count) {
        return new BasicEnumerable<TSource>(new SkipIterable<TSource>(iterable,
                count));
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public Enumerable<TSource> skipWhile(final Func<TSource, Boolean> predicate) {
        return new BasicEnumerable<TSource>(new SkipWhileIterable<TSource>(
                iterable, predicate));
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public Enumerable<TSource> skipWhile(
            final Func2<TSource, Integer, Boolean> predicate) {

        return skipWhile(new Func<TSource, Boolean>() {

            private int index;

            public Boolean call(TSource arg1) {
                return predicate.call(arg1, index++);
            }
        });
    }

    public int sum() {
        int sum = 0;

        for (TSource source : iterable) {
            sum += (Integer) source;
        }

        return sum;
    }

    public long sumLong() {
        long sum = 0;

        for (TSource source : iterable) {
            sum += (Long) source;
        }

        return sum;
    }

    public float sumFloat() {
        float sum = 0;

        for (TSource source : iterable) {
            sum += (Float) source;
        }

        return sum;
    }

    public double sumDouble() {
        double sum = 0;

        for (TSource source : iterable) {
            sum += (Double) source;
        }

        return sum;
    }

    public int sum(Func<TSource, Integer> selector) {
        int sum = 0;

        for (TSource source : iterable) {
            sum += selector.call(source);
        }

        return sum;
    }

    public long sumLong(Func<TSource, Long> selector) {
        long sum = 0;

        for (TSource source : iterable) {
            sum += selector.call(source);
        }

        return sum;
    }

    public float sumFloat(Func<TSource, Float> selector) {
        float sum = 0;

        for (TSource source : iterable) {
            sum += selector.call(source);
        }

        return sum;
    }

    public double sumDouble(Func<TSource, Double> selector) {
        double sum = 0;

        for (TSource source : iterable) {
            sum += selector.call(source);
        }

        return sum;
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public Enumerable<TSource> take(final int count) {
        return new BasicEnumerable<TSource>(new TakeIterable<TSource>(iterable,
                count));
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public Enumerable<TSource> takeWhile(final Func<TSource, Boolean> predicate) {
        return new BasicEnumerable<TSource>(new TakeWhileIterable<TSource>(
                iterable, predicate));
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public Enumerable<TSource> takeWhile(
            final Func2<TSource, Integer, Boolean> predicate) {

        return takeWhile(new Func<TSource, Boolean>() {

            private int index;

            public Boolean call(TSource arg1) {
                return predicate.call(arg1, index++);
            }
        });
    }

    public Object[] toArray() {
        return toList().toArray();
    }

    @SuppressWarnings("unchecked")
    public <R> R[] toArray(Class<R> resultClass) {
        List<TSource> list = toList();
        R[] array = (R[]) Array.newInstance(resultClass, list.size());
        return list.toArray(array);
    }

    public List<TSource> toList() {
        ArrayList<TSource> list = new ArrayList<TSource>();

        for (TSource e : iterable) {
            list.add(e);
        }

        return list;
    }

    public EnumerableList<TSource> toEnumerableList() {
        return new BasicEnumerableList<TSource>(toList());
    }

    public <K> Map<K, List<TSource>> toListMap(Func<TSource, K> keySelector) {
        return toListMap(keySelector, getPassThroughFunc());
    }

    public <K, E> Map<K, List<E>> toListMap(Func<TSource, K> keySelector,
            Func<TSource, E> elementSelector) {

        Map<K, List<E>> map = new LinkedHashMap<K, List<E>>();

        for (TSource source : iterable) {
            K k = keySelector.call(source);
            List<E> list = map.get(k);

            if (list == null) {
                list = new ArrayList<E>();
                map.put(k, list);
            }

            list.add(elementSelector.call(source));
        }

        return map;
    }

    public <K> Lookup<K, TSource> toLookup(Func<TSource, K> keySelector) {
        return toLookup(keySelector, getPassThroughFunc());
    }

    public <K, E> Lookup<K, E> toLookup(Func<TSource, K> keySelector,
            Func<TSource, E> elementSelector) {

        return new BasicLookup<K, E>(toListMap(keySelector, elementSelector));
    }

    public <K> Map<K, TSource> toMap(Func<TSource, K> keySelector) {
        return toMap(keySelector, getPassThroughFunc());
    }

    public <K, E> Map<K, E> toMap(Func<TSource, K> keySelector,
            Func<TSource, E> elementSelector) {

        Map<K, E> map = new LinkedHashMap<K, E>();

        for (TSource source : iterable) {
            K k = keySelector.call(source);

            if (map.containsKey(k)) {
                throw new IllegalArgumentException();
            }

            map.put(k, elementSelector.call(source));
        }

        return map;
    }

    public <K> Dictionary<K, TSource> toDictionary(Func<TSource, K> keySelector) {
        return toDictionary(keySelector, getPassThroughFunc());
    }

    public <K, E> Dictionary<K, E> toDictionary(Func<TSource, K> keySelector,
            Func<TSource, E> elementSelector) {

        return new BasicDictionary<K, E>(toMap(keySelector, elementSelector));
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public Enumerable<TSource> union(final Enumerable<TSource> second) {
        return new BasicEnumerable<TSource>(new UnionIterable<TSource>(
                iterable, second));
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public Enumerable<TSource> where(final Func<TSource, Boolean> predicate) {
        return new BasicEnumerable<TSource>(new WhereIterable<TSource>(
                iterable, predicate));
    }

    /**
     * This method is Deferred Streaming Execution.
     */
    public Enumerable<TSource> where(
            final Func2<TSource, Integer, Boolean> predicate) {

        return where(new Func<TSource, Boolean>() {

            private int index;

            public Boolean call(TSource arg1) {
                return predicate.call(arg1, index++);
            }
        });
    }

    private boolean equals(Object o1, Object o2) {

        if (o2 == null) {

            if (o1 == null) {
                return true;
            }

        } else {

            if (o2.equals(o1)) {
                return true;
            }
        }

        return false;
    }

    private Func<TSource, TSource> getPassThroughFunc() {
        return new Func<TSource, TSource>() {

            public TSource call(TSource arg1) {
                return arg1;
            }
        };
    }
}
