/* Copyright(C) 2004-2007 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "lib/com.h"
#include "lib/ql.h"
#include <string.h>
#include <stdio.h>
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif /* HAVE_SYS_WAIT_H */

#define DEFAULT_PORT 10041
#define DEFAULT_PATH "." PATH_SEPARATOR "senna.db"
#define DEFAULT_DEST "localhost"

int port = DEFAULT_PORT;
int batchmode;
sen_encoding enc = sen_enc_default;

static void
usage(void)
{
  fprintf(stderr,
          "Usage: senna [options...] [dest]\n"
          "options:\n"
          "  -a:                 run in standalone mode (default)\n"
          "  -c:                 run in client mode\n"
          "  -s:                 run in server mode\n"
          "  -p <port number>:   server port number (default: %d)\n"
          "dest: <db pathname> or <dest hostname>\n"
          "  <db pathname>: when standalone/server mode (default: \"%s\")\n"
          "  <dest hostname>: when client mode (default: \"%s\")\n",
          DEFAULT_PORT, DEFAULT_PATH, DEFAULT_DEST);
}

inline static void
prompt(void)
{
  if (!batchmode) { fputs("> ", stderr); }
}

static void
output(sen_ctx *c, int flags, void *dummy)
{
  sen_com_sqtp_header header;
  sen_rbuf *buf = &c->outbuf;
  header.size = SEN_RBUF_VSIZE(buf);
  header.flags = (flags & SEN_QL_MORE) ? SEN_COM_SQTP_MORE : SEN_COM_SQTP_TAIL;
  sen_com_sqtp_send((sen_com_sqtp *)c->data.ptr, &header, buf->head);
  SEN_RBUF_REWIND(buf);
}

static void
msg_handler(sen_com_event *ev, sen_com *c)
{
  sen_com_sqtp *cs = (sen_com_sqtp *)c;
  sen_ctx *ctx = (sen_ctx *)cs->userdata;
  if (cs->rc) { goto exit; }
  if (!ctx) {
    ctx = sen_ctx_open((sen_db *)ev->userdata, SEN_CTX_USEQL);
    if (!ctx) { goto exit; }
    sen_ctx_recv_handler_set(ctx, output, cs);
    sen_ctx_load(ctx, NULL);
    cs->userdata = ctx;
  }
  {
    char *body = SEN_COM_SQTP_MSG_BODY(&cs->msg);
    sen_com_sqtp_header *header = SEN_COM_SQTP_MSG_HEADER(&cs->msg);
    uint32_t size = header->size;
    uint16_t flags = header->flags;
    sen_ctx_send(ctx, body, size, flags);
    return;
  }
exit :
  SEN_LOG(sen_log_notice, "connection closed..");
  if (ctx) { sen_ctx_close(ctx); }
  sen_com_sqtp_close(ev, cs);
}

#define BUFSIZE 0x100000
#define MAX_CON 0x100

static int
do_alone(char *path)
{
  int rc = -1;
  sen_db *db;
  if ((db = sen_db_open(path ? path : DEFAULT_PATH)) ||
      (path && (db = sen_db_create(path, 0, enc)))) {
    sen_ctx *ctx;
    if ((ctx = sen_ctx_open(db, SEN_CTX_USEQL|(batchmode ? SEN_CTX_BATCHMODE : 0)))) {
      sen_ctx_info info;
      sen_ctx_info_get(ctx, &info);
      if (!sen_rbuf_reinit(info.outbuf, BUFSIZE)) {
        sen_ctx_recv_handler_set(ctx, sen_ctx_stream_out_func, stdout);
        sen_ctx_load(ctx, NULL);
        while (prompt(), fgets(info.outbuf->head, SEN_RBUF_WSIZE(info.outbuf), stdin)) {
          uint32_t size = strlen(info.outbuf->head) - 1;
          info.outbuf->head[size] = '\0';
          sen_ctx_send(ctx, info.outbuf->head, size, 0);
        }
        rc = 0;
      } else {
        fprintf(stderr, "sen_rbuf_reinit failed (%d)\n", BUFSIZE);
      }
      sen_ctx_close(ctx);
    } else {
      fprintf(stderr, "db_ctx open failed (%s)\n",  path);
    }
    sen_db_close(db);
  } else {
    fprintf(stderr, "db open failed (%s)\n", path);
  }
  return rc;
}

static int
do_client(char *hostname)
{
  int rc = -1;
  sen_ctx *ctx;
  if ((ctx = sen_ctx_connect(hostname, port, 0))) {
    sen_ctx_info info;
    sen_ctx_info_get(ctx, &info);
    if (!sen_rbuf_reinit(info.outbuf, BUFSIZE)) {
      while (prompt(), fgets(info.outbuf->head, SEN_RBUF_WSIZE(info.outbuf), stdin)) {
        int flags;
        char *str;
        unsigned int str_len;
        uint32_t size = strlen(info.outbuf->head) - 1;
        if (sen_ctx_send(ctx, info.outbuf->head, size, 0)) { break; }
        do {
          if (sen_ctx_recv(ctx, &str, &str_len, &flags)) {
            fprintf(stderr, "sen_ctx_recv failed\n");
            goto exit;
          }
          if (str_len) {
            fwrite(str, 1, str_len, stdout);
            putchar('\n');
          }
        } while ((flags & SEN_CTX_MORE));
      }
      rc = 0;
    } else {
      fprintf(stderr, "sen_rbuf_reinit failed (%d)\n", BUFSIZE);
    }
    sen_ctx_close(ctx);
  } else {
    fprintf(stderr, "sen_ctx_connect failed (%s:%d)\n", hostname, port);
  }
exit :
  return rc;
}

// todo : use thread
static int
server(char *path)
{
  int rc = -1;
  sen_com_event ev;
  if (!sen_com_event_init(&ev, MAX_CON, sizeof(sen_com_sqtp))) {
    sen_db *db;
    if ((db = sen_db_open(path ? path : DEFAULT_PATH)) ||
        (path && (db = sen_db_create(path, 0, enc)))) {
      sen_com_sqtp *cs;
      ev.userdata = db;
      if ((cs = sen_com_sqtp_sopen(&ev, port, msg_handler))) {
        while (!sen_com_event_poll(&ev, -1)) ;
        sen_com_sqtp_close(&ev, cs);
        rc = 0;
      } else {
        fprintf(stderr, "sen_com_sqtp_sopen failed (%d)\n", port);
      }
      sen_db_close(db);
    } else {
      fprintf(stderr, "db open failed (%s)\n", path);
    }
    sen_com_event_fin(&ev);
  } else {
    fprintf(stderr, "sen_com_event_init failed\n");
  }
  return rc;
}

#ifndef WIN32
static int
do_server(char *path)
{
  pid_t pid;
  switch (fork()) {
  case 0:
    break;
  case -1:
    perror("fork");
    return -1;
  default:
    wait(NULL);
    return 0;
  }
  switch ((pid = fork())) {
  case 0:
    break;
  case -1:
    perror("fork");
    return -1;
  default:
    fprintf(stderr, "%d\n", pid);
    _exit(0);
  }
  return server(path);
}
#endif /* WIN32 */

enum {
  mode_alone,
  mode_client,
  mode_server,
  mode_usage
};

int
main(int argc, char **argv)
{
  char *portstr = NULL, *encstr = NULL;
  int r, i, mode = mode_alone;
  static sen_str_getopt_opt opts[] = {
    {'p', NULL, NULL, 0, getopt_op_none},
    {'e', NULL, NULL, 0, getopt_op_none},
    {'h', NULL, NULL, mode_usage, getopt_op_update},
    {'a', NULL, NULL, mode_alone, getopt_op_update},
    {'c', NULL, NULL, mode_client, getopt_op_update},
    {'s', NULL, NULL, mode_server, getopt_op_update},
    {'\0', NULL, NULL, 0, 0}
  };
  opts[0].arg = &portstr;
  opts[1].arg = &encstr;
  i = sen_str_getopt(argc, argv, opts, &mode);
  if (i < 0) { mode = mode_usage; }
  if (portstr) { port = atoi(portstr); }
  if (encstr) {
    switch (*encstr) {
    case 'n' :
    case 'N' :
      enc = sen_enc_none;
      break;
    case 'e' :
    case 'E' :
      enc = sen_enc_euc_jp;
      break;
    case 'u' :
    case 'U' :
      enc = sen_enc_utf8;
      break;
    case 's' :
    case 'S' :
      enc = sen_enc_sjis;
      break;
    case 'l' :
    case 'L' :
      enc = sen_enc_latin1;
      break;
    case 'k' :
    case 'K' :
      enc = sen_enc_koi8r;
      break;
    }
  }
  batchmode = !isatty(0);
  if (sen_init()) { return -1; }
  switch (mode) {
  case mode_alone :
    r = do_alone(argc <= i ? NULL : argv[i]);
    break;
  case mode_client :
    r = do_client(argc <= i ? DEFAULT_DEST : argv[i]);
    break;
  case mode_server :
#ifdef WIN32
    r = server(argc <= i ? NULL : argv[i]);
#else /* WIN32 */
    r = do_server(argc <= i ? NULL : argv[i]);
#endif /* WIN32 */
    break;
  default :
    usage(); r = -1;
    break;
  }
  sen_fin();
  return r;
}
