/* Copyright(C) 2004-2007 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "senna_in.h"
#include "str.h"
#include "inv.h"
#include "sym.h"
#include "str.h"
#include "store.h"
#include <string.h>

/* rectangular arrays */

#define SEN_RA_IDSTR "SENNA:RA:01.000"
#define SEN_RA_SEGMENT_SIZE (1 << 22)

#define SEN_RA_MAX_CACHE (4294967295U)

sen_ra *
sen_ra_create(const char *path, unsigned int element_size)
{
  sen_io *io;
  int max_segments, n_elm, w_elm;
  sen_ra *ra = NULL;
  struct sen_ra_header *header;
  unsigned actual_size;
  if (element_size > SEN_RA_SEGMENT_SIZE) {
    SEN_LOG(sen_log_error, "element_size too large (%d)", element_size);
    return NULL;
  }
  for (actual_size = 1; actual_size < element_size; actual_size *= 2) ;
  max_segments = ((SEN_ID_MAX + 1) / SEN_RA_SEGMENT_SIZE) * actual_size;
  io = sen_io_create(path, sizeof(struct sen_ra_header),
                     SEN_RA_SEGMENT_SIZE, max_segments, sen_io_auto, SEN_RA_MAX_CACHE);
  if (!io) { return NULL; }
  header = sen_io_header(io);
  memcpy(header->idstr, SEN_RA_IDSTR, 16);
  header->element_size = actual_size;
  header->curr_max = 0;
  if (!(ra = SEN_MALLOC(sizeof(sen_ra)))) {
    sen_io_close(io);
    return NULL;
  }
  n_elm = SEN_RA_SEGMENT_SIZE / header->element_size;
  for (w_elm = 22; (1 << w_elm) > n_elm; w_elm--);
  ra->io = io;
  ra->header = header;
  ra->element_mask =  n_elm - 1;
  ra->element_width = w_elm;
  return ra;
}

sen_ra *
sen_ra_open(const char *path)
{
  sen_io *io;
  int n_elm, w_elm;
  sen_ra *ra = NULL;
  struct sen_ra_header *header;
  io = sen_io_open(path, sen_io_auto, SEN_RA_MAX_CACHE);
  if (!io) { return NULL; }
  header = sen_io_header(io);
  if (memcmp(header->idstr, SEN_RA_IDSTR, 16)) {
    SEN_LOG(sen_log_error, "ra_idstr (%s)", header->idstr);
    sen_io_close(io);
    return NULL;
  }
  if (!(ra = SEN_MALLOC(sizeof(sen_ra)))) {
    sen_io_close(io);
    return NULL;
  }
  n_elm = SEN_RA_SEGMENT_SIZE / header->element_size;
  for (w_elm = 22; (1 << w_elm) > n_elm; w_elm--);
  ra->io = io;
  ra->header = header;
  ra->element_mask =  n_elm - 1;
  ra->element_width = w_elm;
  return ra;
}

sen_rc
sen_ra_info(sen_ra *ra, unsigned int *element_size, sen_id *curr_max)
{
  if (!ra) { return sen_invalid_argument; }
  if (element_size) { *element_size = ra->header->element_size; }
  if (curr_max) { *curr_max = ra->header->curr_max; }
  return sen_success;
}

sen_rc
sen_ra_close(sen_ra *ra)
{
  sen_rc rc;
  if (!ra) { return sen_invalid_argument; }
  rc = sen_io_close(ra->io);
  SEN_FREE(ra);
  return rc;
}

sen_rc
sen_ra_remove(const char *path)
{
  if (!path) { return sen_invalid_argument; }
  return sen_io_remove(path);
}

void *
sen_ra_get(sen_ra *ra, sen_id id)
{
  void *p;
  uint16_t seg;
  if (id > SEN_ID_MAX) { return NULL; }
  seg = id >> ra->element_width;
  SEN_IO_SEG_MAP(ra->io, seg, p);
  if (!p) { return NULL; }
  if (id > ra->header->curr_max) { ra->header->curr_max = id; }
  return (void *)(((byte *)p) + ((id & ra->element_mask) * ra->header->element_size));
}

void *
sen_ra_at(sen_ra *ra, sen_id id)
{
  void *p;
  uint16_t seg;
  if (id > ra->header->curr_max) { return NULL; }
  seg = id >> ra->element_width;
  SEN_IO_SEG_MAP(ra->io, seg, p);
  if (!p) { return NULL; }
  return (void *)(((byte *)p) + ((id & ra->element_mask) * ra->header->element_size));
}

/**** jagged arrays ****/

#define SEN_JA_IDSTR "SENNA:JA:01.000"

#define W_OF_JA_MAX 38
#define W_OF_JA_SEGMENT 22
#define W_OF_JA_MAX_SEGMENTS (W_OF_JA_MAX - W_OF_JA_SEGMENT)

#define W_OF_JA_EINFO 3
#define W_OF_JA_EINFO_IN_A_SEGMENT (W_OF_JA_SEGMENT - W_OF_JA_EINFO)
#define N_OF_JA_EINFO_IN_A_SEGMENT (1U << W_OF_JA_EINFO_IN_A_SEGMENT)
#define JA_EINFO_MASK (N_OF_JA_EINFO_IN_A_SEGMENT - 1)

#define JA_SEGMENT_SIZE (1U << W_OF_JA_SEGMENT)
#define JA_MAX_SEGMENTS (1U << W_OF_JA_MAX_SEGMENTS)

#define JA_BSA_SIZE (1U << (W_OF_JA_SEGMENT - 7))
#define JA_N_BSEGMENTS (1U << (W_OF_JA_MAX_SEGMENTS - 7))

#define JA_N_ESEGMENTS (1U << 9)

#define SEN_JA_MAX_CACHE (4294967295U)

struct _sen_ja_einfo {
  union {
    uint64_t ll;
    struct {
      uint16_t seg;
      uint16_t pos;
      uint16_t size;
      uint8_t tail[2];
    } s;
  } u;
};

#define EINFO_SET(e,_seg,_pos,_size) {\
  (e)->u.s.seg = _seg;\
  (e)->u.s.pos = (_pos) >> 4;\
  (e)->u.s.size = _size;\
  (e)->u.s.tail[0] = (((_pos) >> 14) & 0xc0) + ((_size) >> 16);\
  (e)->u.s.tail[1] = 0;\
}

#define EINFO_GET(e,_seg,_pos,_size) {\
  _seg = (e)->u.s.seg;\
  _pos = ((e)->u.s.pos + (((e)->u.s.tail[0] & 0xc0) << 10)) << 4;\
  _size = (e)->u.s.size + (((e)->u.s.tail[0] & 0x3f) << 16);\
}

typedef struct {
  uint32_t seg;
  uint32_t pos;
} ja_pos;

struct sen_ja_header {
  char idstr[16];
  unsigned max_element_size;
  unsigned max_segments;
  ja_pos free_elements[24];
  uint8_t segments[JA_MAX_SEGMENTS];
  uint32_t esegs[JA_N_ESEGMENTS];
  uint32_t bsegs[JA_N_BSEGMENTS];
};


#define JA_SEG_ESEG 1;
#define JA_SEG_BSEG 2;
#define SEG_NOT_ASSIGNED 0xffffffff

sen_ja *
sen_ja_create(const char *path, unsigned int max_element_size)
{
  int i;
  sen_io *io;
  int max_segments;
  sen_ja *ja = NULL;
  struct sen_ja_header *header;
  if (max_element_size > JA_SEGMENT_SIZE) {
    SEN_LOG(sen_log_error, "max_element_size too large (%d)", max_element_size);
    return NULL;
  }
  max_segments = max_element_size * 128;
  if (max_segments > JA_MAX_SEGMENTS) { max_segments = JA_MAX_SEGMENTS; }
  io = sen_io_create(path, sizeof(struct sen_ja_header),
                     JA_SEGMENT_SIZE, max_segments, sen_io_auto, SEN_JA_MAX_CACHE);
  if (!io) { return NULL; }
  header = sen_io_header(io);
  memcpy(header->idstr, SEN_JA_IDSTR, 16);
  for (i = 0; i < JA_N_ESEGMENTS; i++) { header->esegs[i] = SEG_NOT_ASSIGNED; }
  for (i = 0; i < JA_N_BSEGMENTS; i++) { header->bsegs[i] = SEG_NOT_ASSIGNED; }
  header->max_element_size = max_element_size;
  header->max_segments = max_segments;
  header->segments[0] = JA_SEG_ESEG;
  header->esegs[0] = 0;
  if (!(ja = SEN_MALLOC(sizeof(sen_ja)))) {
    sen_io_close(io);
    return NULL;
  }
  ja->io = io;
  ja->header = header;
  return ja;
}

sen_ja *
sen_ja_open(const char *path)
{
  sen_io *io;
  sen_ja *ja = NULL;
  struct sen_ja_header *header;
  io = sen_io_open(path, sen_io_auto, SEN_JA_MAX_CACHE);
  if (!io) { return NULL; }
  header = sen_io_header(io);
  if (memcmp(header->idstr, SEN_JA_IDSTR, 16)) {
    SEN_LOG(sen_log_error, "ja_idstr (%s)", header->idstr);
    sen_io_close(io);
    return NULL;
  }
  if (!(ja = SEN_MALLOC(sizeof(sen_ja)))) {
    sen_io_close(io);
    return NULL;
  }
  ja->io = io;
  ja->header = header;
  return ja;
}

sen_rc
sen_ja_info(sen_ja *ja, unsigned int *max_element_size)
{
  if (!ja) { return sen_invalid_argument; }
  return sen_success;
}

sen_rc
sen_ja_close(sen_ja *ja)
{
  sen_rc rc;
  if (!ja) { return sen_invalid_argument; }
  rc = sen_io_close(ja->io);
  SEN_FREE(ja);
  return rc;
}

sen_rc
sen_ja_remove(const char *path)
{
  if (!path) { return sen_invalid_argument; }
  return sen_io_remove(path);
}

sen_rc
sen_ja_put(sen_ja *ja, sen_id id, const void *value, int value_len, int flags)
{
  int rc;
  void *buf;
  sen_ja_einfo einfo;
  if ((flags & SEN_ST_APPEND)) {
    uint32_t old_len;
    const void *oldvalue = sen_ja_ref(ja, id, &old_len);
    if (oldvalue) {
      if ((rc = sen_ja_alloc(ja, value_len + old_len, &einfo, &buf))) { return rc; }
      memcpy(buf, oldvalue, old_len);
      memcpy((byte *)buf + old_len, value, value_len);
      sen_ja_unref(ja, id);
    } else {
      if ((rc = sen_ja_alloc(ja, value_len, &einfo, &buf))) { return rc; }
      memcpy(buf, value, value_len);
    }
  } else {
    if ((rc = sen_ja_alloc(ja, value_len, &einfo, &buf))) { return rc; }
    // printf("put id=%d, value_len=%d value=%p ei=%p(%d:%d)\n", id, value_len, buf, &einfo, einfo.u.s.pos, einfo.u.s.tail[0]);
    memcpy(buf, value, value_len);
  }
  return sen_ja_replace(ja, id, &einfo);
}

int
sen_ja_at(sen_ja *ja, sen_id id, void *valbuf, int buf_size)
{
  uint32_t len;
  const void *value = sen_ja_ref(ja, id, &len);
  if (!value) { return -1; }
  if (buf_size >= len) { memcpy(valbuf, value, len); }
  sen_ja_unref(ja, id);
  return (int) len;
}

const void *
sen_ja_ref(sen_ja *ja, sen_id id, uint32_t *value_len)
{
  sen_ja_einfo *einfo;
  uint32_t lseg, *pseg, pos;
  lseg = id >> W_OF_JA_EINFO_IN_A_SEGMENT;
  pos = id & JA_EINFO_MASK;
  pseg = &ja->header->esegs[lseg];
  if (*pseg == SEG_NOT_ASSIGNED) { *value_len = 0; return NULL; }
  SEN_IO_SEG_MAP(ja->io, *pseg, einfo);
  if (!einfo) { *value_len = 0; return NULL; }
  if (einfo[pos].u.s.tail[1] & 1) {
    *value_len = einfo[pos].u.s.tail[1] >> 1;
    return (void *) &einfo[pos];
  }
  {
    void *value;
    uint32_t jag, vpos, vsize;
    EINFO_GET(&einfo[pos], jag, vpos, vsize);
    SEN_IO_SEG_MAP(ja->io, jag, value);
    // printf("at id=%d value=%p jag=%d vpos=%d ei=%p(%d:%d)\n", id, value, jag, vpos, &einfo[pos], einfo[pos].u.s.pos, einfo[pos].u.s.tail[0]);
    if (!value) { *value_len = 0; return NULL; }
    *value_len = vsize;
    return (byte *)value + vpos;
  }
}

sen_rc
sen_ja_unref(sen_ja *ja, sen_id id)
{
  // todo
  return sen_success;
}

int
sen_ja_size(sen_ja *ja, sen_id id)
{
  sen_ja_einfo *einfo;
  uint32_t lseg, *pseg, pos;
  lseg = id >> W_OF_JA_EINFO_IN_A_SEGMENT;
  pos = id & JA_EINFO_MASK;
  pseg = &ja->header->esegs[lseg];
  if (*pseg == SEG_NOT_ASSIGNED) { return -1; }
  SEN_IO_SEG_MAP(ja->io, *pseg, einfo);
  if (!einfo) { return -1; }
  if (einfo[pos].u.s.tail[1] & 1) {
    return einfo[pos].u.s.tail[1] >> 1;
  } else {
    return einfo[pos].u.s.size + ((einfo[pos].u.s.tail[0] & 0x3f) << 16);
  }
}

sen_rc
sen_ja_alloc(sen_ja *ja, int element_size, sen_ja_einfo *einfo, void **value)
{
  int m, size;
  void *addr;
  ja_pos *vp;
  if (element_size < 8) {
    einfo->u.s.tail[1] = element_size * 2 + 1;
    *value = (void *)einfo;
    return sen_success;
  }
  if (element_size >= ja->header->max_element_size) {
    return sen_invalid_argument;
  }
  for (m = 4, size = 16; size < element_size; m++, size *= 2);
  vp = &ja->header->free_elements[m];
  if (!vp->seg) {
    int i = 0;
    while (ja->header->segments[i]) {
      if (++i >= ja->header->max_segments) { return sen_memory_exhausted; }
    }
    ja->header->segments[i] = m;
    vp->seg = i;
    vp->pos = 0;
  }
  EINFO_SET(einfo, vp->seg, vp->pos, element_size);
  SEN_IO_SEG_MAP(ja->io, vp->seg, addr);
  // printf("addr=%p seg=%d pos=%d\n", addr, vp->seg, vp->pos);
  if (!addr) { return sen_memory_exhausted; }
  *value = (byte *)addr + vp->pos;
  if ((vp->pos += size) == JA_SEGMENT_SIZE) {
    vp->seg = 0;
    vp->pos = 0;
  }
  return sen_success;
}

sen_rc
sen_ja_free(sen_ja *ja, sen_ja_einfo *einfo)
{
  uint32_t seg, pos, size;
  if (einfo->u.s.tail[1] & 1) { return sen_success; }
  EINFO_GET(einfo, seg, pos, size);
  // free
  return sen_success;
}

sen_rc
sen_ja_replace(sen_ja *ja, sen_id id, sen_ja_einfo *ei)
{
  uint32_t lseg, *pseg, pos;
  sen_ja_einfo *einfo, eback;
  lseg = id >> W_OF_JA_EINFO_IN_A_SEGMENT;
  pos = id & JA_EINFO_MASK;
  pseg = &ja->header->esegs[lseg];
  if (*pseg == SEG_NOT_ASSIGNED) {
    int i = 0;
    while (ja->header->segments[i]) {
      if (++i >= ja->header->max_segments) { return sen_memory_exhausted; }
    }
    ja->header->segments[i] = 1;
    *pseg = i;
  }
  SEN_IO_SEG_MAP(ja->io, *pseg, einfo);
  if (!einfo) { return sen_memory_exhausted; }
  eback = einfo[pos];
  einfo[pos] = *ei;
  // todo: SEN_ATOMIC_SET64
  sen_ja_free(ja, &eback);
  return sen_success;
}

/**** db ****/

inline static void
gen_pathname(const char *path, char *buffer, int fno)
{
  size_t len = strlen(path);
  memcpy(buffer, path, len);
  if (fno >= 0) {
    buffer[len] = '.';
    sen_str_itoh(fno, buffer + len + 1, 7);
  } else {
    buffer[len] = '\0';
  }
}

sen_db_store *
sen_db_store_by_id(sen_db *s, sen_id id)
{
  sen_db_store *slot;
  const char *name;
  if (sen_set_at(s->stores, &id, (void **) &slot)) { return slot; }
  if (!(name = _sen_sym_key(s->keys, id))) { return NULL; }
  return sen_db_store_open(s, name);
}

// todo : slot should has cache class id
sen_db_store *
sen_db_slot_class(sen_db *s, const char *slot)
{
  int i = SEN_SYM_MAX_KEY_SIZE;
  char buf[SEN_SYM_MAX_KEY_SIZE], *dst = buf;
  while (*slot != '.') {
    if (!*slot || !--i) { return NULL; }
    *dst++ = *slot++;
  }
  *dst = '\0';
  return sen_db_store_open(s, buf);
}

sen_db_store *
sen_db_store_open(sen_db *s, const char *name)
{
  sen_id id;
  sen_set_eh *ep;
  uint32_t spec_len;
  sen_db_store *e;
  char buffer[PATH_MAX];
  const sen_db_store_spec *spec;
  if (!s || !(id = sen_sym_at(s->keys, name))) { return NULL; }
  if (sen_set_at(s->stores, &id, (void **) &e)) { return e; }
  if (!(spec = sen_ja_ref(s->values, id, &spec_len))) { return NULL; }
  ep = sen_set_get(s->stores, &id, (void **) &e);
  e->type = spec->type;
  e->db = s;
  e->id = id;
  e->triggers = NULL;
  gen_pathname(s->keys->io->path, buffer, id);
  switch (spec->type) {
  case sen_db_raw_class :
    e->u.bc.element_size = spec->u.c.size;
    break;
  case sen_db_class :
    if (!(e->u.c.keys = sen_sym_open(buffer))) { goto exit; }
    break;
  case sen_db_obj_slot :
    e->u.o.class = spec->u.s.class;
    if (!(e->u.o.ra = sen_ra_open(buffer))) { goto exit; }
    break;
  case sen_db_ra_slot :
    e->u.f.class = spec->u.s.class;
    if (!(e->u.f.ra = sen_ra_open(buffer))) { goto exit; }
    break;
  case sen_db_ja_slot :
    e->u.v.class = spec->u.s.class;
    if (!(e->u.v.ja = sen_ja_open(buffer))) { goto exit; }
    break;
  case sen_db_idx_slot :
    e->u.i.class = spec->u.s.class;
    {
      sen_db_store *l, *k;
      if (!(k = sen_db_store_by_id(s, spec->u.s.class))) { goto exit; }
      if (!(l = sen_db_slot_class(s, name))) { goto exit; }
      if (!(e->u.i.index =
            sen_index_open_with_keys_lexicon(buffer, k->u.c.keys, l->u.c.keys))) {
        goto exit;
      }
    }
    break;
  case sen_db_rel1 :
    e->u.f.class = spec->u.s.class;
    if (!(e->u.f.ra = sen_ra_open(buffer))) { goto exit; }
    break;
  default :
    goto exit;
  }
  {
    int i;
    for (i = 0; i < spec->n_triggers; i++) {
      sen_db_store *target = sen_db_store_by_id(s, spec->triggers[i].target);
      if (target) {
        sen_db_trigger *r = SEN_MALLOC(sizeof(sen_db_trigger));
        if (!r) { goto exit; }
        r->next = e->triggers;
        r->type = spec->triggers[i].type;
        r->target = target;
        e->triggers = r;
      }
    }
  }
  sen_ja_unref(s->values, id);
  return e;
exit :
  sen_set_del(s->stores, ep);
  sen_ja_unref(s->values, id);
  return NULL;
}

sen_db_store *
sen_db_store_create(sen_db *s, const char *name, sen_db_store_spec *spec)
{
  sen_id id;
  sen_set_eh *ep;
  sen_db_store *e;
  char buffer[PATH_MAX];
  if (strlen(name) >= SEN_SYM_MAX_KEY_SIZE) {
    SEN_LOG(sen_log_error, "too long store name (%s)", name);
    return NULL;
  }
  if (strchr(name, '.') &&
      ((spec->type == sen_db_raw_class) ||
       (spec->type == sen_db_class))) {
    SEN_LOG(sen_log_error, "class name must not include '.' (%s)", name);
    return NULL;
  }
  if (!(id = sen_sym_get(s->keys, name))) { return NULL; }
  if (!(ep = sen_set_get(s->stores, &id, (void **) &e))) { goto exit; }
  spec->n_triggers = 0;
  if (sen_ja_put(s->values, id, spec, SEN_DB_STORE_SPEC_SIZE(0), 0)) { goto exit; }
  e->type = spec->type;
  e->db = s;
  e->id = id;
  e->triggers = NULL;
  gen_pathname(s->keys->io->path, buffer, id);
  switch (spec->type) {
  case sen_db_raw_class :
    e->u.bc.element_size = spec->u.c.size;
    break;
  case sen_db_class :
    if (!(e->u.c.keys = sen_sym_create(buffer,
                                       spec->u.c.size,
                                       spec->u.c.flags,
                                       spec->u.c.encoding))) { goto exit; }
    break;
  case sen_db_obj_slot :
    e->u.o.class = spec->u.s.class;
    if (!(e->u.o.ra = sen_ra_create(buffer, sizeof(sen_id)))) { goto exit; }
    break;
  case sen_db_ra_slot :
    e->u.f.class = spec->u.s.class;
    if (!(e->u.f.ra = sen_ra_create(buffer, spec->u.s.size))) { goto exit; }
    break;
  case sen_db_ja_slot :
    e->u.v.class = spec->u.s.class;
    if (!(e->u.v.ja = sen_ja_create(buffer, spec->u.s.size))) { goto exit; }
    break;
  case sen_db_idx_slot :
    e->u.i.class = spec->u.s.class;
    {
      sen_db_store *l, *k;
      if (!(k = sen_db_store_by_id(s, spec->u.s.class))) { goto exit; }
      if (!(l = sen_db_slot_class(s, name))) { goto exit; }
      if (!(e->u.i.index =
            sen_index_create_with_keys_lexicon(buffer, k->u.c.keys, l->u.c.keys,
                                               spec->u.s.size))) {
        goto exit;
      }
    }
    break;
  case sen_db_rel1 :
    e->u.f.class = spec->u.s.class;
    if (!(e->u.f.ra = sen_ra_create(buffer, spec->u.s.size))) { goto exit; }
    break;
  default :
    goto exit;
  }
  return e;
exit :
  if (ep) { sen_set_del(s->stores, ep); }
  sen_sym_del(s->keys, name);
  // todo : sen_ja_put(s->values, id, NULL, 0, 0);
  return NULL;
}

sen_rc
sen_db_store_add_trigger(sen_db_store *e, sen_db_store_rel_spec *t)
{
  sen_rc rc;
  sen_db *s = e->db;
  uint32_t spec_len, newspec_len;
  const sen_db_store_spec *spec;
  sen_db_store_spec *newspec;
  if (!(spec = sen_ja_ref(s->values, e->id, &spec_len))) {
    return sen_invalid_argument;
  }
  newspec_len = SEN_DB_STORE_SPEC_SIZE(spec->n_triggers + 1);
  if (!(newspec = SEN_MALLOC(newspec_len))) { return sen_memory_exhausted; }
  memcpy(newspec, spec, spec_len);
  memcpy(&newspec->triggers[spec->n_triggers], t, sizeof(sen_db_store_rel_spec));
  newspec->n_triggers++;
  sen_ja_unref(s->values, e->id);
  if ((rc = sen_ja_put(s->values, e->id, newspec, newspec_len, 0))) {
    goto exit;
  }
  {
    sen_db_store *target = sen_db_store_by_id(s, t->target);
    if (target) {
      sen_db_trigger *r = SEN_MALLOC(sizeof(sen_db_trigger));
      if (!r) { rc = sen_memory_exhausted; goto exit; }
      r->next = e->triggers;
      r->type = t->type;
      r->target = target;
      e->triggers = r;
      if (t->type == sen_db_index_target) {
        sen_db_store_rel_spec invrs;
        invrs.type = sen_db_before_update_trigger;
        invrs.target = e->id;
        rc = sen_db_store_add_trigger(target, &invrs);
      }
    }
  }
exit :
  SEN_FREE(newspec);
  return rc;
}

sen_db_store *
sen_db_slot_class_by_id(sen_db *s, sen_id slot)
{
  return sen_db_slot_class(s, _sen_sym_key(s->keys, slot));
}

sen_rc
sen_db_class_slotpath(sen_db *s, sen_id class, const char *name, char *buf)
{
  char *dst;
  const char *src = _sen_sym_key(s->keys, class);
  if (!src) { return sen_invalid_argument; }
  strcpy(buf, src);
  dst = buf + strlen(src);
  *dst++ = '.';
  strcpy(dst, name);
  return sen_success;
}

sen_db_store *
sen_db_class_slot(sen_db *s, sen_id class, const char *name)
{
  char buf[SEN_SYM_MAX_KEY_SIZE];
  if (sen_db_class_slotpath(s, class, name, buf)) { return NULL; }
  return sen_db_store_open(s, buf);
}

sen_db_store *
sen_db_class_add_slot(sen_db *s, sen_id class, const char *name, sen_db_store_spec *spec)
{
  char buf[SEN_SYM_MAX_KEY_SIZE];
  if (sen_db_class_slotpath(s, class, name, buf)) { return NULL; }
  return sen_db_store_create(s, buf, spec);
}

sen_rc
sen_db_store_close(sen_db_store *slot, int all)
{
  sen_db *s = slot->db;
  sen_db_trigger *t, *t_;
  switch (slot->type) {
  case sen_db_obj_slot :
    // sen_db_class_close(slot->u.o.class);
    sen_ra_close(slot->u.o.ra);
    break;
  case sen_db_ra_slot :
    sen_ra_close(slot->u.f.ra);
    break;
  case sen_db_ja_slot :
    sen_ja_close(slot->u.v.ja);
    break;
  case sen_db_idx_slot :
    sen_index_close(slot->u.i.index);
    break;
  case sen_db_class :
    sen_sym_close(slot->u.c.keys);
    break;
  case sen_db_rel1 :
    sen_ra_close(slot->u.f.ra);
    break;
  default :
    return sen_invalid_argument;
  }
  for (t = slot->triggers; t; t = t_) {
    t_ = t->next;
    SEN_FREE(t);
  }
  if (!all) {
    sen_set_eh *ep;
    if ((ep = sen_set_at(s->stores, &slot->id, NULL))) {
      sen_set_del(s->stores, ep);
    }
  }
  return sen_success;
}

sen_rc
sen_db_prepare_builtin_class(sen_db *db)
{
  sen_db_store_spec spec;
  spec.type = sen_db_raw_class;
  spec.u.c.size = sizeof(int);
  if (!sen_db_store_create(db, "<int>", &spec)) { return sen_memory_exhausted; }
  if (!sen_db_store_create(db, "<uint>", &spec)) { return sen_memory_exhausted; }
  spec.u.c.size = sizeof(int64_t);
  if (!sen_db_store_create(db, "<int64>", &spec)) { return sen_memory_exhausted; }
  spec.u.c.size = sizeof(double);
  if (!sen_db_store_create(db, "<float>", &spec)) { return sen_memory_exhausted; }
  spec.u.c.size = SEN_SYM_MAX_KEY_SIZE;
  if (!sen_db_store_create(db, "<shorttext>", &spec)) { return sen_memory_exhausted; }
  spec.u.c.size = 1 << 16;
  if (!sen_db_store_create(db, "<text>", &spec)) { return sen_memory_exhausted; }
  spec.u.c.size = JA_SEGMENT_SIZE;
  if (!sen_db_store_create(db, "<longtext>", &spec)) { return sen_memory_exhausted; }
  return sen_success;
}

sen_db *
sen_db_create(const char *path, int flags, sen_encoding encoding)
{
  sen_db *s;
  char buffer[PATH_MAX];
  if (strlen(path) > PATH_MAX - 14) { return NULL; }
  if (!(s = SEN_MALLOC(sizeof(sen_db)))) { return NULL; }
  if ((s->stores = sen_set_open(sizeof(sen_id), sizeof(sen_db_store), 0))) {
    if ((s->keys = sen_sym_create(path, 0, flags, encoding))) {
      gen_pathname(path, buffer, 0);
      if ((s->values = sen_ja_create(buffer, JA_SEGMENT_SIZE))) {
        sen_db_prepare_builtin_class(s);
        SEN_LOG(sen_log_notice, "db created (%s) flags=%x", path, s->keys->flags);
        return s;
      }
      sen_sym_close(s->keys);
    }
    sen_set_close(s->stores);
  }
  SEN_FREE(s);
  return NULL;
}

sen_db *
sen_db_open(const char *path)
{
  sen_db *s;
  char buffer[PATH_MAX];
  if (strlen(path) > PATH_MAX - 14) { return NULL; }
  if (!(s = SEN_MALLOC(sizeof(sen_db)))) { return NULL; }
  if ((s->stores = sen_set_open(sizeof(sen_id), sizeof(sen_db_store), 0))) {
    if ((s->keys = sen_sym_open(path))) {
      gen_pathname(path, buffer, 0);
      if ((s->values = sen_ja_open(buffer))) {
        SEN_LOG(sen_log_notice, "db opened (%s) flags=%x", path, s->keys->flags);
        return s;
      }
      sen_sym_close(s->keys);
    }
    sen_set_close(s->stores);
  }
  SEN_FREE(s);
  return NULL;
}

sen_rc
sen_db_close(sen_db *s)
{
  sen_db_store *e;
  sen_set_cursor *c;
  sen_sym_close(s->keys);
  sen_ja_close(s->values);
  if ((c = sen_set_cursor_open(s->stores))) {
    while (sen_set_cursor_next(c, NULL, (void **) &e)) {
      sen_db_store_close(e, 1);
    }
    sen_set_cursor_close(c);
  }
  sen_set_close(s->stores);
  SEN_FREE(s);
  return sen_success;
}

/**** vgram ****/

static int len_sum = 0;
static int img_sum = 0;
static int simple_sum = 0;
static int skip_sum = 0;

sen_vgram *
sen_vgram_create(const char *path)
{
  sen_vgram *s;
  if (!(s = SEN_MALLOC(sizeof(sen_vgram)))) { return NULL; }
  s->vgram = sen_sym_create(path, sizeof(sen_id) * 2, 0, sen_enc_none);
  if (!s->vgram) {
    SEN_FREE(s);
    return NULL;
  }
  return s;
}

sen_vgram *
sen_vgram_open(const char *path)
{
  sen_vgram *s;
  if (!(s = SEN_MALLOC(sizeof(sen_vgram)))) { return NULL; }
  s->vgram = sen_sym_open(path);
  if (!s->vgram) {
    SEN_FREE(s);
    return NULL;
  }
  return s;
}

sen_vgram_buf *
sen_vgram_buf_open(size_t len)
{
  sen_vgram_buf *b;
  if (!(b = SEN_MALLOC(sizeof(sen_vgram_buf)))) { return NULL; }
  b->len = len;
  b->tvs = b->tvp = SEN_MALLOC(sizeof(sen_id) * len);
  if (!b->tvp) { SEN_FREE(b); return NULL; }
  b->tve = b->tvs + len;
  b->vps = b->vpp = SEN_MALLOC(sizeof(sen_vgram_vnode) * len * 2);
  if (!b->vpp) { SEN_FREE(b->tvp); SEN_FREE(b); return NULL; }
  b->vpe = b->vps + len;
  return b;
}

sen_rc
sen_vgram_buf_add(sen_vgram_buf *b, sen_id tid)
{
  uint8_t dummybuf[8], *dummyp;
  if (b->tvp < b->tve) { *b->tvp++ = tid; }
  dummyp = dummybuf;
  SEN_B_ENC(tid, dummyp);
  simple_sum += dummyp - dummybuf;
  return sen_success;
}

typedef struct {
  sen_id vid;
  sen_id tid;
} vgram_key;

sen_rc
sen_vgram_update(sen_vgram *vgram, sen_id rid, sen_vgram_buf *b, sen_set *terms)
{
  sen_inv_updspec **u;
  if (b && b->tvs < b->tvp) {
    sen_id *t0, *tn;
    for (t0 = b->tvs; t0 < b->tvp - 1; t0++) {
      sen_vgram_vnode *v, **vp;
      sen_set_at(terms, t0, (void **) &u);
      vp = &(*u)->vnodes;
      for (tn = t0 + 1; tn < b->tvp; tn++) {
        for (v = *vp; v && v->tid != *tn; v = v->cdr) ;
        if (!v) {
          if (b->vpp < b->vpe) {
            v = b->vpp++;
          } else {
            // todo;
            break;
          }
          v->car = NULL;
          v->cdr = *vp;
          *vp = v;
          v->tid = *tn;
          v->vid = 0;
          v->freq = 0;
          v->len = tn - t0;
        }
        v->freq++;
        if (v->vid) {
          vp = &v->car;
        } else {
          break;
        }
      }
    }
    {
      sen_set *th = sen_set_open(sizeof(sen_id), sizeof(int), 0);
      if (!th) { return sen_memory_exhausted; }
      if (t0 == b->tvp) { SEN_LOG(sen_log_debug, "t0 == tvp"); }
      for (t0 = b->tvs; t0 < b->tvp; t0++) {
        sen_id vid, vid0 = *t0, vid1 = 0;
        sen_vgram_vnode *v, *v2 = NULL, **vp;
        sen_set_at(terms, t0, (void **) &u);
        vp = &(*u)->vnodes;
        for (tn = t0 + 1; tn < b->tvp; tn++) {
          for (v = *vp; v; v = v->cdr) {
            if (!v->vid && (v->freq < 2 || v->freq * v->len < 4)) {
              *vp = v->cdr;
              v->freq = 0;
            }
            if (v->tid == *tn) { break; }
            vp = &v->cdr;
          }
          if (v) {
            if (v->freq) {
              v2 = v;
              vid1 = vid0;
              vid0 = v->vid;
            }
            if (v->vid) {
              vp = &v->car;
              continue;
            }
          }
          break;
        }
        if (v2) {
          if (!v2->vid) {
            vgram_key key;
            key.vid = vid1;
            key.tid = v2->tid;
            v2->vid = sen_sym_get(vgram->vgram, (char *)&key);
          }
          vid = *t0 = v2->vid * 2 + 1;
          memset(t0 + 1, 0, sizeof(sen_id) * v2->len);
          t0 += v2->len;
        } else {
          vid = *t0 *= 2;
        }
        {
          int *tf;
          sen_set_get(th, &vid, (void **) &tf);
          (*tf)++;
        }
      }
      if (!th->n_entries) { SEN_LOG(sen_log_debug, "th->n_entries == 0"); }
      {
        int j = 0;
        int skip = 0;
        sen_set_eh *ehs, *ehp, *ehe;
        sen_set_sort_optarg arg;
        uint8_t *ps = SEN_MALLOC(b->len * 2), *pp, *pe;
        if (!ps) {
          sen_set_close(th);
          return sen_memory_exhausted;
        }
        pp = ps;
        pe = ps + b->len * 2;
        arg.mode = sen_sort_descending;
        arg.compar = NULL;
        arg.compar_arg = (void *)(intptr_t)sizeof(sen_id);
        arg.compar_arg0 = NULL;
        ehs = sen_set_sort(th, 0, &arg);
        if (!ehs) {
          SEN_FREE(ps);
          sen_set_close(th);
          return sen_memory_exhausted;
        }
        SEN_B_ENC(th->n_entries, pp);
        for (ehp = ehs, ehe = ehs + th->n_entries; ehp < ehe; ehp++, j++) {
          int *id = (int *)SEN_SET_INTVAL(*ehp);
          SEN_B_ENC(*SEN_SET_INTKEY(*ehp), pp);
          *id = j;
        }
        for (t0 = b->tvs; t0 < b->tvp; t0++) {
          if (*t0) {
            int *id;
            if (!sen_set_at(th, t0, (void **) &id)) {
              SEN_LOG(sen_log_error, "lookup error (%d)", *t0);
            }
            SEN_B_ENC(*id, pp);
          } else {
            skip++;
          }
        }
        len_sum += b->len;
        img_sum += pp - ps;
        skip_sum += skip;
        SEN_FREE(ehs);
        SEN_FREE(ps);
      }
      sen_set_close(th);
    }
  }
  return sen_success;
}

sen_rc
sen_vgram_buf_close(sen_vgram_buf *b)
{
  if (!b) { return sen_invalid_argument; }
  if (b->tvs) { SEN_FREE(b->tvs); }
  if (b->vps) { SEN_FREE(b->vps); }
  SEN_FREE(b);
  return sen_success;
}

sen_rc
sen_vgram_close(sen_vgram *vgram)
{
  if (!vgram) { return sen_invalid_argument; }
  SEN_LOG(sen_log_debug, "len=%d img=%d skip=%d simple=%d", len_sum, img_sum, skip_sum, simple_sum);
  sen_sym_close(vgram->vgram);
  SEN_FREE(vgram);
  return sen_success;
}
