/* Copyright(C) 2006 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*  Senna Query Language is based on Mini-Scheme, original credits follow  */

/*
 *      ---------- Mini-Scheme Interpreter Version 0.85 ----------
 *
 *                coded by Atsushi Moriwaki (11/5/1989)
 *
 *            E-MAIL :  moriwaki@kurims.kurims.kyoto-u.ac.jp
 *
 *               THIS SOFTWARE IS IN THE PUBLIC DOMAIN
 *               ------------------------------------
 * This software is completely free to copy, modify and/or re-distribute.
 * But I would appreciate it if you left my name on the code as the author.
 *
 */
/*--
 *
 *  This version has been modified by R.C. Secrist.
 *
 *  Mini-Scheme is now maintained by Akira KIDA.
 *
 *  This is a revised and modified version by Akira KIDA.
 *  current version is 0.85k4 (15 May 1994)
 *
 *  Please send suggestions, bug reports and/or requests to:
 *    <SDI00379@niftyserve.or.jp>
 *--
 */

#include "senna_in.h"
#include <fcntl.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "ql.h"

#define InitFile "init.scm"

/* global variables */

sen_obj *sen_ql_nil;  /* special cell representing empty cell */
sen_obj *sen_ql_t;    /* special cell representing #t */
sen_obj *sen_ql_f;    /* special cell representing #f */

/* sen query language */

inline static void
obj_ref(sen_obj *o)
{
  if (o->nrefs < 0xffff) { o->nrefs++; }
  if (LISTP(o)) { // todo : check cycle
    if (CAR(o) != NIL) { obj_ref(CAR(o)); }
    if (CDR(o) != NIL) { obj_ref(CDR(o)); }
  }
}

inline static void
obj_unref(sen_obj *o)
{
  if (!o->nrefs) {
    SEN_LOG(sen_log_error, "o->nrefs corrupt");
    return;
  }
  if (o->nrefs < 0xffff) { o->nrefs--; }
  if (LISTP(o)) { // todo : check cycle
    if (CAR(o) != NIL) { obj_unref(CAR(o)); }
    if (CDR(o) != NIL) { obj_unref(CDR(o)); }
  }
}

/* todo : update set-car! set-cdr!
inline static void
rplaca(sen_ctx *c, sen_obj *a, sen_obj *b)
{
  if (a->nrefs) {
    c->nbinds++;
    if (a->u.l.car) {
      c->nunbinds++;
      obj_unref(a->u.l.car);
    }
    if (b) { obj_ref(b); }
  }
  a->u.l.car = b;
}

inline static void
rplacd(sen_ctx *c, sen_obj *a, sen_obj *b)
{
  if (a->nrefs) {
    c->nbinds++;
    if (a->u.l.cdr) {
      c->nunbinds++;
      obj_unref(a->u.l.cdr);
    }
    if (b) { obj_ref(b); }
  }
  a->u.l.cdr = b;
}
*/

sen_rc
sen_obj2int(sen_obj *o)
{
  sen_rc rc = sen_invalid_argument;
  if (o) {
    switch (o->type) {
    case sen_ql_bulk :
      if (o->u.b.size) {
        const char *end = o->u.b.value + o->u.b.size, *rest;
        int i = sen_atoi(o->u.b.value, end, &rest);
        if (rest == end) {
          sen_obj_clear(o);
          o->type = sen_ql_int;
          o->u.i.i = i;
          rc = sen_success;
        }
      }
      break;
    case sen_ql_int :
      rc = sen_success;
      break;
    default :
      break;
    }
  }
  return rc;
}

#define symname(p)      (SEN_SET_STRKEY_BY_VAL(p))

/* get new symbol */
sen_obj *
sen_ql_mk_symbol(sen_ctx *c, const char *name)
{
  sen_obj *x;
  sen_set_eh *eh = sen_set_get(c->symbols, name, (void **) &x);
  if (!eh) { /* todo : must be handled */ }
  if (!x->flags) {
    x->flags |= SEN_OBJ_SYMBOL;
    x->type = sen_ql_void;
  }
  if (x->type == sen_ql_void) {
    sen_db_store *slot = sen_db_store_open(c->db, symname(x));
    if (slot) { sen_ql_bind_symbol(slot, x); }
  }
  return x;
}

sen_obj *
sen_ql_at(sen_ctx *c, const char *key)
{
  sen_obj *o;
  if (!sen_set_at(c->symbols, key, (void **) &o)) {
    return NULL;
  }
  return o;
}

void
sen_ql_def_native_method(sen_ctx *c, const char *name, sen_ql_method_func *func)
{
  sen_obj *o = sen_ql_mk_symbol(c, name);
  o->type = sen_ql_native_method;
  o->flags |= SEN_OBJ_NATIVE;
  o->u.o.func = func;
}

inline static void
sen_ctx_igc(sen_ctx *c)
{
  uint32_t i;
  sen_obj *o;
  sen_set_eh *ep;
  for (i = c->lseqno; i != c->seqno; i++) {
    if ((ep = sen_set_at(c->objects, &i, (void **) &o))) {
      if (c->nbinds &&
          (o->nrefs ||
           (BULKP(o) && (o->flags & SEN_OBJ_ALLOCATED)))) { continue; }
      sen_obj_clear(o);
      sen_set_del(c->objects, ep);
    }
  }
  c->lseqno = c->seqno;
  c->nbinds = 0;
}

#define ismark(x) ((x)->flags & SEN_OBJ_MARKED)

#define car(p)          ((p)->u.l.car)
#define cdr(p)          ((p)->u.l.cdr)


#define type(p)         ((p)->type)
#define flags(p)         ((p)->flags)

#define isreferer(p)    (flags(p) & SEN_OBJ_REFERER)
#define setreferer(p)   (flags(p) |= SEN_OBJ_REFERER)
#define unsetreferer(p)   (flags(p) &= ~SEN_OBJ_REFERER)

/*--
 *  We use algorithm E (Knuth, The Art of Computer Programming Vol.1,
 *  sec.3.5) for marking.
 */
inline static void
obj_mark(sen_obj *o)
{
  sen_obj *t, *q, *p;
  t = NULL;
  p = o;
  // if (ismark(o)) { return; }
E2:
  p->flags |= SEN_OBJ_MARKED;
  // if (!o->nrefs) { SEN_LOG(sen_log_error, "obj->nrefs corrupt"); }
  if (BULKP(o) && !(o->flags & SEN_OBJ_ALLOCATED)) {
    char *b = SEN_MALLOC(o->u.b.size + 1);
    memcpy(b, o->u.b.value, o->u.b.size);
    b[o->u.b.size] = '\0';
    o->u.b.value = b;
    o->flags |= SEN_OBJ_ALLOCATED;
  }
  if (!isreferer(p)) { goto E6; }
  q = car(p);
  if (q && !ismark(q)) {
    unsetreferer(p);
    car(p) = t;
    t = p;
    p = q;
    goto E2;
  }
E5:
  q = cdr(p);
  if (q && !ismark(q)) {
    cdr(p) = t;
    t = p;
    p = q;
    goto E2;
  }
E6:
  if (!t) { return; }
  q = t;
  if (!isreferer(q)) {
    setreferer(q);
    t = car(q);
    car(q) = p;
    p = q;
    goto E5;
  } else {
    t = cdr(q);
    cdr(q) = p;
    p = q;
    goto E6;
  }
}

inline static sen_rc
sen_ctx_mgc(sen_ctx *c)
{
  sen_set_cursor *sc;
  /*
  if (!(sc = sen_set_cursor_open(c->symbols))) { return sen_memory_exhausted; }
  {
    sen_obj *o;
    while (sen_set_cursor_next(sc, NULL, (void **) &o)) { obj_mark(o); }
    sen_set_cursor_close(sc);
  }
  */
  obj_mark(c->global_env);

  /* mark current registers */
  obj_mark(c->args);
  obj_mark(c->envir);
  obj_mark(c->code);
  obj_mark(c->dump);
  //  obj_mark(c->value);
  //  obj_mark(c->phs);

  if (!(sc = sen_set_cursor_open(c->objects))) { return sen_memory_exhausted; }
  {
    sen_obj *o;
    sen_set_eh *ep;
    while ((ep = sen_set_cursor_next(sc, NULL, (void **) &o))) {
      if (o->flags & SEN_OBJ_MARKED) {
        o->flags &= ~SEN_OBJ_MARKED;
      } else {
        sen_obj_clear(o);
        sen_set_del(c->objects, ep);
      }
    }
  }
  sen_set_cursor_close(sc);
  c->lseqno = c->seqno;
  c->nbinds = 0;
  c->nunbinds = 0;
  return sen_success;
}

static void Eval_Cycle(sen_ctx *c);

/* ========== Evaluation Cycle ========== */

/* operator code */
#define  OP_LOAD      0
#define  OP_T0LVL    1
#define  OP_T1LVL    2
#define  OP_READ      3
#define  OP_VALUEPRINT    4
#define  OP_EVAL      5
#define  OP_E0ARGS    6
#define  OP_E1ARGS    7
#define  OP_APPLY    8
#define  OP_DOMACRO    9
#define  OP_LAMBDA    10
#define  OP_QUOTE    11
#define  OP_DEF0      12
#define  OP_DEF1      13
#define  OP_BEGIN    14
#define  OP_IF0      15
#define  OP_IF1      16
#define  OP_SET0      17
#define  OP_SET1      18
#define  OP_LET0      19
#define  OP_LET1      20
#define  OP_LET2      21
#define  OP_LET0AST    22
#define  OP_LET1AST    23
#define  OP_LET2AST    24
#define  OP_LET0REC    25
#define  OP_LET1REC    26
#define  OP_LET2REC    27
#define  OP_COND0    28
#define  OP_COND1    29
#define  OP_DELAY    30
#define  OP_AND0      31
#define  OP_AND1      32
#define  OP_OR0      33
#define  OP_OR1      34
#define  OP_C0STREAM    35
#define  OP_C1STREAM    36
#define  OP_0MACRO    37
#define  OP_1MACRO    38
#define  OP_CASE0    39
#define  OP_CASE1    40
#define  OP_CASE2    41
#define  OP_PEVAL    42
#define  OP_PAPPLY    43
#define  OP_CONTINUATION    44
#define  OP_ADD      45
#define  OP_SUB      46
#define  OP_MUL      47
#define  OP_DIV      48
#define  OP_REM      49
#define  OP_CAR      50
#define  OP_CDR      51
#define  OP_CONS      52
#define  OP_SETCAR    53
#define  OP_SETCDR    54
#define  OP_NOT      55
#define  OP_BOOL      56
#define  OP_NULL      57
#define  OP_ZEROP    58
#define  OP_POSP      59
#define  OP_NEGP      60
#define  OP_NEQ      61
#define  OP_LESS      62
#define  OP_GRE      63
#define  OP_LEQ      64
#define  OP_GEQ      65
#define  OP_SYMBOL    66
#define  OP_NUMBER    67
#define  OP_STRING    68
#define  OP_PROC      69
#define  OP_PAIR      70
#define  OP_EQ      71
#define  OP_EQV      72
#define  OP_FORCE    73
#define  OP_WRITE    74
#define  OP_DISPLAY    75
#define  OP_NEWLINE    76
#define  OP_ERR0      77
#define  OP_ERR1      78
#define  OP_REVERSE    79
#define  OP_APPEND    80
#define  OP_PUT      81
#define  OP_GET      82
#define  OP_QUIT      83
#define  OP_GC      84
#define  OP_GCVERB    85
#define  OP_NATIVEP    86
#define  OP_RDSEXPR    87
#define  OP_RDLIST    88
#define  OP_RDDOT    89
#define  OP_RDQUOTE    90
#define  OP_RDQQUOTE    91
#define  OP_RDUNQUOTE    92
#define  OP_RDUQTSP    93


#define  OP_LIST_LENGTH    96
#define  OP_ASSQ      97



#define  OP_GET_CLOSURE    101
#define  OP_CLOSUREP    102
#define  OP_MACROP    103
#define  OP_NATIVE    104
#define  OP_VOIDP    105

sen_obj *
sen_ql_feed(sen_ctx *c, char *str, uint32_t str_size, int mode)
{
  if (SEN_QL_WAITINGP(c)) {
    SEN_RBUF_REWIND(&c->outbuf);
    SEN_RBUF_REWIND(&c->subbuf);
    c->bufcur = 0;
  }
  // op = setjmp(c->error_jmp);
  for (;;) {
    switch (c->stat) {
    case SEN_QL_TOPLEVEL :
      c->co.mode &= ~SEN_QL_HEAD;
      Eval_Cycle(c);
      break;
    case SEN_QL_WAIT_EXPR :
      c->co.mode = mode;
      c->cur = str;
      c->str_end = str + str_size;
      Eval_Cycle(c);
      break;
    case SEN_QL_WAIT_ARG :
      c->co.mode = mode;
      if ((mode & SEN_QL_HEAD)) {
        c->cur = str;
        c->str_end = str + str_size;
      } else {
        char *buf;
        sen_obj *ph = CAR(c->phs);
        if (!(buf = SEN_MALLOC(str_size + 1))) {
          return NIL; /* longjmp(c->error_jmp, 0); */
        }
        memcpy(buf, str, str_size);
        buf[str_size] = '\0';
        ph->flags |= SEN_OBJ_ALLOCATED;
        ph->u.b.value = buf;
        ph->u.b.size = str_size;
        c->phs = CDR(c->phs);
      }
      if ((c->phs == NIL) || (mode & (SEN_QL_HEAD|SEN_QL_TAIL))) {
        c->stat = SEN_QL_EVAL;
      }
      break;
    case SEN_QL_EVAL :
      Eval_Cycle(c);
      break;
    case SEN_QL_WAIT_DATA :
      c->co.mode = mode;
      if ((mode & SEN_QL_HEAD)) {
        c->args = NIL;
        c->cur = str;
        c->str_end = str + str_size;
      } else {
        c->arg.u.b.value = str;
        c->arg.u.b.size = str_size;
        c->arg.type = sen_ql_bulk;
        c->args = &c->arg;
      }
      /* fall through */
    case SEN_QL_NATIVE :
      SEN_ASSERT(c->doing);
      c->value = c->doing(c, c->args, &c->co);
      if (c->co.last && !(c->co.mode & (SEN_QL_HEAD|SEN_QL_TAIL))) {
        c->stat = SEN_QL_WAIT_DATA;
      } else {
        c->co.mode = 0;
        Eval_Cycle(c);
      }
      break;
    }
    if (SEN_QL_WAITINGP(c)) { /* waiting input data */
      if (c->inbuf) {
        SEN_FREE(c->inbuf);
        c->inbuf = NULL;
      }
      break;
    }
    if ((c->stat & 0x40) && SEN_QL_GET_MODE(c) == sen_ql_step) {
      break;
    }
  }
  return NIL;
}

/**** sexp parser ****/

typedef sen_obj cell;

inline static cell *
cons(sen_ctx *c, cell *a, cell *b)
{
  cell *o = sen_obj_new(c);
  o->type = sen_ql_list;
  o->flags = SEN_OBJ_REFERER;
  car(o) = a;
  cdr(o) = b;
  return o;
}

inline static void
skipspace(sen_ctx *c)
{
  unsigned int len;
  while (c->cur < c->str_end && sen_isspace(c->cur, c->encoding)) {
    /* null check and length check */
    if (!(len = sen_str_charlen_nonnull(c->cur, c->str_end, c->encoding))) {
      c->cur = c->str_end;
      break;
    }
    c->cur += len;
  }
}

inline static void
skipline(sen_ctx *c)
{
  while (c->cur < c->str_end) {
    if (*c->cur++ == '\n') { break; }
  }
}

/*************** scheme interpreter ***************/

# define BACKQUOTE '`'

#include <stdio.h>
#include <ctype.h>

/* macros for cell operations */
#define strvalue(p)     ((p)->u.b.value)
#define keynum(p)       ((p)->class)

#define isnumber(p)     (type(p) == sen_ql_int)
#define ivalue(p)       ((p)->u.i.i)

#define ispair(p)       (type(p) & sen_ql_list)

#define issymbol(p)     (flags(p) & SEN_OBJ_SYMBOL)

#define hasprop(p)      (flags(p) & SEN_OBJ_SYMBOL)
#define symprop(p)      cdr(p)

#define issyntax(p)     (type(p) == sen_ql_syntax)
#define isproc(p)       (type(p) == sen_ql_proc)
#define syntaxname(p)   symname(p)
#define syntaxnum(p)    keynum(p)
#define procnum(p)      ivalue(p)

#define isclosure(p)    (type(p) == sen_ql_closure)
#define ismacro(p)      (flags(p) & SEN_OBJ_MACRO)
#define closure_code(p) car(p)
#define closure_env(p)  cdr(p)

#define iscontinuation(p) (type(p) == sen_ql_continuation)
#define cont_dump(p)    cdr(p)

#define ispromise(p)    (flags(p) & SEN_OBJ_PROMISE)
#define setpromise(p)   flags(p) |= SEN_OBJ_PROMISE


#define caar(p)         car(car(p))
#define cadr(p)         car(cdr(p))
#define cdar(p)         cdr(car(p))
#define cddr(p)         cdr(cdr(p))
#define cadar(p)        car(cdr(car(p)))
#define caddr(p)        car(cdr(cdr(p)))
#define cadaar(p)       car(cdr(car(car(p))))
#define cadddr(p)       car(cdr(cdr(cdr(p))))
#define cddddr(p)       cdr(cdr(cdr(cdr(p))))

#define LAMBDA (sen_ql_mk_symbol(c, "lambda"))
#define QUOTE  (sen_ql_mk_symbol(c, "quote"))
#define QQUOTE (sen_ql_mk_symbol(c, "quasiquote"))
#define UNQUOTE (sen_ql_mk_symbol(c, "unquote"))
#define UNQUOTESP (sen_ql_mk_symbol(c, "unquote-splicing"))

/* get new cell.  parameter a, b is marked by gc. */
inline static cell *
get_cell(sen_ctx *c, cell *a, cell *b)
{
  return sen_obj_new(c);
}

/* get number atom */
inline static cell *
mk_number(sen_ctx *c, int num)
{
  cell *x = sen_obj_new(c);
  x->type = sen_ql_int;
  x->u.i.i = num;
  return x;
}

/* get new string */
sen_obj *
sen_ql_mk_string(sen_ctx *c, const char *str, unsigned int len)
{
  cell *x = sen_obj_alloc(c, len);
  memcpy(x->u.b.value, str, len);
  x->u.b.value[len] = '\0';
  return x;
}

inline static cell *
mk_const_string(sen_ctx *c, const char *str)
{
  cell *x = sen_obj_new(c);
  x->flags = 0;
  x->type = sen_ql_bulk;
  x->u.b.value = (char *)str;
  x->u.b.size = strlen(str);
  return x;
}

inline static cell *
sen_ql_mk_symbol2(sen_ctx *ctx, const char *q, unsigned int len, int kwdp)
{
  char buf[SEN_SYM_MAX_KEY_SIZE], *p = buf;
  if (len + 1 >= SEN_SYM_MAX_KEY_SIZE) { return NIL; /* todo : too long */ }
  if (kwdp) { *p++ = ':'; }
  memcpy(p, q, len);
  p[len] = '\0';
  return sen_ql_mk_symbol(ctx, buf);
}

/* make symbol or number atom from string */
inline static cell *
mk_atom(sen_ctx *c, char *str, unsigned int len, cell *v)
{
  cell **vp = &v;
  const char *cur, *last, *str_end = str + len;
  int ivalue = sen_atoi(str, str_end, &cur);
  if (cur == str_end) { return mk_number(c, ivalue); }
  for (last = cur = str; cur < str_end; cur += len) {
    if (!(len = sen_str_charlen_nonnull(cur, str_end, c->encoding))) { break; }
    if (*cur == '.') {
      if (last < cur) { *vp = sen_ql_mk_symbol2(c, last, cur - last, str != last); }
      v = cons(c, v, cons(c, NIL, NIL));
      vp = &CADR(v);
      last = cur + 1;
    }
  }
  if (last < cur) { *vp = sen_ql_mk_symbol2(c, last, cur - last, str != last); }
  return v;
}

/* make constant */
inline static cell *
mk_const(sen_ctx *c, char *name, unsigned int len)
{
  long    x;
  char    tmp[256];
  char    tmp2[256];
  /* todo : rewirte with sen_str_* functions */
  if (len == 1) {
    if (*name == 't') {
      return T;
    } else if (*name == 'f') {
      return F;
    }
  } else if (len > 1) {
    if (*name == 'p' && name[1] == '<' && name[12] == '>') {/* #p (sen_ql_object) */
      sen_id cls = sen_str_btoi(name + 2);
      if (cls) {
        sen_id self = sen_str_btoi(name + 7);
        if (self) {
          cell * v = sen_ql_mk_obj(c, cls, self);
          if (len > 13 && name[13] == '.') {
            return mk_atom(c, name + 13, len - 13, v);
          } else {
            return v;
          }
        }
      }
    } else if (*name == 'o') {/* #o (octal) */
      len = (len > 255) ? 255 : len - 1;
      memcpy(tmp2, name + 1, len);
      tmp2[len] = '\0';
      sprintf(tmp, "0%s", tmp2);
      sscanf(tmp, "%lo", &x);
      return mk_number(c, x);
    } else if (*name == 'd') {  /* #d (decimal) */
      sscanf(&name[1], "%ld", &x);
      return mk_number(c, x);
    } else if (*name == 'x') {  /* #x (hex) */
      len = (len > 255) ? 255 : len - 1;
      memcpy(tmp2, name + 1, len);
      tmp2[len] = '\0';
      sprintf(tmp, "0x%s", tmp2);
      sscanf(tmp, "%lx", &x);
      return mk_number(c, x);
    }
  }
  return NIL;
}

void
sen_ctx_concat_func(sen_ctx *c, int flags, void *dummy)
{
  if (flags & SEN_CTX_MORE) {
    unsigned int size = SEN_RBUF_VSIZE(&c->outbuf);
    sen_rbuf_write(&c->subbuf, (char *) &size, sizeof(unsigned int));
  }
}

void
sen_ctx_stream_out_func(sen_ctx *c, int flags, void *stream)
{
  sen_rbuf *buf = &c->outbuf;
  uint32_t size = SEN_RBUF_VSIZE(buf);
  if (size) {
    fwrite(buf->head, 1, size, (FILE *)stream);
    fputc('\n', (FILE *)stream);
  }
  SEN_RBUF_REWIND(buf);
}

sen_rc
sen_ctx_load(sen_ctx *c, const char *filename)
{
  if (!filename) { filename = InitFile; }
  c->args = cons(c, mk_const_string(c, filename), NIL);
  c->stat = SEN_QL_TOPLEVEL;
  c->op = 0; //OP_LOAD
  return sen_ql_feed(c, "init", 4, 0) == NIL ? sen_success : sen_other_error;
}

/* ========== Routines for Reading ========== */

#define TOK_LPAREN  0
#define TOK_RPAREN  1
#define TOK_DOT     2
#define TOK_ATOM    3
#define TOK_QUOTE   4
#define TOK_COMMENT 5
#define TOK_DQUOTE  6
#define TOK_BQUOTE  7
#define TOK_COMMA   8
#define TOK_ATMARK  9
#define TOK_SHARP   10
#define TOK_EOS     11
#define TOK_QUESTION 12

#define lparenp(c) ((c) == '(' || (c) == '[')
#define rparenp(c) ((c) == ')' || (c) == ']')

/* read chacters to delimiter */
inline static char
readstr(sen_ctx *c, char **str, unsigned int *size)
{
  char *start, *end;
  for (start = end = c->cur;;) {
    unsigned int len;
    /* null check and length check */
    if (!(len = sen_str_charlen_nonnull(end, c->str_end, c->encoding))) {
      c->cur = c->str_end;
      break;
    }
    if (sen_isspace(end, c->encoding) ||
        *end == ';' || lparenp(*end) || rparenp(*end)) {
      c->cur = end;
      break;
    }
    end += len;
  }
  if (start < end || c->cur < c->str_end) {
    *str = start;
    *size = (unsigned int)(end - start);
    return TOK_ATOM;
  } else {
    return TOK_EOS;
  }
}

/* read string expression "xxx...xxx" */
inline static char
readstrexp(sen_ctx *c, char **str, unsigned int *size)
{
  char *start, *src, *dest;
  for (start = src = dest = c->cur;;) {
    unsigned int len;
    /* null check and length check */
    if (!(len = sen_str_charlen_nonnull(src, c->str_end, c->encoding))) {
      c->cur = c->str_end;
      if (start < dest) {
        *str = start;
        *size = (unsigned int)(dest - start);
        return TOK_ATOM;
      }
      return TOK_EOS;
    }
    if (src[0] == '"' && len == 1) {
      c->cur = src + 1;
      *str = start;
      *size = (unsigned int)(dest - start);
      return TOK_ATOM;
    } else if (src[0] == '\\' && src + 1 < c->str_end && len == 1) {
      src++;
      *dest++ = *src++;
    } else {
      while (len--) { *dest++ = *src++; }
    }
  }
}

/* get token */
inline static char
token(sen_ctx *c)
{
  skipspace(c);
  if (c->cur >= c->str_end) { return TOK_EOS; }
  switch (*c->cur) {
  case '(':
  case '[':
    c->cur++;
    return TOK_LPAREN;
  case ')':
  case ']':
    c->cur++;
    return TOK_RPAREN;
  case '.':
    c->cur++;
    if (c->cur == c->str_end ||
        sen_isspace(c->cur, c->encoding) ||
        *c->cur == ';' || lparenp(*c->cur) || rparenp(*c->cur)) {
      return TOK_DOT;
    } else {
      c->cur--;
      return TOK_ATOM;
    }
  case '\'':
    c->cur++;
    return TOK_QUOTE;
  case ';':
    c->cur++;
    return TOK_COMMENT;
  case '"':
    c->cur++;
    return TOK_DQUOTE;
  case BACKQUOTE:
    c->cur++;
    return TOK_BQUOTE;
  case ',':
    c->cur++;
    if (c->cur < c->str_end && *c->cur == '@') {
      c->cur++;
      return TOK_ATMARK;
    } else {
      return TOK_COMMA;
    }
  case '#':
    c->cur++;
    return TOK_SHARP;
  case '?':
    c->cur++;
    return TOK_QUESTION;
  default:
    return TOK_ATOM;
  }
}

/* ========== Routines for Printing ========== */
#define  ok_abbrev(x)  (ispair(x) && cdr(x) == NIL)

void
sen_obj_inspect(sen_ctx *c, sen_obj *obj, sen_rbuf *buf, int flags)
{
  if (!obj) {
    SEN_RBUF_PUTS(buf, "NULL");
  } else if (obj == NIL) {
    SEN_RBUF_PUTS(buf, "()");
  } else if (obj == T) {
    SEN_RBUF_PUTS(buf, "#t");
  } else if (obj == F) {
    SEN_RBUF_PUTS(buf, "#f");
  } else {
    if (issymbol(obj)) {
      const char *sym = SEN_SET_STRKEY_BY_VAL(obj);
      if (sym) {
        if (flags & SEN_OBJ_INSPECT_SYM_AS_STR) {
          sen_rbuf_str_esc(buf, (*sym == ':') ? sym + 1 : sym, -1, c->encoding);
        } else {
          SEN_RBUF_PUTS(buf, sym);
        }
        return;
      }
    }
    switch (obj->type) {
    case sen_ql_void :
      SEN_RBUF_PUTS(buf, issymbol(obj) ? SEN_SET_STRKEY_BY_VAL(obj) : "#<VOID>");
      break;
    case sen_ql_object :
      if (flags & SEN_OBJ_INSPECT_ESC) {
        SEN_RBUF_PUTS(buf, "#p<");
        sen_rbuf_itob(buf, obj->class);
        sen_rbuf_itob(buf, obj->u.o.self);
        SEN_RBUF_PUTC(buf, '>');
      } else {
        SEN_RBUF_PUTS(buf, _sen_obj_key(c, obj));
      }
      break;
    case sen_ql_records :
      SEN_RBUF_PUTS(buf, "#<RECORDS>");
      break;
    case sen_ql_bulk :
      if (flags & SEN_OBJ_INSPECT_ESC) {
        sen_rbuf_str_esc(buf, obj->u.b.value, obj->u.b.size, c->encoding);
      } else {
        sen_rbuf_write(buf, obj->u.b.value, obj->u.b.size);
      }
      break;
    case sen_ql_int :
      sen_rbuf_itoa(buf, obj->u.i.i);
      break;
    case sen_ql_native_method :
      SEN_RBUF_PUTS(buf, "#<NATIVE_METHOD>");
      break;
    case sen_ql_query :
      SEN_RBUF_PUTS(buf, "#<QUERY>");
      break;
    case sen_ql_op :
      SEN_RBUF_PUTS(buf, "#<OP>");
      break;
    case sen_ql_syntax :
      SEN_RBUF_PUTS(buf, "#<SYNTAX>");
      break;
    case sen_ql_proc :
      SEN_RBUF_PUTS(buf, "#<PROCEDURE ");
      sen_rbuf_itoa(buf, procnum(obj));
      SEN_RBUF_PUTS(buf, ">");
      break;
    case sen_ql_closure :
      if (ismacro(obj)) {
        SEN_RBUF_PUTS(buf, "#<MACRO>");
      } else {
        SEN_RBUF_PUTS(buf, "#<CLOSURE>");
      }
      break;
    case sen_ql_continuation :
      SEN_RBUF_PUTS(buf, "#<CONTINUATION>");
      break;
    case sen_db_raw_class :
      SEN_RBUF_PUTS(buf, "#<RAW_CLASS>");
      break;
    case sen_db_class :
      SEN_RBUF_PUTS(buf, "#<CLASS>");
      break;
    case sen_db_obj_slot :
      SEN_RBUF_PUTS(buf, "#<OBJ_SLOT>");
      break;
    case sen_db_ra_slot :
      SEN_RBUF_PUTS(buf, "#<RA_SLOT>");
      break;
    case sen_db_ja_slot :
      SEN_RBUF_PUTS(buf, "#<JA_SLOT>");
      break;
    case sen_db_idx_slot :
      SEN_RBUF_PUTS(buf, "#<IDX_SLOT>");
      break;
    case sen_ql_list :
      /* todo : detect loop */
      if (car(obj) == QUOTE && ok_abbrev(cdr(obj))) {
        SEN_RBUF_PUTC(buf, '\'');
        sen_obj_inspect(c, cadr(obj), buf, flags);
      } else if (car(obj) == QQUOTE && ok_abbrev(cdr(obj))) {
        SEN_RBUF_PUTC(buf, '`');
        sen_obj_inspect(c, cadr(obj), buf, flags);
      } else if (car(obj) == UNQUOTE && ok_abbrev(cdr(obj))) {
        SEN_RBUF_PUTC(buf, ',');
        sen_obj_inspect(c, cadr(obj), buf, flags);
      } else if (car(obj) == UNQUOTESP && ok_abbrev(cdr(obj))) {
        SEN_RBUF_PUTS(buf, ",@");
        sen_obj_inspect(c, cadr(obj), buf, flags);
      } else {
        SEN_RBUF_PUTC(buf, '(');
        for (;;) {
          sen_obj_inspect(c, car(obj), buf, flags);
          if ((obj = cdr(obj)) && (obj != NIL)) {
            if (LISTP(obj)) {
              SEN_RBUF_PUTC(buf, ' ');
            } else {
              SEN_RBUF_PUTS(buf, " . ");
              sen_obj_inspect(c, obj, buf, flags);
              SEN_RBUF_PUTC(buf, ')');
              break;
            }
          } else {
            SEN_RBUF_PUTC(buf, ')');
            break;
          }
        }
      }
      break;
    default :
      if (issymbol(obj)) {
        SEN_RBUF_PUTS(buf, SEN_SET_STRKEY_BY_VAL(obj));
      } else {
        SEN_RBUF_PUTS(buf, "#<?(");
        sen_rbuf_itoa(buf, obj->type);
        SEN_RBUF_PUTS(buf, ")?>");
      }
      break;
    }
  }
}

/* ========== Routines for Evaluation Cycle ========== */

/* make closure. c is code. e is environment */
inline static cell *
mk_closure(sen_ctx *ctx, cell *c, cell *e)
{
  cell *x = get_cell(ctx, c, e);
  type(x) = sen_ql_closure;
  x->flags = SEN_OBJ_REFERER;
  car(x) = c;
  cdr(x) = e;
  return x;
}

/* make continuation. */
inline static cell *
mk_continuation(sen_ctx *ctx, cell *d)
{
  cell *x = get_cell(ctx, NIL, d);
  type(x) = sen_ql_continuation;
  x->flags = SEN_OBJ_REFERER;
  cont_dump(x) = d;
  return x;
}

/* reverse list -- make new cells */
inline static cell *
reverse(sen_ctx *ctx, cell *a) /* a must be checked by gc */
{
  cell *p = NIL;
  for ( ; ispair(a); a = cdr(a)) {
    p = cons(ctx, car(a), p);
  }
  return p;
}

/* reverse list --- no make new cells */
inline static cell *
non_alloc_rev(cell *term, cell *list)
{
  cell *p = list, *result = term, *q;
  while (p != NIL) {
    q = cdr(p);
    cdr(p) = result;
    result = p;
    p = q;
  }
  return result;
}

/* append list -- make new cells */
inline static cell *
append(sen_ctx *ctx, cell *a, cell *b)
{
  cell *p = b, *q;
  if (a != NIL) {
    a = reverse(ctx, a);
    while (a != NIL) {
      q = cdr(a);
      cdr(a) = p;
      p = a;
      a = q;
    }
  }
  return p;
}

/* equivalence of atoms */
inline static int
eqv(cell *a, cell *b)
{
  if (BULKP(a)) {
    if (BULKP(b)) {
      return strvalue(a) == strvalue(b) ||
             (a->u.b.size == b->u.b.size &&
              !memcmp(strvalue(a), strvalue(b), a->u.b.size));
    } else {
      return 0;
    }
  } else if (isnumber(a)) {
    if (isnumber(b))
      return ivalue(a) == ivalue(b);
    else
      return 0;
  } else {
    return a == b;
  }
}

/* true or false value macro */
#define istrue(p)       ((p) != NIL && (p) != F)
#define isfalse(p)      ((p) == NIL || (p) == F)

#define Error_0(c,s) do {\
  c->cur = c->str_end;\
  c->args = cons(c, sen_ql_mk_string((c), (s), strlen(s)), NIL);\
  c->op = OP_ERR0;\
  return T;\
} while (0)

#define Error_1(c,s,a) do {\
    c->cur = c->str_end;\
    c->args = cons(c, (a), NIL);\
    c->args = cons(c, sen_ql_mk_string((c), (s), strlen(s)), c->args);\
    c->op = OP_ERR0;\
    return T;\
} while (0)

/* control macros for Eval_Cycle */
#define s_goto(c,a) do {\
  c->op = (a);\
  return T;\
} while (0)

#define s_save(c,a,b,args) (\
    c->dump = cons(c, c->envir, cons(c, (args), c->dump)),\
    c->dump = cons(c, (b), c->dump),\
    c->dump = cons(c, mk_number(c, (int)(a)), c->dump))

#define s_return(c,a) do {\
    c->value = (a);\
    c->op = ivalue(car(c->dump));\
    c->args = cadr(c->dump);\
    c->envir = caddr(c->dump);\
    c->code = cadddr(c->dump);\
    c->dump = cddddr(c->dump);\
    return T;\
} while (0)

#define s_retbool(tf)  s_return(c, (tf) ? T : F)

#define isnative(c)  ((c)->flags & SEN_OBJ_NATIVE)

#define RTN_NIL_IF_HEAD(c) do {\
  if (((c)->co.mode & SEN_QL_HEAD)) { s_return((c), NIL); }\
} while (0)

#define RTN_NIL_IF_TAIL(c) do {\
  if (((c)->co.mode & SEN_QL_TAIL)) { s_return((c), NIL); } else { return NIL; }\
} while (0)

static cell *
opexe(sen_ctx *c)
{
  register cell *x, *y;
  register long v;

  switch (c->op) {
  case OP_LOAD:    /* load */
    if (BULKP(car(c->args))) {
      struct stat st;
      char *fname = strvalue(car(c->args));
      if (fname && !stat(fname, &st)) {
        if (c->inbuf) { SEN_FREE(c->inbuf); }
        if ((c->inbuf = SEN_MALLOC(st.st_size))) {
          int fd;
          if ((fd = open(fname, O_RDONLY)) != -1) {
            if (read(fd, c->inbuf, st.st_size) == st.st_size) {
              SEN_RBUF_PUTS(&c->outbuf, "loading ");
              SEN_RBUF_PUTS(&c->outbuf, strvalue(car(c->args)));
              c->cur = c->inbuf;
              c->str_end = c->inbuf + st.st_size;
            }
            close(fd);
          }
          if (c->cur != c->inbuf) {
            SEN_FREE(c->inbuf);
            c->inbuf = NULL;
          }
        }
      }
    }
    s_goto(c, OP_T0LVL);

  case OP_T0LVL:  /* top level */
    c->dump = NIL;
    c->envir = c->global_env;
    /*
    if (c->gc_verbose) {
      sen_rbuf buf;
      sen_rbuf_init(&buf, 0);
      sen_obj_inspect(c, c->envir, &buf, SEN_OBJ_INSPECT_ESC);
      *buf.curr = '\0';
      SEN_LOG(sen_log_notice, "globals=<%s>", buf.head);
      sen_rbuf_fin(&buf);
    }
    */
    {
      int n = c->objects->n_entries;
      sen_ctx_mgc(c);
      if (c->gc_verbose) {
        SEN_LOG(sen_log_notice, "mgc: %d => %d", n, c->objects->n_entries);
      }
    }
    if (c->batchmode) {
      s_save(c, OP_T0LVL, NIL, NIL);
    } else {
      s_save(c, OP_VALUEPRINT, NIL, NIL);
    }
    s_save(c, OP_T1LVL, NIL, NIL);
    // if (infp == stdin) printf("hoge>\n");
    c->pht = &c->phs;
    *c->pht = NIL;
    s_goto(c, OP_READ);

  case OP_T1LVL:  /* top level */
    // verbose check?
    if (c->phs != NIL &&
        !(c->co.mode & (SEN_QL_HEAD|SEN_QL_TAIL))) { RTN_NIL_IF_TAIL(c); }
    // SEN_RBUF_PUTC(&c->outbuf, '\n');
    c->code = c->value;
    s_goto(c, OP_EVAL);

  case OP_READ:    /* read */
    RTN_NIL_IF_HEAD(c);
    if ((c->tok = token(c)) == TOK_EOS) { RTN_NIL_IF_TAIL(c); }
    s_goto(c, OP_RDSEXPR);

  case OP_VALUEPRINT:  /* print evalution result */
    c->args = c->value;
    s_save(c, OP_T0LVL, NIL, NIL);
    sen_obj_inspect(c, c->args, &c->outbuf, SEN_OBJ_INSPECT_ESC);
    s_return(c, T);

  case OP_EVAL:    /* main part of evalution */
    // fixme : quick hack.
    if (issymbol(c->code)) {  /* symbol */
      if (*symname(c->code) == ':') { s_return(c, c->code); }
      for (x = c->envir; x != NIL; x = cdr(x)) {
        for (y = car(x); y != NIL; y = cdr(y))
          if (caar(y) == c->code)
            break;
        if (y != NIL)
          break;
      }
      if (x != NIL) {
        s_return(c, cdar(y));
      } else {
        if (isproc(c->code)) { s_return(c, c->code); }
        if (isnative(c->code)) { s_return(c, c->code); }
        Error_1(c, "Unbounded variable", c->code);
      }
    } else if (ispair(c->code)) {
      if (issyntax(x = car(c->code))) {  /* SYNTAX */
        c->code = cdr(c->code);
        s_goto(c, syntaxnum(x));
      } else {/* first, eval top element and eval arguments */
        s_save(c, OP_E0ARGS, NIL, c->code);
        c->code = car(c->code);
        // if (isnative(c->code)) { s_return(c, c->code); } /* call native funcs. fast */
        s_goto(c, OP_EVAL);
      }
    } else {
      s_return(c, c->code);
    }

  case OP_E0ARGS:  /* eval arguments */
    if (ismacro(c->value)) {  /* macro expansion */
      s_save(c, OP_DOMACRO, NIL, NIL);
      c->args = cons(c, c->code, NIL);
      c->code = c->value;
      s_goto(c, OP_APPLY);
    } else {
      c->code = cdr(c->code);
      s_goto(c, OP_E1ARGS);
    }

  case OP_E1ARGS:  /* eval arguments */
    c->args = cons(c, c->value, c->args);
    if (ispair(c->code)) {  /* continue */
      s_save(c, OP_E1ARGS, c->args, cdr(c->code));
      c->code = car(c->code);
      c->args = NIL;
      s_goto(c, OP_EVAL);
    } else {  /* end */
      c->args = non_alloc_rev(NIL, c->args);
      c->code = car(c->args);
      c->args = cdr(c->args);
      s_goto(c, OP_APPLY);
    }

  case OP_APPLY:    /* apply 'code' to 'args' */
    if (isnative(c->code)) {
      c->doing = c->code->u.o.func;
      s_goto(c, OP_NATIVE);
    } else if (isproc(c->code)) {
      s_goto(c, procnum(c->code));  /* PROCEDURE */
    } else if (isclosure(c->code)) {  /* CLOSURE */
      /* make environment */
      c->envir = cons(c, NIL, closure_env(c->code));
      for (x = car(closure_code(c->code)), y = c->args;
           ispair(x); x = cdr(x), y = cdr(y)) {
        if (y == NIL) {
          Error_0(c, "Few arguments");
        } else {
          car(c->envir) = cons(c, cons(c, car(x), car(y)), car(c->envir));
        }
      }
      if (x == NIL) {
        /*--
         * if (y != NIL) {
         *   Error_0(c, "Many arguments");
         * }
         */
      } else if (issymbol(x))
        car(c->envir) = cons(c, cons(c, x, y), car(c->envir));
      else {
        Error_0(c, "Syntax error in closure");
      }
      c->code = cdr(closure_code(c->code));
      c->args = NIL;
      s_goto(c, OP_BEGIN);
    } else if (iscontinuation(c->code)) {  /* CONTINUATION */
      c->dump = cont_dump(c->code);
      s_return(c, c->args != NIL ? car(c->args) : NIL);
    } else {
      Error_0(c, "Illegal function");
    }

  case OP_DOMACRO:  /* do macro */
    c->code = c->value;
    s_goto(c, OP_EVAL);

  case OP_LAMBDA:  /* lambda */
    s_return(c, mk_closure(c, c->code, c->envir));

  case OP_QUOTE:    /* quote */
    s_return(c, car(c->code));

  case OP_DEF0:  /* define */
    if (ispair(car(c->code))) {
      x = caar(c->code);
      c->code = cons(c, LAMBDA, cons(c, cdar(c->code), cdr(c->code)));
    } else {
      x = car(c->code);
      c->code = cadr(c->code);
    }
    if (!issymbol(x)) {
      Error_0(c, "Variable is not symbol");
    }
    s_save(c, OP_DEF1, NIL, x);
    s_goto(c, OP_EVAL);

  case OP_DEF1:  /* define */
    for (x = car(c->envir); x != NIL; x = cdr(x))
      if (caar(x) == c->code)
        break;
    if (x != NIL)
      cdar(x) = c->value;
    else
      car(c->envir) = cons(c, cons(c, c->code, c->value), car(c->envir));
    s_return(c, c->code);

  case OP_SET0:    /* set! */
    s_save(c, OP_SET1, NIL, car(c->code));
    c->code = cadr(c->code);
    s_goto(c, OP_EVAL);

  case OP_SET1:    /* set! */
    for (x = c->envir; x != NIL; x = cdr(x)) {
      for (y = car(x); y != NIL; y = cdr(y))
        if (caar(y) == c->code)
          break;
      if (y != NIL)
        break;
    }
    if (x != NIL) {
      cdar(y) = c->value;
      s_return(c, c->value);
    } else {
      Error_1(c, "Unbounded variable", c->code);
    }

  case OP_BEGIN:    /* begin */
    if (!ispair(c->code)) {
      s_return(c, c->code);
    }
    if (cdr(c->code) != NIL) {
      s_save(c, OP_BEGIN, NIL, cdr(c->code));
    }
    c->code = car(c->code);
    s_goto(c, OP_EVAL);

  case OP_IF0:    /* if */
    s_save(c, OP_IF1, NIL, cdr(c->code));
    c->code = car(c->code);
    s_goto(c, OP_EVAL);

  case OP_IF1:    /* if */
    if (istrue(c->value))
      c->code = car(c->code);
    else
      c->code = cadr(c->code);  /* (if #f 1) ==> () because
             * car(NIL) = NIL */
    s_goto(c, OP_EVAL);

  case OP_LET0:    /* let */
    c->args = NIL;
    c->value = c->code;
    c->code = issymbol(car(c->code)) ? cadr(c->code) : car(c->code);
    s_goto(c, OP_LET1);

  case OP_LET1:    /* let (caluculate parameters) */
    c->args = cons(c, c->value, c->args);
    if (ispair(c->code)) {  /* continue */
      s_save(c, OP_LET1, c->args, cdr(c->code));
      c->code = cadar(c->code);
      c->args = NIL;
      s_goto(c, OP_EVAL);
    } else {  /* end */
      c->args = non_alloc_rev(NIL, c->args);
      c->code = car(c->args);
      c->args = cdr(c->args);
      s_goto(c, OP_LET2);
    }

  case OP_LET2:    /* let */
    c->envir = cons(c, NIL, c->envir);
    for (x = issymbol(car(c->code)) ? cadr(c->code) : car(c->code), y = c->args;
         y != NIL; x = cdr(x), y = cdr(y))
      car(c->envir) = cons(c, cons(c, caar(x), car(y)), car(c->envir));
    if (issymbol(car(c->code))) {  /* named let */
      for (x = cadr(c->code), c->args = NIL; x != NIL; x = cdr(x))
        c->args = cons(c, caar(x), c->args);
      x = mk_closure(c, cons(c, non_alloc_rev(NIL, c->args), cddr(c->code)), c->envir);
      car(c->envir) = cons(c, cons(c, car(c->code), x), car(c->envir));
      c->code = cddr(c->code);
      c->args = NIL;
    } else {
      c->code = cdr(c->code);
      c->args = NIL;
    }
    s_goto(c, OP_BEGIN);

  case OP_LET0AST:  /* let* */
    if (car(c->code) == NIL) {
      c->envir = cons(c, NIL, c->envir);
      c->code = cdr(c->code);
      s_goto(c, OP_BEGIN);
    }
    s_save(c, OP_LET1AST, cdr(c->code), car(c->code));
    c->code = cadaar(c->code);
    s_goto(c, OP_EVAL);

  case OP_LET1AST:  /* let* (make new frame) */
    c->envir = cons(c, NIL, c->envir);
    s_goto(c, OP_LET2AST);

  case OP_LET2AST:  /* let* (caluculate parameters) */
    car(c->envir) = cons(c, cons(c, caar(c->code), c->value), car(c->envir));
    c->code = cdr(c->code);
    if (ispair(c->code)) {  /* continue */
      s_save(c, OP_LET2AST, c->args, c->code);
      c->code = cadar(c->code);
      c->args = NIL;
      s_goto(c, OP_EVAL);
    } else {  /* end */
      c->code = c->args;
      c->args = NIL;
      s_goto(c, OP_BEGIN);
    }

  case OP_LET0REC:  /* letrec */
    c->envir = cons(c, NIL, c->envir);
    c->args = NIL;
    c->value = c->code;
    c->code = car(c->code);
    s_goto(c, OP_LET1REC);

  case OP_LET1REC:  /* letrec (caluculate parameters) */
    c->args = cons(c, c->value, c->args);
    if (ispair(c->code)) {  /* continue */
      s_save(c, OP_LET1REC, c->args, cdr(c->code));
      c->code = cadar(c->code);
      c->args = NIL;
      s_goto(c, OP_EVAL);
    } else {  /* end */
      c->args = non_alloc_rev(NIL, c->args);
      c->code = car(c->args);
      c->args = cdr(c->args);
      s_goto(c, OP_LET2REC);
    }

  case OP_LET2REC:  /* letrec */
    for (x = car(c->code), y = c->args; y != NIL; x = cdr(x), y = cdr(y))
      car(c->envir) = cons(c, cons(c, caar(x), car(y)), car(c->envir));
    c->code = cdr(c->code);
    c->args = NIL;
    s_goto(c, OP_BEGIN);

  case OP_COND0:    /* cond */
    if (!ispair(c->code)) {
      Error_0(c, "Syntax error in cond");
    }
    s_save(c, OP_COND1, NIL, c->code);
    c->code = caar(c->code);
    s_goto(c, OP_EVAL);

  case OP_COND1:    /* cond */
    if (istrue(c->value)) {
      if ((c->code = cdar(c->code)) == NIL) {
        s_return(c, c->value);
      }
      s_goto(c, OP_BEGIN);
    } else {
      if ((c->code = cdr(c->code)) == NIL) {
        s_return(c, NIL);
      } else {
        s_save(c, OP_COND1, NIL, c->code);
        c->code = caar(c->code);
        s_goto(c, OP_EVAL);
      }
    }

  case OP_DELAY:    /* delay */
    x = mk_closure(c, cons(c, NIL, c->code), c->envir);
    setpromise(x);
    s_return(c, x);

  case OP_AND0:    /* and */
    if (c->code == NIL) {
      s_return(c, T);
    }
    s_save(c, OP_AND1, NIL, cdr(c->code));
    c->code = car(c->code);
    s_goto(c, OP_EVAL);

  case OP_AND1:    /* and */
    if (isfalse(c->value)) {
      s_return(c, c->value);
    } else if (c->code == NIL) {
      s_return(c, c->value);
    } else {
      s_save(c, OP_AND1, NIL, cdr(c->code));
      c->code = car(c->code);
      s_goto(c, OP_EVAL);
    }

  case OP_OR0:    /* or */
    if (c->code == NIL) {
      s_return(c, F);
    }
    s_save(c, OP_OR1, NIL, cdr(c->code));
    c->code = car(c->code);
    s_goto(c, OP_EVAL);

  case OP_OR1:    /* or */
    if (istrue(c->value)) {
      s_return(c, c->value);
    } else if (c->code == NIL) {
      s_return(c, c->value);
    } else {
      s_save(c, OP_OR1, NIL, cdr(c->code));
      c->code = car(c->code);
      s_goto(c, OP_EVAL);
    }

  case OP_C0STREAM:  /* cons-stream */
    s_save(c, OP_C1STREAM, NIL, cdr(c->code));
    c->code = car(c->code);
    s_goto(c, OP_EVAL);

  case OP_C1STREAM:  /* cons-stream */
    c->args = c->value;  /* save c->value to register c->args for gc */
    x = mk_closure(c, cons(c, NIL, c->code), c->envir);
    setpromise(x);
    s_return(c, cons(c, c->args, x));

  case OP_0MACRO:  /* macro */
    x = car(c->code);
    c->code = cadr(c->code);
    if (!issymbol(x)) {
      Error_0(c, "Variable is not symbol");
    }
    s_save(c, OP_1MACRO, NIL, x);
    s_goto(c, OP_EVAL);

  case OP_1MACRO:  /* macro */
    flags(c->value) |= SEN_OBJ_MACRO;
    for (x = car(c->envir); x != NIL; x = cdr(x))
      if (caar(x) == c->code)
        break;
    if (x != NIL)
      cdar(x) = c->value;
    else
      car(c->envir) = cons(c, cons(c, c->code, c->value), car(c->envir));
    s_return(c, c->code);

  case OP_CASE0:    /* case */
    s_save(c, OP_CASE1, NIL, cdr(c->code));
    c->code = car(c->code);
    s_goto(c, OP_EVAL);

  case OP_CASE1:    /* case */
    for (x = c->code; x != NIL; x = cdr(x)) {
      if (!ispair(y = caar(x)))
        break;
      for ( ; y != NIL; y = cdr(y))
        if (eqv(car(y), c->value))
          break;
      if (y != NIL)
        break;
    }
    if (x != NIL) {
      if (ispair(caar(x))) {
        c->code = cdar(x);
        s_goto(c, OP_BEGIN);
      } else {/* else */
        s_save(c, OP_CASE2, NIL, cdar(x));
        c->code = caar(x);
        s_goto(c, OP_EVAL);
      }
    } else {
      s_return(c, NIL);
    }

  case OP_CASE2:    /* case */
    if (istrue(c->value)) {
      s_goto(c, OP_BEGIN);
    } else {
      s_return(c, NIL);
    }
  case OP_PAPPLY:  /* apply */
    c->code = car(c->args);
    c->args = cadr(c->args);
    s_goto(c, OP_APPLY);

  case OP_PEVAL:  /* eval */
    c->code = car(c->args);
    c->args = NIL;
    s_goto(c, OP_EVAL);

  case OP_CONTINUATION:  /* call-with-current-continuation */
    c->code = car(c->args);
    c->args = cons(c, mk_continuation(c, c->dump), NIL);
    s_goto(c, OP_APPLY);

  case OP_ADD:    /* + */
    for (x = c->args, v = 0; x != NIL; x = cdr(x)) {
      if (sen_obj2int(car(x))) { Error_0(c, "can't convert into integer"); }
      v += ivalue(car(x));
    }
    s_return(c, mk_number(c, v));

  case OP_SUB:    /* - */
    x = c->args;
    if (sen_obj2int(car(x))) { Error_0(c, "can't convert into integer"); }
    v = ivalue(car(x));
    while (cdr(x) != NIL) {
      x = cdr(x);
      if (sen_obj2int(car(x))) { Error_0(c, "can't convert into integer"); }
      v -= ivalue(car(x));
    }
    s_return(c, mk_number(c, v));

  case OP_MUL:    /* * */
    for (x = c->args, v = 1; x != NIL; x = cdr(x)) {
      if (sen_obj2int(car(x))) { Error_0(c, "can't convert into integer"); }
      v *= ivalue(car(x));
    }
    s_return(c, mk_number(c, v));

  case OP_DIV:    /* / */
    x = c->args;
    if (sen_obj2int(car(x))) { Error_0(c, "can't convert into integer"); }
    v = ivalue(car(x));
    while (cdr(x) != NIL) {
      x = cdr(x);
      if (sen_obj2int(car(x))) { Error_0(c, "can't convert into integer"); }
      if (ivalue(car(x)) != 0)
        v /= ivalue(car(x));
      else {
        Error_0(c, "Divided by zero");
      }
    }
    s_return(c, mk_number(c, v));

  case OP_REM:    /* remainder */
    x = c->args;
    if (sen_obj2int(car(x))) { Error_0(c, "can't convert into integer"); }
    v = ivalue(x);
    while (cdr(x) != NIL) {
      x = cdr(x);
      if (sen_obj2int(car(x))) { Error_0(c, "can't convert into integer"); }
      if (ivalue(car(x)) != 0)
        v %= ivalue(car(x));
      else {
        Error_0(c, "Divided by zero");
      }
    }
    s_return(c, mk_number(c, v));

  case OP_CAR:    /* car */
    if (ispair(car(c->args))) {
      s_return(c, caar(c->args));
    } else {
      Error_0(c, "Unable to car for non-cons cell");
    }

  case OP_CDR:    /* cdr */
    if (ispair(car(c->args))) {
      s_return(c, cdar(c->args));
    } else {
      Error_0(c, "Unable to cdr for non-cons cell");
    }

  case OP_CONS:    /* cons */
    cdr(c->args) = cadr(c->args);
    s_return(c, c->args);

  case OP_SETCAR:  /* set-car! */
    if (ispair(car(c->args))) {
      caar(c->args) = cadr(c->args);
      s_return(c, car(c->args));
    } else {
      Error_0(c, "Unable to set-car! for non-cons cell");
    }

  case OP_SETCDR:  /* set-cdr! */
    if (ispair(car(c->args))) {
      cdar(c->args) = cadr(c->args);
      s_return(c, car(c->args));
    } else {
      Error_0(c, "Unable to set-cdr! for non-cons cell");
    }

  case OP_NOT:    /* not */
    s_retbool(isfalse(car(c->args)));
  case OP_BOOL:    /* boolean? */
    s_retbool(car(c->args) == F || car(c->args) == T);
  case OP_NULL:    /* null? */
    s_retbool(car(c->args) == NIL);
  case OP_ZEROP:    /* zero? */
    s_retbool(ivalue(car(c->args)) == 0);
  case OP_POSP:    /* positive? */
    s_retbool(ivalue(car(c->args)) > 0);
  case OP_NEGP:    /* negative? */
    s_retbool(ivalue(car(c->args)) < 0);
  case OP_NEQ:    /* = */
    s_retbool(ivalue(car(c->args)) == ivalue(cadr(c->args)));
  case OP_LESS:    /* < */
    s_retbool(ivalue(car(c->args)) < ivalue(cadr(c->args)));
  case OP_GRE:    /* > */
    s_retbool(ivalue(car(c->args)) > ivalue(cadr(c->args)));
  case OP_LEQ:    /* <= */
    s_retbool(ivalue(car(c->args)) <= ivalue(cadr(c->args)));
  case OP_GEQ:    /* >= */
    s_retbool(ivalue(car(c->args)) >= ivalue(cadr(c->args)));
  case OP_SYMBOL:  /* symbol? */
    s_retbool(issymbol(car(c->args)));
  case OP_NUMBER:  /* number? */
    s_retbool(isnumber(car(c->args)));
  case OP_STRING:  /* string? */
    s_retbool(BULKP(car(c->args)));
  case OP_PROC:    /* procedure? */
    /*--
           * continuation should be procedure by the example
           * (call-with-current-continuation procedure?) ==> #t
                 * in R^3 report sec. 6.9
           */
    s_retbool(isproc(car(c->args)) || isclosure(car(c->args))
        || iscontinuation(car(c->args)));
  case OP_PAIR:    /* pair? */
    s_retbool(ispair(car(c->args)));
  case OP_EQ:    /* eq? */
    s_retbool(car(c->args) == cadr(c->args));
  case OP_EQV:    /* eqv? */
    s_retbool(eqv(car(c->args), cadr(c->args)));

  case OP_FORCE:    /* force */
    c->code = car(c->args);
    if (ispromise(c->code)) {
      c->args = NIL;
      s_goto(c, OP_APPLY);
    } else {
      s_return(c, c->code);
    }

  case OP_WRITE:    /* write */
    c->args = car(c->args);
    sen_obj_inspect(c, c->args, &c->outbuf, SEN_OBJ_INSPECT_ESC);
    s_return(c, T);

  case OP_DISPLAY:  /* display */
    c->args = car(c->args);
    sen_obj_inspect(c, c->args, &c->outbuf, 0);
    s_return(c, T);

  case OP_NEWLINE:  /* newline */
    SEN_RBUF_PUTC(&c->outbuf, '\n');
    s_return(c, T);

  case OP_ERR0:  /* error */
    if (!BULKP(car(c->args))) {
      Error_0(c, "error -- first argument must be string");
    }
    SEN_RBUF_PUTS(&c->outbuf, "Error: ");
    SEN_RBUF_PUTS(&c->outbuf, strvalue(car(c->args)));
    c->args = cdr(c->args);
    s_goto(c, OP_ERR1);

  case OP_ERR1:  /* error */
    SEN_RBUF_PUTC(&c->outbuf, ' ');
    if (c->args != NIL) {
      s_save(c, OP_ERR1, cdr(c->args), NIL);
      c->args = car(c->args);
      sen_obj_inspect(c, c->args, &c->outbuf, SEN_OBJ_INSPECT_ESC);
      s_return(c, T);
    } else {
      SEN_RBUF_PUTC(&c->outbuf, '\n');
      s_goto(c, OP_T0LVL);
    }

  case OP_REVERSE:  /* reverse */
    s_return(c, reverse(c, car(c->args)));

  case OP_APPEND:  /* append */
    s_return(c, append(c, car(c->args), cadr(c->args)));

  case OP_PUT:    /* put */
    if (!hasprop(car(c->args)) || !hasprop(cadr(c->args))) {
      Error_0(c, "Illegal use of put");
    }
    for (x = symprop(car(c->args)), y = cadr(c->args); x != NIL; x = cdr(x))
      if (caar(x) == y)
        break;
    if (x != NIL)
      cdar(x) = caddr(c->args);
    else
      symprop(car(c->args)) = cons(c, cons(c, y, caddr(c->args)),
              symprop(car(c->args)));
    s_return(c, T);

  case OP_GET:    /* get */
    if (!hasprop(car(c->args)) || !hasprop(cadr(c->args))) {
      Error_0(c, "Illegal use of get");
    }
    for (x = symprop(car(c->args)), y = cadr(c->args); x != NIL; x = cdr(x))
      if (caar(x) == y)
        break;
    if (x != NIL) {
      s_return(c, cdar(x));
    } else {
      s_return(c, NIL);
    }

  case OP_QUIT:    /* quit */
    return (NIL);

  case OP_GC:    /* gc */
    sen_ctx_mgc(c);
    // gc(NIL, NIL);
    s_return(c, T);

  case OP_GCVERB:    /* gc-verbose */
  {  int  was = c->gc_verbose;
    c->gc_verbose = (car(c->args) != F);
    s_retbool(was);
  }

  case OP_NATIVEP:    /* native? */
    s_retbool(isnative(car(c->args)));

  case OP_RDSEXPR:
    {
      char tok, *str;
      unsigned len;
      RTN_NIL_IF_HEAD(c);
      switch (c->tok) {
      case TOK_COMMENT:
        skipline(c);
        if ((c->tok = token(c)) == TOK_EOS) { RTN_NIL_IF_TAIL(c); }
        s_goto(c, OP_RDSEXPR);
      case TOK_LPAREN:
        if ((tok = token(c)) == TOK_EOS) { RTN_NIL_IF_TAIL(c); }
        c->tok = tok;
        if (c->tok == TOK_RPAREN) {
          s_return(c, NIL);
        } else if (c->tok == TOK_DOT) {
          Error_0(c, "syntax error -- illegal dot expression");
        } else {
          s_save(c, OP_RDLIST, NIL, NIL);
          s_goto(c, OP_RDSEXPR);
        }
      case TOK_QUOTE:
        s_save(c, OP_RDQUOTE, NIL, NIL);
        if ((c->tok = token(c)) == TOK_EOS) { RTN_NIL_IF_TAIL(c); }
        s_goto(c, OP_RDSEXPR);
      case TOK_BQUOTE:
        s_save(c, OP_RDQQUOTE, NIL, NIL);
        if ((c->tok = token(c)) == TOK_EOS) { RTN_NIL_IF_TAIL(c); }
        s_goto(c, OP_RDSEXPR);
      case TOK_COMMA:
        s_save(c, OP_RDUNQUOTE, NIL, NIL);
        if ((c->tok = token(c)) == TOK_EOS) { RTN_NIL_IF_TAIL(c); }
        s_goto(c, OP_RDSEXPR);
      case TOK_ATMARK:
        s_save(c, OP_RDUQTSP, NIL, NIL);
        if ((c->tok = token(c)) == TOK_EOS) { RTN_NIL_IF_TAIL(c); }
        s_goto(c, OP_RDSEXPR);
      case TOK_ATOM:
        if (readstr(c, &str, &len) == TOK_EOS) { c->tok = TOK_EOS; RTN_NIL_IF_TAIL(c); }
        s_return(c, mk_atom(c, str, len, NIL));
      case TOK_DQUOTE:
        if (readstrexp(c, &str, &len) == TOK_EOS) {
          Error_0(c, "unterminated string meets end of line");
        }
        s_return(c, sen_ql_mk_string(c, str, len));
      case TOK_SHARP:
        if ((readstr(c, &str, &len) == TOK_EOS) ||
            (x = mk_const(c, str, len)) == NIL) {
          Error_0(c, "Undefined sharp expression");
        } else {
          s_return(c, x);
        }
      case TOK_EOS :
        if ((c->tok = token(c)) == TOK_EOS) { RTN_NIL_IF_TAIL(c); }
        s_goto(c, OP_RDSEXPR);
      case TOK_QUESTION:
        {
          cell *o = sen_obj_new(c);
          cell *p = cons(c, o, NIL);
          o->type = sen_ql_bulk;
          o->flags = 0;
          o->u.b.size = 1;
          o->u.b.value = "?";
          *c->pht = p;
          c->pht = &cdr(p);
          s_return(c, o);
        }
      default:
        Error_0(c, "syntax error -- illegal token");
      }
    }
    break;

  case OP_RDLIST:
    RTN_NIL_IF_HEAD(c);
    if ((c->tok = token(c)) == TOK_EOS) { RTN_NIL_IF_TAIL(c); }
    c->args = cons(c, c->value, c->args);
    /*
    if (tok == TOK_COMMENT) {
      skipline(c);
      tok = token();
    }
    */
    if (c->tok == TOK_RPAREN) {
      cell *v = non_alloc_rev(NIL, c->args);
      if (c->cur < c->str_end && *c->cur == '.') {
        char *str;
        unsigned len;
        if (readstr(c, &str, &len) != TOK_ATOM) { /* error */ }
        s_return(c, mk_atom(c, str, len, v));
      } else {
        s_return(c, v);
      }
    } else if (c->tok == TOK_DOT) {
      s_save(c, OP_RDDOT, c->args, NIL);
      if ((c->tok = token(c)) == TOK_EOS) { c->op = OP_RDSEXPR; RTN_NIL_IF_TAIL(c); }
      s_goto(c, OP_RDSEXPR);
    } else {
      s_save(c, OP_RDLIST, c->args, NIL);;
      s_goto(c, OP_RDSEXPR);
    }

  case OP_RDDOT:
    RTN_NIL_IF_HEAD(c);
    if ((c->tok = token(c)) == TOK_EOS) { RTN_NIL_IF_TAIL(c); }
    if (c->tok != TOK_RPAREN) {
      Error_0(c, "syntax error -- illegal dot expression");
    } else {
      cell *v = non_alloc_rev(c->value, c->args);
      if (c->cur < c->str_end && *c->cur == '.') {
        char *str;
        unsigned len;
        if (readstr(c, &str, &len) != TOK_ATOM) { /* error */ }
        s_return(c, mk_atom(c, str, len, v));
      } else {
        s_return(c, v);
      }
    }

  case OP_RDQUOTE:
    s_return(c, cons(c, QUOTE, cons(c, c->value, NIL)));

  case OP_RDQQUOTE:
    s_return(c, cons(c, QQUOTE, cons(c, c->value, NIL)));

  case OP_RDUNQUOTE:
    s_return(c, cons(c, UNQUOTE, cons(c, c->value, NIL)));

  case OP_RDUQTSP:
    s_return(c, cons(c, UNQUOTESP, cons(c, c->value, NIL)));

  case OP_LIST_LENGTH:  /* list-length */  /* a.k */
    for (x = car(c->args), v = 0; ispair(x); x = cdr(x))
      ++v;
    s_return(c, mk_number(c, v));

  case OP_ASSQ:    /* assq */  /* a.k */
    x = car(c->args);
    for (y = cadr(c->args); ispair(y); y = cdr(y)) {
      if (!ispair(car(y))) {
        Error_0(c, "Unable to handle non pair element");
      }
      if (x == caar(y))
        break;
    }
    if (ispair(y)) {
      s_return(c, car(y));
    } else {
      s_return(c, F);
    }

  case OP_GET_CLOSURE:  /* get-closure-code */  /* a.k */
    c->args = car(c->args);
    if (c->args == NIL) {
      s_return(c, F);
    } else if (isclosure(c->args)) {
      s_return(c, cons(c, LAMBDA, closure_code(c->value)));
    } else if (ismacro(c->args)) {
      s_return(c, cons(c, LAMBDA, closure_code(c->value)));
    } else {
      s_return(c, F);
    }
  case OP_CLOSUREP:    /* closure? */
    /*
     * Note, macro object is also a closure.
     * Therefore, (closure? <#MACRO>) ==> #t
     */
    if (car(c->args) == NIL) {
        s_return(c, F);
    }
    s_retbool(isclosure(car(c->args)));
  case OP_MACROP:    /* macro? */
    if (car(c->args) == NIL) {
        s_return(c, F);
    }
    s_retbool(ismacro(car(c->args)));
  case OP_NATIVE:
    s_return(c, c->value);
  case OP_VOIDP:    /* void? */
    s_retbool(car(c->args)->type == sen_ql_void);
  }
  SEN_LOG(sen_log_error, "illegal op (%d)", c->op);
  return NIL;
}

/* kernel of this intepreter */
static void
Eval_Cycle(sen_ctx *c)
{
  c->doing = NULL;
  c->co.last = 0;
  while (opexe(c) != NIL) {
    switch (c->op) {
    case OP_NATIVE :
      c->stat = SEN_QL_NATIVE;
      return;
    case OP_T0LVL :
      c->stat = SEN_QL_TOPLEVEL;
      return;
    case OP_T1LVL :
      c->stat = (c->phs != NIL) ? SEN_QL_WAIT_ARG : SEN_QL_EVAL;
      return;
    default :
      break;
    }
  }
  c->stat = SEN_QL_WAIT_EXPR;
}

/* ========== Initialization of internal keywords ========== */

inline static void
mk_syntax(sen_ctx *c, uint8_t op, char *name)
{
  cell *x;
  x = sen_ql_mk_symbol(c, name);
  type(x) = sen_ql_syntax;
  syntaxnum(x) = op;
}

inline static void
mk_proc(sen_ctx *c, uint8_t op, char *name)
{
  cell *x = sen_ql_mk_symbol(c, name);
  type(x) = sen_ql_proc;
  ivalue(x) = (long) op;
}

void
sen_ql_init_const(void)
{
  static sen_obj _NIL, _T, _F;
  /* init NIL */
  NIL = &_NIL;
  type(NIL) = sen_ql_void;
  car(NIL) = cdr(NIL) = NIL;
  /* init T */
  T = &_T;
  type(T) = sen_ql_void;
  car(T) = cdr(T) = T;
  /* init F */
  F = &_F;
  type(F) = sen_ql_void;
  car(F) = cdr(F) = F;
}

inline static void
init_vars_global(sen_ctx *c)
{
  cell *x;
  /* init global_env */
  c->global_env = cons(c, NIL, NIL);
  /* init else */
  x = sen_ql_mk_symbol(c, "else");
  car(c->global_env) = cons(c, cons(c, x, T), car(c->global_env));
}

inline static void
init_syntax(sen_ctx *c)
{
  /* init syntax */
  mk_syntax(c, OP_LAMBDA, "lambda");
  mk_syntax(c, OP_QUOTE, "quote");
  mk_syntax(c, OP_DEF0, "define");
  mk_syntax(c, OP_IF0, "if");
  mk_syntax(c, OP_BEGIN, "begin");
  mk_syntax(c, OP_SET0, "set!");
  mk_syntax(c, OP_LET0, "let");
  mk_syntax(c, OP_LET0AST, "let*");
  mk_syntax(c, OP_LET0REC, "letrec");
  mk_syntax(c, OP_COND0, "cond");
  mk_syntax(c, OP_DELAY, "delay");
  mk_syntax(c, OP_AND0, "and");
  mk_syntax(c, OP_OR0, "or");
  mk_syntax(c, OP_C0STREAM, "cons-stream");
  mk_syntax(c, OP_0MACRO, "define-macro");
  mk_syntax(c, OP_CASE0, "case");
}

inline static void
init_procs(sen_ctx *c)
{
  /* init procedure */
  mk_proc(c, OP_PEVAL, "eval");
  mk_proc(c, OP_PAPPLY, "apply");
  mk_proc(c, OP_CONTINUATION, "call-with-current-continuation");
  mk_proc(c, OP_FORCE, "force");
  mk_proc(c, OP_CAR, "car");
  mk_proc(c, OP_CDR, "cdr");
  mk_proc(c, OP_CONS, "cons");
  mk_proc(c, OP_SETCAR, "set-car!");
  mk_proc(c, OP_SETCDR, "set-cdr!");
  mk_proc(c, OP_ADD, "+");
  mk_proc(c, OP_SUB, "-");
  mk_proc(c, OP_MUL, "*");
  mk_proc(c, OP_DIV, "/");
  mk_proc(c, OP_REM, "remainder");
  mk_proc(c, OP_NOT, "not");
  mk_proc(c, OP_BOOL, "boolean?");
  mk_proc(c, OP_SYMBOL, "symbol?");
  mk_proc(c, OP_NUMBER, "number?");
  mk_proc(c, OP_STRING, "string?");
  mk_proc(c, OP_PROC, "procedure?");
  mk_proc(c, OP_PAIR, "pair?");
  mk_proc(c, OP_EQV, "eqv?");
  mk_proc(c, OP_EQ, "eq?");
  mk_proc(c, OP_NULL, "null?");
  mk_proc(c, OP_ZEROP, "zero?");
  mk_proc(c, OP_POSP, "positive?");
  mk_proc(c, OP_NEGP, "negative?");
  mk_proc(c, OP_NEQ, "=");
  mk_proc(c, OP_LESS, "<");
  mk_proc(c, OP_GRE, ">");
  mk_proc(c, OP_LEQ, "<=");
  mk_proc(c, OP_GEQ, ">=");
  mk_proc(c, OP_READ, "read");
  mk_proc(c, OP_WRITE, "write");
  mk_proc(c, OP_DISPLAY, "display");
  mk_proc(c, OP_NEWLINE, "newline");
  mk_proc(c, OP_LOAD, "load");
  mk_proc(c, OP_ERR0, "error");
  mk_proc(c, OP_REVERSE, "reverse");
  mk_proc(c, OP_APPEND, "append");
  mk_proc(c, OP_PUT, "put");
  mk_proc(c, OP_GET, "get");
  mk_proc(c, OP_GC, "gc");
  mk_proc(c, OP_GCVERB, "gc-verbose");
  mk_proc(c, OP_NATIVEP, "native?");
  mk_proc(c, OP_LIST_LENGTH, "list-length");  /* a.k */
  mk_proc(c, OP_ASSQ, "assq");  /* a.k */
  mk_proc(c, OP_GET_CLOSURE, "get-closure-code");  /* a.k */
  mk_proc(c, OP_CLOSUREP, "closure?");  /* a.k */
  mk_proc(c, OP_MACROP, "macro?");  /* a.k */
  mk_proc(c, OP_QUIT, "quit");
  mk_proc(c, OP_VOIDP, "void?");
}

/* initialize several globals */
void
sen_ql_init_globals(sen_ctx *c)
{
  init_vars_global(c);
  init_syntax(c);
  init_procs(c);
  c->output = sen_ctx_concat_func;
  /* intialization of global pointers to special symbols */
}
