/* Copyright(C) 2004-2007 Brazil

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "senna_in.h"

#include <errno.h>
#include <string.h>
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif /* HAVE_NETDB_H */
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif /* HAVE_SYS_SOCKET_H */
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif /* HAVE_NETINET_IN_H */
#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif /* HAVE_NETINET_TCP_H */
#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif /* HAVE_SIGNAL_H */

#include "com.h"
#include "set.h"
#include "str.h"

#ifndef PF_INET
#define PF_INET AF_INET
#endif /* PF_INET */

#ifdef USE_MSG_NOSIGNAL
#if __FreeBSD__ >= 2 && __FreeBSD_version >= 600020
#define MSG_NOSIGNAL 0x20000
#endif /* __FreeBSD__ >= 2 && __FreeBSD_version >= 600020 */
#else /* USE_MSG_NOSIGNAL */
#define MSG_NOSIGNAL 0
#endif /* USE_MSG_NOSIGNAL */

#ifdef WIN32
void
log_sockerr(const char *msg)
{
  int e = WSAGetLastError();
  const char *m;
  switch (e) {
  case WSANOTINITIALISED: m = "please call sen_com_init first"; break;
  case WSAEFAULT: m = "bad address"; break;
  case WSAEINVAL: m = "invalid argument"; break;
  case WSAEMFILE: m = "too many sockets"; break;
  case WSAEWOULDBLOCK: m = "operation would block"; break;
  case WSAENOTSOCK: m = "given fd is not socket fd"; break;
  case WSAEOPNOTSUPP: m = "operation is not supported"; break;
  case WSAEADDRINUSE: m = "address is already in use"; break;
  case WSAEADDRNOTAVAIL: m = "address is not available"; break;
  case WSAENETDOWN: m = "network is down"; break;
  case WSAENOBUFS: m = "no buffer"; break;
  case WSAEISCONN: m = "socket is already connected"; break;
  case WSAENOTCONN: m = "socket is not connected"; break;
  case WSAESHUTDOWN: m = "socket is already shutdowned"; break;
  case WSAETIMEDOUT: m = "connection time out"; break;
  case WSAECONNREFUSED: m = "connection refused"; break;
  default:
    SEN_LOG(sen_log_error, "%s: socket error (%d)", msg, e);
    return;
  }
  SEN_LOG(sen_log_error, "%s: %s", msg, m);
}
#define LOG_SOCKERR(m) log_sockerr((m))
#else /* WIN32 */
#define LOG_SOCKERR(m) SEN_LOG(sen_log_error, "%s: %s", (m), strerror(errno))
#endif /* WIN32 */

/******* sen_com ********/

sen_rc
sen_com_init(void)
{
  sen_rc rc = sen_success;
#ifdef WIN32
  WSADATA wd;
  if (WSAStartup(MAKEWORD(2, 0), &wd) != 0) {
    rc = sen_other_error;
  }
#else /* WIN32 */
#ifndef USE_MSG_NOSIGNAL
  if (signal(SIGPIPE, SIG_IGN) == SIG_ERR) {
    rc = sen_other_error;
  }
#endif /* USE_MSG_NOSIGNAL */
#endif /* WIN32 */
  return rc;
}

void
sen_com_fin(void)
{
#ifdef WIN32
  WSACleanup();
#endif /* WIN32 */
}

sen_rc
sen_com_event_init(sen_com_event *ev, int max_nevents, int data_size)
{
  sen_rc rc = sen_memory_exhausted;
  ev->max_nevents = max_nevents;
  if ((ev->set = sen_set_open(sizeof(int), data_size, 0))) {
#ifndef USE_SELECT
#ifdef USE_EPOLL
    if ((ev->events = SEN_MALLOC(sizeof(struct epoll_event) * max_nevents))) {
      if ((ev->epfd = epoll_create(max_nevents)) != -1) {
        rc = sen_success;
        goto exit;
      } else {
        LOG_SOCKERR("epoll_create");
        rc = sen_other_error;
      }
      SEN_FREE(ev->events);
    }
#else /* USE_EPOLL */
    if ((ev->events = SEN_MALLOC(sizeof(struct pollfd) * max_nevents))) {
      rc = sen_success;
      goto exit;
    }
#endif /* USE_EPOLL */
    sen_set_close(ev->set);
    ev->set = NULL;
    ev->events = NULL;
#else /* USE_SELECT */
    rc = sen_success;
#endif /* USE_SELECT */
  }
exit :
  return rc;
}

sen_rc
sen_com_event_fin(sen_com_event *ev)
{
  if (ev->set) { sen_set_close(ev->set); }
#ifndef USE_SELECT
  if (ev->events) { SEN_FREE(ev->events); }
#endif /* USE_SELECT */
  return sen_success;
}

sen_rc
sen_com_event_add(sen_com_event *ev, int fd, sen_com **com)
{
  sen_com *c;
  if (!ev || ev->set->n_entries == ev->max_nevents) { return sen_invalid_argument; }
#ifdef USE_EPOLL
  {
    struct epoll_event e;
    memset(&e, 0, sizeof(struct epoll_event));
    e.events = EPOLLIN;
    e.data.fd = (fd);
    if (epoll_ctl(ev->epfd, EPOLL_CTL_ADD, (fd), &e) == -1) {
      LOG_SOCKERR("epoll_ctl");
      return sen_other_error;
    }
  }
#endif /* USE_EPOLL*/
  if (sen_set_get(ev->set, &fd, (void **)&c)) {
    c->fd = fd;
    if (com) { *com = c; }
    return sen_success;
  }
  return sen_other_error;
}

sen_rc
sen_com_event_del(sen_com_event *ev, int fd)
{
  if (!ev) { return sen_invalid_argument; }
#ifdef USE_EPOLL
  {
    struct epoll_event e;
    memset(&e, 0, sizeof(struct epoll_event));
    e.data.fd = fd;
    if (epoll_ctl(ev->epfd, EPOLL_CTL_DEL, fd, &e) == -1) {
      LOG_SOCKERR("epoll_ctl");
      return sen_other_error;
    }
  }
#endif /* USE_EPOLL*/
  return sen_set_del(ev->set, sen_set_at(ev->set, &fd, NULL));
}

sen_rc
sen_com_event_poll(sen_com_event *ev, int timeout)
{
  int nevents;
#ifdef USE_SELECT
  int *pfd, nfds = 0;
  fd_set rfds;
  struct timeval tv;
  if (timeout >= 0) {
    tv.tv_sec = timeout / 1000;
    tv.tv_usec = (timeout % 1000) * 1000;
  }
  FD_ZERO(&rfds);
  {
    sen_set_cursor *sc = sen_set_cursor_open(ev->set);
    while (sen_set_cursor_next(sc, (void *)&pfd, NULL)) {
      FD_SET(*pfd, &rfds);
      if (*pfd > nfds) { nfds = *pfd; }
    }
    sen_set_cursor_close(sc);
  }
  nevents = select(nfds + 1, &rfds, NULL, NULL, (timeout >= 0) ? &tv : NULL);
  if (nevents < 0) {
#ifdef WIN32
    if (WSAGetLastError() == WSAEINTR) { return sen_success; }
#else /* WIN32 */
    if (errno == EINTR) { return sen_success; }
#endif /* WIN32 */
    LOG_SOCKERR("select");
    return sen_other_error;
  }
  if (timeout < 0 && !nevents) { SEN_LOG(sen_log_notice, "select returns 0 events"); }
  {
    sen_com *com;
    sen_set_cursor *sc = sen_set_cursor_open(ev->set);
    while (sen_set_cursor_next(sc, (void *)&pfd, (void *)&com)) {
      if (FD_ISSET(*pfd, &rfds)) {
        com->ev_in(ev, com);
      }
    }
    sen_set_cursor_close(sc);
  }
#else /* USE_SELECT */
#ifdef USE_EPOLL
  struct epoll_event *ep;
  nevents = epoll_wait(ev->epfd, ev->events, ev->max_nevents, timeout);
#else /* USE_EPOLL */
  int nfd = 0, *pfd;
  struct pollfd *ep = ev->events;
  sen_set_cursor *sc = sen_set_cursor_open(ev->set);
  while (sen_set_cursor_next(sc, (void *)&pfd, NULL)) {
    ep->fd = *pfd;
    ep->events = POLLIN;
    ep->revents = 0;
    ep++;
    nfd++;
  }
  sen_set_cursor_close(sc);
  nevents = poll(ev->events, nfd, timeout);
#endif /* USE_EPOLL */
  if (nevents < 0) {
    if (errno == EINTR) { return sen_success; }
    LOG_SOCKERR("poll");
    return sen_other_error;
  }
  if (timeout < 0 && !nevents) { SEN_LOG(sen_log_notice, "poll returns 0 events"); }
  for (ep = ev->events; nevents; ep++) {
    int efd;
    sen_com *com;
#ifdef USE_EPOLL
    if (!(ep->events & EPOLLIN)) {
      SEN_LOG(sen_log_error, "unexpected epoll event: %x\n", ep->events);
      return sen_other_error;
    }
    efd = ep->data.fd;
#else /* USE_EPOLL */
    if (!(ep->revents & POLLIN)) { continue; }
    efd = ep->fd;
#endif /* USE_EPOLL */
    nevents--;
    if (!sen_set_at(ev->set, &efd, (void *)&com)) {
      SEN_LOG(sen_log_error, "fd(%d) not found in ev->set", efd);
      return sen_other_error;
    }
    com->ev_in(ev, com);
  }
#endif /* USE_SELECT */
  return sen_success;
}

/******* sen_com_sqtp ********/

sen_rc
sen_com_sqtp_send(sen_com_sqtp *cs, sen_com_sqtp_header *header, char *body)
{
  ssize_t ret, whole_size = sizeof(sen_com_sqtp_header) + header->size;
  header->proto = SEN_COM_PROTO_SQTP;
  if (header->size) {
#ifdef WIN32
    ssize_t reth;
    if ((reth = send(cs->com.fd, header, sizeof(sen_com_sqtp_header), 0)) == -1) {
      LOG_SOCKERR("send size");
      cs->rc = sen_other_error;
      goto exit;
    }
    if ((ret = send(cs->com.fd, body, header->size, 0)) == -1) {
      LOG_SOCKERR("send body");
      cs->rc = sen_other_error;
      goto exit;
    }
    ret += reth;
#else /* WIN32 */
    struct iovec msg_iov[2];
    struct msghdr msg;
    msg.msg_name = NULL;
    msg.msg_namelen = 0;
    msg.msg_iov = msg_iov;
    msg.msg_iovlen = 2;
    msg.msg_control = NULL;
    msg.msg_controllen = 0;
    msg.msg_flags = 0;
    msg_iov[0].iov_base = header;
    msg_iov[0].iov_len = sizeof(sen_com_sqtp_header);
    msg_iov[1].iov_base = body;
    msg_iov[1].iov_len = header->size;
    while ((ret = sendmsg(cs->com.fd, &msg, MSG_NOSIGNAL)) == -1) {
      LOG_SOCKERR("sendmsg");
      if (errno == EAGAIN || errno == EINTR) { continue; }
      cs->rc = sen_other_error;
      goto exit;
    }
#endif /* WIN32 */
  } else {
    while ((ret = send(cs->com.fd, header, whole_size, MSG_NOSIGNAL)) == -1) {
#ifdef WIN32
      int e = WSAGetLastError();
      LOG_SOCKERR("send");
      if (e == WSAEWOULDBLOCK || e == WSAEINTR) { continue; }
#else /* WIN32 */
      LOG_SOCKERR("send");
      if (errno == EAGAIN || errno == EINTR) { continue; }
#endif /* WIN32 */
      cs->rc = sen_other_error;
      goto exit;
    }
  }
  if (ret != whole_size) {
    SEN_LOG(sen_log_error, "sendmsg: %d < %d", ret, whole_size);
    cs->rc = sen_other_error;
    goto exit;
  }
  cs->rc = sen_success;
exit :
  return cs->rc;
}

sen_rc
sen_com_sqtp_recv(sen_com_sqtp *cs, sen_rbuf *buf,
                  unsigned int *status, unsigned int *info)
{
  ssize_t ret;
  size_t rest = sizeof(sen_com_sqtp_header);
  if (SEN_RBUF_WSIZE(buf) < rest) {
    if ((cs->rc = sen_rbuf_reinit(buf, rest))) {
      *status = sen_com_emem; *info = 1; goto exit;
    }
  } else {
    SEN_RBUF_REWIND(buf);
  }
  do {
    // todo : also support non blocking mode (use MSG_DONTWAIT)
    if ((ret = recv(cs->com.fd, buf->curr, rest, MSG_WAITALL)) <= 0) {
      if (ret < 0) {
#ifdef WIN32
        int e = WSAGetLastError();
        LOG_SOCKERR("recv size");
        if (e == WSAEWOULDBLOCK || e == WSAEINTR) { continue; }
        *info = e;
#else /* WIN32 */
        LOG_SOCKERR("recv size");
        if (errno == EAGAIN || errno == EINTR) { continue; }
        *info = errno;
#endif /* WIN32 */
      }
      cs->rc = sen_other_error;
      *status = sen_com_erecv_head;
      goto exit;
    }
    rest -= ret, buf->curr += ret;
  } while (rest);
  {
    sen_com_sqtp_header *header = SEN_COM_SQTP_MSG_HEADER(buf);
    size_t value_size = header->size;
    size_t whole_size = sizeof(sen_com_sqtp_header) + value_size;
    if (header->proto != SEN_COM_PROTO_SQTP) {
      SEN_LOG(sen_log_error, "illegal header: %d", header->proto);
      cs->rc = sen_invalid_format;
      *status = sen_com_eproto;
      *info = header->proto;
      goto exit;
    }
    if (SEN_RBUF_WSIZE(buf) < whole_size) {
      if ((cs->rc = sen_rbuf_resize(buf, whole_size))) {
        *status = sen_com_emem; *info = 2;
        goto exit;
      }
    }
    for (rest = value_size; rest;) {
      if ((ret = recv(cs->com.fd, buf->curr, rest, MSG_WAITALL)) <= 0) {
        if (ret < 0) {
#ifdef WIN32
          int e = WSAGetLastError();
          LOG_SOCKERR("recv body");
          if (e == WSAEWOULDBLOCK || e == WSAEINTR) { continue; }
          *info = e;
#else /* WIN32 */
          LOG_SOCKERR("recv body");
          if (errno == EAGAIN || errno == EINTR) { continue; }
          *info = errno;
#endif /* WIN32 */
        }
        cs->rc = sen_other_error;
        *status = sen_com_erecv_body;
        goto exit;
      }
      rest -= ret, buf->curr += ret;
    }
    *buf->curr = '\0';
    *status = header->status;
    *info = header->info;
  }
  cs->rc = sen_success;
exit :
  return cs->rc;
}

sen_com_sqtp *
sen_com_sqtp_copen(sen_com_event *ev, const char *dest, int port)
{
  int fd;
  sen_com_sqtp *cs = NULL;
  struct hostent *he;
  struct sockaddr_in addr;
  if (!(he = gethostbyname(dest))) {
    LOG_SOCKERR("gethostbyname");
  }
  addr.sin_family = AF_INET;
  memcpy(&addr.sin_addr, he->h_addr, he->h_length);
  addr.sin_port = htons(port);
  if ((fd = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
    LOG_SOCKERR("socket");
  }
  {
    int v = 1;
    if (setsockopt(fd, 6, TCP_NODELAY, (void *) &v, sizeof(int)) == -1) {
      LOG_SOCKERR("setsockopt");
    }
  }
  while (connect(fd, (struct sockaddr *)&addr, sizeof addr) == -1) {
#ifdef WIN32
    if (WSAGetLastError() == WSAECONNREFUSED)
#else /* WIN32 */
    if (errno == ECONNREFUSED)
#endif /* WIN32 */
    {
      SEN_LOG(sen_log_notice, "connect retrying..");
      sleep(2);
      continue;
    }
    LOG_SOCKERR("connect");
    goto exit;
  }
  if (ev) {
    if (sen_com_event_add(ev, fd, (sen_com **)&cs)) { goto exit; }
  } else {
    if (!(cs = SEN_CALLOC(sizeof(sen_com_sqtp)))) { goto exit; }
    cs->com.fd = fd;
  }
  sen_rbuf_init(&cs->msg, 0);
exit :
  if (!cs) { close(fd); }
  return cs;
}

static void
sen_com_sqtp_receiver(sen_com_event *ev, sen_com *c)
{
  unsigned int status, info;
  sen_com_sqtp *cs = (sen_com_sqtp *)c;
  if (sen_com_sqtp_recv(cs, &cs->msg, &status, &info)) {
    // todo : must be handled
  }
  cs->msg_in(ev, c);
}

static void
sen_com_sqtp_acceptor(sen_com_event *ev, sen_com *c)
{
  sen_com_sqtp *cs = (sen_com_sqtp *)c, *ncs;
  int fd = accept(cs->com.fd, NULL, NULL);
  if (fd == -1) {
    LOG_SOCKERR("accept");
    return;
  }
  if (sen_com_event_add(ev, fd, (sen_com **)&ncs)) {
    close(fd);
    return;
  }
  ncs->com.ev_in = sen_com_sqtp_receiver;
  sen_rbuf_init(&ncs->msg, 0);
  ncs->msg_in = cs->msg_in;
}

#define LISTEN_BACKLOG 50

sen_com_sqtp *
sen_com_sqtp_sopen(sen_com_event *ev, int port, sen_com_callback *func)
{
  int lfd;
  sen_com_sqtp *cs = NULL;
  struct sockaddr_in addr;
  memset(&addr, 0, sizeof(struct sockaddr_in));
  addr.sin_family = AF_INET;
  addr.sin_addr.s_addr = INADDR_ANY;
  addr.sin_port = htons(port);
  if ((lfd = socket(PF_INET, SOCK_STREAM, 0)) == -1) {
    LOG_SOCKERR("socket");
    return NULL;
  }
  {
    int v = 1;
    if (setsockopt(lfd, 6, TCP_NODELAY, (void *) &v, sizeof(int)) == -1) {
      LOG_SOCKERR("setsockopt");
      goto exit;
    }
  }
  {
    int retry = 0;
    for (;;) {
      if (bind(lfd, (struct sockaddr *) &addr, sizeof addr) < 0) {
#ifdef WIN32
        if (WSAGetLastError() == WSAEADDRINUSE)
#else /* WIN32 */
        if (errno == EADDRINUSE)
#endif /* WIN32 */
        {
          SEN_LOG(sen_log_notice, "bind retrying..(%d)", port);
          if (++retry < 10) { sleep(2); continue; }
        }
        LOG_SOCKERR("bind");
        goto exit;
      }
      break;
    }
  }
  if (listen(lfd, LISTEN_BACKLOG) < 0) {
    LOG_SOCKERR("listen");
    goto exit;
  }
  if (ev) {
    if (sen_com_event_add(ev, lfd, (sen_com **)&cs)) { goto exit; }
  } else {
    if (!(cs = SEN_MALLOC(sizeof(sen_com_sqtp)))) { goto exit; }
    cs->com.fd = lfd;
  }
exit :
  if (cs) {
    cs->com.ev_in = sen_com_sqtp_acceptor;
    cs->msg_in = func;
  } else {
    close(lfd);
  }
  return cs;
}

sen_rc
sen_com_sqtp_close(sen_com_event *ev, sen_com_sqtp *cs)
{
  int fd = cs->com.fd;
  sen_rbuf_fin(&cs->msg);
  if (ev) {
    sen_com_event_del(ev, fd);
  } else {
    SEN_FREE(cs);
  }
  return close(fd) ? sen_other_error : sen_success;
}
