/*
 * All Rights Reserved, Copyright (C) 2003, Hitachi Software Engineering Co., Ltd.
 */
/* $Id: initial_policy.c,v 1.1.1.1 2003/01/20 02:39:36 ueno Exp $ */

#include <stdio.h>
#include <string.h>
#include "initial_policy.h"
#include "security_class.h"
#include "initial_sids.h"
#include "file_label.h"

/**
 *  list of attribute
 */
static char ATTRIBUTE[][32]={
	"global",
	"file_type",
	"fs_type",
	"netif_type",
	"netmsg_type",
	"node_type",
	"ttyfile",
	"port_type",
	"ptyfile",
	"sysadmfile",
	"tmpfsfile",
	"sysctl_type",
	"device_type",
	"mlstrustedobject",
	"serial_device",
	"root_dir_type",
	"noexattrfile"
};

/**
 *  @name:	include_file
 *  @about:	output the content of "filename" to "outfp"
 *  @args:	filename (char *) -> filename
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return:	return 0 on success, return -1 in failure.
 */
int
include_file(char *filename, FILE *outfp)
{
	FILE *fp;
	char buf[BUFSIZE];

	if ((fp=fopen(filename,"r")) == NULL)
	{
		fprintf(stderr, "file open error %s\n", filename);
		return -1;
	}

	fprintf(outfp, "#start of file:%s\n\n", filename);
	while (fgets(buf, sizeof(buf), fp) != NULL)
	{
		fprintf(outfp, "%s", buf);
	}
	fprintf(outfp, "#end of file:%s\n\n", filename);

	fclose(fp);

	return 0;
}

/**
 *  @name:	declare_attributes
 *  @about:	declare default attribute
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return:	none
 */
void
declare_attributes(FILE *outfp)
{
	int i;
	fprintf(outfp, "###attributes\n");
	for (i = 0; i < sizeof(ATTRIBUTE)/32; i++)
	{
		fprintf(outfp, "attribute %s;\n", ATTRIBUTE[i]);
	}
}

/**
 *  @name:	register_initial_types
 *  @about:	register label list reserved by system in defined_label_table.
 *  @args:	none
 *  @return:	none
 */
void
register_initial_types()
{
	int i, n;
	CONTEXT con;

	n = get_initial_contexts_num();

	for (i = 0; i < n; i++)
	{
		con = get_initial_contexts(i);

		/* kernel_t,kmod_t,init_t is used by domain */
		if (strcmp(con.type, "kernel_t") == 0)
		{
			continue;
		}
		if (strcmp(con.type, "init_t") == 0)
		{
			continue;
		}
		if (strcmp(con.type, "kmod_t") == 0)
		{
			continue;
		}
		register_label_table(con.type);
	}

	/* labels for tty/pts */
	register_label_table("tty_device_t");
	register_label_table("devtty_t");
	register_label_table("devpts_t");
	register_label_table("ptmx_t");

	/* labels for procfs */
	register_label_table("proc_t");
	register_label_table("proc_kmsg_t");
	register_label_table("proc_kcore_t");

	/* labels for tmpfs */
	register_label_table("tmpfs_t");

	/* labels for devfs */
	register_label_table("devfs_t");

	/* dummy label */
	register_label_table("dummy_label_t");
}

/**
 *  @name:	declare_initial_types
 *  @about:	declare default types
 *  @args:	outfp (FILE *) -> output file pointer
 *  @return:	none
 */
void
declare_initial_types(FILE *outfp)
{
	int i;
	int n;
	CONTEXT con;

	n = get_initial_contexts_num();

	fprintf(outfp, "#declare initial types\n");

	for (i = 0; i < n; i++)
	{
		con = get_initial_contexts(i);
		if (strcmp(con.role, "object_r") != 0)
		{
			fprintf(outfp, "type %s,global;\n", con.type);
		}
		else
		{
			if (strcmp(con.type, "netif_t") == 0)
			{
				fprintf(outfp, "type netif_t,netif_type;\n");
			}
			else if (strcmp(con.type, "netmsg_t") == 0)
			{
				fprintf(outfp, "type netmsg_t,netmsg_type;\n");
			}
			else if (strcmp(con.type, "node_t") == 0)
			{
				fprintf(outfp, "type node_t,node_type;\n");
			}
			else if (strcmp(con.type, "fs_t") == 0)
			{
				fprintf(outfp, "type fs_t,fs_type;\n");
			}
			else
			{
				fprintf(outfp, "type %s,file_type,fs_type;\n", con.type);
			}
		}
	}

	/* labels for tty/pts	*/
	fprintf(outfp, "#labels for terminals\n");
	fprintf(outfp, "type tty_device_t,file_type;\n");
	fprintf(outfp, "type devtty_t,file_type,ttyfile;\n");
	fprintf(outfp, "type devpts_t,file_type,ptyfile;\n");
	fprintf(outfp, "type ptmx_t,file_type;\n");

	/* labels for procfs	*/
	fprintf(outfp, "type proc_t,file_type,fs_type;\n");
	fprintf(outfp, "type proc_kmsg_t;\n");
	fprintf(outfp, "type proc_kcore_t;\n");

	/* labels for tmpfs	*/
	fprintf(outfp, "type tmpfs_t,fs_type,file_type,tmpfsfile;\n");

	/* labels for devfs	*/
	fprintf(outfp, "type devfs_t,fs_type,file_type;\n");

	/* dummy label		*/
	fprintf(outfp, "#dummy label\n");
	fprintf(outfp, "type dummy_label_t_t;\n");

	/**
 	 * add onabuta 2004/9/10 <onabuta@selinux.gr.jp
	 */
	fprintf(outfp, "type policy_src_t,file_type;\n");
	fprintf(outfp, "type no_access_t, file_type, sysadmfile;\n");
	fprintf(outfp, "type sysctl_hotplug_t, sysctl_type;\n");
	fprintf(outfp, "type sysctl_rpc_t, sysctl_type;\n");
	fprintf(outfp, "type sysctl_irq_t, sysctl_type;\n");
	fprintf(outfp, "role system_r types scmp_packet_t;\n");
	fprintf(outfp, "type sysfs_t, fs_type, root_dir_type, sysadmfile;\n");
	fprintf(outfp, "type usbdevfs_t, fs_type, root_dir_type, noexattrfile, sysadmfile;\n");
	fprintf(outfp, "type iso9660_t, fs_type, root_dir_type, noexattrfile, sysadmfile;\n");
	fprintf(outfp, "type romfs_t, fs_type, root_dir_type, sysadmfile;\n");
	fprintf(outfp, "type ramfs_t, fs_type, root_dir_type, sysadmfile;\n");
	fprintf(outfp, "type dosfs_t, fs_type, root_dir_type, noexattrfile, sysadmfile;\n");
	fprintf(outfp, "type sambafs_t, fs_type, root_dir_type, noexattrfile, sysadmfile;\n");
	fprintf(outfp, "type nfs_t, fs_type, root_dir_type;\n");
	fprintf(outfp, "type eventpollfs_t, fs_type;\n");
	fprintf(outfp, "type futexfs_t, fs_type;\n");
	fprintf(outfp, "type bdev_t, fs_type;\n");	
	fprintf(outfp, "type usbfs_t, fs_type;\n");
	fprintf(outfp, "type nfsd_fs_t, fs_type;\n");
	fprintf(outfp, "type rpc_pipefs_t, fs_type;\n");
	fprintf(outfp, "type device_t, file_type;\n");
	fprintf(outfp, "type null_device_t, file_type, device_type, mlstrustedobject;\n");
	fprintf(outfp, "type zero_device_t, file_type, device_type, mlstrustedobject;\n");
	fprintf(outfp, "type console_device_t, file_type, device_type;\n");
	fprintf(outfp, "type memory_device_t, file_type, device_type;\n");
	fprintf(outfp, "type random_device_t, file_type, device_type;\n");
	fprintf(outfp, "type urandom_device_t, file_type, device_type;\n");
	fprintf(outfp, "type usbtty_device_t, file_type, serial_device, device_type;\n");
	fprintf(outfp, "type fixed_disk_device_t, file_type, device_type;\n");
	fprintf(outfp, "type scsi_generic_device_t, file_type, device_type;\n");
	fprintf(outfp, "type removable_device_t, file_type, device_type;\n");
	fprintf(outfp, "type clock_device_t, file_type, device_type;\n");
	fprintf(outfp, "type tun_tap_device_t, file_type, device_type;\n");
	fprintf(outfp, "type misc_device_t, file_type, device_type;\n");
	fprintf(outfp, "type mouse_device_t, file_type, device_type;\n");
	fprintf(outfp, "type event_device_t, file_type, device_type;\n");
	fprintf(outfp, "type agp_device_t, file_type, device_type;\n");
	fprintf(outfp, "type dri_device_t, file_type, device_type;\n");
	fprintf(outfp, "type sound_device_t, file_type, device_type;\n");
	fprintf(outfp, "type ppp_device_t, file_type, device_type;\n");
	fprintf(outfp, "type framebuf_device_t, file_type, device_type;\n");
	fprintf(outfp, "type devfs_control_t, file_type, device_type;\n");
	fprintf(outfp, "type mtrr_device_t, file_type, device_type;\n");
	fprintf(outfp, "type apm_bios_t, file_type, device_type;\n");
	fprintf(outfp, "type v4l_device_t, file_type, device_type;\n");
	fprintf(outfp, "type tape_device_t, file_type, device_type;\n");
	fprintf(outfp, "type scanner_device_t, file_type, device_type;\n");
	fprintf(outfp, "type cpu_device_t, file_type, device_type;\n");
	fprintf(outfp, "type xserver_misc_device_t, file_type, device_type;\n");
	fprintf(outfp, "type auditd_device_t, file_type, device_type, sysadmfile;\n");
	fprintf(outfp, "type net_conf_t, file_type, sysadmfile;\n");
	fprintf(outfp, "typealias net_conf_t alias resolv_conf_t;\n");

	fprintf(outfp, "\n\n");

	return;
}

/**
 *  @name:	declare_initial_roles
 *  @about:	declare default roles
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return:	none
 */
void
declare_initial_roles(FILE *outfp)
{
	CONTEXT con;
	int i, n;

	n = get_initial_contexts_num();
	fprintf(outfp, "#declare initial roles\n");
	fprintf(outfp, "role system_r types{\n");

	for (i = 0; i < n; i++)
	{
		con = get_initial_contexts(i);
		if (strcmp(con.role, "system_r") == 0)
		{
			fprintf(outfp, "%s\n",con.type);
		}
	}
	fprintf(outfp, "};\n\n");

	return ;
}

/**
 *  @name:	declare_initial_users
 *  @about:	declare default users
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return:	none
 */
void
declare_initial_users(FILE *outfp)
{
	fprintf(outfp, "user system_u roles system_r;\n\n\n");
}

/**
 *  @name:	declare_initial_constrains
 *  @about:	print dummy constrains
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return:	none
 */
void
declare_initial_constrains(FILE *outfp)
{
	fprintf(outfp, "##dummy constrain\n");
	fprintf(outfp, "constrain process transition\n( u1 == u2 or u1 != u2 );\n\n\n");
}

/**
 *  @name:	allow_class
 *  @about:	print "allow"
 *  @args:	type (char *) -> type
 *  @args:	class (char *) -> class name
 *  @args:	fd (FILE *) -> output file descripter
 *  @return:	none
 */
static void
allow_class(char *type, char *class, FILE *fd)
{
	fprintf(fd, "allow %s *:%s *;\n", type, class);

	return;
}

/**
 *  @name:	allow_all
 *  @about:	type can access all classes
 *  @args:	type (char *) -> type
 *  @args:	fd (FILE *) -> output file descripter
 *  @return:	none
 */
static void
allow_all(char *type, FILE *fd)
{
	int i;

	for (i = 0; i < get_class_num(); i++)
	{
		allow_class(type, get_security_class(i), fd);
	}

	return ;
}

/**
 *  @name:	test_allow
 *  @about:	print test allow
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return:	none
 */
void
test_allow(FILE *outfp)
{
	CONTEXT con;
	int i,n;

	n = get_initial_contexts_num();

	for (i = 0; i < n; i++)
	{
		con = get_initial_contexts(i);
		if (strcmp(con.role, "system_r") == 0)
		{
			allow_all(con.type, outfp);
		}
	}

	return;
}

/**
 *  @name:	allow_socket
 *  @about:	allow kernel to use socket
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return:	none
 */
static void
allow_socket(FILE *outfp)
{
	/* icmp */
	fprintf(outfp, "allow icmp_socket_t netif_t:netif *;\n");
	fprintf(outfp, "allow icmp_socket_t node_t:node *; \n");
	fprintf(outfp, "allow icmp_socket_t netmsg_t:rawip_socket *;\n");

	/* tcp,udp */
	fprintf(outfp, "allow tcp_socket_t netif_t:netif *;\n"
		"allow tcp_socket_t netmsg_t:tcp_socket *;\n"
		"allow tcp_socket_t node_t:node *;\n"
		"allow netmsg_t netif_t:netif *;\n");

	/* network node */
	fprintf(outfp, "allow netmsg_t node_t:node *;\n");

	return;
}

/**
 *  @name:	default_allow
 *  @about:	print default allow 
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return:	none
 */
void
default_allow(FILE *outfp)
{
	/* necessary to use filesystem	*/
	fprintf(outfp, "allow file_type fs_t:filesystem associate;\n");

	allow_socket(outfp);
}
