/*
 * All Rights Reserved, Copyright (C) 2003, Hitachi Software Engineering Co., Ltd.
 */
/* $Id: file_label.c,v 1.1.1.1 2003/01/20 02:39:36 ueno Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "hashtab.h"
#include "global.h"
#include "file_label.h"

/**
 ***********************************************
 *   Functions to associate label with file 
 ***********************************************
 */

/**
 *  hash table of association of label with file
 *  key:file name
 */
HASH_TABLE *file_label_table;

HASH_TABLE *defined_label_table;

#define FILE_LABEL_TABLE_SIZE 1024*8

/**
 *  @name:	make_label
 *  @about:	make label name from filename.
 *  @args:	name (char *) -> label name
 *  @return:	returns label name by malloc.
 *  @notes:	!!Warning!! Conflict of label name isn't considered
 */
static char *
make_label(char *name)
{
	char *head = NULL;
	char *label;
	int len;
	int i;
	char *filename;

	filename = strdup(name);

	if (filename[0] != '/')
	{
		/**
		 * if the top of filename isn't "/",
		 * filename is "allow exclusive"
		 */
		label = strdup(name);
		return label;
	}

	/* the label of root directory */
	if (strcmp(name, "/") == 0)
	{
		label = strdup(ROOT_LABEL);
		return label;
	}

	/* chop "/" at the top of filename,at the end of filename */
	if (filename[0] == '/')
		head = filename+1;

	len = strlen(head);

	if (head[len-1] == '/')
	{
		head[len-1]='\0';
	}

	/* convert "/" to "_","." to "d" */
	for (i = 0; i < len; i++)
	{
		if (head[i] == '/')
		{
			head[i]='_';
		}
		if(head[i] == '.')
		{
			head[i]='d';
		}
		if (head[i] == '+')
		{
			head[i] = 'p';
		}
		if (head[i] == '-')
		{
			head[i] = 'm';
		}
	}

	label = (char *)malloc(sizeof(char)*(len+3));
	if (label == NULL)
	{
		perror("malloc");
		exit(1);
	}

	sprintf(label, "%s_t", head);
	free(filename);

	return label;
}

#if 0
/*
 * convert "path" to labelname.
 * return label name by malloc.
 * label is <program name>_exec_t
 */
static char *
make_exec_label(char *path)
{
	char *p;
	char *result;
	const char tail[] = "_exec_t";
	int len;
	int i;

	p = strrchr(path, '/');
	if (p == NULL)
	{
		yyerror("invalid file name\n");
		exit(1);
	}
	p = p+1;

	len = strlen(p);
	result = (char *)malloc(sizeof(char) * (len + sizeof(tail) + 1));
	if (result == NULL)
	{
		perror("malloc");
		exit(1);
	}

	sprintf(result, "%s%s", p, tail);

	len = strlen(result);
	for (i = 0; i < len; i++)
	{
		if (result[i] == '.')
		{
			result[i] = '_';
		}
	}

	//fprintf(stderr, "%s", result);

	return result;
}
#endif

/**
 *  @name:	add_file_label
 *  @about;	register label with FILE_LABEL.
 * 		label is generated by path in "file_acl".
 *  @args:	file_acl (void *) -> 
 *  @return:	return 0 on success
 */
static int
add_file_label(void *file_acl)
{
	FILE_ACL_RULE *acl;
	FILE_LABEL *label;
	FILE_LABEL *tmp;
	char *s;

	acl = file_acl;

	/* create label buffer */
	if ((label = (FILE_LABEL *)malloc(sizeof(FILE_LABEL))) == NULL)
	{
		perror("malloc");
		exit(1);
	}

	/* store path name and label name */
	label->filename = strdup(acl->path);
	label->labelname = make_label(acl->path);

	if (acl->only_flag == 1)
	{
		label->rec_flag = 0;
	}
	else
	{
		label->rec_flag = 1;
	}

	/* search same file info */
	if ((tmp = search_element(file_label_table, label->filename)) == NULL)
	{
		/* when filename isn't exclusive name, check label name conflict */
		if (strchr(label->filename, '/') != NULL)
		{
			if (register_label_table(label->labelname) == -2)
			{
				s=resolve_label_conflict(label->labelname);
				if (strcmp(label->filename, "/proc") != 0)
				{
					fprintf(stderr,
						"Warning label name conflict is detected.Label for %s was %s but replaced by %s.\n",
						label->filename, label->labelname, s);
				}
				free(label->labelname);
				label->labelname = s;
			}
		}

		insert_element(file_label_table, label, label->filename);
	}
	else
	{
		if (tmp->rec_flag != 0)
		{
			tmp->rec_flag = label->rec_flag;
		}
	}

	return 0;
}

/**
 *  @name:	one_domain_file_label
 *  @about:	
 *  @args:	domain (void *) -> domain
 *  @return:	return 0 on success 
 */
static int
one_domain_file_label(void *domain)
{
	DOMAIN *d;
	HASH_TABLE *acl;

	d = domain;
	acl = d->file_acl;
	handle_all_element(acl, add_file_label);

	return 0;
}

/**
 *  @name:	insert_force
 *  @about:	insert "filename,labelname" in file_label hash table
 *  @args:	filename (char *) -> file name
 *  @args:	labelname (char *) -> label name
 *  @return:	none
 */
static void
insert_force(char *filename, char *labelname)
{
	FILE_LABEL *l;

	if ((l = search_element(file_label_table, filename)) == NULL)
	{
		l = (FILE_LABEL *)malloc(sizeof(FILE_LABEL));
		l->filename = strdup(filename);
		l->labelname = strdup(labelname);
		l->rec_flag = 1;
		insert_element(file_label_table, l, l->filename);
	}
	else
	{
		free(l->labelname);
		l->labelname = strdup(labelname);
	}
}

/**
 *  @name:	set_domain_trans_label
 *  @about:	register label named by domain_trans
 *  @args:	none
 *  @return:	none
 */
static void
set_domain_trans_label()
{
	int i;
	TRANS_RULE t;
	FILE_LABEL *label;
	FILE_LABEL *tmp;
	int st;
	char *s;

	for (i = 0; i < domain_trans_rule_num; i++)
	{
		t = rulebuf[i];
		if ((label = (FILE_LABEL *)malloc(sizeof(FILE_LABEL))) == NULL)
		{
			perror("malloc");
			exit(1);
		}
		label->filename = strdup(t.path);
		label->labelname = make_label(t.path);

		tmp = search_element(file_label_table, label->filename);
		if (tmp == NULL)
		{
			if (register_label_table(label->labelname) == -2)
			{
				s=resolve_label_conflict(label->labelname);
				free(label->labelname);
				label->labelname = s;
				fprintf(stderr,
					"Warning label name conflict is detected.Label for %s was %s but replaced by %s.\n",
					label->filename, label->labelname, s);
			}
			st = insert_element(file_label_table, label, label->filename);
		}
		else
		{
			tmp->rec_flag=0;	//???OK?
		}
	}

	return;
}

/**
 *  @name:	create_file_label_table
 *  @about:	major function to label files
 *  @args:	domaintab (HASH_TABLE *) -> hash table
 *  @return:	return 0 on success
 */
int
create_file_label_table(HASH_TABLE *domaintab)
{
	/* create file hash table */
	file_label_table = create_hash_table(FILE_LABEL_TABLE_SIZE);

	/* store handler function to the domain table */
	handle_all_element(domaintab, one_domain_file_label);

	/*  /...security and  /etc/security/selinux are labeled by default.  */
	insert_force(PSID_LABEL_FILE, "file_labels_t");
	insert_force(SECURITY_CONF_FILE, "policy_config_t");
	insert_force("/proc", "proc_t");

	set_domain_trans_label();

	return 0;
}

/**
 *  @name:	print_file_label
 *  @about:	print file label
 *  @args:	n (void *) -> file data
 *  @retrun:	return 0
 */
static int
print_file_label(void *n)
{
	FILE_LABEL *l;

	l = n;
	printf("%s,%s\n", l->filename, l->labelname);

	return 0;
}

/**
 *  @name:	print_file_label_tab
 *  @about:	print file label table
 *  @args:	none
 *  @return:	none
 */
void
print_file_label_tab()
{
	handle_all_element(file_label_table, print_file_label);
}

/**
 *  @name:	free_file_label
 *  @about:	free file label buffer
 *  @args:	l (void *) -> label buffer
 *  @return:	return 0
 */
static int
free_file_label(void *l)
{
	FILE_LABEL *n;

	n = l;
	free(n->filename);
	free(n->labelname);
	free(n);

	return 0;
}

/**
 *  @name:	delete_file_label_tab
 *  @about:	delete file label table
 *  @args:	none
 *  @return:	none
 */
void
delete_file_label_tab()
{
	handle_all_element(file_label_table, free_file_label);
	delete_hash_table(file_label_table);
}

/**
 *  output file pointer (static)
 */
static FILE *out_fp;

/**
 *  @name:	print_type
 *  @about:	print file type
 *  @args:	n (void *) -> FILE_LABEL pointer 
 *  @return:	return 0 on success
 */	
static int
print_type(void *n)
{
	FILE_LABEL *l;

	l = n;

	/**
	 *  these labels are already declared in "declare_initial_types" function
	 */
	if (strcmp(l->filename, PSID_LABEL_FILE) == 0 ||
	    strcmp(l->filename, SECURITY_CONF_FILE) == 0 ||
	    strcmp(l->filename, "/proc") == 0)
	{
		return 0;
	}

	fprintf(out_fp, "type %s,file_type;\n", l->labelname);

	return 0;
}

/**
 *  @name:	out_file_type
 *  @about:	output file type
 *  @args:	out (FILE *) -> output file descripter
 *  @return:	none
 */
void
out_file_type(FILE *out)
{
	out_fp = out;
	handle_all_element(file_label_table, print_type);
}

/**
 *  Functions related to file_context
 */
static char **file_name_buf;
static FILE *context_out_fp;

/**
 *  @name:	set_buf
 *  @about:	to set value in array to  sort
 * 		This function is supposed to be called only once because of "static"
 *  @args:	e (void *) -> file label
 *  @return:	return 0 on success
 */
static int
set_buf(void *e)
{
	static int i=0;
	FILE_LABEL *l;
	l = e;
	file_name_buf[i] = l->filename;
	i++;

	return 0;
}

/**
 *  @name:	compar
 *  @about:	compare strings
 *  @args:	a (void *) -> first target
 *  @args:	b (void *) -> second target
 *  @return:	return 0 if a < b, otherwise return 1
 */
static int
compar(const void *a, const void *b)
{
	return strcmp(*(char **)a, *(char **)b);
}

/**
 *  @name:	out_file_context
 *  @about:	output file context
 *  @args:	out (FILE *) -> output file descripter
 *  @return:	none
 */
void
out_file_context(FILE *out)
{
	int i;
	FILE_LABEL *fl;
	struct stat buf;
	char tmp[1024];
	FILE *fp;
	context_out_fp=out;


	file_name_buf = (char **)malloc(sizeof(char *)*(file_label_table->element_num));
	if (file_name_buf == NULL)
	{
		perror("malloc");
		exit(1);
	}

	handle_all_element(file_label_table, set_buf);

	qsort(file_name_buf, file_label_table->element_num, sizeof(char *), compar);

	/*print file_contexts*/
	fprintf(out, "#This file is generated by policyedit converter\n");

	fprintf(out, "/(|.*)\t system_u:object_r:file_t\n");

	for (i = 0; i < file_label_table->element_num; i++)
	{
		fl = (FILE_LABEL *)search_element(file_label_table, file_name_buf[i]);
		if (fl == NULL)
		{
			fprintf(stderr, "bug???error\n");
			exit(1);
		}

		/*skip if file doesn't exist */
		if (stat(file_name_buf[i], &buf) == -1)
		{
			continue;
		}

		if (buf.st_mode & S_IFDIR)
		{
			if (fl->rec_flag == 1)
			{
				fprintf(out, "%s(|/.*)\tsystem_u:object_r:%s\n", file_name_buf[i], fl->labelname);
			}
			else
			{
				if (strcmp(file_name_buf[i], "/") == 0)
				{
					fprintf(out, "%s(|[^/]*) \t system_u:object_r:%s\n", file_name_buf[i], fl->labelname);
				}
				else
				{
					fprintf(out, "%s(|/[^/]*) \t system_u:object_r:%s\n", file_name_buf[i], fl->labelname);
				}
			}
		}
		else
		{
			fprintf(out, "%s\tsystem_u:object_r:%s\n", file_name_buf[i], fl->labelname);
		}
	}

	fprintf(out, "#These labels are to protect Terminal\n");
	fprintf(out, "/dev/[^/]*tty[^/]* \tsystem_u:object_r:tty_device_t\n");
	fprintf(out, "/dev/[^/]*pty[^/]* \tsystem_u:object_r:tty_device_t\n");
	fprintf(out, "/dev/vcs[^/]*\tsystem_u:object_r:tty_device_t\n");
	fprintf(out, "/dev/tty\tsystem_u:object_r:devtty_t\n");
	fprintf(out, "/dev/ptmx\tsystem_u:object_r:ptmx_t\n");

	free(file_name_buf);


	if ((fp=fopen(TMP_FILE_NAME,"r")) == NULL)
	{
		return;
	}

	fprintf(out, "#These files are labeled by file_type_auto_trans\n");

	while (fgets(tmp, sizeof(tmp), fp) != NULL)
	{
		fputs(tmp, out);
	}

	fclose(fp);

	return;
}

/**
 *  @name:	registar_label_table
 *  @about;	register "name" with defined_label_table.and return -2
 * 		if "name" already exists
 *  @args:	name (char *) -> label name
 *  @return:	return 0 on success, return -2 if label already exists
 */
int
register_label_table(char *name)
{
	int *v;
	v=(int *)malloc(sizeof(int));
	*v=1;

	return insert_element(defined_label_table, v, name);
}

/**
 *  @name:	print_label_table
 *  @about:	print label table
 *  @args:	none
 *  @return:	none
 */
void
print_label_table()
{
	int num,i;
	HASH_NODE **n;

	n = create_hash_array(defined_label_table);
	num = defined_label_table->element_num;

	for (i = 0; i < num; i++)
	{
		printf("%s\n", (char *)(n[i]->key));
	}
}

/**
 *  @name:	resovle_label_conflict
 *  @about:	resolve label name conflict and returns new label
 * 		new label name is automatically registered.
 *  @args:	name (char *) -> label name
 *  @return:	??
 */
char *
resolve_label_conflict(char *name)
{
	char *tmp;
	char *tmp2;
	static char foot[] = "_t";

	tmp = strdup(name);

	do {
		tmp2 = (char *)malloc(sizeof(char)*(strlen(tmp)+sizeof(foot)));
		sprintf(tmp2, "%s%s", tmp, foot);
		free(tmp);
		tmp=tmp2;
	} while(register_label_table(tmp2) == -2);

	return tmp2;
}
