/*
 * All Rights Reserved, Copyright (C) 2003, Hitachi Software Engineering Co., Ltd.
 */
/* $Id: convert.c,v 1.2 2003/02/26 02:05:49 ynakam Exp $ */

/**
 *  Convert DOMAIN hash table to SELinux configuration language.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/types.h>
#include "global.h"
#include "file_label.h"
#include "action.h"
#include "hashtab.h"
#include "convert.h"
#include "debug.h"
#include "out_file_acl.h"

/**
 *  These classes are not supported enough.
 */
char NOT_SUPPORTED_CLASSES[][128] = {
	"security",
	"process",
	"system",
	"capability",
	"netlink_socket",
	"packet_socket",
	"key_socket",
	"ipc",
	"filesystem",
	"fd"
};

/**
 *  these domains are reserved by SELinux
 */
char DEFAULT_DOMAIN[][32] = {
	"kernel_t",
	"init_t",
	"kmod_t"
};

static void convert_file_acl();
static void out_allow(FILE *);
static void out_domain_trans(FILE *);
//static void out_dummy_user_conf(FILE *);
static void out_net_type(FILE *);
static void out_rbac(FILE *);
char *get_prefix(char *);

/**
 *  @name:	convert
 *  @about:	Major process of converter
 *  @args:	policy_fp (FILE *) -> FILE pointer to print SELinux configuration.
 *  @args:	file_context_fp (FILE *) -> :FILE pointer to print file_contexts.
 *  @return:	none
 */
void
convert(FILE *policy_fp, FILE *file_context_fp)
{
	/* open temporally file */
	TMP_fp = fopen(TMP_FILE_NAME, "w");
	if (TMP_fp == NULL)
	{
		TMP_fp = stdout;
	}

	if (domain_hash_table == NULL)
	{
		fprintf(stderr, "bug? domain table is not initialized.\n");
		exit(1);
	}

	/* calculate relationship between file and label. */
	create_file_label_table(domain_hash_table);
#ifdef DEBUG
//	print_file_label_tab();
#endif

	/* print "type .." of label on files */
	out_file_type(policy_fp);
	out_net_type(policy_fp);

	/**
	 *  find "access right to some file of global">"access right to some file of domain"
	 *  and convert such access rights to "access right to some file of global"<"access
	 *  right to some file of domain"
	 */
	convert_file_acl();

	/* print "allow ..." */
	out_allow(policy_fp);

	/* print "domain_auto_trans or domain_trans " */
	out_domain_trans(policy_fp);

	/* print rbac configuration */
	out_rbac(policy_fp);

	if (TMP_fp != stdout)
	{
		fclose(TMP_fp);
	}

	/* print file_contexts */
	out_file_context(file_context_fp);
#ifdef DEBUG
	//print_user();
#endif
}

/**
 *  @name:	out_net_type
 *  @about:	print labels of port numbers
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return:	none
 *  @notes:	used_tcp_ports and used_udp_ports is global variable
 */
static void
out_net_type(FILE *outfp)
{
	int i;

	/* label unreserved Well-Known ports default name  */
	fprintf(outfp, "##Labels for Well-Known ports\n");
	fprintf(outfp, "type well_known_tcp_port_t,port_type;\n");
	fprintf(outfp, "type well_known_udp_port_t,port_type;\n");

	/*  when reserved by allownet,new name is given  */
	for (i = 0; i < sizeof(used_tcp_ports); i++)
	{
		if (used_tcp_ports[i] == 1)
		{
			fprintf(outfp, "type tcp%d_port_t,port_type;\n", i);
		}
		if (used_udp_ports[i] == 1)
		{
			fprintf(outfp, "type udp%d_port_t,port_type;\n", i);
		}
	}
}

/**
 *  @name:	out_net_acl
 *  @about:	print allow about network ACL
 *  @args:	outfp (FILE *) -> output file descripter
 *  @args:	d (DOMAIN *) -> DOMAIN structure
 *  @return:	none
 */
static void
out_net_acl(FILE *outfp, DOMAIN *d)
{
	NET_RULE *net;
	int i;

	net=d->net_acl;
	if (net == NULL)
		return;

	/* allownet (1) */
	if ((net->perm & (CAN_TCP|CAN_UDP)) != 0)
	{
		fprintf(outfp, "\n###can use tcp and udp\n");
		fprintf(outfp, "can_network(%s)\n", d->name);
	}

	/* allownet(2) */
	if (((net->perm) & CAN_RAW) != 0)
	{
		fprintf(outfp, "\n##can use raw socket\n");
		fprintf(outfp, "allow %s *:rawip_socket { create bind setopt getopt write read };\n",
			d->name);
		fprintf(outfp, "allow %s any_socket_t:rawip_socket sendto;\n", d->name);
		fprintf(outfp, "allow %s icmp_socket_t:rawip_socket recvfrom;\n", d->name);
		fprintf(outfp, "allow %s %s:capability net_raw;\n", d->name, d->name);
	}

	/* allownet(3) */
	if (((net->perm) & CAN_WELLKNOWN) != 0)
	{
		fprintf(outfp, "\n##can use wellknown port\n");
		fprintf(outfp, "allow %s %s:capability net_bind_service;\n", d->name, d->name);
	}

	/* allownet(4) */
	if (net->flag == 1)
	{
		fprintf(outfp, "\n##can bind well-known port\n");
		for (i = 0; i < sizeof(used_tcp_ports); i++)
		{
			if (net->tcp_port[i] == 1)
				fprintf(outfp, "allow %s tcp%d_port_t:tcp_socket name_bind;\n", d->name, i);
			if (net->udp_port[i] == 1)
				fprintf(outfp, "allow %s udp%d_port_t:udp_socket name_bind;\n", d->name, i);
		}
	}

	/* allownet(5) */
	if (((net->perm) & CAN_WELLKNOWN_TCP) != 0)
	{
		fprintf(outfp, "\n##can use unreserved well-known tcp port\n");
		fprintf(outfp, "allow %s well_known_tcp_port_t:tcp_socket name_bind;\n", d->name);
	}
	if (((net->perm) & CAN_WELLKNOWN_UDP) != 0)
	{
		fprintf(outfp, "\n##can use unreserved well-known udp port\n");
		fprintf(outfp, "allow %s well_known_udp_port_t:udp_socket name_bind;\n", d->name);
	}

	return;
}

/**
 *  @name:	out_netcontext
 *  @about:	print labels about network
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return:	none
 */
void
out_netcontext(FILE *outfp)
{
	int i;

	fprintf(outfp, "####net_contexts\n");

	/* label well-known ports */
	for (i = 0; i < 1024; i++)
	{
		if (used_tcp_ports[i] == 1)
		{
			fprintf(outfp, "portcon tcp %d system_u:object_r:tcp%d_port_t\n", i, i);
		}
		else
		{
			fprintf(outfp, "portcon tcp %d system_u:object_r:well_known_tcp_port_t\n", i);
		}
		if (used_udp_ports[i] == 1)
		{
			fprintf(outfp, "portcon udp %d system_u:object_r:udp%d_port_t\n", i, i);
		}
		else
		{
			fprintf(outfp, "portcon udp %d system_u:object_r:well_known_udp_port_t\n", i);
		}
	}

	/* dummy labels: to satisfy syntax  */
	fprintf(outfp, "netifcon lo system_u:object_r:netif_t system_u:object_r:netmsg_t\n");
	fprintf(outfp, "nodecon 127.0.0.1 255.255.255.255 system_u:object_r:node_t\n");
}

/**
 *  @name:	out_one_tmpfs_rw
 *  @name:	print "allow tmpfs r/w"
 *  @args:	outfp (FILE *) -> output file descripter
 *  @args:	from (char *) -> 
 *  @args:	to (char *) -> 
 *  @args:	perm (int) -> permission
 *  @return:	none
 */
static void
out_one_tmpfs_rw(FILE *outfp, char *from, char *to, int perm)
{
	char *prefix;
	char *tmpfs_label;

	/* permission type check */
	if ((perm & READ_PRM) == 0 && (perm & WRITE_PRM) == 0)
	{
		return;
	}

	prefix = get_prefix(to);

	if (strcmp(to, "global") == 0)
	{
		tmpfs_label = strdup("tmpfsfile");
	}
	else if(strcmp(to, "general") == 0)
	{
		tmpfs_label = strdup("tmpfs_t");
	}
	else
	{
		tmpfs_label = (char *)malloc(strlen(prefix) + strlen("_tmpfs_t") + 1);
		sprintf(tmpfs_label, "%s_tmpfs_t", prefix);
	}

	free(prefix);

	if (perm & READ_PRM)
	{
		fprintf(outfp, "##%s can read %s 's tmpfs\n", from, to);
	}
	if (perm & WRITE_PRM)
	{
		fprintf(outfp, "##%s can write %s 's tmpfs\n", from, to);
	}

	fprintf(outfp, "allow %s %s:dir_file_class_set ", from, tmpfs_label);
	fprintf(outfp, "{");
	fprintf(outfp, " getattr setattr lock  ioctl ");

	if (perm & READ_PRM)
	{
		fprintf(outfp, "read execute  ");
	}

	if (perm & WRITE_PRM)
	{
		fprintf(outfp, "write create unlink link rename ");
	}

	fprintf(outfp, "};\n");

	if (perm & READ_PRM)
	{
		fprintf(outfp, "allow %s %s:dir search;\n", from, tmpfs_label);
		fprintf(outfp, "allow %s %s:file execute_no_trans;\n", from, tmpfs_label);
	}
	if(perm & WRITE_PRM)
	{
		fprintf(outfp, "allow %s %s:dir {add_name remove_name reparent rmdir };\n",
			from, tmpfs_label);
	}

	free(tmpfs_label);
}

/**
 *  @name:	out_one_com_acl
 *  @about:	print "allow" that means "domain_name" is allowed to use "acl"
 *  @args:	outfp (FILE *) -> output file descripter
 *  @args:	domain_name (char *) -> domain name
 *  @args:	acl (COM_ACL_RULE *) -> communication acl rule data
 *  @return:	none
 */
static void
out_one_com_acl(FILE *outfp, char *domain_name, COM_ACL_RULE *acl)
{
	int flag;
	char *to_domain = NULL;
	char *prefix = NULL;
	flag = acl->flag;

	//  to_domain=acl->domain_name;
	if (strcmp(acl->domain_name, "self") == 0 ||
	    strcmp(acl->domain_name, "global") == 0 ||
	    strcmp(acl->domain_name, "general") == 0)
	{
		prefix = strdup("dummy");
		to_domain = strdup(acl->domain_name);
	}
	else
	{
		prefix = get_prefix(acl->domain_name);
		to_domain = malloc(strlen(prefix) + 3);
		sprintf(to_domain, "%s_t", prefix);
	}

	switch (flag)
	{
	case TCP_ACL:
		fprintf(outfp, "\n#%s and  %s can communicate tcp \n", domain_name, to_domain);
		fprintf(outfp, "allow %s %s:tcp_socket *;\n", domain_name, to_domain);
		break;

	case UDP_ACL:
		fprintf(outfp, "\n#%s and  %s can communicate udp \n", domain_name, to_domain);
		fprintf(outfp, "allow %s %s:udp_socket *;\n", domain_name, to_domain);
		break;

	case UNIX_ACL:
		fprintf(outfp, "\n#%s and  %s can communicate unix domain \n", domain_name, to_domain);
		fprintf(outfp, "allow %s %s:unix_dgram_socket *;\n", domain_name, to_domain);
		fprintf(outfp, "allow %s %s:unix_stream_socket *;\n", domain_name, to_domain);
		break;

	case SEM_ACL:
		if (acl->perm & READ_PRM)
		{
			fprintf(outfp, "\n#sem read\n");
			fprintf(outfp, "allow %s %s:sem r_sem_perms;\n", domain_name, to_domain);
		}
		if(acl->perm & WRITE_PRM)
		{
			fprintf(outfp, "\n#sem write\n");
			fprintf(outfp, "allow %s %s:sem {setattr create destroy  write  unix_write};\n",
				domain_name, to_domain);
		}
		break;

	case MSG_ACL:
		if (acl->perm & READ_PRM)
		{
			fprintf(outfp, "\n#msg read\n");
			fprintf(outfp, "allow %s %s:msg send;\n", domain_name, to_domain);
		}

		if (acl->perm & WRITE_PRM)
		{
			fprintf(outfp, "\n#msg write\n");
			fprintf(outfp, "allow %s %s:msg receive;\n", domain_name, to_domain);
		}
		break;

	case MSGQ_ACL:
		if (acl->perm & READ_PRM)
		{
			fprintf(outfp, "\n#msgq read\n");
			fprintf(outfp, "allow %s %s:msgq r_msgq_perms;\n", domain_name, to_domain);
		}
		if(acl->perm & WRITE_PRM)
		{
			fprintf(outfp, "\n#msgq write\n");
			fprintf(outfp, "allow %s %s:msgq { setattr create destroy  write enqueue unix_write };\n",
				domain_name, to_domain);
		}
		break;

	case SHM_ACL:
		if (acl->perm & READ_PRM)
		{
			fprintf(outfp, "\n#shm read\n");
			fprintf(outfp, "allow %s %s:shm r_shm_perms;\n", domain_name, to_domain);
		}
		if(acl->perm & WRITE_PRM)
		{
			fprintf(outfp, "\n#shm write\n");
			fprintf(outfp, "allow %s %s:shm { setattr create destroy  write  unix_write lock};\n",
				domain_name, to_domain);
		}
		break;

	case PIPE_ACL:
		if (acl->perm & READ_PRM)
		{
			fprintf(outfp, "\n#pipe read\n");
			fprintf(outfp, "allow %s %s:fifo_file r_file_perms;\n", domain_name, to_domain);
		}
		if(acl->perm & WRITE_PRM)
		{
			fprintf(outfp, "\n#pipe write\n");
			fprintf(outfp, "allow %s %s:fifo_file { ioctl write create setattr lock relabelfrom "
				"relabelto append  unlink link rename execute swapon quotaon mounton };\n",
				domain_name, to_domain);
		}
		break;

	case SIG_ACL:
		if(acl->perm & CHID_PRM)
		{
			fprintf(outfp, "\n#sigchid\n");
			fprintf(outfp, "allow %s %s:process sigchld;\n", domain_name, to_domain);
		}
		if(acl->perm & KILL_PRM)
		{
			fprintf(outfp, "\n#sigkill\n");
			fprintf(outfp, "allow %s %s:process sigkill;\n", domain_name, to_domain);
		}
		if(acl->perm & STOP_PRM)
		{
			fprintf(outfp, "\n#sigstop\n");
			fprintf(outfp, "allow %s %s:process sigstop;\n", domain_name, to_domain);
		}
		if(acl->perm & OTHERSIG_PRM)
		{
			fprintf(outfp, "\n#other signals\n");
			fprintf(outfp, "allow %s %s:process signal;\n", domain_name, to_domain);
		}
		break;

	case TMPFS_ACL:
		out_one_tmpfs_rw(outfp, domain_name, to_domain, acl->perm);
		break;

	default:
		break;
	}

	free(prefix);
	free(to_domain);
}

/**
 *  @name:	out_com_acl
 *  @about:	print "allow" from acl_array in DOMAIN structure.
 *  @args:	outfp (FILE *) -> output file descripter
 *  @args:	d (DOMAIN *) -> domain data
 *  @return:	none
 */
static void
out_com_acl(FILE *outfp, DOMAIN *d)
{
	int i;
	COM_ACL_RULE *acl;

	for (i = 0; i < d->com_acl_array_num; i++)
	{
		acl = &(d->com_acl_array[i]);
		out_one_com_acl(outfp, d->name, acl);
	}
}

/**
 *  @name:	out_adm_acl
 *  @about:	print "allow" from admin_rule in DOMAIN structure
 *  @args:	outfp (FILE *) -> output file descripter
 *  @args:	d (DOMAIN *) -> domain data
 *  @return:	none
 */
static void
out_adm_acl(FILE *outfp, DOMAIN *d)
{
	ADMIN_RULE r;
	char *name;

	r = d->admin_rule;
	name = d->name;

	fprintf(outfp, "\n#Administrative permissions \n");
	if (r.relabel)
	{
		fprintf(outfp, "\n#for relabel permission\n");
		fprintf(outfp, "allow %s *:dir { relabelfrom relabelto } ;\n", name);
		fprintf(outfp, "allow %s *:file_class_set { relabelfrom relabelto };\n", name);
	}
	if (r.getsecurity)
	{
		fprintf(outfp, "\n#for getsecurity\n");
		//fprintf(outfp, "allow %s *:security compute_relabel;\n", name);
		fprintf(outfp, "can_getsecurity(%s)\n", name);
	}

	if (r.setenforce)
	{
		fprintf(outfp, "\n#for setenforce\n");
		fprintf(outfp, "can_setenforce(%s)\n", name);
	}

	if (r.load_policy)
	{
		fprintf(outfp, "\n#for load_policy\n");
		fprintf(outfp, "allow %s security_t:security load_policy;\n", name);
	}

	if (r.net)
	{
		fprintf(outfp, "\n#for net admin\n");
		fprintf(outfp, "allow %s self:capability net_admin;\n", name);
	}

	if (r.boot)
	{
		fprintf(outfp, "\n#for boot\n");
		fprintf(outfp, "allow %s self:capability sys_boot;\n", name);
	}

	if (r.insmod)
	{
		fprintf(outfp, "\n#for insmod\n");
		fprintf(outfp, "allow %s self:capability sys_module;\n", name);
	}

	if (r.quotaon)
	{
		fprintf(outfp, "\n#for quotaon\n");
		fprintf(outfp, "allow %s *:dir quotaon;\n", name);
	}

	if (r.swapon)
	{
		fprintf(outfp, "\n#for swapon\n");
		fprintf(outfp, "allow %s *:blk_file swapon;\n", name);
	}

	if (r.mount)
	{
		fprintf(outfp, "\n#for mount\n");
		fprintf(outfp, "allow %s *:dir mounton;\n", name);
		fprintf(outfp, "allow %s *:filesystem mount_fs_perms;\n", name);
		fprintf(outfp, "allow %s file_t:dir search;\n", name);
	}

	if (r.raw_io)
	{
		fprintf(outfp, "\n#for raw I/O\n");
		fprintf(outfp, "allow %s self:capability sys_rawio;\n", name);
	}

	if (r.ptrace)
	{
		fprintf(outfp, "\n#for ptrace\n");
		fprintf(outfp, "allow %s self:capability sys_ptrace;\n", name);
	}

	if (r.chroot)
	{
		fprintf(outfp, "\n#for chroot\n");
		fprintf(outfp, "allow %s self:capability sys_chroot;\n", name);
	}

	if (r.search)
	{
		fprintf(outfp, "\n#can search all directories\n");
		fprintf(outfp, "allow %s *:dir { getattr read search } ;\n", name);
		fprintf(outfp, "allow %s *:file_class_set { getattr };\n", name);
		fprintf(outfp, "allow %s *:lnk_file { read };\n", name);
	}
	if (r.unlabel)
	{
		fprintf(outfp, "\n#can read/write unlabeled files\n");
		fprintf(outfp, "allow %s unlabeled_t:file_class_set r_file_perms;\n", name);
		fprintf(outfp, "allow %s unlabeled_t:dir r_dir_perms;\n", name);
		fprintf(outfp, "allow %s unlabeled_t:dir_file_class_set", name);
		fprintf(outfp, "{ write setattr append create unlink link rename };\n");
		fprintf(outfp, "allow %s unlabeled_t:dir { add_name remove_name reparent rmdir };\n", name);
		fprintf(outfp, "allow unlabeled_t unlabeled_t:filesystem { associate };\n");
	}

	if (r.read)
	{
		fprintf(outfp,"\n#can read all directories\n");
		fprintf(outfp,"allow %s *:dir_file_class_set { read getattr ioctl lock  };\n", name);
	}

	if (r.write)
	{
		fprintf(outfp,"\n#can write all directories\n");
		fprintf(outfp,"allow %s *:dir_file_class_set "
			"{ write setattr append create unlink link rename };\n", name);
		fprintf(outfp,"allow %s *:dir { add_name remove_name reparent rmdir };\n", name);
		/*part_relabel is in print_allow*/
	}
}

/**
 *  @name:	out_not_suppoerted_allow
 *  @about:	print "allow" about  unsupported configuration.
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return:	none
 */
static void
out_not_supported_allow(FILE *outfp)
{
	int j;

	fprintf(outfp,"\n##########These are not supported configurations\n");

	for (j = 0; j < sizeof(NOT_SUPPORTED_CLASSES)/128; j++)
	{
		/* capability,process,system,security,filesystem */
		if (strcmp(NOT_SUPPORTED_CLASSES[j], "capability") == 0)
		{
			fprintf(outfp, "allow global *:%s ~{ net_raw net_bind_service net_admin sys_boot "
				"sys_module sys_rawio sys_ptrace sys_chroot };\n", NOT_SUPPORTED_CLASSES[j]);
		}
		else if (strcmp(NOT_SUPPORTED_CLASSES[j], "process") == 0)
		{
			fprintf(outfp, "allow global *:%s ~{ transition sigchld sigkill sigstop signal };\n",
				NOT_SUPPORTED_CLASSES[j]);
		}
		else if (strcmp(NOT_SUPPORTED_CLASSES[j], "security") == 0)
		{
			fprintf(outfp, "allow global *:%s ~{ compute_relabel load_policy setenforce };\n",
				NOT_SUPPORTED_CLASSES[j]);
		}
		else if (strcmp(NOT_SUPPORTED_CLASSES[j], "filesystem") == 0)
		{
			fprintf(outfp, "allow global *:%s ~{ mount remount unmount };\n",
				NOT_SUPPORTED_CLASSES[j]);
		}
		else
		{
			fprintf(outfp, "allow global *:%s *;\n", NOT_SUPPORTED_CLASSES[j]);
		}
	}
}

/**
 *  @name:	role_to_termlabel
 *  @about:	Convert role name to tty's label name.
 * 		return tty's label by malloc.
 *  @args:	role (char *) -> role name
 *  @return:	terminal role
*/
char *
role_to_termlabel(char *role)
{
	char *tmp;
	char *result;
	int len;
	char tail[] = "_tty_device_t";

	tmp = strdup(role);
	len = strlen(tmp);
	if (len < 2)
	{
		fprintf(stderr, "Invalid rolename %s\n", role);
		exit(1);
	}

	/*chop "_r"*/
	tmp[len-2] = '\0';

	result = (char *)malloc(sizeof(char)*(len+strlen(tail)+1));
	sprintf(result, "%s%s", tmp, tail);

	free(tmp);

	return result;
}

/**
 *  @name:	role_to_pts_termlabel
 *  @about:	Convert role name to pty's label name.
 * 		return tty's label by malloc.
 *  @args:	role (char *) -> role name
 *  @return:	role name
 */
char *
role_to_pts_termlabel(char *role)
{
	char *tmp;
	char *result;
	int len;
	char tail[] = "_devpts_t";

	tmp = strdup(role);
	len = strlen(tmp);
	if (len < 2)
	{
		fprintf(stderr, "Invalid rolename %s\n", role);
		exit(1);
	}

	/*chop "_r"*/
	tmp[len-2] = '\0';

	result = (char *)malloc(sizeof(char)*(len+strlen(tail)+1));
	sprintf(result, "%s%s", tmp, tail);

	free(tmp);
	return result;
}

/**
 *  ALLOWTTY ACL
 */
static void out_tty_acl_global(FILE *, DOMAIN *, int);

/**
 *  @name:	out_tty_acl
 *  @about:	output tty acl rule
 *  @args:	outfp (FILE *) -> output file descripter
 *  @args:	domain (DOMAIN *) -> domain name
 *  @return	none
 */
static void
out_tty_acl(FILE *outfp, DOMAIN *domain)
{
	TTY_RULE *t;
	int i;

	fprintf(outfp,"\n##TTY configurations \n");
	if (domain->tty_create_flag == 1)
	{
		/*allowtty -create*/
		fprintf(outfp, "#%s can create its own terminal \n", domain->name);
		fprintf(outfp, "type %s,file_type,sysadmfile,ttyfile;\n", role_to_termlabel(domain->name));
		fprintf(outfp, "type_change %s tty_device_t:chr_file %s;\n",
			domain->name, role_to_termlabel(domain->name));
		fprintf(outfp, "type_change %s ttyfile:chr_file %s;\n",
			domain->name, role_to_termlabel(domain->name));
	}


	for(i = 0; i < domain->tty_acl_array_num; i++)
	{
		t = &(domain->tty_acl_array)[i];

		if (strcmp(t->rolename, "general") == 0)
		{						/*allowtty general*/
			if (t->perm & READ_PRM)
			{
				fprintf(outfp, "#%s can read general tty\n", domain->name);
				fprintf(outfp, "allow %s {devtty_t tty_device_t }:{ chr_file lnk_file }r_file_perms;\n",
					domain->name);
			}
			if (t->perm & WRITE_PRM)
			{
				fprintf(outfp, "#%s can write role general tty\n", domain->name);
				fprintf(outfp, "allow %s {devtty_t tty_device_t }:{ chr_file lnk_file }{ write setattr append };\n",
					domain->name);
			}

			if (t->perm & CHANGE_PRM)
			{
				fprintf(outfp, "#%s can change general tty's label\n", domain->name);
				fprintf(outfp, "allow %s { devtty_t tty_device_t }:{ chr_file lnk_file} { getattr relabelfrom relabelto };\n",
					domain->name);
			}
			continue;
		}

		if (strcmp(t->rolename, "global") == 0)
		{						/*allowtty global*/
			out_tty_acl_global(outfp, domain, t->perm);
			continue;
		}

		/*normal domain*/
		if (t->perm & READ_PRM)
		{
			fprintf(outfp, "#%s can read role %s's tty\n", domain->name, t->rolename);
			fprintf(outfp, "allow %s %s:{ chr_file lnk_file } r_file_perms;\n",
				domain->name, role_to_termlabel(t->rolename));
		}
		if (t->perm & WRITE_PRM)
		{
			fprintf(outfp, "#%s can write role %s's tty\n", domain->name, t->rolename);
			fprintf(outfp, "allow %s %s:{ chr_file lnk_file} { write setattr append };\n",
				domain->name, role_to_termlabel(t->rolename));
		}

		if (t->perm & CHANGE_PRM)
		{
			fprintf(outfp, "#%s can change %s's tty label\n", domain->name, t->rolename);
			fprintf(outfp, "allow %s %s:{ chr_file lnk_file} { getattr relabelfrom relabelto };\n",
				domain->name, role_to_termlabel(t->rolename));
		}
	}
}

/**
 *  @name:	out_tty_acl_global
 *  @about:	handle allowtty global
 *  @args:	outfp (FILE *) -> output file pointer
 *  @args:	domain (DOMAIN *) -> domain data
 *  @args:	perm (int) -> permission
 *  @return:	none
 */
static void
out_tty_acl_global(FILE *outfp, DOMAIN *domain, int perm)
{
	HASH_NODE **n;
	int num;

	n = create_hash_array(rbac_hash_table);
	num = rbac_hash_table->element_num;

	if (perm & READ_PRM)
	{
		fprintf(outfp, "#%s can read all named ttys\n", domain->name);
		fprintf(outfp, "allow %s ttyfile:{ chr_file lnk_file } r_file_perms;\n", domain->name);
	}

	if (perm & WRITE_PRM)
	{
		fprintf(outfp, "#%s can write all named ttys\n", domain->name);
		fprintf(outfp, "allow %s ttyfile:{ chr_file lnk_file } { write setattr append };\n", domain->name);
	}

	if (perm & CHANGE_PRM)
	{
		fprintf(outfp, "#%s can relabel all named ttys\n", domain->name);
		fprintf(outfp, "allow %s ttyfile:{ chr_file lnk_file }{ getattr relabelfrom relabelto };\n",
			domain->name);
	}

	free(n);
}

/**
 *  allowpts
 */

/**
 *  @name:	get_prefix
 *  @about:	chop "_t" or "_r" and return it(by malloc)
 *  @args:	p (char *) -> path name
 *  @return:	prefix
 */
char *
get_prefix(char *p)
{
	int len;
	char *result;
	len = strlen(p);

	if (len < 3)
	{
		fprintf(stderr, "invalid domain %s\n", p);
		exit(1);
	}

	result = (char *)malloc(sizeof(char) * (len+1));
	strcpy(result, p);

	result[len-2] = '\0';

	return result;
}

/**
 *  @name:	out_pts_acl_one_global
 *  @about:	allowpts global
 *  @args:	outfp (FILE *) -> output file descripter
 *  @args:	p (PTS_RULE) -> pts rule data
 *  @return:	none
 */
static void
out_pts_acl_one_global(FILE *outfp, PTS_RULE p)
{
	HASH_NODE **n;
	int num;

	n = create_hash_array(rbac_hash_table);
	num = rbac_hash_table->element_num;

	if (p.perm & READ_PRM)
	{
		fprintf(outfp, "#%s can read all named pts\n", p.domain->name);
		fprintf(outfp, "allow %s ptyfile:{ chr_file lnk_file } r_file_perms;\n", p.domain->name);
		fprintf(outfp, "allow %s ptyfile:dir r_dir_perms;\n", p.domain->name);
	}

	if (p.perm & WRITE_PRM)
	{
		fprintf(outfp, "#%s can write all named pts\n", p.domain->name);
		fprintf(outfp, "allow %s ptyfile:{ chr_file lnk_file } { write setattr append };\n",
			p.domain->name);
		fprintf(outfp, "allow %s ptyfile:dir { write setattr append create };\n", p.domain->name);
	}

	if (p.perm & CHANGE_PRM)
	{
		fprintf(outfp, "#%s can relabel all named pts\n", p.domain->name);
		fprintf(outfp, "allow %s ptyfile:{ chr_file lnk_file }{ getattr relabelfrom relabelto };\n",
			p.domain->name);
		fprintf(outfp, "allow %s ptyfile:dir { getattr relabelfrom relabelto };\n", p.domain->name);
	}

	return;
}

/**
 *  @name:	out_pts_acl_one
 *  @about:	output pts acl function
 *  @args:	outfp (FILE *) -> output file descripter
 *  @args:	p (PTS_RULE) -> pts rule data
 *  @return:	none
 */
static void
out_pts_acl_one(FILE *outfp, PTS_RULE p)
{
	char *prefix1;
	char *prefix2 = NULL;

	if (strcmp(p.domain_name, "global") == 0)
	{
		out_pts_acl_one_global(outfp, p);
		return;
	}

	/* allowpts <domain or role> r,w */
	/* general                       */
	if (strcmp(p.domain_name, "general") == 0)
	{
		if (p.perm & READ_PRM)
		{
			fprintf(outfp, "#%s can read general pts\n", p.domain->name);
			fprintf(outfp, "allow %s { devpts_t ptmx_t }:{ chr_file lnk_file }r_file_perms;\n",
				p.domain->name);
			fprintf(outfp, "allow %s devpts_t:dir r_dir_perms;\n", p.domain->name);
		}

		if (p.perm & WRITE_PRM)
		{
			fprintf(outfp, "#%s can write general pts\n", p.domain->name);
			fprintf(outfp, "allow %s { devpts_t ptmx_t }:{ chr_file lnk_file }{ write setattr append };\n",
				p.domain->name);
			fprintf(outfp, "allow %s devpts_t:dir { write setattr append create };\n", p.domain->name);
		}

		if (p.perm & CHANGE_PRM)
		{
			fprintf(outfp, "\n#%s can change general pts's label\n", p.domain->name);
			fprintf(outfp, "allow %s devpts_t:{ chr_file lnk_file } { getattr relabelfrom relabelto };\n",
				p.domain->name);
		}

	}
	else
	{							/*normal domain*/
		prefix1 = get_prefix(p.domain_name);
		if (p.domain_name != NULL)
		{
			prefix2 = get_prefix(p.domain_name);
		}

		if (p.perm & READ_PRM)
		{
			fprintf(outfp, "#%s can read  %s's pts\n", p.domain->name, p.domain_name);
			fprintf(outfp, "allow %s %s_devpts_t:{ chr_file lnk_file } r_file_perms;\n",
				p.domain->name, prefix1);
		}

		if (p.perm & WRITE_PRM)
		{
			fprintf(outfp, "#%s can write  %s's pts\n", p.domain->name, p.domain_name);
			fprintf(outfp, "allow %s %s_devpts_t:{ chr_file lnk_file } {write setattr append};\n",
				p.domain->name, prefix1);
		}

		if (p.perm & CHANGE_PRM)
		{
			fprintf(outfp, "\n#%s can change pts label derived from %s\n",
				p.domain->name, p.domain_name);
			fprintf(outfp, "allow %s %s_devpts_t:{ chr_file lnk_file } {getattr relabelfrom relabelto};\n",
				p.domain->name, prefix2);
		}

		free(prefix1);
		if (prefix2 != NULL)
		{
			free(prefix2);
		}
	}

	return;
}

/**
 *  @name:	out_pts_acl
 *  @about:	output pts acl
 *  @args:	outfp (FILE *) -> output file descripter
 *  @args:	domain (DOMAIN *) -> domain data
 *  @return:	none
 */
static void
out_pts_acl(FILE *outfp, DOMAIN *domain)
{
	int i;
	char *prefix;
	char *prefix2=NULL;
	HASH_NODE **domain_array;
	DOMAIN *d;

	domain_array = create_hash_array(domain_hash_table);

	prefix = get_prefix(domain->name);
	/* if(domain->pts_change_name!=NULL){
	   prefix2=get_prefix(domain->pts_change_name);
	   } */
	fprintf(outfp, "\n##PTS configurations \n");

	/* allowpts -create */
	if (domain->pts_create_flag == 1)
	{
		fprintf(outfp,"\n#can create its own pseudo terminal\n");
		fprintf(outfp,"can_create_pty(%s)\n",prefix);

		/*to tell rlogind or ssh which domain to change.*/
		for (i = 0; i < domain_hash_table->element_num; i++)
		{
			d = (DOMAIN *)domain_array[i]->data;
			if (d->pts_create_flag == 1)
			{
				prefix2 = get_prefix(d->name);
				fprintf(outfp, "type_change %s_t %s_devpts_t:chr_file %s_devpts_t;\n", prefix, prefix2, prefix);
				free(prefix2);
			}
		}
	}

	/* allowpts r,w,-change */
	for (i = 0; i < domain->pts_acl_array_num; i++)
	{
		out_pts_acl_one(outfp, domain->pts_acl_array[i]);
	}

	free(prefix);
	free(domain_array);
}

/**
 *  allowproc
 */

/**
 *  @name:	out_proc_acl_one
 *  @about:	print "allow" from "allowproc"
 *  @args:	outfp (FILE *) -> output file descripter
 *  @args:	domain (DOMAIN *) -> domain data
 *  @args:	proc (PROC_RULE *) -> proc rule
 *  @return:	none
 */
static void
out_proc_acl_one(FILE *outfp, DOMAIN *domain, PROC_RULE *proc)
{
	//PROC_RULE tmp;
	char *name;

	name = domain->name;

	switch (proc->type)
	{
	case PROC_SELF:
		if (proc->perm & READ_PRM)
		{
			fprintf(outfp, "##can read self /proc/PID\n");
			fprintf(outfp, "allow %s self:dir r_dir_perms;\n", name);
			fprintf(outfp, "allow %s self:notdevfile_class_set r_file_perms;\n", name);
		}
		if (proc->perm & WRITE_PRM)
		{
			fprintf(outfp, "##can write self /proc/PID\n");
			fprintf(outfp, "allow %s self:dir {setattr write append create};\n", name);
			fprintf(outfp, "allow %s self:notdevfile_class_set {setattr write append create};\n", name);
		}
		break;

	case PROC_OTHER:
		if (proc->perm & READ_PRM)
		{
			fprintf(outfp, "##can read all /proc/PID\n");
			fprintf(outfp, "allow %s global:dir r_dir_perms;\n", name);
			fprintf(outfp, "allow %s global:notdevfile_class_set r_file_perms;", name);
		}
		if (proc->perm & WRITE_PRM)
		{
			fprintf(outfp, "##can write self /proc/PID\n");
			fprintf(outfp, "allow %s global:dir {setattr write append create};\n", name);
			fprintf(outfp, "allow %s global:notdevfile_class_set {setattr write append create};\n", name);
		}
		break;

	case PROC_SYSTEM:
		if (proc->perm & READ_PRM)
		{
			fprintf(outfp, "#can read /proc assosiated with system\n");
			fprintf(outfp, "general_proc_read_access(%s)\n", name);
		}
		if (proc->perm & WRITE_PRM)
		{
			fprintf(outfp, "#can write /proc assosiated with system\n");
			fprintf(outfp, "can_sysctl(%s)\n", name);
		}
		break;

	case PROC_KMSG:
		if (proc->perm & READ_PRM)
		{
			fprintf(outfp, "#can read /proc/kmsg assosiated with system\n");
			fprintf(outfp, "allow %s proc_kmsg_t:file r_file_perms;\n", name);
		}
		if (proc->perm & WRITE_PRM)
		{
			fprintf(outfp, "#can write /proc/kmsg assosiated with system\n");
			fprintf(outfp, "allow %s proc_kmsg_t:file {setattr write append};\n", name);
		}
		break;

	case PROC_PROC:
		if (proc->perm & READ_PRM)
		{
			fprintf(outfp, "#can read proc_t\n");
			fprintf(outfp, "allow %s proc_t:dir r_dir_perms;\n", name);
			fprintf(outfp, "allow %s proc_t:notdevfile_class_set r_file_perms;\n", name);
		}
		if (proc->perm & WRITE_PRM)
		{
			fprintf(outfp, "#can write proc_t\n");
			fprintf(outfp, "allow %s proc_t:{file lnk_file} {setattr write append};\n", name);
		}
		break;

	default:
		fprintf(stderr, "unexpected error\n");
		exit(1);
		break;
	}

	return ;
}

static int
out_proc_acl(FILE *outfp, DOMAIN *d)
{
	int i;

	fprintf(outfp, "\n##/proc configuration\n");
	for (i = 0; i < d->proc_acl_array_num; i++)
	{
		out_proc_acl_one(outfp, d, &(d->proc_acl_array[i]));
	}

	return 0;
}

static int
out_tmpfs_create(FILE *outfp, DOMAIN *d)
{
	char *prefix;

	if (d->tmpfs_create_flag == 1)
	{
		prefix = get_prefix(d->name);

		fprintf(outfp, "\n##TMPFS create\n");
		fprintf(outfp, "#%s can create its own file in tmpfs\n", d->name);
		fprintf(outfp, "type %s_tmpfs_t,file_type,tmpfsfile;\n", prefix);
		fprintf(outfp, "file_type_auto_trans(%s,tmpfs_t,%s_tmpfs_t)\n", d->name, prefix);
		fprintf(outfp, "allow %s_tmpfs_t tmpfs_t:filesystem associate;\n", prefix);

		free(prefix);
	}

	return 0;
}



/*
 * if s in included in default domains,
 * return 1 others 0
 */
static int
chk_default_type(char *s)
{
	int i;

	for (i = 0; i < sizeof(DEFAULT_DOMAIN)/32; i++)
	{
		if (strcmp(DEFAULT_DOMAIN[i], s) == 0)
			return 1;
	}

	return 0;
}

/**
 *  @name:	out_acls
 *  @about:	print all "allow"
 *  @args:	outfp (FILE *) -> output file descripter
 *  @args:	domain (DOMAIN*) -> domain data
 *  @return:	none
 */
static void
out_acls(FILE *outfp, DOMAIN *domain)
{
	out_file_acl(outfp, domain);
	out_net_acl(outfp, domain);
	out_com_acl(outfp, domain);
	out_adm_acl(outfp, domain);
	out_tty_acl(outfp, domain);
	out_pts_acl(outfp, domain);
	out_proc_acl(outfp, domain);
	out_tmpfs_create(outfp, domain);
}

/**
 *  @name:	out_allow
 *  @about:	print "allow"
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return:	none
 */
void
out_allow(FILE *outfp)
{
	HASH_NODE **domain_array;
	DOMAIN *domain;
	int i;

	domain_array = create_hash_array(domain_hash_table);

	domain = (DOMAIN *)search_element(domain_hash_table, "global");

	/********"allow" of global domain*****************/
	fprintf(outfp, "\n############################\n");
	fprintf(outfp, "#####domain attribute this is global\n");

	out_acls(outfp,domain);

	out_not_supported_allow(outfp);

	fprintf(outfp, "\n#############endof domain attribute\n\n\n");


	/****************"allow" of normal domain***************************/

	for (i = 0; i < domain_hash_table->element_num; i++)
	{
		domain = (DOMAIN *)domain_array[i]->data;

		/* Don't print dummy domain */
		if (strcmp(domain->name, "global") == 0 ||
		    strcmp(domain->name, DUMMY_DOMAIN_NAME) == 0)
			continue;

		fprintf(outfp, "###########################\n");
		fprintf(outfp, "############## %s domain\n", domain->name);

		/* declaration of domain */
		if (chk_default_type(domain->name) == 0)
		{
			fprintf(outfp, "type %s,global;\n", domain->name);
			fprintf(outfp, "role system_r types %s;\n", domain->name);
		}

		/* print acl */
		out_acls(outfp, domain);

		fprintf(outfp, "\n##### endof %s domain\n\n\n", domain->name);

	}
	free(domain_array);
}

/**
 *  @name:	free_all_objets
 *  @about:	free all (domain file-label domain trans) objects
 *  @args:	none
 *  @return:	none
 */
void
free_all_objects()
{
	free_rbac_hash_table();
	free_user_hash_table();
	free_domain_tab();
	delete_file_label_tab();
	free_domain_trans();
}

/**
 *  @name:	add_global_acl
 *  @about:	Add global domain's ACL to all domains,
 * 		and delete the ACL from global domain's ACL.
 *  @args:	a (void *) -> acl data
 *  @return:	return 0 on success
 */
static int
add_global_acl(void *a)
{
	FILE_ACL_RULE *acl_global;
	FILE_ACL_RULE *acl_tmp;
	FILE_ACL_RULE *acl_del;
	FILE_ACL_RULE *acl_new;
	HASH_NODE **domain_array;
	DOMAIN *domain;
	char *path_global;
	int allowed_global;
	int only_global;
	int i;

	acl_global = a;
	path_global = strdup(acl_global->path);
	allowed_global = acl_global->allowed;
	only_global = acl_global->only_flag;

	domain_array = create_hash_array(domain_hash_table);
	for (i = 0; i < domain_hash_table->element_num; i++)
	{
		domain = (DOMAIN *)(domain_array[i]->data);

		acl_tmp = search_element(domain->file_acl, path_global);
		if (acl_tmp != NULL)
		{				/*don't overwrite*/
			continue;
		}

		/* add new element to domain's File ACL */
		if ((acl_new = (FILE_ACL_RULE *)malloc(sizeof(FILE_ACL_RULE))) == NULL)
		{
			perror("malloc");
			exit(1);
		}
		memset(acl_new, 0, sizeof(FILE_ACL_RULE));
		acl_new->path = strdup(path_global);
		acl_new->allowed = allowed_global;
		acl_new->domain = domain;
		acl_new->only_flag = only_global;

		if (domain->file_acl == NULL)
		{
			if ((domain->file_acl = create_hash_table(FILE_ACL_TABLE_SIZE)) == NULL)
			{
				perror("malloc");
				exit(1);
			}
		}

		insert_element(domain->file_acl, acl_new, acl_new->path);
	}

	/* delete File ACL from global domain */
	domain = (DOMAIN *)search_element(domain_hash_table, "global");
	acl_del = (FILE_ACL_RULE *)search_element(domain->file_acl, path_global);
	free(acl_del->path);
	delete_element(domain->file_acl, path_global);
	free(path_global);

	/*
	  acl_tmp = (FILE_ACL_RULE *)search_element(domain->file_acl, path_global);
	  acl_tmp->allowed = DENY_PRM;
	  acl_tmp->only_flag = 0;
	*/

	return 0;
}

/**
 *  @name:	check_acl
 *  @about:	convert access rights "global>domain" to access rights "global<domain"
 *		this converts one ACL
 *  @args:	a (void *) -> acl data
 *  @return:	return 0 on success
 */
static int
check_acl(void *a)
{
	FILE_ACL_RULE *acl_global;
	FILE_ACL_RULE *acl_tmp;
	FILE_ACL_RULE *acl_new;
	FILE_ACL_RULE *acl_del;
	HASH_NODE **domain_array;
	DOMAIN *domain;
	char *path_global;
	int allowed_global;
	int only_global;
	int i, tmp;
	int exist=0;

	acl_global = a;
	path_global = strdup(acl_global->path);
	allowed_global = acl_global->allowed;
	only_global = acl_global->only_flag;

	/**
	 *  search the domain which has less access rights than global,
	 *  and set NULL in  relevant ACL.
	 */
	domain_array = create_hash_array(domain_hash_table);
	for (i = 0; i < domain_hash_table->element_num; i++)
	{
		domain = (DOMAIN *)domain_array[i]->data;

		acl_tmp = search_element(domain->file_acl, path_global);
		if (acl_tmp == NULL)
			continue;

		if (acl_tmp->allowed == allowed_global)
			continue;

		tmp = (acl_tmp->allowed)&allowed_global;

		if (tmp < allowed_global)
		{
			domain_array[i] = NULL;
			exist = 1;
		}
	}

	/*If the domain - which has less access rights than global - exists, convert access rights.*/
	if (exist == 1)
	{
		for (i = 0; i < domain_hash_table->element_num; i++)
		{
			if (domain_array[i] == NULL)
				continue;

			domain = (DOMAIN *)(domain_array[i]->data);

			/*add new element*/
			if ((acl_new = (FILE_ACL_RULE *)malloc(sizeof(FILE_ACL_RULE))) == NULL)
			{
				perror("malloc");
				exit(1);
			}
			memset(acl_new, 0, sizeof(FILE_ACL_RULE));

			acl_new->path = strdup(path_global);
			acl_new->allowed = allowed_global;
			acl_new->domain = domain;
			acl_new->only_flag = only_global;

			if (domain->file_acl == NULL)
			{
				if ((domain->file_acl = create_hash_table(FILE_ACL_TABLE_SIZE)) == NULL)
				{
					perror("malloc");
					exit(1);
				}
			}

			insert_element(domain->file_acl, acl_new,acl_new->path);
#ifdef DEBUG
//			debug_print(__FILE__, __LINE__, "domain:%s,path:%s:only:%d", domain->name, acl_new->path, acl_new->only_flag);
#endif
		}

		/*delete acl from global*/
		domain = (DOMAIN *)search_element(domain_hash_table, "global");
		acl_del = (FILE_ACL_RULE *)search_element(domain->file_acl, path_global);
		free(acl_del->path);
		delete_element(domain->file_acl, path_global);
		free(path_global);
		/* acl_tmp = (FILE_ACL_RULE *)search_element(domain->file_acl, path_global);
		   acl_tmp->allowed = DENY_PRM;
		   acl_tmp->only_flag = 0;
		*/
	}

	free(domain_array);

	return 0;
}

/**
 *  @name:	convert_file_acl
 *  @about:	find "access right to some file in global" > "access right to some file 
 *   		in normal domain" and convert such access rights to "access right to some file
 *		in global"<"access right to some file in normal domain"
 *  @args:	none
 *  @return:	none
 */
void
convert_file_acl()
{
	DOMAIN *global_domain;

	/* search "global" domain data */
	global_domain = search_element(domain_hash_table, "global");

	/* convert access rights to be global<domain. */
	handle_all_element(global_domain->file_acl, check_acl);

	/* delete file acl from global,and add it to all domains */
	handle_all_element(global_domain->file_acl, add_global_acl);

}

void out_domain_trans_child_dir(FILE *, TRANS_RULE *, char *);

/**
 *  @name:	out_domain_trans
 *  @about:	print domain_trans by using rulebuf
 * 		attention:wildcard isn't supported,the program of the same name isn't supported.
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return:	none
 *  @notes:	rulebuf is global variable
 */
void
out_domain_trans(FILE *outfp)
{
	int i;
	TRANS_RULE t;
	FILE_LABEL *label;
	char *name;/*domain name*/
	int len;
	struct stat buf;

	/* print domain_auto_trans */
	for (i = 0; i < domain_trans_rule_num; i++)
	{
		/* search target path's file label */
		t = rulebuf[i];
		label = search_element(file_label_table, t.path);
		if (label == NULL)
		{
			fprintf(stderr, "bug line %d\n", __LINE__);
			exit(1);
		}

		/* convert "_r" to "_t" */
		name = strdup(t.parent);
		len = strlen(name);
		if (len < 2)
		{
			fprintf(stderr, "bug line %d\n", __LINE__);
		}
		name[len-1] = 't';

		if (t.auto_flag == 1)
		{
			fprintf(outfp, "domain_auto_trans(%s,%s,%s)\n", name,label->labelname, t.child);
		}
		else
		{
			fprintf(outfp, "domain_trans(%s,%s,%s)\n", name, label->labelname, t.child);
		}

		stat(t.path, &buf);
		if (buf.st_mode & S_IFDIR)
		{
			/*
			 * when entry point is directory and a file under the directory is labeled,
			 * then output domain_auto_trans for label of the file.
			 */
			out_domain_trans_child_dir(outfp, &t,name);
		}
		free(name);
	}
}

/**
 *  @name:	out_domain_trans_child_dir
 *  @about:	print domain_trans by using rulebuf
 * 		attention:wildcard isn't supported,the program of the same name isn't supported.
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return:	none
 *  @notes:	rulebuf is global variable
 */
void
out_domain_trans_child_dir(FILE *outfp, TRANS_RULE *t, char *domain_name)
{
	FILE_LABEL *label;
	HASH_NODE **file_label_array;
	int i;

	file_label_array = create_hash_array(file_label_table);

	for (i = 0; i < file_label_table->element_num; i++)
	{
		label = (FILE_LABEL *)file_label_array[i]->data;

		if (chk_child_file(t->path, label->filename) == 1)
		{
			fprintf(outfp, "domain_trans(%s,%s,%s)\n", domain_name, label->labelname, t->child);
		}
	}

	free(file_label_array);
}

/**
 *  Global variable
 */
static FILE *rbac_out;

/**
 *  @name:	out_one_user_role
 *  @about:	print "user" based on USER_ROLE structure
 *  @args:	u (void *) -> user data
 *  @return:	none
 */
static int
out_one_user_role(void *u)
{
	USER_ROLE *ur;
	int i;
	int max;
	char **str_a;

	ur = (USER_ROLE*)u;
	max = ur->role_name_array_num;
	str_a = ur->role_name_array;

	fprintf(rbac_out, "user %s roles {", ur->username);

	for (i = 0; i < max; i++)
	{
		fprintf(rbac_out, " %s ", str_a[i]);
	}

	fprintf(rbac_out,"};\n");

	return 0;
}

/**
 *  @name:	out_role_allow
 *  @about:	print role allow
 *  @args:	u (void *) -> user data
 *  @return:	none
 */
static int
out_role_allow(void *u)
{
	RBAC *rbac;
	HASH_NODE **rbac_array;
	RBAC *element;
	char *rname;
	int num;
	int i;

	rbac = u;
	rbac_array = create_hash_array(rbac_hash_table);
	num = rbac_hash_table->element_num;

	for (i = 0; i < num; i++)
	{
		element = (RBAC *)rbac_array[i]->data;
		rname = element->rolename;
		fprintf(rbac_out, "allow %s %s;\n", rbac->rolename, rname);
	}

	free(rbac_array);

	return 0;
}

/**
 *  @name:	out_one_role
 *  @about:	called by handle_all_element
 *		print "type" based on USER_ROLE structure
 *  @args:	u (void *) -> user data
 *  @return:	none
 */
static int
out_one_role(void *u)
{
	RBAC *rbac;
	char *tmp;
	int len;
	HASH_NODE **rbac_array;
	RBAC *element;
	char *rname;
	char *dname;
	int num;
	int i;

	rbac = u;

	rbac_array = create_hash_array(rbac_hash_table);
	num = rbac_hash_table->element_num;

	tmp = strdup(rbac->rolename);

	/* this prohibit login with unrelated type */
	fprintf(rbac_out, "role %s types ~{", rbac->rolename);
	for (i = 0; i <num; i++)
	{
		element = (RBAC *)rbac_array[i]->data;
		rname = element->rolename;
		dname = element->default_domain->name;

		if (strcmp(rbac->rolename, rname) == 0)
		{
			continue;
		}
		fprintf(rbac_out," %s ", dname);
	}
	fprintf(rbac_out,"};\n");

	free(rbac_array);

	/* allow role transition */
	fprintf(rbac_out, "allow system_r %s;\n", rbac->rolename);
	fprintf(rbac_out, "allow %s system_r;\n", rbac->rolename);

	/* chop "_r" */
	len = strlen(tmp);
	if ( len < 2)
	{
		fprintf(stderr, "Invalid rolename %s\n", tmp);
		exit(1);
	}
	tmp[len-2] = '\0';
	//  fprintf(rbac_out, "user_tty_domain(%s)\n", tmp);

	free(tmp);

	return 0;
}

/**
 *  @name:	out_rbac
 *  @about:	print rbac rule
 *  @args:	outfp (FILE *) -> output file descripter
 *  @return:	none
 */
static void
out_rbac(FILE *outfp)
{
	rbac_out = outfp;

	if (rbac_hash_table == NULL)
		return;

	fprintf(rbac_out, "\n#RBAC related configration\n");

	/* print "role" */
	handle_all_element(rbac_hash_table, out_one_role);

	/* print allow <role> <role>; */
	handle_all_element(rbac_hash_table, out_role_allow);

	/* print "user" */
	handle_all_element(user_hash_table, out_one_user_role);

	fprintf(rbac_out, "user system_u roles system_r;\n");
}
