/*
 * All Rights Reserved, Copyright (C) 2003, Hitachi Software Engineering Co., Ltd.
 */
/* $Id: action.c,v 1.2 2003/01/24 00:16:56 ynakam Exp $ */

/*the action of yacc*/
/*build DOMAIN structure(domain_hash_table)*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>

#include <stdarg.h>
#include "global.h"
#include "action.h"
#include "common.h"
#include "file_label.h"
#include "hashtab.h"
#include "debug.h"

/**
 *  Buffer for error message
 */
static char errmsg[1024];

/**
 *  the name of domain currently handled
 */
static char *current_domain = NULL;

/**
 *  Dummy domain used to enforce file labeing.
 */
static DOMAIN dummy_domain;

HASH_TABLE *exc_label_table = NULL;

char *make_role_to_domain(char *name);

/**
 *  @name:	init_dummy_domain
 *  @about:	initialize dummy domain
 *  @args:	none
 *  @return:	none
 */
static void
init_dummy_domain()
{
	static int init = 0;

	if (init == 0)
	{
		memset(&dummy_domain, 0, sizeof(DOMAIN));
		dummy_domain.name = strdup(DUMMY_DOMAIN_NAME);
		init=1;
	}
}

/**
 *  @name:	action_error
 *  @about:	yyerror like printf
 *  @args:	fmt (char *) -> format
 *  @return:	none
 */
void
action_error(char *fmt, ...)
{
	va_list ap;
	char buf[1024];

	va_start(ap, fmt);

	vsnprintf(buf, sizeof(buf), fmt, ap);
	va_end(ap);

	yyerror(buf);
}

/**
 *  @name:	check_domain_name
 *  @about:	chech whether domain name is valid.
 *  @args:	name (char *) -> domain name
 *  @return: 	return 1 if domain name is valid.	
 */
int
check_domain_name(char *name)
{
	int len;
	char *tmp;
	len = strlen(name);

	tmp = strdup(name);

	/**
         *  domain format have to be formed xxxxx_t
	 */
	if (len < 2 || !(tmp[len-2] == '_' && tmp[len-1] == 't'))
	{
		/**
		 *  exclude "global" domain
		 */
		if (strcmp(name, "global") == 0)
		{
			return 1;
		}

		action_error("Invalid domain name:%s\n", name);
		exit(1);
	}
	free(tmp);
	return 1;
}

/**
 *  @name:	register_domain
 *  @about:	register domain name
 *  @args:	dname (char *) -> domain name
 *  @args:      role_flags (int) -> If role_flag=1 "name" is a domain which belong to a role.
 *  @return:	none
 */ 
void
registar_domain(char *dname, int role_flag)
{
	DOMAIN *d;
	char *name = dname;
	char *tmp = NULL;

	check_domain_name(name);

	if (register_label_table(name) == -2)
	{
		tmp = resolve_label_conflict(name);
		fprintf(stderr,
			"Warning! Domain %s conflicts label reserved by system. Domain name %s is used instead.\n",
			name, tmp);
		name = tmp;
	}

	if (insert_domain_hash_by_name(name) == -2)
	{
		action_error("multiple definition of domain %s\n", name);
		exit(1);
	}

	if (current_domain != NULL)
	{
		free(current_domain);
	}
	current_domain = strdup(name);

	if (current_domain == NULL)
	{
		action_error(errmsg, sizeof(errmsg), "Too long domain name:%s", name);
		exit(1);
	}

	if (role_flag == 1)
	{
		d = search_domain_hash(name);
		d->roleflag = 1;
		insert_element(domain_hash_table, d, name);
	}

	if (tmp != NULL)
	{
		free(tmp);
	}

	return;
}

/**
 *  Variable used to analyze permission
 */
static int tmp_perm = DENY_PRM;

/**
 *  @name:	str_to_perm
 *  @about:	convert permission string to number
 *  @args:	s (char *) -> permission string
 *  @return:	permission number
 */
int str_to_perm(char *s){

	if (strcmp(s, READ_STR) == 0)
	{
		return READ_PRM;
	}
	else if (strcmp(s, WRITE_STR) == 0)
	{
		return WRITE_PRM;
	}
	else if (strcmp(s, EXECUTE_STR) == 0)
	{
		return EXECUTE_PRM;
	}
	else if (strcmp(s, APPEND_STR) == 0)
	{
		return APPEND_PRM;
	}
	else if (strcmp(s, SEARCH_STR) == 0)
	{
		return SEARCH_PRM;
	}
	return -1;
}

/**
 *  @name:	perm_to_str
 *  @about:	convert permission number to string
 *  @args:	allowed (int) -> permission number
 *  @return:	return string as static buffer
 */
char *
perm_to_str(int allowed)
{
	static char buf[256];
	char str[] = {'-', '-', '-', '-', '\0'};

	if (allowed & READ_PRM)
	{
		str[0] = 'r';
	}

	if (allowed & WRITE_PRM)
	{
		str[1] = 'w';
	}

	if (allowed & EXECUTE_PRM)
	{
		str[2] = 'x';
	}

	if (allowed & SEARCH_PRM)
	{
		str[3]='s';
	}

	strcpy(buf, str);

	return buf;
}

/**
 *  @name:	str_to_rw_perm
 *  @about:	convert rw permission string to rw permission number
 *  @args:	s (char *) -> permission string
 *  @return:	return permisstion number on success, return -1 in failure.
 */
int
str_to_rw_perm(char *s)
{
	if (strcmp(s, READ_STR) == 0)
	{
		return READ_PRM;
	}
	else if (strcmp(s, WRITE_STR) == 0)
	{
		return WRITE_PRM;
	}
	return -1;
}

/**
 *  @name:	str_to_sig_perm
 *  @about:	convert sig permission string to sig permission number
 *  @args:	s (char *) -> permission string
 *  @return:	return permisstion number on success, return -1 in failure.
 */
int
str_to_sig_perm(char *s)
{
	if (strcmp(s, CHID_STR) == 0)
	{
		return CHID_PRM;
	}
	else if (strcmp(s, KILL_STR) == 0)
	{
		return KILL_PRM;
	}
	else if (strcmp(s, STOP_STR) == 0)
	{
		return STOP_PRM;
	}
	else if (strcmp(s, OTHERSIG_STR) == 0)
	{
		return OTHERSIG_PRM;
	}

	return -1;
}

/**
 *  @name:	str_to_adm_perm
 *  @about:	convert admin permission string to admin permission number
 *  @args:	s (char *) -> permission string
 *  @args:	tmp (ADMIN_RULE *) -> output buffer
 *  @return:	return permisstion number on success, return -1 in failure.
 */
int
str_to_adm_perm(char *s, ADMIN_RULE *tmp)
{

	if (strcmp(s, RELABEL_STR) == 0)
	{
		(*tmp).relabel = 1;
	}
	else if (strcmp(s, GETSECURITY_STR) == 0)
	{
		(*tmp).getsecurity = 1;
	}
	else if (strcmp(s, SETENFORCE_STR) == 0)
	{
		(*tmp).setenforce = 1;
	}
	else if (strcmp(s, LOAD_POLICY_STR) == 0)
	{
		(*tmp).load_policy = 1;
	}
	else if (strcmp(s, NET_STR) == 0)
	{
		(*tmp).net = 1;
	}
	else if (strcmp(s, BOOT_STR) == 0)
	{
		(*tmp).boot = 1;
	}
	else if (strcmp(s, INSMOD_STR) == 0)
	{
		(*tmp).insmod = 1;
	}
	else if (strcmp(s, QUOTAON_STR) == 0)
	{
		(*tmp).quotaon = 1;
	}
	else if (strcmp(s, SWAPON_STR) == 0)
	{
		(*tmp).swapon = 1;
	}
	else if (strcmp(s, MOUNT_STR) == 0)
	{
		(*tmp).mount = 1;
	}
	else if (strcmp(s, RAW_IO_STR) == 0)
	{
		(*tmp).raw_io = 1;
	}
	else if (strcmp(s, PTRACE_STR) == 0)
	{
		(*tmp).ptrace = 1;
	}
	else if (strcmp(s, CHROOT_STR) == 0)
	{
		(*tmp).chroot = 1;
	}
	else if (strcmp(s, SEARCH_ALL_STR) == 0)
	{
		(*tmp).search = 1;
	}
	else if (strcmp(s, UNLABEL_STR) == 0)
	{
		(*tmp).unlabel = 1;
	}
	else if (strcmp(s, READ_ALL_STR) == 0)
	{
		(*tmp).read = 1;
	}
	else if (strcmp(s, WRITE_ALL_STR) == 0)
	{
		(*tmp).write = 1;
	}
	else if (strcmp(s, PART_RELABEL_STR) == 0)
	{
		(*tmp).part_relabel = 1;
	}
	else
	{
		return -1;
	}

	return 0;
}

/**
 *   ADMIN_RULE tempolary buffer
 */
static ADMIN_RULE tmp_adm_perm;

/**
 *  @name:	add_adm_permisstion
 *  @about:	add admin permittion into the rule buffer
 *  @args:	s (char *) -> admin permisstion string
 *  @args:	flag (int) -> 
 *  @args:	init (int) -> whether ADMIN_RULE buffer is initialized..
 *  @return:	return 0 on success.
 */
int
add_adm_permission(char *s, int flag, int init)
{
	int p;

	/* initialize temp buffer */
	if (init == 1)
	{
		memset(&tmp_adm_perm, 0, sizeof(ADMIN_RULE));
	}

	/* change admin permission from string to id */
	p = str_to_adm_perm(s, &tmp_adm_perm);	/*update tmp_adm_perm*/

	if (p < 0)
	{
		action_error(errmsg, sizeof(errmsg), "no such permission %s\n", s);
		exit(1);
	}

	return 0;
}

/**
 *  @name:	add_permisstion
 *  @about:	add permittion 
 *  @args:	s (char *) -> admin permisstion string
 *  @args:	flag (int) -> 
 *  @args:	init (int) -> whether ADMIN_RULE buffer is initialized..
 *  @return:	return 0 on success.
 */
int
add_permission(char *s, int flag, int init)
{
	int p = DENY_PRM;

	/* this means to start analyze permission */
	if (init == 1)
	{
		tmp_perm = DENY_PRM;
	}

	/* get permission number */
	if (flag == FILE_PERM)
	{
		p = str_to_perm(s);
	}
	if (flag == RW_PERM)
	{
		p = str_to_rw_perm(s);
	}
	if (flag == SIG_PERM)
	{
		p = str_to_sig_perm(s);
	}
	if (flag == ADM_PERM)
	{
		//    p=str_to_adm_perm(s);
		debug_print(__FILE__, __LINE__, "bug!!!");
		exit(1);
	}

	/* error check */
	if (p < 0)
	{
		action_error(errmsg, sizeof(errmsg), "no such permission %s\n", s);
		exit(1);
	}

	/* Add permission */
	tmp_perm |= p;

	return 0;
}

/**
 *  @name:	strip_slash
 *  @about:	Remove "/" at the end of "s".
 *              But when s is "/",this function does nothing.
 *  @args:	s (char *) -> target data
 *  @return:	none
 */
static void
strip_slash(char *s)
{
	int len;

	len = strlen(s);
	if (s[len-1] == '/' && len > 1)
	{
		s[len-1] = '\0';
	}
}

/**
 *  @name:	add_fileacl_to_domain	
 *  @about:	Add File ACL to DOMAIN struct related to "domain_name",
 *		by using path,perm. 
 *  @args:	domain_name (char *) -> string for domain name
 *  @args:	path (char *) -> full path name 
 *  @args:	perm (int) -> permission
 *  @args:	only_flag (int) -> flag
 *  @return:	return 0 on success.
 */
static int
add_fileacl_to_domain(char *domain_name, char *path, int perm, int only_flag)
{
	DOMAIN *domain;
	HASH_TABLE *acl;/*this hash table stores File ACL of domain*/
	FILE_ACL_RULE *element;
	int s;

	/* striping shash from specifyed path name */
	strip_slash(path);

	/* registar new acl with domain's file_acl */
	domain = search_domain_hash(domain_name);
	if (domain == NULL)
	{
		action_error("domain %s isn't defined\n", domain_name);
		exit(1);
	}

	/* create hash table for acl */
	acl = domain->file_acl;
	if (acl == NULL)
	{
		acl = create_hash_table(FILE_ACL_TABLE_SIZE);
		if (acl == NULL)
		{
			yyerror("memory shortage\n");
			exit(1);
		}
		domain->file_acl = acl;
	}

	/* create acl rule buffer */
	if ((element = (FILE_ACL_RULE *)malloc(sizeof(FILE_ACL_RULE))) == NULL)
	{
		yyerror("memory shortage\n");
		exit(1);
	}

	memset(element, 0, sizeof(FILE_ACL_RULE));

	/* store file rule acl */
	element->domain = domain;
	element->path = strdup(path);
	element->allowed = perm;
	element->only_flag = only_flag;

	/* insert file rule acl into acl hash table */
	s = insert_element(acl, element, element->path);
	if (s == -2)
	{
		/**
		 *  if key:element->path already exists in hash,
		 *  add permission
		 */
		free(element->path);
		element = search_element(acl, path);
		if (element == NULL)
		{
			error_print(__FILE__, __LINE__, "bug!!!\n");
		}
		element->allowed |= perm;

	}
	else if (s < 0)
	{
		fprintf(stderr, "system error in line %d\n", __LINE__);
		exit(1);
	}

	return 0;
}

/**
 *  @name:	registar_deny	
 *  @about:	register Deny permission to the target path's acl
 *  @args:	path (char *) -> full path name 
 *  @args:	only_flag (int) -> flag
 *  @return:	return 0 on success.
 */
int
registar_deny(char *path, int only_flag)
{
	tmp_perm = DENY_PRM;
	registar_file_acl(path, only_flag);
	return 0;
}

/**
 *  @name:	label_child_dir	
 *  @about:	To label child directory of "path",
 *              this function registars ACL in dummy domain.
 *  @args:	path (char *) -> full path name 
 *  @return:	none
 */
void
label_child_dir(char *path)
{
	DIR *dp;
	struct dirent *p;
	struct stat buf;
	char fullname[PATH_MAX];

	strip_slash(path);

	if ((dp = opendir(path)) == NULL)
	{
		fprintf(stderr, "Warning!! dir open err %s\n", path);
		return ;
	}

	while ((p = readdir(dp)) != NULL)
	{
		if (strcmp(p->d_name, "..") == 0 ||
		    strcmp(p->d_name, ".") == 0)
		{
			continue;
		}

		if (strcmp(path, "/") == 0)
		{
			sprintf(fullname, "%s%s", path, p->d_name);
		}
		else
		{
			sprintf(fullname, "%s/%s", path, p->d_name);
		}

		stat(fullname, &buf);

		if (buf.st_mode & S_IFDIR)
		{
			add_fileacl_to_domain(DUMMY_DOMAIN_NAME, fullname, READ_PRM, 0);
		}
	}

	closedir(dp);
}

/**
 *  @name:	registar_file_acl
 *  @about:	This function adds File ACL of path to current_domain,
 *		and uses tmp_perm as permission calculated in parse_permission.
 *              this function registars ACL in dummy domain.
 *  @args:	path (char *) -> full path name 
 *  @args:	only_flag (int) -> flag
 *  @return:	return 0 on success.
 */
int
registar_file_acl(char *path, int only_flag)
{

	add_fileacl_to_domain(current_domain, path, tmp_perm, only_flag);
	//tmp_perm = P_DENY;

	if (only_flag == 1)
	{
		/* when allowonly is described, add dummy permission to dummy domain. */
		label_child_dir(path);
	}

	return 0;
}

/**
 *  @name:	registar_exec_file_acl
 *  @about:	registar exc_rule_array in DOMAIN structure.
 *  @args:	path (char *) -> full path name 
 *  @args:	e_name (char *) -> 
 *  @return:	return 0 on success.
 */
int
registar_exc_file_acl(char *path, char *e_name)
{
	struct stat buf;
	DOMAIN *domain;
	FILE_EXC_RULE work;
	FILE_EXC_RULE *tmp;
	char *tmpname = NULL;
	char *exc_name = e_name;
	int *v;

	/* exclusive grobal domain */
	if (strcmp(current_domain, "global") == 0)
	{
		yyerror("exclusive can't be used in global domain\n");
		exit(1);
	}

	strip_slash(path);

	/* if the file named "path" doesn't exist or isn't directory */
	if (stat(path, &buf) == -1 ||
	    !(buf.st_mode & S_IFDIR))
	{
		action_error("Filename %s must be directory\n", path);
		exit(1);
	}

	/* check label name conflict */
	if (register_label_table(exc_name) == -2)
	{
		if (search_element(exc_label_table, exc_name) == NULL)
		{
			tmpname = resolve_label_conflict(exc_name);
			fprintf(stderr,
				"Warning! Exclusive label %s conflicts label reserved by system. Domain name %s is used instead.\n",
				exc_name, tmpname);
			exc_name = tmpname;
		}
	}

	/* set exc_name in exc_label_table */
	if (exc_label_table == NULL)
	{
		exc_label_table = create_hash_table(LABEL_LIST_SIZE / 2);
	}
	v = (int *)malloc(sizeof(int));
	*v = 1;
	insert_element(exc_label_table, v, exc_name);


	/*
	 * register P_DENY with global domain's File ACL
	 */
	add_fileacl_to_domain("global", exc_name, DENY_PRM, 0);

	/*
	 * This solves bug which happens when "allow exclusive" is described
	 * to unlabeled directory. To enforce labeling ,this registers dummy acl
	 */
	add_fileacl_to_domain(dummy_domain.name, path, READ_PRM, 0);

	/* add to exc_rule_array */
	domain = search_domain_hash(current_domain);
	work.domain = domain;
	work.path = strdup(path);
	if (tmpname == NULL)
	{
		work.name = strdup(exc_name);
	}
	else
	{
		work.name=exc_name;
	}


	tmp = (FILE_EXC_RULE *)extend_array(domain->exc_rule_array,
					    &(domain->exc_rule_array_num),
					    sizeof(FILE_EXC_RULE));
	tmp[domain->exc_rule_array_num-1] = work;
	domain->exc_rule_array = tmp;

	return 0;
}

/**
 *  Information of domain is registerd.
 *  This is very important.
 */
#define DOMAIN_HASH_TABLE_SIZE 1024
HASH_TABLE *domain_hash_table = NULL; 

/**
 *  @name:	insert_domain_hash_by_name
 *  @about:	register new domain named "name" with hash table
 *  @args:	name (char *) -> domain name
 *  @return:	none
 */
int
insert_domain_hash_by_name(char *name)
{

	DOMAIN *newdomain;

	if (domain_hash_table == NULL)
	{
		domain_hash_table = create_hash_table(DOMAIN_HASH_TABLE_SIZE);
		if (domain_hash_table == NULL)
		{
			perror("malloc");
			exit(1);
			return -1;
		}
		/* dummy domain is created */
		init_dummy_domain();
		insert_element(domain_hash_table, &dummy_domain, dummy_domain.name);
	}

	if ((newdomain = (DOMAIN *)malloc(sizeof(DOMAIN))) == NULL)
	{
		perror("malloc");
		exit(1);
	}

	memset(newdomain, 0, sizeof(DOMAIN));

	newdomain->name = strdup(name);
	newdomain->roleflag = 0;
	newdomain->file_acl = NULL;
	newdomain->exc_rule_array = NULL;
	newdomain->net_acl = NULL;
	newdomain->com_acl_array = NULL;
	newdomain->tty_create_flag = 0;
	newdomain->tty_acl_array = NULL;
	newdomain->tty_acl_array_num = 0;
	newdomain->pts_create_flag = 0;
	//  newdomain->pts_change_name = NULL;
	newdomain->pts_acl_array = NULL;
	newdomain->pts_acl_array_num = 0;

	return insert_element(domain_hash_table, newdomain, newdomain->name);
}

/**
 *  @name:	search_domain_hash
 *  @about:	return pointer to domain named "key".
 *  @args:	key (char *) -> hash key
 *  @return:	domain data on success, return NULL in failure
 */
DOMAIN *
search_domain_hash(char *key)
{
	if (domain_hash_table == NULL)
	{
		fprintf(stderr, "domain hash table isn't initialized\n");
		exit(1);
	}
	return (DOMAIN *)search_element(domain_hash_table, key);
}

/**
 *  @name:	free_file_acl_node
 *  @about:	free node of file_acl hash table.
 *  @args:	node (void *) -> node data
 *  @return:	return 0
 */
static int
free_file_acl_node(void *node)
{
	FILE_ACL_RULE *n;

	n = node;
	free(n->path);

	return 0;
}

/**
 *  @name:	free_file_acl
 *  @about:	free file_acl element in DOMAIN-structure
 *  @args:	file_acl (HASH_TABLE *) -> hash table for file acl
 *  @return:	none
 */
static void
free_file_acl(HASH_TABLE *file_acl)
{
	handle_all_element(file_acl, free_file_acl_node);
	return ;
}

/**
 *  @name:	free_file_exc
 *  @about:	free exc_array in DOMAIN structure
 *  @args:	exc_array (FILE_EXC_RULE *) -> file sec rule
 *  @args:	array_num (int) -> a number of elements of array 
 */
static void
free_file_exc(FILE_EXC_RULE *exc_array, int array_num)
{
	int i;
	FILE_EXC_RULE tmp;

	for (i = 0; i < array_num; i++)
	{
		tmp = exc_array[i];
		free(tmp.path);
		free(tmp.name);
	}
	free(exc_array);
}

static void free_pts_acl(DOMAIN *d);
static void free_tty_acl(DOMAIN *d);
static void free_proc_acl(DOMAIN *d);
static void free_com_acl(COM_ACL_RULE *com_array, int array_num);

/**
 *  @name:	free_domain
 *  @about:	free DOMAIN structure
 *  @args:	domain (void *) -> domain data
 *  @return:	return 0
 */
int
free_domain(void *domain)
{
	DOMAIN *d;
	d = domain;

	free_file_acl(d->file_acl);
	free_file_exc(d->exc_rule_array, d->exc_rule_array_num);
	free_com_acl(d->com_acl_array, d->com_acl_array_num);
	free(d->net_acl);

	free_tty_acl(domain);
	free_pts_acl(domain);
	free_proc_acl(domain);

	return 0;
}

/**
 *  @name:	print_acl
 *  @about:	print acl
 *  @args:	p (void *) -> rule data
 *  @return:	return 0 on success
 */
static int
print_acl(void *p)
{
	FILE_ACL_RULE *r;
	r = (FILE_ACL_RULE *)p;

	printf("%s,%d\n", r->path, r->allowed);

	return 0;
}

/**
 *  @name:	print_domain_info
 *  @about:	print domain information
 *  @args:	d (DOMAIN) -> domain data
 *  @return:	none
 */
void
print_domain_info(DOMAIN d)
{
	int i;
	printf("***domain name %s***\n", d.name);

	printf("****************file acls permission*********\n");
	if (d.file_acl != NULL)
		handle_all_element(d.file_acl, print_acl);

	printf("******file_trans*********\n");
	if (d.exc_rule_array != NULL)
	{
		for (i = 0; i < d.exc_rule_array_num; i++)
		{
			printf("%s,%s\n", d.exc_rule_array[i].domain->name,
			       d.exc_rule_array[i].path);
		}
	}

	printf("network\n");
	if (d.net_acl != NULL)
		printf("%d\n", d.net_acl->perm);

	return;
}

/**
 *  @name:	print_all_domain
 *  @about:	print information about hash table
 *  @args:	none
 *  @return:	none
 */
void
print_all_domain()
{
	int i;
	HASH_NODE *p;
	HASH_NODE *hashbuf;

	printf("number of domain%d\n", domain_hash_table->element_num);

	hashbuf = domain_hash_table->buf;

	for (i = 0; i < DOMAIN_HASH_TABLE_SIZE; i++)
	{
		if (hashbuf[i].data != NULL)
		{
			p = &hashbuf[i];
			while (p->next != NULL)
			{
				print_domain_info(*((DOMAIN *)(p->data)));
				p = p->next;
			}
			print_domain_info(*((DOMAIN *)(p->data)));
		}
	}
}

/**
 *  @name:	free_domain_tab
 *  @about:	free "domain_hash_table"
 *  @args:	none
 *  @return:	none
 */
void
free_domain_tab()
{
	handle_all_element(domain_hash_table, free_domain);
	delete_hash_table(domain_hash_table);
}

/**
 *  Functions related to domain transition
 */
TRANS_RULE *rulebuf=NULL;
int domain_trans_rule_num=0;

/**
 *  @name:	print_domain_trans
 *  @about:	print domai trans rule
 *  @args:	none
 *  @return:	none
 */
void
print_domain_trans()
{
	int i;
	TRANS_RULE t;

	for (i = 0; i < domain_trans_rule_num; i++)
	{
		t = rulebuf[i];
		printf("trans:%s %s %s\n", t.parent,t.path, t.child);
	}
}

/**
 *  @name:	registar_domain_trans
 *  @about:	When yacc finds "domain_trans",register information about domain transition
 * 		with "rulebuf" as dynamic array.
 *  @args:	from_domain (char *) -> from domain name
 *  @args:	path (char *) -> path name
 *  @return:	none
 */
void
registar_domain_trans(char *from_domain, char *path)
{
	TRANS_RULE *tmp;
	TRANS_RULE work;
	DOMAIN *d;
	struct stat buf;
	int exist;

	d = search_domain_hash(current_domain);

	work.parent = strdup(from_domain);
	work.path = strdup(path);
	work.child = strdup(current_domain);

	if (d->roleflag == 1)
	{			/*domain_trans or domain_auto_trans*/
		work.auto_flag = 0;
	}
	else
	{
		work.auto_flag = 1;
	}

	tmp = (TRANS_RULE *)extend_array(rulebuf, &(domain_trans_rule_num), sizeof(TRANS_RULE));

	tmp[domain_trans_rule_num-1] = work;

	rulebuf = tmp;

	// printf("??????????%s:%s\n", DUMMY_DOMAIN_NAME, work.path);
	// add_fileacl_to_domain(DUMMY_DOMAIN_NAME, work.path, P_READ, 1);
	exist = stat(work.path,&buf);

	if ((buf.st_mode & S_IFDIR) && exist == 0)
	{
		label_child_dir(work.path);
	}

	return;
}

/**
 *  @name:	free_domain_trans
 *  @about:	free TRANS_RULE
 *  @args:	none
 *  @return:	none
 */
void
free_domain_trans()
{
	int i;
	TRANS_RULE t;

	for (i = 0; i < domain_trans_rule_num; i++)
	{
		t = rulebuf[i];
		free(t.parent);
		free(t.path);
		free(t.child);
	}
	free(rulebuf);
	domain_trans_rule_num = 0;
}

/**
 *  register network ACL
 */
char used_tcp_ports[1024+1];
char used_udp_ports[1024+1];

/**
 *  @name:	registar_net_acl
 *  @about:	registar network acl
 *  @args:	flag (int) ->
 *			0:allownet
 *     			1:allownet -raw
 *     			2:allownet -tcp -port <port>
 *     			3:allownet -udp -port <port>
 *  @args:	port (int) -> port number
 *  @return:	return 0 on success
 */ 
int
registar_net_acl(int flag, int port)
{
	DOMAIN *domain;
	NET_RULE *net;
	static int init = 0;
	int i;

	if (init == 0)
	{
		for (i = 0; i < sizeof(used_tcp_ports); i++)
		{
			used_tcp_ports[i] = 0;
			used_udp_ports[i] = 0;
		}
		init = 1;
	}


	domain = search_domain_hash(current_domain);
	if (domain == NULL)
	{
		action_error("domain %s isn't defined\n", current_domain);
		exit(1);
	}

	if (domain->net_acl == NULL)
	{
		if ((net = (NET_RULE *)malloc(sizeof(NET_RULE))) == NULL)
		{
			perror("malloc");
			yyerror("malloc error\n");
			exit(1);
		}
		memset(net, 0, sizeof(NET_RULE));

		net->domain = domain;
		domain->net_acl=net;
	}

	if (port > 1024)
	{
		snprintf(errmsg,sizeof(errmsg), "Port number>1024 is not yet supported\n");
		yyerror(errmsg);
		exit(1);
	}

	switch (flag)
	{
	case 0:
		domain->net_acl->perm |= CAN_TCP|CAN_UDP;
		break;

	case 1:
		domain->net_acl->perm |= CAN_RAW;
		break;

	case 2:
		domain->net_acl->tcp_port[port] = 1;
		domain->net_acl->flag = 1;
		used_tcp_ports[port] = 1;
		break;

	case 3:
		domain->net_acl->udp_port[port] = 1;
		domain->net_acl->flag = 1;
		used_udp_ports[port] = 1;
		break;

	case 4:
		domain->net_acl->perm |= CAN_WELLKNOWN;
		break;

	case 5:
		domain->net_acl->perm |= CAN_WELLKNOWN_TCP;
		break;

	case 6:
		domain->net_acl->perm |= CAN_WELLKNOWN_UDP;
		break;

	default:
		break;
	}

	return 0;
}

/**
 *  register IPC ACL
 */

/**
 *  @name:	registar_com_acl
 *  @about:	registar communication acl
 *  @args:	flag (int) ->
 *  @args:	to_domain (char *) ->
 *  @return:	return 0 on success
 */
int
registar_com_acl(int flag, char *to_domain)
{
	DOMAIN *domain;
	COM_ACL_RULE *tmp;
	COM_ACL_RULE work;

	domain = search_domain_hash(current_domain);
	work.domain = domain;
	work.flag = flag;
	work.domain_name = strdup(to_domain);
	work.perm = tmp_perm;


	tmp = (COM_ACL_RULE *)extend_array(domain->com_acl_array, &(domain->com_acl_array_num),
					   sizeof(COM_ACL_RULE));
	tmp[domain->com_acl_array_num-1] = work;
	domain->com_acl_array = tmp;

	return 0;
}

/**
 *  @name:	free_com_acl
 *  @about:	free communication acl
 *  @args:	com_array (COM_ACL_RULE *) -> array of rule
 *  @args:	array_num (int) -> number of elements
 *  @return:	none
 */
static void
free_com_acl(COM_ACL_RULE *com_array, int array_num)
{
	int i;

	COM_ACL_RULE tmp;
	for (i = 0; i < array_num; i++)
	{
		tmp = com_array[i];
		free(tmp.domain_name);
	}
	free(com_array);
}

/**
 *  @name:	registar_adm_acl
 *  @about:	register Administration ACL
 *  @args:	flag (int) ->
 *  @return:	return 0 on success
 */
int
registar_adm_acl(int flag)
{
	DOMAIN *domain;
	ADMIN_RULE r;

	domain = search_domain_hash(current_domain);
	r = domain->admin_rule;


	if (flag == ADM_PERM)
	{
		if (tmp_adm_perm.relabel == 1)
		{
			r.relabel = 1;
		}
		if (tmp_adm_perm.getsecurity == 1)
		{
			r.getsecurity = 1;
		}
		if (tmp_adm_perm.setenforce == 1)
		{
			r.setenforce = 1;
		}

		if (tmp_adm_perm.load_policy == 1)
		{
			r.load_policy = 1;
		}
		if (tmp_adm_perm.net == 1)
		{
			r.net = 1;
		}
		if (tmp_adm_perm.boot == 1)
		{
			r.boot = 1;
		}
		if (tmp_adm_perm.insmod == 1)
		{
			r.insmod = 1;
		}
		if (tmp_adm_perm.quotaon == 1)
		{
			r.quotaon = 1;
		}
		if (tmp_adm_perm.swapon == 1)
		{
			r.swapon = 1;
		}
		if (tmp_adm_perm.mount == 1)
		{
			r.mount = 1;
		}
		if (tmp_adm_perm.raw_io == 1)
		{
			r.raw_io = 1;
		}
		if (tmp_adm_perm.ptrace == 1)
		{
			r.ptrace = 1;
		}
		if (tmp_adm_perm.chroot == 1)
		{
			r.chroot = 1;
		}
		if (tmp_adm_perm.search == 1)
		{
			r.search = 1;
		}
		if (tmp_adm_perm.unlabel == 1)
		{
			r.unlabel = 1;
		}
		if (tmp_adm_perm.read == 1)
		{
			r.read = 1;
		}
		if (tmp_adm_perm.write == 1)
		{
			r.write = 1;
		}
		if (tmp_adm_perm.part_relabel == 1)
		{
			r.part_relabel = 1;
		}

		domain->admin_rule = r;
	}

	return 0;
}

/**
 *  tty ACL
 */

/**
 *  @name:	registar_tty_acl
 *  @about:	role=NULL only tty_create_flag is set.
 * 		flag=0:-create,flag=1:-change,flag=2:r,w
 *  @arsg:	role (char *) -> role name
 *  @args:	flag (int) ->
 :  @return:	return 0 on success 
 */
int
registar_tty_acl(char *role, int flag)
{
	DOMAIN *d;
	TTY_RULE *tmp_tty_rule;
	TTY_RULE work;

	d = search_domain_hash(current_domain);

	if (flag == 0)
	{
		d->tty_create_flag = 1;
		return 0;
	}

	if (flag == 1)
	{
		work.domain = d;
		work.rolename = strdup(role);
		work.perm = CHANGE_PRM;
	}

	if (flag == 2)
	{
		work.domain = d;
		work.rolename = strdup(role);
		work.perm = tmp_perm;
	}

	/* update */
	tmp_tty_rule = (TTY_RULE *)extend_array(d->tty_acl_array,&(d->tty_acl_array_num),
						sizeof(TTY_RULE));
	tmp_tty_rule[d->tty_acl_array_num - 1] = work;
	d->tty_acl_array = tmp_tty_rule;

	return 0;
}

/**
 *  @name:	free_tty_acl
 *  @about:	free tty acl
 *  @args:	d (DOMAIN *) -> domain data
 *  @return:	none
 */
static void
free_tty_acl(DOMAIN *d)
{
	int i;

	if (d->tty_acl_array == NULL)
		return ;

	for (i = 0; i < d->tty_acl_array_num; i++)
	{
		if (d->tty_acl_array[i].rolename != NULL)
			free(d->tty_acl_array[i].rolename);
	}
	free(d->tty_acl_array);
}

/**
 *  PTS ACL
 */

/**
 *  @name:	registar_pts_acl
 *  @about:	registar pts acl
 *  @args:	domain (char *) -> domain name
 *  @args:	flag (int) ->
 *  @return:	return 0 on success
 */
int
registar_pts_acl(char *domain, int flag)
{
	DOMAIN *d;
	PTS_RULE *tmp;
	PTS_RULE work;

	d = search_domain_hash(current_domain);
	work.domain = d;
	work.domain_name = NULL;
	work.perm = 0;

	if (flag == 0)
	{
		d->pts_create_flag = 1;
		return 0;
	}

	if (flag == 1)
	{
		work.domain_name = strdup(domain);
		work.perm = CHANGE_PRM;
	}

	if (flag == 2)
	{
		work.domain_name = strdup(domain);
		work.perm = tmp_perm;
	}
	/*update*/
	tmp = (PTS_RULE *)extend_array(d->pts_acl_array, &(d->pts_acl_array_num),
				       sizeof(PTS_RULE));
	tmp[d->pts_acl_array_num - 1] = work;

	d->pts_acl_array = tmp;

	return 0;
}

/**
 *  @name:	free_pts_acl
 *  @about:	free pts acl
 *  @args:	d (DOMAIN *) -> domain data
 *  @return:	none
 */
static void
free_pts_acl(DOMAIN *d)
{
	int i;
	char *p;

	if (d->pts_acl_array == NULL)
		return;

	for (i = 0; i < d->pts_acl_array_num; i++)
	{
		p = d->pts_acl_array[i].domain_name;
		if (p != NULL)
			free(p);
	}

	/*  if (d->pts_change_name != NULL)
	    {
	    free(d->pts_change_name);
	    }*/
	free(d->pts_acl_array);
}

/**
 *  ALLOWPROC
 */

/**
 *  @name:	registart_proc_acl
 *  @about:	registar process acl
 *  @args:	flag (int) ->
 *  @return:	return 0 on success
 */
int
 registar_proc_acl(int flag)
{
	DOMAIN *d;
	PROC_RULE work;
	PROC_RULE *tmp;

	d = search_domain_hash(current_domain);

	work.domain = d;
	work.perm = tmp_perm;
	work.type = flag;

	/* update */
	tmp = (PROC_RULE *)extend_array(d->proc_acl_array, &(d->proc_acl_array_num),
					sizeof(PROC_RULE));
	tmp[d->proc_acl_array_num - 1] = work;

	d->proc_acl_array = tmp;

	return 0;
}

/**
 *  @name:	free_proc_acl
 *  @about:	free process acl
 *  @args:	d (DOMAIN *) -> domain data
 *  @return:	none
 */
static void
free_proc_acl(DOMAIN *d)
{
	if (d->proc_acl_array == NULL)
		return;

	free(d->proc_acl_array);
}

/**
 *  tmpfs
 */

/**
 *  @name:	registar_tmpfs_create
 *  @about:	allowtmpfs -create
 * 		others are handled by registar_com_acl
 *  @args:	none
 *  @return:	return 0
 */
int
registar_tmpfs_create()
{
	DOMAIN *domain;
	domain = search_domain_hash(current_domain);
	domain->tmpfs_create_flag = 1;

	return 0;
}

/**
 *  RBAC
 */

/**
 *  This points to rolename currently handled.
 *  This is initialized by register_role.
 *  This is used by register_user.
 */
static char *tmp_role_name=NULL;

/**
 *  user-role
 */
HASH_TABLE *user_hash_table=NULL;

/**
 *  rbac information
 */
HASH_TABLE *rbac_hash_table=NULL;
#define USER_HASH_TABLE_SIZE 1024
#define RBAC_HASH_TABLE_SIZE 1024

/**
 *  @name:	registar_role
 *  @about:	register new role with role_hash_table
 *  @args:	name (char *) -> role name
 *  @return:	nreturn 0 on success
 */
int
registar_role(char *name)
{
	char *domain_name;
	DOMAIN *default_domain;
	RBAC *rbac;
	int len;

	len = strlen(name);
	if (len < 2 || !(name[len - 2] == '_' && name[len - 1] == 'r'))
	{
		snprintf(errmsg,sizeof(errmsg),
			 "role name %s is invalid. role name must be *_r\n",name);
		yyerror(errmsg);

		exit(1);
	}

	if (tmp_role_name != NULL)
	{
		free(tmp_role_name);
	}

	tmp_role_name = strdup(name);

	/* first call */
	if (rbac_hash_table == NULL)
	{
		rbac_hash_table = create_hash_table(RBAC_HASH_TABLE_SIZE);
		if (rbac_hash_table == NULL)
		{
			perror("malloc");
			exit(1);
		}
	}

	/* role name to <rolename prefix>_t */
	domain_name=make_role_to_domain(name);

	if ((rbac = (RBAC *)malloc(sizeof(RBAC))) == NULL)
	{
		perror("malloc");
		exit(1);
	}

	/*  */
	registar_domain(domain_name, 1);

	/* set rbac */
	rbac->rolename = strdup(name);
	default_domain = search_domain_hash(domain_name);
	rbac->default_domain = default_domain;

	if (insert_element(rbac_hash_table, rbac, name) == -2)
	{
		snprintf(errmsg, sizeof(errmsg), "multiple definition of role %s\n", name);
		yyerror(errmsg);
		exit(1);
	}

	return 0;
}

/**
 *  @name:	register_user
 *  @about:	register new name with user_hash_table
 *  @args:	name (char *) -> user name
 *  @return:	nreturn 0 on success
 */
int
registar_user(char *name)
{
	int n;
	USER_ROLE *u;
	//RBAC *r;
	char **new_array;

	/* first call */
	if (user_hash_table == NULL)
	{
		user_hash_table = create_hash_table(USER_HASH_TABLE_SIZE);
		if (user_hash_table == NULL)
		{
			perror("malloc");
			exit(1);
		}
	}

	u = (USER_ROLE *)search_element(user_hash_table, name);
	if (u == NULL)
	{
		if ((u = (USER_ROLE *)malloc(sizeof(USER_ROLE))) == NULL)
		{
			perror("malloc");
			exit(1);
		}
		u->username = strdup(name);

		if ((new_array = (char **)malloc(sizeof(char *))) == NULL)
		{
			perror("malloc");
			exit(1);
		}
		new_array[0] = strdup(tmp_role_name);

		u->role_name_array = new_array;

		u->role_name_array_num = 1;
		insert_element(user_hash_table, u, u->username);

		return 0;
	}
	else
	{
		/* update rbac_array */
		n = u->role_name_array_num;
		new_array = (char **)realloc(u->role_name_array, sizeof(char *)*(n+1));
		if (new_array == NULL)
		{
			perror("realloc");
			exit(1);
		}
		u->role_name_array_num++;

		new_array[u->role_name_array_num - 1] = strdup(tmp_role_name);

		u->role_name_array = new_array;

		return 0;
	}

	return 0;
}

/**
 *  @name:	free_rback
 *  @about:	free rbac data
 *  @args:	v (void *) -> rbac data
 *  @return:	return 0 on success
 */
static int
free_rbac(void *v)
{
	RBAC *r;
	r = v;
	free(r->rolename);

	return 0;
}

/**
 *  @name:	free_user_role
 *  @about:	free user role data
 *  @args:	v (void *) -> user role data
 *  @return:	return 0 on success
 */
static int
free_user_role(void *v)
{
	int i;
	USER_ROLE *r;

	r = v;

	free(r->username);

	for (i = 0; i < r->role_name_array_num; i++)
	{
		free((r->role_name_array)[i]);
	}

	free(r->role_name_array);

	return 0;
}

/**
 *  @name:	free_rback_hash_table
 *  @about:	free rbac_hash_table
 *  @args:	none
 *  @return:	none
 */
void
free_rbac_hash_table()
{
	if (rbac_hash_table == NULL)
		return;

	handle_all_element(rbac_hash_table, free_rbac);
	delete_hash_table(rbac_hash_table);
}

/**
 *  @name:	free_user_hash_table
 *  @about:	free user hash table
 *  @args:	none
 *  @return:	none
 */
void
free_user_hash_table()
{
	if (user_hash_table == NULL)
		return ;
	handle_all_element(user_hash_table, free_user_role);

	delete_hash_table(user_hash_table);
}

/**
 *  @name:	make_role_to_domain
 *  @about:	This generates domain name by malloc from rolename.
 * 		domain name is <role name prefix>_t
 *  @args:	name (char *) -> role
 *  @return:	return domain name exchanged role name
 */
char *
make_role_to_domain(char *name)
{
	int len;
	char *domain_name;

	len = strlen(name);
	if ((domain_name = (char *)malloc(len+1)) == NULL)
	{
		perror("malloc");
		exit(1);
	}

	strcpy(domain_name,name);
	domain_name[len-1] = 't';

	return domain_name;
}
