package hayashi.kiban;
import hayashi.osm.api.HttpGET;
import javax.xml.parsers.*;
import javax.xml.transform.TransformerException;

import org.w3c.dom.*;
import org.xml.sax.*;

import java.io.*;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;

import jp.co.areaweb.tools.database.*;

public class ConvBusstopGML {

	String filter = "";
	String urlStr = "";

	// 近くのバス停を探す範囲（バス停を中心としたNEER×２ｍ四方の領域
	static final int NEER = 75;

	/**
	 * 国土数値情報　バス停留所データ
	 * JPGIS2.1(GML)準拠フォーマット :
	 * http://nlftp.mlit.go.jp/ksj/gml/datalist/KsjTmplt-P11.html
	 *
	 * @throws IOException
	 * @throws SQLException
	 * @throws ClassNotFoundException
	 * @throws FileNotFoundException
	 * @throws TransformerException
	 * @throws SAXException
	 * @throws ParserConfigurationException
	 */
	public static void main(String[] args) throws FileNotFoundException, ClassNotFoundException, SQLException, IOException, ParserConfigurationException, SAXException, TransformerException
	{
		int index = 0;
		boolean updateDb = false;
		boolean docheck = true;
		if (args.length > index) {
			if (args[index].equals("-init")) {
				updateDb = true;
				index++;
			}
			if (args[index].equals("-nocheck")) {
				docheck = false;
				index++;
			}
		}

		/**
		 * アプリケーション [ConvBusstop]
		 * > java -jar ReadKIBAN.jar <オプション>
		 * 	オプション： -exp	実行する直前にデータベースを初期化する（省略可能）
		 */
		Connection con = DatabaseTool.openDb("database");
		try {
			/**
			 * バス停データ変換のメイン処理
			 */

			if (updateDb) {
				File dbdir = new File("database");
				if (dbdir.isDirectory()) {
					dbdir.delete();
				}
				dbdir.mkdir();

				ConvBusstopGML.initDb(con);
			}

			int fcounter = 0;
			if (args.length > index) {
				File iFile = new File(args[index]);
				fcounter += proc(con, iFile, docheck);
			}
			else {
				File dir = new File(".");
				File[] files = dir.listFiles();

				for (File iFile : files) {
					fcounter += proc(con, iFile, docheck);
				}
			}
			System.out.println("["+ fcounter +"]つのファイルをインポートしました。");
		}
		finally {
			DatabaseTool.closeDb(con);
		}
	}

	/**
	 * メイン処理ルーチン
	 * @param con
	 * @param iFile
	 * @return
	 * @throws FileNotFoundException
	 * @throws ClassNotFoundException
	 * @throws SQLException
	 * @throws IOException
	 */
	static int proc(Connection con, File iFile, boolean docheck) throws FileNotFoundException, ClassNotFoundException, SQLException, IOException {
		if (checkFile(iFile)) {
			// GMLファイルを読み取ってローカルベータベースへ記録する
			inputFile(con, iFile, docheck);

			// ローカルデータベース内の情報を出力する
			String iStr = iFile.getName();
			outputDb(con, iStr.substring(0, iStr.length() - 4));

			outputAllOsm(con, iStr.substring(0, iStr.length() - 4));
			return 1;
		}
		else {
			return 0;
		}
	}

	/**
	 * 各テーブルを初期化する
	 * @param con
	 */
	public static void initDb(Connection con) {
		DbBusstop.initDb(con);
		DbBuscourse.initDb(con);
		DbBusref.initDb(con);
	}

	/**
	 * ソースファイルを読み取ってローカルベータベースへ記録する
	 * @param con
	 * @param iFile
	 * @throws FileNotFoundException
	 * @throws ClassNotFoundException
	 * @throws SQLException
	 * @throws IOException
	 */
	public static void inputFile (Connection con, File iFile, boolean docheck) throws FileNotFoundException, ClassNotFoundException, SQLException, IOException {
		int iCounter = 0;
		String timeStampStr = null;

		String iStr = iFile.getName();
		File dir = new File(iStr.substring(0, iStr.length() - 4));
		dir.mkdir();

		DocumentBuilderFactory factory;
		DocumentBuilder        builder;
		Node root;
		try {
			factory = DocumentBuilderFactory.newInstance();
			builder = factory.newDocumentBuilder();
			factory.setIgnoringElementContentWhitespace(true);
			factory.setIgnoringComments(true);
			factory.setValidating(true);
			root    = builder.parse(iStr);

			iCounter += showNodes(con, root, iStr.substring(0, iStr.length() - 4), timeStampStr, docheck);
		} catch (ParserConfigurationException e0) {
			System.out.println(e0.getMessage());
		} catch (SAXException e1){
			System.out.println(e1.getMessage());
		} catch (IOException e2) {
			System.out.println(e2.getMessage());
		} catch (SQLException e2) {
			System.out.println(e2.getMessage());
		}
		System.out.println("バス停数["+ iCounter +"]");
	}


	/**
	 * ローカルデータベース内の情報を出力する
	 * @param con
	 * @param iCode
	 * @throws IOException
	 * @throws SQLException
	 */
	public static void outputDb(Connection con, String iCode) throws IOException, SQLException {
		SimpleDateFormat timeStampFmt = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
		String timeStampStr = timeStampFmt.format(new Date(Calendar.getInstance().getTimeInMillis()));
		File dir = new File(iCode);
		dir.mkdir();

		BufferedWriter ow = null;
		BufferedWriter gw = null;

		File htmlFile = new File(iCode  +".html");
		BufferedWriter hw = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(htmlFile), "UTF-8"));
		hw.write("<!DOCTYPE html>");
		hw.newLine();
		hw.write("<html><head><meta charset=\"utf-8\" /></head>");
		hw.newLine();
		hw.write("<body><table border='1'>");
		hw.newLine();
		hw.write("<tr>");
		hw.write("<td>type</td>");
		hw.write("<td>corp</td>");
		hw.write("<td>course</td>");
		hw.write("<td>GPX</td>");
		hw.write("<td>SAMPLE</td>");
		hw.write("<td>バス停数</td>");
		hw.write("<td>未入力</td>");
		hw.write("<td>既存</td>");
		hw.write("</tr>");
		hw.newLine();

		PreparedStatement ps7 = con.prepareStatement("SELECT code,type,corp,course,ifile FROM bus_course WHERE ifile=? ORDER BY corp,course");
		PreparedStatement ps9 = con.prepareStatement("SELECT idref FROM bus_ref WHERE code=?");
		//PreparedStatement ps8 = con.prepareStatement("SELECT name,lat,lon,fixed FROM bus_stop WHERE idref=?");
		ps7.setString(1, iCode);
		ResultSet rset7 = ps7.executeQuery();
		while (rset7.next()) {
			int code = rset7.getInt(1);
			int type = rset7.getInt(2);
			String corp = rset7.getString(3);
			String course = rset7.getString(4);

			File osmFile = new File(dir, iCode + String.format("_%1$08d", code) +".osm");
			File gpxFile = new File(dir, iCode + String.format("_%1$08d", code) +".gpx");
			File osmSample = new File(dir, iCode + String.format("_s%1$08d", code) +".osm");

			System.out.println("course = "+ course);
			int stopCount = 0;
			int fixedCount = 0;
			int unfixedCount = 0;

			// index file header
			hw.write("<tr>");
			hw.write("<td>"+ type +"</td>");
			hw.write("<td>"+ corp +"</td>");
			hw.write("<td><a href='"+ dir.getName() +"/"+ osmFile.getName() +"'>"+ course +"</a></td>");
			hw.write("<td><a href='"+ dir.getName() +"/"+ gpxFile.getName() +"'>"+ gpxFile.getName() +"</a></td>");
			hw.write("<td><a href='"+ dir.getName() +"/"+ osmSample.getName() +"'>"+ osmSample.getName() +"</a></td>");
			hw.newLine();

			// OSM file header
			ow = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(osmFile), "UTF-8"));
			ow.write("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>");
			ow.newLine();
			ow.write("<osm version=\"0.5\" generator=\"ReadKIBAN\">");
			ow.newLine();

			// GPX file header
			gw = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(gpxFile), "UTF-8"));
			gw.write("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>");
			gw.newLine();
			gw.write("<gpx xmlns=\"http://www.topografix.com/GPX/1/1\" version=\"1.1\" creator=\"osmtracker-android\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.topografix.com/GPX/1/1 http://www.topografix.com/GPX/1/1/gpx.xsd \">");
			gw.newLine();

			double maxLat = 0.0;
			double minLat = 180.0;
			double maxLon = 0.0;
			double minLon = 180.0;

			ps9.setInt(1, code);
			ResultSet rset9 = ps9.executeQuery();
			while (rset9.next()) {
				String idref = rset9.getString(1);
				if ((idref != null) && (!idref.equals(""))) {
					DbBusstop bean = new DbBusstop();
					DbBusstop.get(con,  bean, idref);
					if (bean.idref.equals(idref)) {
						stopCount++;

						if (bean.lat > maxLat) {
							maxLat = bean.lat;
						}
						if (bean.lon > maxLon) {
							maxLon = bean.lon;
						}
						if (bean.lat < minLat) {
							minLat = bean.lat;
						}
						if (bean.lon < minLon) {
							minLon = bean.lon;
						}

						System.out.println("\tway point = "+ idref);
						fixedCount += bean.fixed;
						if (bean.fixed == 0) {
							unfixedCount++;

							// OSM node
							int nodeid = Integer.parseInt(idref.substring(1)) * -1;
							String osm_node = nodeBusstop(nodeid, bean);
							ow.write(osm_node);
							ow.newLine();

							// TEXT node
							File txtFile = new File(dir, iCode + idref +".txt");
							BufferedWriter gw2 = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(txtFile), "UTF-8"));
							gw2.write(osm_node);
							gw2.newLine();
							gw2.close();

							// GPX waypoint
							gw.write("<wpt lat=\""+ bean.lat +"\" lon=\""+ bean.lon +"\">\n");
							gw.write(" <time>"+ timeStampStr +"Z</time>\n");
							gw.write(" <name><![CDATA["+ bean.name +"]]></name>\n");
							gw.write(" <link href=\""+ txtFile.getName() +"\"><text>"+ idref +"</text></link>\n");
							gw.write("</wpt>\n");
							gw.newLine();
						}
					}
				}
			}
			rset9.close();

			// INDEX file
			hw.write("<td>"+ stopCount +"</td>");
			hw.write("<td>"+ unfixedCount +"</td>");
			hw.write("<td>"+ fixedCount +"</td>");
			hw.write("</tr>");
			hw.newLine();

			// OSM file header
			BufferedWriter ow2 = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(osmSample), "UTF-8"));
			ow2.write("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>");
			ow2.newLine();
			ow2.write("<osm version=\"0.5\" generator=\"ReadKIBAN\">");
			ow2.newLine();

			DbBusstop sample = new DbBusstop();
			sample.name = "SAMPLE";
			sample.lat = ((maxLat+minLat)/2);
			sample.lon = ((maxLon+minLon)/2);
			ow2.write(nodeBusstop(code, sample));
			ow2.newLine();
			ow2.write("</osm>");
			ow2.newLine();
			ow2.close();

			// OSM file footer
			ow.write("</osm>");
			ow.newLine();
			ow.close();

			// GPX file footer
			gw.write("</gpx>");
			gw.newLine();
			gw.close();
		}
		rset7.close();

		// index file footer
		hw.write("</table></body></html>");
		hw.newLine();
		hw.close();
	}

	/**
	 * ローカルデータベース内の情報を出力する
	 * @param con
	 * @param iCode
	 * @throws IOException
	 * @throws SQLException
	 */
	public static void outputAllOsm(Connection con, String iCode) throws IOException, SQLException {
		File dir = new File(iCode);
		dir.mkdir();

		File osmFile = new File(dir, iCode +".osm");

		// OSM file header
		BufferedWriter ow = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(osmFile), "UTF-8"));
		ow.write("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>");
		ow.newLine();
		ow.write("<osm version=\"0.5\" generator=\"ReadKIBAN\">");
		ow.newLine();


		DbBusstop bean = new DbBusstop();
		PreparedStatement ps8 = con.prepareStatement("SELECT name,lat,lon,fixed,ifile,idref FROM bus_stop ORDER BY ifile,idref");
		ResultSet rset8 = ps8.executeQuery();
		while (rset8.next()) {
			// OSM node
			bean.name = rset8.getString(1);
			bean.lat = rset8.getDouble(2);
			bean.lon = rset8.getDouble(3);
			bean.fixed = rset8.getInt(4);
			bean.ifile = rset8.getString(5);
			bean.idref = rset8.getString(6);
			int nodeid = Integer.parseInt(bean.idref.substring(1)) * -1;
			if (bean.fixed == 0) {
				String osm_node = nodeBusstop(nodeid, bean);
				ow.write(osm_node);
				ow.newLine();
			}
		}
		rset8.close();

		// OSM file footer
		ow.write("</osm>");
		ow.newLine();
		ow.close();
	}

	public static String nodeBusstop(int code, DbBusstop bean) {
		SimpleDateFormat timeStampFmt = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
		String timeStampStr = timeStampFmt.format(new Date(Calendar.getInstance().getTimeInMillis()));

		String osm_node = ("<node id=\""+ code +"\" timestamp=\""+ timeStampStr +"Z\" lat=\""+ bean.lat +"\" lon=\""+ bean.lon +"\">\n");
		osm_node += "<tag k=\"name\" v=\""+ bean.name +"\"/>\n";
		osm_node += "<tag k=\"highway\" v=\"bus_stop\"/>\n";
		osm_node += "<tag k=\"bus\" v=\"yes\"/>\n";
		osm_node += "<tag k=\"fixme\" v=\"このバス停を正しい位置に移動させて欲しい\"/>\n";
		osm_node += ("<tag k=\"source\" v=\"KSJ2\"/>\n");
		osm_node += ("<tag k=\"source_ref\" v=\"http://nlftp.mlit.go.jp/ksj/gml/datalist/KsjTmplt-P11.html\"/>\n");
		osm_node += ("<tag k=\"created_by\" v=\"National-Land-Numerical-Information_MLIT_Japan\"/>\n");
		osm_node += ("<tag k=\"note\" v=\"National-Land Numerical Information (Bus stop) 2012, MLIT Japan\"/>\n");
		osm_node += ("<tag k=\"note:ja\" v=\"国土数値情報（バス停留所）平成２４年　国土交通省\"/>\n");
		osm_node += ("<tag k=\"public_transport\" v=\"stop_position\"/>\n");
		osm_node += ("</node>\n");
		return osm_node;
	}

	/**
	 *
	 * <ksj:Dataset>
	 *   <gml:description>国土数値情報 バス停留所 インスタンス文書</gml:description>
	 *   <gml:boundedBy>
	 *     <gml:EnvelopeWithTimePeriod srsName="JGD2000 / (B, L)" frame="GC / JST">
	 *       <gml:lowerCorner>20.0 123.0</gml:lowerCorner>
	 *       <gml:upperCorner>46.0 154.0</gml:upperCorner>
	 *       <gml:beginPosition calendarEraName="西暦">1900</gml:beginPosition>
	 *       <gml:endPosition indeterminatePosition="unknown"/>
	 *     </gml:EnvelopeWithTimePeriod>
	 *   </gml:boundedBy>
	 *
	 *   <gml:Point gml:id="n1">
	 *     <gml:pos>40.55992791 140.49739301</gml:pos>
	 *   </gml:Point>
	 *
	 *   <ksj:BusStop gml:id="ED01_1">
	 *     <ksj:position xlink:href="#n1"/>
	 *     <ksj:busStopName>松木平駅前</ksj:busStopName>
	 *     <ksj:busRouteInformation>
	 *       <ksj:BusRouteInformation>
	 *         anaComm("ksj:BusRouteInformation")
	 *       </ksj:BusRouteInformation>
	 *     </ksj:busRouteInformation>
	 *     <ksj:busRouteInformation>
	 *       <ksj:BusRouteInformation>
	 *         anaComm("ksj:BusRouteInformation")
	 *       </ksj:BusRouteInformation>
	 *     </ksj:busRouteInformation>
	 *   </ksj:BusStop>
	 *
	 * </ksj:Dataset>
	 *
	 * @param con
	 * @param node
	 * @param iFileName		// ソースファイル名（拡張子を含まない）
	 * @param timeStampStr
	 * @return
	 * @throws IOException
	 * @throws SQLException
	 */
	public static int showNodes(Connection con, Node node, String iFileName, String timeStampStr, boolean docheck) throws IOException, SQLException {
		int iCounter = 0;
		NamedNodeMap nodeMap = node.getAttributes();
		if ( null != nodeMap ) {
			for ( int j=0; j < nodeMap.getLength(); j++ ) {
				if (nodeMap.item(j).getNodeName().equals("timeStamp")) {
					timeStampStr = nodeMap.item(j).getNodeValue();
				}
			}
		}

		NodeList nodes = node.getChildNodes();
		for (int i=0; i<nodes.getLength(); i++) {
			Node node2 = nodes.item(i);
			if (node2.getNodeName().equals("gml:Point")) {
				showGmPoint(con, node2, docheck);
			}
			else if (node2.getNodeName().equals("ksj:BusStop")) {
				iCounter++;
				showED01(con, node2, iFileName);
			}
			else {
				iCounter += showNodes(con, node2, iFileName, timeStampStr, docheck);
			}
		}
		return iCounter;
	}

	/**
	 *
	 *   <ksj:BusStop gml:id="ED01_1">
	 *     <ksj:position xlink:href="#n1"/>
	 *     <ksj:busStopName>松木平駅前</ksj:busStopName>
	 *     <ksj:busRouteInformation>
	 *       <ksj:BusRouteInformation>
	 *         anaComm("ksj:BusRouteInformation")
	 *       </ksj:BusRouteInformation>
	 *     </ksj:busRouteInformation>
	 *     <ksj:busRouteInformation>
	 *       <ksj:BusRouteInformation>
	 *         anaComm("ksj:BusRouteInformation")
	 *       </ksj:BusRouteInformation>
	 *     </ksj:busRouteInformation>
	 *   </ksj:BusStop>
	 *
	 * @param con
	 * @param node
	 * @param iFileName		// ソースファイル名（拡張子を含まない）
	 * @throws IOException
	 * @throws SQLException
	 */
	public static void showED01(Connection con, Node node, String iFileName) throws IOException, SQLException {
		String idrefStr = "";
		String nameStr = "";
		//PreparedStatement ps1 = con.prepareStatement("SELECT idref FROM bus_stop WHERE idref=?");
		//PreparedStatement ps2 = con.prepareStatement("INSERT INTO bus_stop (idref,name,ifile) VALUES (?,?,?)");
		PreparedStatement ps3 = con.prepareStatement("SELECT code FROM bus_course WHERE course=? AND corp=? AND ifile=?");
		PreparedStatement ps4 = con.prepareStatement("INSERT INTO bus_course (code,type,corp,course,ifile) VALUES (?,?,?,?,?)");
		PreparedStatement ps5 = con.prepareStatement("INSERT INTO bus_ref (idref,code) VALUES (?,?)");
		PreparedStatement ps6 = con.prepareStatement("SELECT max(code) FROM bus_course WHERE ifile=?");

		ArrayList<String[]> bris = new ArrayList<String[]>();
		NodeList nodes = node.getChildNodes();
		for (int i=0; i < nodes.getLength(); i++) {
			Node node2 = nodes.item(i);
			if (node2.getNodeName().equals("ksj:position")) {
				NamedNodeMap nodeMap = node2.getAttributes();
				if (null != nodeMap) {
					for ( int j=0; j < nodeMap.getLength(); j++ ) {
						if (nodeMap.item(j).getNodeName().equals("xlink:href")) {
							idrefStr = nodeMap.item(j).getNodeValue().substring(1);
							System.out.println("found idref='"+ idrefStr +"'");
							break;
						}
					}
				}
			}
			else if (node2.getNodeName().equals("ksj:busStopName")) {
				nameStr = node2.getTextContent();
			}
			else if (node2.getNodeName().equals("ksj:busRouteInformation")) {
				String[] rtn = anaComm(node2);
				if (rtn != null) {
					bris.add(rtn);
				}
			}
		}

		// idref と nameStr をデータベースに格納する
		DbBusstop.setName(con, idrefStr, nameStr, iFileName);

		for (String[] rtn : bris) {
			int code = 0;
			ps3.setString(1, rtn[1]);
			ps3.setString(2, rtn[2]);
			ps3.setString(3, iFileName);
			ResultSet rset = ps3.executeQuery();
			if (rset.next()) {
				code = rset.getInt(1);
			}
			rset.close();

			if (code == 0) {
				ps6.setString(1, iFileName);
				ResultSet rset6 = ps6.executeQuery();
				if (rset6.next()) {
					code = rset6.getInt(1);
				}
				rset6.close();
				code++;

				System.out.println("code="+code);
				ps4.setInt(1, code);
				ps4.setInt(2, Integer.parseInt(rtn[0]));
				ps4.setString(3, rtn[2]);
				ps4.setString(4, rtn[1]);
				ps4.setString(5, iFileName);
				ps4.executeUpdate();
			}

			ps5.setString(1, idrefStr);
			ps5.setInt(2, code);
			ps5.executeUpdate();
		}

		//ps1.close();
		//ps2.close();
		ps3.close();
		ps4.close();
		ps5.close();
	}

	/**
	 * <ksj:busRouteInformation>
	 *   <ksj:BusRouteInformation>
	 *     <ksj:busType>1</ksj:busType>
	 *     <ksj:busOperationCompany>弘南バス（株）</ksj:busOperationCompany>
	 *     <ksj:busLineName>清原・安原～小栗山線　75</ksj:busLineName>
	 *   </ksj:BusRouteInformation>
	 * </ksj:busRouteInformation>
	 *
	 * @param briNode
	 * @return
	 */
	public static String[] anaComm(Node briNode) {
		String[] rtn = new String[3];
		rtn[0] = "0";	// corp type
		rtn[1] = "";	// course name
		rtn[2] = "";	// corp name

		// Node briNode = <ksj:busRouteInformation>
		NodeList nodes = briNode.getChildNodes();
		for (int i=0; i < nodes.getLength(); i++) {
			Node node2 = nodes.item(i);

			if (node2.getNodeName().equals("ksj:BusRouteInformation")) {
				NodeList nodes2 = node2.getChildNodes();
				for (int i2=0; i2 < nodes2.getLength(); i2++) {
					Node node3 = nodes2.item(i2);

					if (node3.getNodeName().equals("ksj:busType")) {
						rtn[0] = node3.getTextContent();
					}
					else if (node3.getNodeName().equals("ksj:busLineName")) {
						rtn[1] = node3.getTextContent();
					}
					else if (node3.getNodeName().equals("ksj:busOperationCompany")) {
						rtn[2] = node3.getTextContent();
					}
				}
				return rtn;
			}

		}
		return null;
	}

	/**
	 *   <gml:Point gml:id="n1">
	 *     <gml:pos>40.55992791 140.49739301</gml:pos>
	 *   </gml:Point>
	 *
	 * @param con
	 * @param node
	 * @throws IOException
	 * @throws SQLException
	 */
	public static void showGmPoint(Connection con, Node node, boolean docheck) throws IOException, SQLException {
		String idStr = "";

		NamedNodeMap nodeMap = node.getAttributes();
		if ( null != nodeMap ) {
			for ( int j=0; j<nodeMap.getLength(); j++ ) {
				if (nodeMap.item(j).getNodeName().equals("gml:id")) {
					idStr = nodeMap.item(j).getNodeValue();
				}
			}
		}
		if (idStr.equals("")) {
			return;
		}

		NodeList nodes1 = node.getChildNodes();
		for (int i1=0; i1 < nodes1.getLength(); i1++) {
			Node node2 = nodes1.item(i1);
			if (node2.getNodeName().equals("gml:pos")) {
				String positionStr = node2.getTextContent();
				String[] str4Ary = positionStr.split(" ");
				String latStr = str4Ary[0];
				String lonStr = str4Ary[1];

				double lat = Double.parseDouble(latStr);
				double lon = Double.parseDouble(lonStr);
				DbBusstop.setPosition(con, idStr, lat, lon, (docheck ? HttpGET.getMap(lat, lon, NEER) : 0));

				break;
			}
		}
	}

	/**
	 * 国土数値情報　バス停留所データ
	 * JPGIS2.1(GML)準拠フォーマット :
	 * http://nlftp.mlit.go.jp/ksj/gml/datalist/KsjTmplt-P11.html
	 *
	 * @param f
	 * @return
	 */
	static boolean checkFile(File f) {
		String name = f.getName();
		if (!name.startsWith("P11-10_")) {
			return false;
		}
		if (!name.toUpperCase().endsWith("-JGD-G.XML")) {
			return false;
		}
		return true;
	}


}