/* 
 * The MIT License
 *
 * Copyright (c) 2014 Yuki SAKAI
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

/*******************************************************************************
 * include ********************************************************************/
#if !defined(BMP_H__)
#  include <bmp.h>
#endif
#if !defined(BGP_H__)
#  include <bgp.h>
#endif
#if !defined(_SYS_TYPES_H) && !defined(_SYS_TYPES_H_)
#  include <sys/types.h>
#endif
#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
#if !defined(_SYS_ENDIAN_H) && !defined(_SYS_ENDIAN_H_)
#  include <sys/endian.h>
#endif
#else
#if !defined(_ENDIAN_H) && !defined(_ENDIAN_H_)
#  include <endian.h>
#endif
#endif

/*+==========================================================================+**
 *: check BMP messages size
 *: arg:
 *:    rdata        : bmp messages start point address.
 *:    maxDataLength: stored bmp messages max data size.
 *: return:
 *:    plus value  : BMP messages size.
 *:    zero        : data is not enough size for bmp messages.
 *:    minus value: error
 *+==========================================================================+*/
ssize_t bmpMessageSizeCheck (u_int8_t *rdata, const ssize_t maxDataLength)
{
  ssize_t     iHeaderLen;
  ssize_t     iMsgLen;
  BMPv1Header *pV1Header;
  BMPv3Header *pV3Header;
  /*==========================================================================*/
  if (maxDataLength<1) {return 0;}
  else if (*rdata == 1 || *rdata == 2) {iHeaderLen = sizeof (BMPv1Header);}
  else if (*rdata == 3               ) {iHeaderLen = sizeof (BMPv3Header);}
  else                                 {return -1;}
  /*==========================================================================*/
  if (maxDataLength<iHeaderLen) {return 0;}
  else if (*rdata == 3) {
    pV3Header = (BMPv3Header *)rdata;
    iMsgLen = be16toh (pV3Header->msgLen);
  }
  else {
    pV1Header = (BMPv1Header *)rdata;
    switch (pV1Header->msgType) {
    case BMP_MSGTYPE_ROUTEMON:
      iMsgLen = bmpRouteMonSize ((u_int8_t *)(pV1Header+1), maxDataLength - iHeaderLen);
      break;
    case BMP_MSGTYPE_STATISTICS:
      iMsgLen = bmpStatisticsSize ((u_int8_t *)(pV1Header+1), maxDataLength - iHeaderLen);
      break;
    case BMP_MSGTYPE_PEERDOWN:
      iMsgLen = bmpPeerDownSize ((u_int8_t *)(pV1Header+1), maxDataLength - iHeaderLen);
      break;
    case BMP_MSGTYPE_PEERUP:
      iMsgLen = bmpPeerUpSize ((u_int8_t *)(pV1Header+1), maxDataLength - iHeaderLen, pV1Header->version);
      break;
    default: return -1;
    }
    if      (iMsgLen==0) {return  0;}
    else if (iMsgLen <0) {return -1;}
    iMsgLen += sizeof (BMPv1Header);
  }
  /*==========================================================================*/
  if (maxDataLength<iMsgLen) {return 0;}
  return iMsgLen;
}

/*+==========================================================================+**
 *: check BMP PeerDown messages size
 *: arg:
 *:    rdata        : bmp peer down messages start point address.
 *:    maxDataLength: stored messages max data size.
 *: return:
 *:    plus value  : BMP peer down messages size.
 *:    zero        : data is not enough size for bmp messages.
 *:    minus value: error
 *+==========================================================================+*/
ssize_t bmpPeerDownSize (u_int8_t *rdata, const ssize_t maxDataLength)
{
  int             iSize;
  BMPMsgPeerDown  *pBMPPeerDownHeader;
  BGPCommonHeader *pBGPHeader;
  iSize = sizeof (BMPMsgPeerDown);
  /*==========================================================================*/
  if (maxDataLength<iSize) {return 0;}
  pBMPPeerDownHeader = (BMPMsgPeerDown *)rdata;
  if      (pBMPPeerDownHeader->reason==BMP_PEER_DOWN_REMOTE_NO_NOTIFICATION) {return iSize;}
  else if (pBMPPeerDownHeader->reason==BMP_PEER_DOWN_LOCAL_NOTIFICATION    ) {}
  else if (pBMPPeerDownHeader->reason==BMP_PEER_DOWN_LOCAL_NO_NOTIFICATION ) {}
  else if (pBMPPeerDownHeader->reason==BMP_PEER_DOWN_REMOTE_NOTIFICATION   ) {}
  else {return -1;}
  /*==========================================================================*/
  pBGPHeader = (BGPCommonHeader *)(pBMPPeerDownHeader+1);
  if (maxDataLength<(iSize+sizeof (BGPCommonHeader))) {return 0;}
  iSize += be16toh (pBGPHeader->length);
  if (maxDataLength<iSize) {return 0;}
  return iSize;
}

/*+==========================================================================+**
 *: check BMP Peer Up messages size
 *: arg:
 *:    rdata        : bmp peer up messages start point address.
 *:    maxDataLength: stored messages max data size.
 *: return:
 *:    plus value  : BMP peer up messages size.
 *:    zero        : data is not enough size for bmp messages.
 *:    minus value: error
 *+==========================================================================+*/
ssize_t bmpPeerUpSize (u_int8_t *rdata, const ssize_t maxDataLength, const u_int8_t ver)
{
  int             iSize;
  BGPCommonHeader *pBGPHeader;
  if      (ver==2) {}
  else if (ver==3) {}
  else {return -1;}
  /*==========================================================================*/
  if (ver==2) {
    pBGPHeader = (BGPCommonHeader *)rdata;
    if (maxDataLength<sizeof (BGPCommonHeader)) {return 0;}
    iSize = be16toh (pBGPHeader->length);
  }
  else if (ver==3) {
    /* unknown */
    iSize = maxDataLength;
  }
  if (maxDataLength<iSize) {return 0;}
  return iSize;
}

/*+==========================================================================+**
 *: check BMP Statistics messages size
 *: arg:
 *:    rdata        : bmp statistics messages start point address.
 *:    maxDataLength: stored messages max data size.
 *: return:
 *:    plus value  : BMP statistics messages size.
 *:    zero        : data is not enough size for bmp messages.
 *:    minus value: error
 *+==========================================================================+*/
ssize_t bmpStatisticsSize (u_int8_t *rdata, const ssize_t maxDataLength)
{
  int               i;
  int               iSize, iNum;;
  u_int8_t          *pTmp;
  BMPMsgStatsNum    *pStatsNum;
  BMPMsgStatsHeader *pStatsHeader;

  iSize = sizeof (BMPMsgStatsNum);
  /*==========================================================================*/
  if (maxDataLength<iSize) {return 0;}
  pStatsNum = (BMPMsgStatsNum *)rdata;
  iNum = be32toh (*pStatsNum);
  for (i=0; i<iNum; i++) {
    pTmp = rdata + iSize;
    pStatsHeader = (BMPMsgStatsHeader *)pTmp;
    /*========================================================================*/
    iSize += sizeof (BMPMsgStatsHeader);
    if (maxDataLength<iSize) {return 0;}
    /*========================================================================*/
    iSize += be16toh (pStatsHeader->statLen);
    if (maxDataLength<iSize) {return 0;}
  }
  return iSize;
}

/*+==========================================================================+**
 *: check BMP route monitoring messages size
 *: arg:
 *:    rdata        : bmp route monitoring  messages start point address.
 *:    maxDataLength: stored messages max data size.
 *: return:
 *:    plus value  : BMP route monitoring messages size.
 *:    zero        : data is not enough size for bmp messages.
 *:    minus value: error
 *+==========================================================================+*/
ssize_t bmpRouteMonSize (u_int8_t *rdata, const ssize_t maxDataLength)
{
  int             iSize;
  BGPCommonHeader *pBGPHeader;

  iSize = sizeof (BGPCommonHeader);
  /*==========================================================================*/
  if (maxDataLength<iSize) {return 0;} 
  pBGPHeader = (BGPCommonHeader *)rdata;
  iSize = be16toh(pBGPHeader->length);
  if (maxDataLength<iSize) {return 0;}
  return iSize;
}
