/* -*- c -*- */
/*
 * The MIT License
 *
 * Copyright (c) 2014 Yuki SAKAI
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

/*******************************************************************************
 * Include ********************************************************************/
#if !defined(SELFLIB_H__)
#  include <selfLib.h>
#endif
#if !defined(_SYS_PARAM_H) &&  !defined(_SYS_PARAM_H_)
#  include <sys/param.h>
#endif
#if !defined(_STRING_H) &&  !defined(_STRING_H_)
#  include <string.h>
#endif
#if !defined(_SYS_STAT_H) && !defined(_SYS_STAT_H_)
#  include <sys/stat.h>
#endif
#if !defined(_ERRNO_H) && !defined(_ERRNO_H_)
#  include <errno.h>
#endif

/*******************************************************************************
 * Get Last Block**************************************************************/
ssize_t getLastPathBlock (const char *dpath, char *lpath, ssize_t llen)
{
  const char *sp, *fp, *tp;
  /*=Search===================================================================*/
  sp = tp = dpath;
  for (fp=strstr (tp, "/"); fp!=NULL; fp=strstr (tp, "/")) {
    if      (tp==fp       ) {tp = fp + 1; sp = tp;}
    else if (*(fp-1)=='\\') {tp = fp + 1;}
    else                    {tp = fp + 1; sp = tp;}
  }
  /*=Copy=====================================================================*/
  return strncpy (lpath, sp, llen-1);
}

/*******************************************************************************
 * check directory path *******************************************************/
int chkdir (const char *path, mode_t mode)
{
  mode_t      nomod;
  struct stat dstat;
  nomod = (~ mode) & 0777;
  /*=Check Parent directory===================================================*/
  errno = 0;
  if (stat (path, &dstat)<0) {  /* no path process ****************************/
    switch (errno) {
    case EACCES: case EFAULT: case ELOOP: case ENAMETOOLONG: case ENOMEM: case ENOTDIR:
      return -1; break;
    }
    return 0;
  }
  else if (!S_ISDIR(dstat.st_mode)) {return -1;}        /* Directory check ****/
  else if ((dstat.st_mode&nomod))   {return -1;}        /* permission check ***/

  return 1;
}

/*******************************************************************************
 * Create directory ***********************************************************/
int mkdirs (const char *path, mode_t mode)
{
  struct stat dstat;
  char        updir[MAXPATHLEN], *pupd;
  const char  *ppath;
  /*=Init=====================================================================*/
  (void) memset ((char *)&dstat, 0, sizeof (struct stat));
  (void) memset ((char *)updir , 0, sizeof (updir      ));
  /*=Get Parent directory=====================================================*/
  for (pupd=updir, ppath=path; *ppath!=0; pupd++, ppath++) {*pupd = *ppath;}
  rmPathLastBlock (updir);

  /*=Check Parent directory===================================================*/
  if (stat (updir, &dstat)<0) { /* no path process ****************************/
    switch (errno) {
    case EACCES: case EFAULT: case ELOOP: case ENAMETOOLONG: case ENOMEM: case ENOTDIR:
      return 0; break;
    }
    return mkdirs (updir, mode);
  }
  /*=Mkdir====================================================================*/
  errno = 0;
  if (mkdir (path, mode)<0) {
    switch (errno) {
    case EACCES: case EFAULT: case ELOOP: case ENAMETOOLONG: case ENOMEM: case ENOSPC: case ENOTDIR: case EPERM: case EROFS:
      return -1; break;
    }
  }

  return 1;
}

/*******************************************************************************
 * remove Last Block***********************************************************/
ssize_t rmPathLastBlock (char *dpath)
{
  char *sp, *fp, *tp;

  rmPathLastSlash  (dpath);
  /*=Search===================================================================*/
  sp = tp = dpath;
  for (fp=strstr (tp, "/"); fp!=NULL; fp=strstr (tp, "/")) {
    if      (tp==fp       ) {tp = fp + 1; sp = tp;}
    else if (*(fp-1)=='\\') {tp = fp + 1;}
    else                    {tp = fp + 1; sp = tp;}
  }
  /*=Copy=====================================================================*/
  *sp = 0;
  if (strcpy (dpath, "/")!=0) {rmPathLastSlash  (dpath);}
  return strlen (dpath);
}

/*******************************************************************************
 * Remove Path Last slash *****************************************************/
ssize_t rmPathLastSlash (char *dpath)
{
  int len, i;
  packPathDupSlash (dpath);
  /*=Remove Last '/'==========================================================*/
  len = strlen (dpath);
  if (len==0) {return -1;}
  for (i=len-1; 0<=i; i--) {
    if (i!=0) {
      if      (*(dpath+i)!='/')           {break;}
      else if (0<i && *(dpath+i-1)=='\\') {break;}
      else                                {*(dpath+i) = 0x00;}
    }
  }

  return strlen (dpath);
}

/*******************************************************************************
 * Pack Duplicate Slash *******************************************************/
ssize_t packPathDupSlash (char *dpath)
{
  char *pntA, *pntB;
  int  find, skip;
  for (pntA=pntB=dpath,find=skip=0;;) {
    if      (skip==1               ) {skip = 0;}
    else if (*pntB=='\\'           ) {skip = 1; find = 0;}
    else if (*pntB=='/'  && find==0) {find = 1;}
    else if (*pntB=='/'  && find==1) {pntB++; continue;}
    else                             {skip = find = 0;}
    if ((*pntA++ = *pntB++)==0) {break;}
  }
  return strlen (dpath);
}
