/* 
 * The MIT License
 *
 * Copyright (c) 2014 Yuki SAKAI
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

/*******************************************************************************
 * include ********************************************************************/
#if !defined(SELFLIB_H__)
#  include <selfLib.h>
#endif
#if !defined(_SYS_PARAM_H) && !defined(_SYS_PARAM_H_)
#  include <sys/param.h>
#endif
#if !defined(_ARPA_INET_H) && !defined(_ARPA_INET_H_)
#  include <arpa/inet.h>
#endif
#if !defined(_NETINET_IN_H) && !defined(_NETINET_IN_H_)
#  include <netinet/in.h>
#endif
#if !defined(_SYS_SOCKET_H) && !defined(_SYS_SOCKET_H_)
#  include <sys/socket.h>
#endif
#if !defined(_STDIO_H) && !defined(_STDIO_H_)
#  include <stdio.h>
#endif
#if !defined(_STDARG_H) && !defined(_STDARG_H_)
#  include <stdarg.h>
#endif
#if !defined(_STRING_H) && !defined(_STRING_H_)
#  include <string.h>
#endif
#if !defined(_STDLIB_H_) && !defined(_STDLIB_H)
#  include <stdlib.h>
#endif
#if !defined(_UNISTD_H) && !defined(_UNISTD_H_)
#  include <unistd.h>
#endif
#if !defined(_TIME_H) && !defined(_TIME_H_)
#  include <time.h>
#endif
#if !defined(_SYS_TYPES_H_) && !defined(_SYS_TYPES_H)
#  include <sys/types.h>
#endif
#if !defined(_SYS_STAT_H_) && !defined(_SYS_STAT_H)
#  include <sys/stat.h>
#endif


/*******************************************************************************
 * define *********************************************************************/
#define _YLOGMAXDATE 30
#define _YLOGMAXIP   64
#define _YLOGMAXLOG  512


/*******************************************************************************
 * socket *********************************************************************/
static void netWriteLog (FILE *w, const char *d, const char *p, const char *r, const char *s)
{
  fprintf (w, "%s %s: Peer %s: %s\n", d, p, r, s);
}

/*******************************************************************************
 * no socket ******************************************************************/
static void nonetWriteLog (FILE *w, const char *d, const char *p, const char *r, const char *s)
{
  fprintf (w, "%s %s: %s\n", d, p, s);
}

/*******************************************************************************
 * Date for log ***************************************************************/
int dateLog (char *date, const int size)
{
  time_t    tv;
  struct tm *tvt;
  tv = time (NULL);
  tvt = localtime (&tv);
  (void) memset (date, 0, size);
  strftime (date, size-1, "%Y/%m/%d %H:%M:%S(%Z)", tvt);

  return 1;
}

/*******************************************************************************
 * Process ********************************************************************/
static int processLog (const char *pgn, char *pstr, const int size)
{
  (void) memset (pstr, 0, size);
  snprintf (pstr, size, "%s(%d)", pgn, (int)getpid ());

  return 1;
}

/*******************************************************************************
 * Remote *********************************************************************/
static int remoteLog (const int soc, char *pstr, const int size)
{
  struct sockaddr_storage peerAddr;
  socklen_t               namelen;
  struct sockaddr_in      *ppF;
  struct sockaddr_in6     *ppS;
  int                     port;
  char                    strRemote[_YLOGMAXIP];
  (void) memset (strRemote, 0, sizeof (strRemote));
  (void) memset (pstr     , 0, size);
  (void) memset (&peerAddr, 0, sizeof (struct sockaddr_storage));
  /*=Get IP address ==========================================================*/
  namelen = sizeof (struct sockaddr_storage);
  if (getpeername (soc, (struct sockaddr *)&peerAddr, &namelen)<0) {return 0;}
  if (peerAddr.ss_family == AF_INET) {
    ppF = (struct sockaddr_in *)&peerAddr;
    port = ntohs (ppF->sin_port);
    inet_ntop (AF_INET, &ppF->sin_addr, strRemote, sizeof (strRemote));
  }
  else if (peerAddr.ss_family == AF_INET6) {
    ppS = (struct sockaddr_in6 *)&peerAddr;
    port = ntohs (ppS->sin6_port);
    inet_ntop (AF_INET6, &ppS->sin6_addr, strRemote, sizeof (strRemote));
  }
  else {return 0;}
  snprintf (pstr, size, "%s(%d)", strRemote, port);

  return 1;
}

/*******************************************************************************
 * write syslog ***************************************************************/
void writeLog (const char *path, const char *prog, const int fd, const char *fmt, ...)
{
  FILE    *wfd;
  char    strdate[_YLOGMAXDATE];
  char    strPrs[MAXPATHLEN];
  char    strRemote[NI_MAXHOST];
  char    *jmp, *ser;
  char    *strLog;
  va_list ap;
  void    (*wfunc) (FILE *, const char*, const char*, const char*, const char*);
  mode_t  mode;

  if (fd<0) {wfunc = nonetWriteLog;}
  else      {wfunc = netWriteLog;}
  /*=Get Infomation===========================================================*/
  if      (!processLog (prog, strPrs   , sizeof (strPrs   ))) {goto end_proc;}
  else if (!dateLog    (strdate        , sizeof (strdate  ))) {goto end_proc;}
  else if (fd<0) {}
  else if (!remoteLog  (fd  , strRemote, sizeof (strRemote))) {goto end_proc;}
  /*=Get Log==================================================================*/
  va_start(ap, fmt);
  if      ((strLog=(char *)malloc (_YLOGMAXLOG))==NULL) {goto fin_va;}
  else if (vsnprintf (strLog, _YLOGMAXLOG, fmt, ap)<0 ) {goto free_data;}
  va_end(ap);
  /*=Open File================================================================*/
  mode = umask (022);
  if      (strcmp (path, "stdout")==0)    {wfd = stdout;}
  else if (strcmp (path, "stderr")==0)    {wfd = stderr;}
  else if ((wfd=fopen (path, "a"))==NULL) {goto free_data2;}

  /*=write Log================================================================*/
  ser = strLog;
  for (jmp=strstr (ser, "\n"); ; jmp=strstr (ser, "\n")) {
    if      (jmp==NULL ) {}
    else if (*jmp=='\n') {*jmp = 0x00;}
    wfunc (wfd, strdate, strPrs, strRemote, ser);
    if (jmp==NULL) {break;}
    ser = jmp + 1;
  }

  if      (strcmp (path, "stdout")==0) {}
  else if (strcmp (path, "stderr")==0) {}
  else                                 {fclose (wfd);}

 free_data2:
  umask (mode);
  free (strLog);
  return ;


 free_data:
  free (strLog);
 fin_va:
  va_end(ap);
 end_proc:
  return ;
}
