/* 
 * The MIT License
 *
 * Copyright (c) 2014 Yuki SAKAI
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

/*******************************************************************************
 * include ********************************************************************/
#if !defined(BMP_STATION_H__)
# include <bmpStation.h>
#endif
#if !defined(BMP_STATION_INT_H__)
# include <bmpStationInt.h>
#endif
#if !defined(SELFLIB_H__)
#  include <selfLib.h>
#endif
#if !defined(_SYS_TYPES_H) && !defined(_SYS_TYPES_H_)
#  include <sys/types.h>
#endif
#if !defined(_SYS_SELECT_H) && !defined(_SYS_SELECT_H_)
#  include <sys/select.h>
#endif
#if !defined(_SYS_WAIT_H) && !defined(_SYS_WAIT_H_)
#  include <sys/wait.h>
#endif
#if !defined(_SYS_SOCKET_H) && !defined(_SYS_SOCKET_H_)
#  include <sys/socket.h>
#endif
#if !defined(_STDIO_H) && !defined(_STDIO_H_)
#  include <stdio.h>
#endif
#if !defined(_UNISTD_H) && !defined(_UNISTD_H_)
#  include <unistd.h>
#endif
#if !defined(_STDLIB_H) && !defined(_STDLIB_H_)
#  include <stdlib.h>
#endif
#if !defined(_STRING_H) && !defined(_STRING_H_)
#  include <string.h>
#endif
#if !defined(_SIGNAL_H) && !defined(_SIGNAL_H_)
#  include <signal.h>
#endif
#if !defined(_ERRNO_H) && !defined(_ERRNO_H_)
#  include <errno.h>
#endif

/*******************************************************************************
 * variables ******************************************************************/
static int       tsoc;         /* TCP Listen socket ***************************/

/*******************************************************************************
 * functions ******************************************************************/
static int  opt_proc   (int, char **);
static void usage      ();
static void hangup     (int);
static void wait_child (int);

/*+==========================================================================+**
 *| hang up
 *+==========================================================================+*/
static void hangup (int pid)
{
  close (tsoc);
  
  exit (0);
}


/*%''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''%**
 *: main process                                                             :**
 *%,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,%*/
int main (int argc, char **argv)
{
  /*****************************************************************************
   * Variables ****************************************************************/
  pid_t  pid;
  /* select */
  struct timeval itime, utime;
  int            fdw  , ret;
  fd_set         iset , uset ; /* iset is initialize, uset for select */
  /* socket */
  int            asoc;         /* accept socket */
  /*****************************************************************************
   * option process ***********************************************************/
  if (!opt_proc (argc, argv)) {goto JUMP_OPT;}

  if (FLGISSET(lastParam.flags, OPTFLAGDAEMONIZE)) {
    if (daemon (0, 0)<0) {
      if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {
	writeLog (syslogFile, prgName, -1, "Fault daemonize");
      }
      goto JUMP_OPT;
    }
    if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {
      writeLog (syslogFile, prgName, -1, "daemonize");
    }
  }

  /*=Signal setup=============================================================*/
  if      (signal (SIGHUP , hangup    )==SIG_ERR) {
    if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {writeLog (syslogFile, prgName, -1, "Failure signal setup: SIGHUP.");}
    goto JUMP_OPT;
  }
  else if (signal (SIGINT , hangup    )==SIG_ERR) {
    if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {writeLog (syslogFile, prgName, -1, "Failure signal setup: SIGINT.");}
    goto JUMP_OPT;
  }
  else if (signal (SIGTERM, hangup    )==SIG_ERR) {
    if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {writeLog (syslogFile, prgName, -1, "Failure signal setup: SIGTERM.");}
    goto JUMP_OPT;
  }
  else if (signal (SIGCHLD, wait_child)==SIG_ERR) {
    if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {writeLog (syslogFile, prgName, -1, "Failure signal setup: SIGCHLD.");}
    goto JUMP_OPT;
  }
  if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {
    writeLog (syslogFile, prgName, -1, "signal setup");
  }
  /*****************************************************************************
   * Socket Listen ************************************************************/
  /*=Socket open==============================================================*/
  if ((tsoc=lsocket (lastParam.host, lastParam.ports, SOCK_STREAM, 0))<0) {
    if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {
      writeLog (syslogFile, prgName, -1, "fault listen socket open.");
    }
    goto JUMP_OPT;
  }
  if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {
    writeLog (syslogFile, prgName, -1, "listen socket open.");
  }
  /*=Select setup=============================================================*/
  fdw = 0;
  FD_ZERO(&iset);
  FD_SET(tsoc, &iset); fdw = tsoc;
  fdw++;
  (void) memset (&itime, 0, sizeof (struct timeval));
  itime = d2tval (10);

  /*=TCP Listen===============================================================*/
  while (1) {
    utime = itime; uset = iset;
    ret = select (fdw, &uset, NULL, NULL, &utime);
    if      (ret< 0) {
      switch (errno) {
      case EBADF:  break;
      case EINTR:  continue; break;
      case EINVAL: break;
      case ENOMEM: break;
      }
      if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {
	writeLog (syslogFile, prgName, -1, "select error.");
      }
      goto ERR_CLOSE_SOC;
    }
    else if (ret==0) {continue ;}
    else if (FD_ISSET(tsoc, &uset)) {
      /*=main BMP proc========================================================*/ 
      if ((asoc=accept (tsoc, NULL, NULL))<0) {
	switch (errno) {
	case EMFILE:
	  if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {
	    writeLog (syslogFile, prgName, -1, "limit tcp session.");
	  }
	case EINTR:
	  continue;
	  break;
	case ENOTSOCK:      case EOPNOTSUPP:  case EPROTO:      case EINVAL:
	case EFAULT:        case EBADF:       case EWOULDBLOCK:
	case ECONNABORTED:  case ENOBUFS:     case ENOMEM:
	default:
	  if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {
	    writeLog (syslogFile, prgName, -1, "Listen socket is error.");
	  }
	  goto ERR_CLOSE_SOC;
	}
      }
      if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {
	writeLog (syslogFile, prgName, asoc, "TCP session is open.");
      }

      pid = fork ();
      if (pid<0) {
	if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {
	  writeLog (syslogFile, prgName, asoc, "Fault new process(fork).");
	}
	close (asoc);
	goto ERR_CLOSE_SOC;
      }
      else if (pid==0) {
	/* main process */
	if (!bmpRead (asoc)) {
	  close (asoc);
	  _exit(1);
	}
	if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {
	  writeLog (syslogFile, prgName, asoc, "Close session.");
	}
	close (asoc);
	_exit (0);
      }
      close (asoc);
    }
  }
  /*=Finish Process===========================================================*/
  close (tsoc);
  
  if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {
    writeLog (syslogFile, prgName, -1, "Close processes.");
  }

  return 0;

 ERR_CLOSE_SOC:
  close (tsoc);
 JUMP_OPT:
  return 1;
}

/*******************************************************************************
 * wait child *****************************************************************/
static void wait_child (int pid)
{
  pid_t _cpid;
#if 0
  int   id;
#endif

  _cpid = wait (NULL);
#if 0
  if (FLGISSET(loglevel, LOGLVLSTATES2)) {
    if (id<0) {/* error proc */}
    else      {/* error proc */}
  }
#endif
  while (SIG_ERR==signal (SIGCHLD, wait_child)) {
    /* error proc */
  }

}

/*******************************************************************************
 * usage **********************************************************************/
static void usage ()
{
  char spaPad[MAXPATHLEN]; /* space padding */
  (void) memset (spaPad, 0  , sizeof (spaPad));
  (void) memset (spaPad, ' ', (strlen (prgName)<sizeof (spaPad)?strlen (prgName):sizeof (spaPad)-1));
  /*=Usage main===============================================================*/
  fprintf (stdout, "%s [\033[1m-H\033[0m] [\033[1m-d\033[0m|\033[1m-D\033[0m] ", prgName);
  fprintf (stdout, " [\033[1m-c\033[0m|\033[1m-C\033[0m] [\033[1m-e\033[0m|\033[1m-E\033[0m] [\033[1m-n\033[0m|\033[1m-N \033[4mListen address\033[0m\033[0m]\n");
  fprintf (stdout, "%s [\033[1m-P \033[4mListen Port\033[0m\033[0m] [\033[1m-W \033[4mWork dirctory\033[0m\033[0m]", spaPad);
  fprintf (stdout, " [\033[1m-A \033[4mAccept TCP Number\033[0m\033[0m]\n");
  fprintf (stdout, "%s -H                   : help(this display)\n", spaPad);
  fprintf (stdout, "%s -[D               |d]: (set|unset) debug mode\n", spaPad);
  fprintf (stdout, "%s -[C               |c]: (set|unset) debug mode\n", spaPad);
  fprintf (stdout, "%s -[E               |e]: (set|unset) damonized mode\n", spaPad);
  fprintf (stdout, "%s -[N \033[4mListen address\033[0m\033[0m|n]: (set|unset) Listen address\n", spaPad);
  fprintf (stdout, "%s -[P \033[4mListen Port\033[0m\033[0m     ]: set Listen Port\n", spaPad);
  fprintf (stdout, "%s -W \033[4mWork Directory\033[0m\033[0m    : set Work directory\n", spaPad);
  fprintf (stdout, "%s -A \033[4mAccept TCP Number\033[0m\033[0m : set Accept TCP Number\n", spaPad);
  
  exit (1);
}

/*******************************************************************************
 * option processes ***********************************************************/
static int opt_proc (int argc, char **argv)
{
  int  ch, rtn;
  char uhdir[MAXPATHLEN]; /* user home directory */
  char curdir[MAXPATHLEN]; /* current directory */
  char *ppname;
  mode_t wdirmode;

  ppname = prgName = *argv;
  while ((ppname=strstr (ppname, "/"))!=NULL) {
    ppname++;
    if (prgName == (ppname-1))    {prgName = ppname;}
    else if (*(ppname - 2) != 92) {prgName = ppname;}
  }
  /*****************************************************************************
   * initialize ***************************************************************/
  wdirmode = 0755;
  (void) memset ((char *)uhdir     , 0, sizeof (uhdir    ));
  (void) memset ((char *)curdir    , 0, sizeof (curdir   ));
  (void) memset ((char *)&defParam , 0, sizeof (defParam ));
  (void) memset ((char *)&cfgParam , 0, sizeof (cfgParam ));
  (void) memset ((char *)&argParam , 0, sizeof (argParam ));
  (void) memset ((char *)&lastParam, 0, sizeof (lastParam));
  ruid = getuid  (); /* real user */
  euid = geteuid (); /* exec user */
  if (getcwd (curdir, sizeof (curdir))==NULL) {
    fprintf (stderr, "Fault current dir\n");
    return 0;
  }

  /*****************************************************************************
   * set default **************************************************************/
  FLGSET(defParam.enableConfigFlag, OPTENABLECFLAGLPORT);
  strncpy (defParam.ports, DEFLPORT, sizeof (((optionParam *)NULL)->ports)-1);
  snprintf (defParam.ports, sizeof (((optionParam *)NULL)->ports)-1, "%s", DEFLPORT);

  FLGSET(defParam.enableConfigFlag, OPTENABLECFLAGLHOST);
  (void) memset (defParam.host, 0, sizeof (((optionParam *)NULL)->host));
  snprintf (defParam.host, sizeof (((optionParam *)NULL)->host)-1, "%s", DEFLHOST);

  FLGSET(defParam.enableConfigFlag, OPTENABLECFLAGDEBUG);  FLGCLR(defParam.flags, OPTFLAGDEBUG);
  FLGSET(defParam.enableConfigFlag, OPTENABLECFLAGCHROOT); FLGCLR(defParam.flags, OPTFLAGCHROOT);
  FLGSET(defParam.enableConfigFlag, OPTENABLECFLAGTCPACCEPTNUM);
  defParam.acceptNum = DEFTCPACCEPTNUM;
  FLGSET(defParam.enableConfigFlag, OPTENABLECFLAGLOGLEVEL);
  defParam.loglevel = DEFLOGLVL;

  GetHomeDir (ruid, uhdir, sizeof (uhdir));
  FLGSET(defParam.enableConfigFlag, OPTENABLECFLAGWORKDIR);
  snprintf (defParam.workdir, sizeof (((optionParam *)NULL)->workdir)-1, "%s/%s", uhdir, DEFWORKDIR);
  

  /*****************************************************************************
   * argument process *********************************************************/
  while ((ch = getopt (argc, argv, "A:cCdDeEHl:nN:P:uU:W:"))!=-1) {
    switch (ch) {
    case 'l': /* set log level */
      if (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGLOGLEVEL)) break;
      FLGSET(defParam.enableConfigFlag, OPTENABLECFLAGLOGLEVEL);
      defParam.loglevel = atoi (optarg);
      break;
    case 'D': /* set   Debug */
      if (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGDEBUG)) break;
      FLGSET(argParam.enableConfigFlag, OPTENABLECFLAGDEBUG);
      FLGSET(argParam.flags           , OPTFLAGDEBUG);
      break;
    case 'd': /* unset Debug */
      if (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGDEBUG)) break;
      FLGSET(argParam.enableConfigFlag, OPTENABLECFLAGDEBUG);
      FLGCLR(argParam.flags           , OPTFLAGDEBUG);
      break;
    case 'C': /* set   Chroot */
      if (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGCHROOT)) break;
      FLGSET(argParam.enableConfigFlag, OPTENABLECFLAGCHROOT);
      FLGSET(argParam.flags           , OPTFLAGCHROOT);
      break;
    case 'c': /* unset Chroot */
      if (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGCHROOT)) break;
      FLGSET(argParam.enableConfigFlag, OPTENABLECFLAGCHROOT);
      FLGCLR(argParam.flags           , OPTFLAGCHROOT);
      break;
    case 'N': /* set   Hostname */
      if (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGLHOST)) break;
      FLGSET(argParam.enableConfigFlag, OPTENABLECFLAGLHOST);
      strncpy (argParam.host, optarg, sizeof (((optionParam *)NULL)->host)-1);
      break;
    case 'n': /* unset Hostname */
      if (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGLHOST)) break;
      FLGSET(argParam.enableConfigFlag, OPTENABLECFLAGLHOST);
      (void) memset (argParam.host, 0, sizeof (((optionParam *)NULL)->host));
      break;
    case 'P': /* set   Service(Port Number) */
      if (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGLPORT)) break;
      FLGSET(argParam.enableConfigFlag, OPTENABLECFLAGLPORT);
      strncpy (argParam.ports, optarg, sizeof (((optionParam *)NULL)->ports)-1);
      break;
    case 'W': /* set   workdir */
      if (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGWORKDIR)) break;
      FLGSET(argParam.enableConfigFlag, OPTENABLECFLAGWORKDIR);
      if (*optarg == '/') {
	strncpy (argParam.workdir, optarg, sizeof (((optionParam *)NULL)->workdir)-1);
      } else {
	snprintf (argParam.workdir, sizeof (((optionParam *)NULL)->workdir)-1, "%s/%s", curdir, optarg);
      }
      break;
    case 'E': /* set   daemonize */
      if (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGDAEMONIZE)) break;
      FLGSET(argParam.enableConfigFlag, OPTENABLECFLAGDAEMONIZE);
      FLGSET(argParam.flags           , OPTFLAGDAEMONIZE);
      break;
    case 'e': /* unset daemonize */
      if (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGDAEMONIZE)) break;
      FLGSET(argParam.enableConfigFlag, OPTENABLECFLAGDAEMONIZE);
      FLGCLR(argParam.flags           , OPTFLAGDAEMONIZE);
      break;
    case 'A': /* set TCP Accept Number */
      if (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGTCPACCEPTNUM)) break;
      FLGSET(argParam.enableConfigFlag, OPTENABLECFLAGTCPACCEPTNUM);
      argParam.acceptNum = atoi (optarg);
      break;
    case 'H': /* Help */
    default:
      usage ();
      return 0;
      break;
    }
  }
  argc -= optind;
  argv += optind;

  /*****************************************************************************
   * Last Option **************************************************************/
  /*=debug Option=*/
  if        (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGDEBUG)) {
    FLGSET(lastParam.enableConfigFlag, OPTENABLECFLAGDEBUG);
    lastParam.flags |= argParam.flags & OPTFLAGDEBUG;
  } else if (FLGISSET(cfgParam.enableConfigFlag, OPTENABLECFLAGDEBUG)) {
    FLGSET(lastParam.enableConfigFlag, OPTENABLECFLAGDEBUG);
    lastParam.flags |= cfgParam.flags & OPTFLAGDEBUG;
  } else {
    lastParam.enableConfigFlag |= defParam.enableConfigFlag & OPTENABLECFLAGDEBUG;
    lastParam.flags            |= defParam.flags & OPTFLAGDEBUG;
  }
  /*=chroot Option=*/
  if        (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGCHROOT)) {
    FLGSET(lastParam.enableConfigFlag, OPTENABLECFLAGCHROOT);
    lastParam.flags |= argParam.flags & OPTFLAGCHROOT;
  } else if (FLGISSET(cfgParam.enableConfigFlag, OPTENABLECFLAGCHROOT)) {
    FLGSET(lastParam.enableConfigFlag, OPTENABLECFLAGCHROOT);
    lastParam.flags |= cfgParam.flags & OPTFLAGCHROOT;
  } else {
    lastParam.enableConfigFlag |= defParam.enableConfigFlag & OPTENABLECFLAGCHROOT;
    lastParam.flags            |= defParam.flags & OPTFLAGCHROOT;
  }
  /*=daemonize Option=*/
  if        (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGDAEMONIZE)) {
    FLGSET(lastParam.enableConfigFlag, OPTENABLECFLAGDAEMONIZE);
    lastParam.flags |= argParam.flags & OPTFLAGDAEMONIZE;
  } else if (FLGISSET(cfgParam.enableConfigFlag, OPTENABLECFLAGDAEMONIZE)) {
    FLGSET(lastParam.enableConfigFlag, OPTENABLECFLAGDAEMONIZE);
    lastParam.flags |= cfgParam.flags & OPTFLAGDAEMONIZE;
  } else {
    lastParam.enableConfigFlag |= defParam.enableConfigFlag & OPTENABLECFLAGDAEMONIZE;
    lastParam.flags            |= defParam.flags & OPTFLAGDAEMONIZE;
  }
  /*=workdir Option=*/
  if        (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGWORKDIR)) {
    FLGSET(lastParam.enableConfigFlag, OPTENABLECFLAGWORKDIR);
    strncpy (lastParam.workdir, argParam.workdir, sizeof (((optionParam *)NULL)->workdir)-1);
  } else if (FLGISSET(cfgParam.enableConfigFlag, OPTENABLECFLAGWORKDIR)) {
    FLGSET(lastParam.enableConfigFlag, OPTENABLECFLAGWORKDIR);
    strncpy (lastParam.workdir, cfgParam.workdir, sizeof (((optionParam *)NULL)->workdir)-1);
  } else {
    lastParam.enableConfigFlag |= defParam.enableConfigFlag & OPTENABLECFLAGWORKDIR;
    strncpy (lastParam.workdir, defParam.workdir, sizeof (((optionParam *)NULL)->workdir)-1);
  }
  /*=host Option=*/
  if        (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGLHOST)) {
    FLGSET(lastParam.enableConfigFlag, OPTENABLECFLAGLHOST);
    strncpy (lastParam.host, argParam.host, sizeof (((optionParam *)NULL)->host)-1);
  } else if (FLGISSET(cfgParam.enableConfigFlag, OPTENABLECFLAGLHOST)) {
    FLGSET(lastParam.enableConfigFlag, OPTENABLECFLAGLHOST);
    strncpy (lastParam.host, cfgParam.host, sizeof (((optionParam *)NULL)->host)-1);
  } else {
    lastParam.enableConfigFlag |= defParam.enableConfigFlag & OPTENABLECFLAGLHOST;
    strncpy (lastParam.host, defParam.host, sizeof (((optionParam *)NULL)->host)-1);
  }
  /*=port Option=*/
  if        (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGLPORT)) {
    FLGSET(lastParam.enableConfigFlag, OPTENABLECFLAGLPORT);
    strncpy (lastParam.ports, argParam.ports, sizeof (((optionParam *)NULL)->ports)-1);
  } else if (FLGISSET(cfgParam.enableConfigFlag, OPTENABLECFLAGLPORT)) {
    FLGSET(lastParam.enableConfigFlag, OPTENABLECFLAGLPORT);
    strncpy (lastParam.ports, cfgParam.ports, sizeof (((optionParam *)NULL)->ports)-1);
  } else {
    lastParam.enableConfigFlag |= defParam.enableConfigFlag & OPTENABLECFLAGLPORT;
    strncpy (lastParam.ports, defParam.ports, sizeof (((optionParam *)NULL)->ports)-1);
  }
  /*=accept number Option=*/
  if        (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGTCPACCEPTNUM)) {
    FLGSET(lastParam.enableConfigFlag, OPTENABLECFLAGTCPACCEPTNUM);
    lastParam.acceptNum = argParam.acceptNum;
  } else if (FLGISSET(cfgParam.enableConfigFlag, OPTENABLECFLAGLPORT)) {
    FLGSET(lastParam.enableConfigFlag, OPTENABLECFLAGTCPACCEPTNUM);
    lastParam.acceptNum = cfgParam.acceptNum;
  } else {
    lastParam.enableConfigFlag |= defParam.enableConfigFlag & OPTENABLECFLAGTCPACCEPTNUM;
    lastParam.acceptNum = defParam.acceptNum;
  }
  /*=log level Option=*/
  if        (FLGISSET(argParam.enableConfigFlag, OPTENABLECFLAGLOGLEVEL)) {
    FLGSET(lastParam.enableConfigFlag, OPTENABLECFLAGLOGLEVEL);
    lastParam.loglevel = argParam.loglevel;
  } else if (FLGISSET(cfgParam.enableConfigFlag, OPTENABLECFLAGLOGLEVEL)) {
    FLGSET(lastParam.enableConfigFlag, OPTENABLECFLAGLOGLEVEL);
    lastParam.loglevel = cfgParam.loglevel;
  } else {
    lastParam.enableConfigFlag |= defParam.enableConfigFlag & OPTENABLECFLAGLOGLEVEL;
    lastParam.loglevel          = defParam.loglevel;
  }

  /*****************************************************************************
   * Setup ********************************************************************/
  /*=Chuser===================================================================*/
  /*=Create workdir===========================================================*/
  if ((rtn=chkdir (lastParam.workdir, wdirmode))<0) {
    fprintf (stderr, "Can't access the workdir: %s", lastParam.workdir);
    return 0;
  }
  else if (rtn==1) {}
  else if ((rtn=mkdirs (lastParam.workdir, wdirmode))<0) {
    fprintf (stderr, "Can't create the workdir: %s", lastParam.workdir);
    return 0;
  }
  else if (rtn==0)                             {
    fprintf (stderr, "Can't create the workdir: %s", lastParam.workdir);
    return 0;
  }
  /*=Chroot workdir===========================================================*/
  if (FLGISSET(lastParam.flags, OPTFLAGCHROOT)) {
    if (chroot (lastParam.workdir)<0) {
      fprintf (stderr, "Can't chroot to workdir: %s", lastParam.workdir);
      return 0;
    }
    else if (chdir ("/")<0) {
      fprintf (stderr, "Can't change to workdir: %s", lastParam.workdir);
      return 0;
    }
    snprintf (lastParam.workdir, sizeof (((optionParam *)NULL)->workdir)-1, "/");
  }
  else if (chdir (lastParam.workdir)<0) {
    fprintf (stderr, "Can't change to workdir: %s", lastParam.workdir);
    return 0;
  }
  /*=System log===============================================================*/
  (void) memset (syslogFile    , 0, sizeof (syslogFile    ));
  (void) memset (statisticsFile, 0, sizeof (statisticsFile));
  (void) memset (peerEventLog  , 0, sizeof (peerEventLog  ));
  snprintf (syslogFile    , sizeof (syslogFile    )-1, "%s/%s", lastParam.workdir, DEFSYSLOG       );
  snprintf (statisticsFile, sizeof (statisticsFile)-1, "%s/%s", lastParam.workdir, DEFSTATISTICSLOG);
  snprintf (peerEventLog  , sizeof (peerEventLog  )-1, "%s/%s", lastParam.workdir, DEFEVENTLOG     );

  if (FLGISSET(lastParam.flags, OPTFLAGDEBUG)) {
    printf ("real user = %d\n", ruid);
    printf ("exec user = %d\n", euid);
    printf ("debug  : %d\n", FLGISSET(lastParam.flags, OPTFLAGDEBUG));
    printf ("chroot : %d\n", FLGISSET(lastParam.flags, OPTFLAGCHROOT));
    printf ("daemon : %d\n", FLGISSET(lastParam.flags, OPTFLAGDAEMONIZE));
    printf ("host   : %s\n", lastParam.host);
    printf ("ports  : %s\n", lastParam.ports);
    printf ("accept#: %d\n", lastParam.acceptNum);
    printf ("workdir: %s\n", lastParam.workdir);
    printf ("log    : %s\n", syslogFile);
    printf ("stati  : %s\n", statisticsFile);
    printf ("event  : %s\n", peerEventLog);
  }
  
  if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {
    writeLog (syslogFile, prgName, -1, "initialize");
  }

  return 1;
}


