/* 
 * The MIT License
 *
 * Copyright (c) 2014 Yuki SAKAI
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

/*******************************************************************************
 * include ********************************************************************/
#if !defined(BMP_STATION_H__)
# include <bmpStation.h>
#endif
#if !defined(BMP_STATION_INT_H__)
# include <bmpStationInt.h>
#endif
#if !defined(SELFLIB_H__)
#  include <selfLib.h>
#endif
#if !defined(BMP_H__)
#  include <bmp.h>
#endif
#if !defined(BGP_H__)
#  include <bgp.h>
#endif
#if !defined(_ARPA_INET_H) && !defined(_ARPA_INET_H_)
#  include <arpa/inet.h>
#endif
#if !defined(_NETINET_IN_H) && !defined(_NETINET_IN_H_)
#  include <netinet/in.h>
#endif
#if !defined(_NETINET_TCP_H) && !defined(_NETINET_TCP_H_)
#  include <netinet/tcp.h>
#endif
#if !defined(_SYS_SOCKET_H) && !defined(_SYS_SOCKET_H_)
#  include <sys/socket.h>
#endif
#if !defined(_SYS_TYPES_H) && !defined(_SYS_TYPES_H_)
#  include <sys/types.h>
#endif
#if !defined(_SYS_TIME_H) && !defined(_SYS_TIME_H_)
#  include <sys/time.h>
#endif
#if !defined(_SYS_FILE_H) && !defined(_SYS_FILE_H_)
#  include <sys/file.h>
#endif
#if !defined(_SYS_SELECT_H) && !defined(_SYS_SELECT_H_)
#  include <sys/select.h>
#endif
#if !defined(_STRING_H) && !defined(_STRING_H_)
#  include <string.h>
#endif
#if !defined(_UNISTD_H) && !defined(_UNISTD_H_)
#  include <unistd.h>
#endif
#if !defined(_STDLIB_H) && !defined(_STDLIB_H_)
#  include <stdlib.h>
#endif
#if !defined(_FCNTL_H) && !defined(_FCNTL_H_)
#  include <fcntl.h>
#endif
#if !defined(_ERRNO_H) && !defined(_ERRNO_H_)
#  include <errno.h>
#endif
#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
#if !defined(_SYS_ENDIAN_H) && !defined(_SYS_ENDIAN_H_)
#  include <sys/endian.h>
#endif
#else
#if !defined(_ENDIAN_H) && !defined(_ENDIAN_H_)
#  include <endian.h>
#endif
#endif

#if 1
#if !defined(_STDIO_H) && !defined(_STDIO_H_)
#  include <stdio.h>
#endif
#endif


/*******************************************************************************
 * Global variables ***********************************************************/
static u_int8_t *pmsgStore;     /* recv messages Data store */
static ssize_t  imsgStoreSize;  /* pmsgStore size */
static u_int8_t *pmsgBondStore; /* recv messages bondig Data store */
static u_int8_t *pmsgBondLast;  /* pmsgBond 1st nodata point */
static u_int8_t *pmsgBondRead;  /* pmsgBond read point */
static ssize_t  iBondStoreSize; /* pmsgBondStore size */
static int      soc4log;
/*=Peer infomation============================================================*/
#define NEW 1
#if NEW
char                    *pStrRemote = NULL;
#endif

/*******************************************************************************
 * functions ******************************************************************/
static int getBMPLogInfo (BMPLogInfoSet *, u_int8_t *, ssize_t);


/* old */
static void HEXoutput (u_int8_t *, int, int); /* debug code */
static void debugBMPinfo (BMPLogInfoSet *); /* debug code */
static int BMPRecvData (int);
#if 1
static int BMPv1Data (BMPLogInfoSet *, u_int8_t *, ssize_t);
static int BMPv3Data (BMPLogInfoSet *, u_int8_t *, ssize_t);
#else
static int BMPv1Data (u_int8_t *, ssize_t);
static int BMPv3Data (u_int8_t *, ssize_t);
#endif

static int BMPInitiation         (u_int8_t *, ssize_t);
static int BMPTermination        (u_int8_t *, ssize_t);
static int BMPRouteMonitoring    (BMPLogInfoSet *, u_int8_t *, ssize_t);
static int BMPRouteMonitoringLog (BMPLogInfoSet *, char *, u_int8_t *, ssize_t);
static int BMPStatisticsData     (BMPLogInfoSet *, u_int8_t *, ssize_t);
static int BMPPeerDown           (BMPLogInfoSet *, u_int8_t *, ssize_t, int);
static int BMPPeerUp             (BMPLogInfoSet *, u_int8_t *, ssize_t, int);

static int getBMPPeerAddr     (BMPLogInfoSet *, char *, const int);

/*******************************************************************************
 * main ***********************************************************************/
int bmpRead (int soc)
{
  int res;
  /* socket */
  int     maxseg; /* TCP MSS size */
  int     optlen; /* maxseg size */
  /* select */
  struct timeval itime, utime;
  int            fdw  , ret;
  fd_set         iset , uset ; /* iset is initialize, uset for select */

  /*=initialize===============================================================*/
  /* get Remote information */
  if ((pStrRemote = imalloc (NI_MAXHOST))==NULL) {
    if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {
      writeLog (syslogFile, prgName, soc, "Fault memory allocate for remote hostname.");
    }
  }
  else if (!getRemoteHost (soc, pStrRemote, NI_MAXHOST)) {
    if (FLGISSET(lastParam.loglevel, LOGLBL_STATS)) {
      writeLog (syslogFile, prgName, soc, "Fault get remote hostname.");
    }
    goto free_remote_name;
  }
#if 1
  soc4log = soc;
#endif
  /* get TCP MSS size */
  optlen=sizeof(maxseg);
  if (getsockopt(soc,IPPROTO_TCP,TCP_MAXSEG, &maxseg, (socklen_t *)&optlen)<0) {
    writeSystemLog (LOGLBL_STATS, pStrRemote, "Fault get TCP MSS.");
    goto free_remote_name;
  }
  /* memory allocate imsgStore */
  imsgStoreSize = maxseg;
  if ((pmsgStore = malloc (imsgStoreSize))==NULL) {
    writeSystemLog (LOGLBL_STATS, pStrRemote, "Fault memory allocate for message store.");
    goto free_remote_name;
  }
  /* memory allocate imsgBondStore */
  iBondStoreSize = LIBBMP_MAXHEADERLEN + LIBBGP_MAXLENGTH + maxseg;
  if ((pmsgBondStore = imalloc (iBondStoreSize))==NULL) {
    writeSystemLog (LOGLBL_STATS, pStrRemote, "Fault memory allocate for message bond store.");
    free (pmsgStore);
    goto free_remote_name;
  }
  pmsgBondRead = pmsgBondLast = pmsgBondStore;
  /* select setup */
  fdw = 0;
  FD_ZERO(&iset);
  FD_SET(soc, &iset); fdw = soc;
  fdw++;
  (void) memset (&itime, 0, sizeof (struct timeval));
  itime = d2tval (10);

  writeSystemLog (LOGLBL_STATS, pStrRemote, "Co-processes initialized.");
  /*=Read BMP data start======================================================*/
  while (1) {
    /*=setup==================================================================*/
    utime = itime; uset = iset;
    ret = select (fdw, &uset, NULL, NULL, &utime);
    if      (ret< 0) {
      switch (errno) {
      case EINTR:  continue; break;
      case EBADF:  break;
      case EINVAL: break;
      case ENOMEM: break;
      }
      writeSystemLog (LOGLBL_STATS, pStrRemote, "select error.");
      goto free_memory;
      break;
    }
    else if (ret==0) {
      continue ;
    }
    else if (FD_ISSET(soc, &uset)) {
      res = BMPRecvData (soc);
      if      (res==1) {}
      else if (res==0) {break;}
      else if (res <0) {goto free_memory;}
    }
  }
  /*=Read BMP data end========================================================*/
  writeSystemLog (LOGLBL_STATS, pStrRemote, "Termination.");

  return 1;

 free_memory:
  free (pmsgStore);
  free (pmsgBondStore);

 free_remote_name:
  free (pStrRemote);
  return 0;
}

/*******************************************************************************
 * Recv BMP data **************************************************************/
static int BMPRecvData (int soc)
{
  int readLen;
  int res;
  int iSize;
  /* BMP Log */
  BMPLogInfoSet BMPLogInfo;
  struct timeval localTime;

  (void) memset (pmsgStore, 0, imsgStoreSize);
  /*=Read data================================================================*/
  readLen = recv (soc, pmsgStore, imsgStoreSize, MSG_DONTWAIT);
  if (readLen<0) {
    writeSystemLog (LOGLBL_STATS, pStrRemote, "BMP data recv error.");
    return -1;
  }
  else if (readLen > (iBondStoreSize-(pmsgBondLast - pmsgBondStore))) {
    writeSystemLog (LOGLBL_STATS, pStrRemote, "BMP data bonding store is not enough area.");
    return -1;
  }
  /*=Data Bonding=============================================================*/
  (void) memcpy (pmsgBondLast, pmsgStore, readLen); pmsgBondLast += readLen;
  pmsgBondRead = pmsgBondStore;
  do {
    /*=BMP data size check====================================================*/
    iSize = bmpMessageSizeCheck (pmsgBondRead, (ssize_t)(pmsgBondLast-pmsgBondRead));
    if      (iSize==0) {break;}
    else if (iSize <0) {return -1;}
    /*=BMP Log Information====================================================*/
    (void) memset ((void *)&BMPLogInfo, 0, sizeof (BMPLogInfoSet));
    (void) memset ((char *)&localTime , 0, sizeof (struct timeval));
    if (gettimeofday (&localTime, NULL)<0) {
      writeSystemLog (LOGLBL_STATS, pStrRemote, "can't get time.");
      return -1;
    }
    BMPLogInfo.lcl_tv_sec  = htobe32 (localTime.tv_sec);
    BMPLogInfo.lcl_tv_usec = htobe32 (localTime.tv_usec);
#if 0
    getBMPLogInfo (&BMPLogInfo, pmsgBondRead, iSize);
#endif

    /*=get BMP data===========================================================*/
    switch (*(pmsgBondRead+0)) {
    case 1: res = BMPv1Data (&BMPLogInfo, pmsgBondRead, iSize); break;
    case 2: res = BMPv1Data (&BMPLogInfo, pmsgBondRead, iSize); break;
    case 3:
      res = BMPv3Data (&BMPLogInfo, pmsgBondRead, iSize);
      break;
    default:
      writeSystemLog (LOGLBL_STATS, pStrRemote, "recv can't understand data.");
      return -1;
      break;
    }
    if      (res==0) {return 0;}
    else if (res <0) {return -1;}
#if 1
    printf ("iSize: %d\n", iSize);
    printf ("res  : %d\n", res);
#endif
    pmsgBondRead += iSize;
  } while (pmsgBondRead<pmsgBondLast);
  (void) memcpy (pmsgBondStore, pmsgBondRead, (pmsgBondLast-pmsgBondRead));
  pmsgBondLast = pmsgBondStore + (pmsgBondLast - pmsgBondRead);

  return 1;
}
/*%''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''%**
 *: get BMP data each version                                                :**
 *%,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,%*/
/*******************************************************************************
 * get BMP log infomation *****************************************************/
static int getBMPLogInfo (BMPLogInfoSet *BMPLogInfo, u_int8_t *rdata, ssize_t dlen)
{
  return 1;
}



/*******************************************************************************
 * BMP v1 data ****************************************************************/
static int BMPv1Data (BMPLogInfoSet *BMPLogInfo, u_int8_t *rdata, ssize_t dlen)
{
  /*=Variables================================================================*/
  int rsize;
  int res = 0;
  /* BMP */
  u_int8_t      *msgData;
  BMPv1Header   *BMPHeader;

  /*=Init=====================================================================*/
  rsize = sizeof (BMPv1Header);
  if (dlen<rsize) {return 0;}
  BMPHeader = (BMPv1Header *)rdata;
  msgData = (u_int8_t *)(BMPHeader+1);
  /*=Format check=============================================================*/
  /* message type check */
  if (!bmpMessageTypeCheck (BMPHeader->version, BMPHeader->msgType)) {
    writeSystemLog (LOGLBL_STATS, pStrRemote, "recv can't understand the BMP message type.");
    return -1;
  }
  else if (!bmpPeerTypeCheck (BMPHeader->version, BMPHeader->peerType)) {
    writeSystemLog (LOGLBL_STATS, pStrRemote, "recv can't understand the BMP peer type.");
    return -1;
  }
  
  /*=Recv Header==============================================================*/
  BMPLogInfo->version   = BMPHeader->version;
  BMPLogInfo->msgType   = BMPHeader->msgType;
  BMPLogInfo->peerType  = BMPHeader->peerType;
  BMPLogInfo->peerFlags = BMPHeader->peerFlags;
  memcpy (&BMPLogInfo->peerDist, BMPHeader->peerDist, sizeof (((BMPLogInfoSet *)NULL)->peerDist));
  memcpy (&BMPLogInfo->peerAddr, BMPHeader->peerAddr, sizeof (((BMPLogInfoSet *)NULL)->peerAddr));
  BMPLogInfo->peerAS      = BMPHeader->peerAS;
  BMPLogInfo->peerBgpID   = BMPHeader->peerBgpID;
  BMPLogInfo->rtr_tv_sec  = BMPHeader->tv_sec;
  BMPLogInfo->rtr_tv_usec = BMPHeader->tv_usec;
  BMPLogInfo->msgLen      = htobe32 (sizeof (BMPLogInfoSet));
#if 1
  HEXoutput (pmsgBondRead, dlen, 256);
  debugBMPinfo (BMPLogInfo);
#endif

  /*=Data Output==============================================================*/
  switch (BMPHeader->msgType) {
  case BMPV1MSGTYPE_ROUTEMON:
    res = BMPRouteMonitoring (BMPLogInfo, msgData, dlen - sizeof (BMPv1Header));
    break;
  case BMPV1MSGTYPE_STATISTICS:
    res = BMPStatisticsData (BMPLogInfo, msgData, dlen - sizeof (BMPv1Header));
    break;
  case BMPV1MSGTYPE_PEERDOWN:
    res = BMPPeerDown (BMPLogInfo, msgData, dlen - sizeof (BMPv1Header), BMPHeader->version);
    break;
  case BMPV1MSGTYPE_PEERUP:
    res = BMPPeerUp   (BMPLogInfo, msgData, dlen - sizeof (BMPv1Header), BMPHeader->version);
    break;
  }
  if      (res <0) {return -1;}
  else if (res==0) {return 1;}
  rsize+=res;

  return 1;
}

/*******************************************************************************
 * BMP v3 data ****************************************************************/
static int BMPv3Data (BMPLogInfoSet *BMPLogInfo, u_int8_t *rdata, ssize_t dlen)
{
  /*=Variables================================================================*/
  int rsize, hlen, msgLen;
  int res = 0;
  /* BMP */
  u_int8_t      *msgData;
  BMPv3Header   *BMPCommonHeader;
  BMPv3PerPeerHeader *BMPHeader;

  /*=Init=====================================================================*/
#if 0
  HEXoutput (rdata, dlen, 256);
#endif
  hlen = rsize = sizeof (BMPv3Header);
  BMPCommonHeader = (BMPv3Header *)rdata;
  msgData = (u_int8_t *)(BMPCommonHeader+1);
  msgLen  = be32toh (BMPCommonHeader->msgLen);
  /*=Format check=============================================================*/
#if 0
  HEXoutput (pmsgBondRead, dlen, 256);
  printf ("version  : 0x%02x : 0x%08x : %d\n"         , BMPCommonHeader->version           , &BMPCommonHeader->version  , sizeof (BMPCommonHeader->version));
  printf ("msgLen   : %d     : 0x%08x : %d, 0x%08x\n" , be32toh (BMPCommonHeader->msgLen)  , &BMPCommonHeader->msgLen   , sizeof (BMPCommonHeader->msgLen ), BMPCommonHeader->msgLen);
  printf ("msgType  : 0x%02x : 0x%08x : %d\n"         , BMPCommonHeader->msgType           , &BMPCommonHeader->msgType  , sizeof (BMPCommonHeader->msgType));
#endif
  /* message type check */
  if (!bmpMessageTypeCheck (BMPCommonHeader->version, BMPCommonHeader->msgType)) {
    writeSystemLog (LOGLBL_STATS, pStrRemote, "recv can't understand the BMP message type.");
    return -1;
  }
  switch (BMPCommonHeader->msgType) {
  case BMPV3MSGTYPE_INIT:
    res = BMPInitiation ((u_int8_t *)(BMPCommonHeader+1), msgLen - hlen);
    if      (res <0) {
      writeSystemLog (LOGLBL_STATS, pStrRemote, "fault BMP initiation.");
      return -1;
    }
    else if (res==0) {return  1;}
    return 0;
  case BMPV3MSGTYPE_TERMINATE:
    if (!BMPTermination ((u_int8_t *)(BMPCommonHeader+1), msgLen - hlen)) {
      writeSystemLog (LOGLBL_STATS, pStrRemote, "fault BMP termination.");
      return -1;
    }
    return 0;
    break;
  default:
    break;
  }

  hlen     += sizeof (BMPv3PerPeerHeader);
  BMPHeader = (BMPv3PerPeerHeader *)msgData;
  msgData   = (u_int8_t *)(BMPHeader+1);
  /* peer type check */
  if (!bmpPeerTypeCheck (BMPCommonHeader->version, BMPHeader->peerType)) {
    writeSystemLog (LOGLBL_STATS, pStrRemote, "recv can't understand the BMP peer type.");
    return -1;
  }
  
  /*=Recv Header==============================================================*/
  BMPLogInfo->version   = BMPCommonHeader->version;
  BMPLogInfo->msgLen    = htobe32 (sizeof (BMPLogInfoSet));
  BMPLogInfo->msgType   = BMPCommonHeader->msgType;
  BMPLogInfo->peerType  = BMPHeader->peerType;
  BMPLogInfo->peerFlags = BMPHeader->peerFlags;
  memcpy (&BMPLogInfo->peerDist, BMPHeader->peerDist, sizeof (((BMPLogInfoSet *)NULL)->peerDist));
  memcpy (&BMPLogInfo->peerAddr, BMPHeader->peerAddr, sizeof (((BMPLogInfoSet *)NULL)->peerAddr));
  BMPLogInfo->peerAS      = BMPHeader->peerAS;
  BMPLogInfo->peerBgpID   = BMPHeader->peerBgpID;
  BMPLogInfo->rtr_tv_sec  = BMPHeader->tv_sec;
  BMPLogInfo->rtr_tv_usec = BMPHeader->tv_usec;

#if 1
  HEXoutput (pmsgBondRead, dlen, 256);
  debugBMPinfo (BMPLogInfo);
#endif

  /*=Data Output==============================================================*/
  res = -1;
  switch (BMPCommonHeader->msgType) {
  case BMPV3MSGTYPE_ROUTEMON:
    res = BMPRouteMonitoring (BMPLogInfo, msgData, be32toh (BMPCommonHeader->msgLen)-hlen);
    break;
  case BMPV3MSGTYPE_STATISTICS:
    res = BMPStatisticsData (BMPLogInfo, msgData, be32toh (BMPCommonHeader->msgLen)-hlen);
    break;
  case BMPV3MSGTYPE_PEERDOWN:
    res = BMPPeerDown (BMPLogInfo, msgData, be32toh (BMPCommonHeader->msgLen)-hlen, BMPCommonHeader->version);
    break;
  case BMPV3MSGTYPE_PEERUP:
    res = BMPPeerUp   (BMPLogInfo, msgData, be32toh (BMPCommonHeader->msgLen)-hlen, BMPCommonHeader->version);
    break;
  }
  if      (res <0) {return -1;}
  else if (res==0) {return 1;}

  return 1;
}

/*%''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''%**
 *: get BMP message                                                          :**
 *%,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,%*/
/*******************************************************************************
 * Route Monitoring ***********************************************************/
static int BMPRouteMonitoring (BMPLogInfoSet *BMPLogInfo, u_int8_t *rdata, ssize_t dlen)
{
  int rsize;
  BGPCommonHeader *BGPHeader;
  u_int8_t        *msgData;
  char            hostAddr[MAXPATHLEN];

  rsize = sizeof (BGPCommonHeader);
  if (dlen<rsize) {return 0;}
  /*=Init=====================================================================*/
  BGPHeader = (BGPCommonHeader *)rdata;
  msgData = (u_int8_t *)(BGPHeader+1);
  rsize = be16toh(BGPHeader->length);
#if 0
  printf ("rsize(BGP headder): %d, 0x%04x\n", (int)rsize, rsize);
  printf ("dlen              : %d\n", (int)dlen);
  HEXoutput (BGPHeader, rsize, 256);
#endif
  if (dlen < rsize) {return 0;}
  BMPLogInfo->msgLen = htobe32 (be32toh (BMPLogInfo->msgLen) + rsize);
#if 0
  printf ("pmsgBondStore  : 0x%016x\n", (int)pmsgBondStore);
  printf ("BGPCommonHeader: 0x%016x\n", (int)BGPHeader);
  printf ("BGPLength      : 0x%d\n"   , (int)be16toh(BGPHeader->length));
  HEXoutput (rdata, (int)be16toh(BGPHeader->length), 256);
#endif
  /*=BGP Data Save============================================================*/
  getRemoteHost (soc4log, hostAddr, sizeof (hostAddr));
  if (!BMPRouteMonitoringLog (BMPLogInfo, hostAddr, rdata, rsize)) {
    return -1;
  }
  
  return rsize;
}


/*******************************************************************************
 * Write Route monitor messages ***********************************************/
static int BMPRouteMonitoringLog (BMPLogInfoSet *BMPLogInfo, char *host, u_int8_t *rdata, ssize_t len)
{
  int wfd;
  int wlen, tlen;
  char routeMonitor[MAXPATHLEN];
  (void) memset (routeMonitor, 0, sizeof (routeMonitor));
  snprintf (routeMonitor, sizeof (routeMonitor)-1, "%s/%s.log", lastParam.workdir, host);
  
  if      ((wfd=open (routeMonitor, O_WRONLY|O_CREAT, 0644))<0) {return 0;}
  else if (flock (wfd, LOCK_EX)<0)                              {close (wfd); return 0;}
  else if (lseek(wfd, 0, SEEK_END)<0)                           {close (wfd); return 0;}
#if 0
  printf ("BMP LogInfo(bef): 0x%016x: tlen: %d\n", (int)BMPLogInfo, tlen);
  HEXoutput (BMPLogInfo, sizeof (BMPLogInfoSet), 256);
#endif
  wlen = tlen = write (wfd, (u_int8_t *)BMPLogInfo, sizeof (BMPLogInfoSet));
#if 0
  printf ("BMP LogInfo(aft): 0x%016x: tlen: %d\n", (int)BMPLogInfo, tlen);
  HEXoutput (BMPLogInfo, sizeof (BMPLogInfoSet), 256);
#endif
  if (tlen != sizeof (BMPLogInfoSet)) {
    flock (wfd, LOCK_UN);
    close (wfd);
    writeSystemLog (LOGLBL_STATS, pStrRemote, "Fault write BGP Monitor Data(BMP Info): %s", routeMonitor);
    return 0;
  }
  tlen = write (wfd, rdata, len);
#if 0
  printf ("BGP data  : 0x%016x: tlen: %d\n", (int)rdata, tlen);
  HEXoutput (rdata, len, 256);
#endif
  wlen += tlen;
  if (tlen != len) {
    flock (wfd, LOCK_UN);
    close (wfd);
    writeSystemLog (LOGLBL_STATS, pStrRemote, "Fault write BGP Monitor Data(BMP Data): %s", routeMonitor);
    return 0;
  }  else if (wlen != be32toh (BMPLogInfo->msgLen)) {
    flock (wfd, LOCK_UN);
    close (wfd);
    writeSystemLog (LOGLBL_STATS, pStrRemote, "Fault write BGP Monitor Data(Fault Data length): %s", routeMonitor);
  }
  
  flock (wfd, LOCK_UN);
  close (wfd);

  return 1;
}

/*******************************************************************************
 * Stats Reports **************************************************************/
static int BMPStatisticsData (BMPLogInfoSet *BMPLogInfo, u_int8_t *rdata, ssize_t dlen)
{
  int      rsize, i, num;
  u_int8_t *ptmp;
  BMPMsgStatsNum    *StatsNum;
  BMPMsgStatsHeader *StatsHeader;
  BMPMsgStatsData   *StatsData;
  char            peerAddr[MAXPATHLEN];
  rsize = sizeof (BMPMsgStatsNum);
  if (dlen<rsize) {return 0;}
  /*=Init=====================================================================*/
  StatsNum    = (BMPMsgStatsNum *)rdata;
  num = be32toh (*StatsNum);
  for (i=0; i<num; i++) {
    ptmp = rdata + rsize;
    rsize += sizeof (BMPMsgStatsHeader);
    if (dlen<rsize) {return 0;}
    StatsHeader = (BMPMsgStatsHeader *)ptmp;
    rsize += be16toh (StatsHeader->statLen);
    if (dlen<rsize) {return 0;}
  }

#if 0
  printf ("BMP statistics data\n");
  printf ("rsize(BGP headder): %d\n", (int)rsize);
  printf ("dlen              : %d\n", (int)dlen);
  HEXoutput (rdata, dlen, 256);
#endif
  getBMPPeerAddr (BMPLogInfo, peerAddr, sizeof (peerAddr));
  ptmp   = (u_int8_t *)(StatsNum+1);
  for (i=0; i<num; i++) {
    StatsHeader = (BMPMsgStatsHeader *)ptmp;
    StatsData   = (BMPMsgStatsData *)(StatsHeader+1);
    ptmp = (u_int8_t *)StatsData;
    ptmp += be16toh (StatsHeader->statLen);
    switch (be16toh (StatsHeader->statType)) {
    case BMPSTATSTYPE_REJECT_BY_INBOUND_POLICY:
      writeLog (statisticsFile, prgName, soc4log, "Adj-RIB-in: %s : # of prefixes rejected by inbound policy: %u", peerAddr, (int)be32toh (StatsData->count32));
      break;
    case BMPSTATSTYPE_DUPLICATE_ADVERTISEMENTS:
      writeLog (statisticsFile, prgName, soc4log, "Adj-RIB-in: %s : # of (known) duplicate prefix: %u", peerAddr, (int)be32toh (StatsData->count32));
      break;
    case BMPSTATSTYPE_DUPLICATE_WITHDRAWS:
      writeLog (statisticsFile, prgName, soc4log, "Adj-RIB-in: %s : # of (known) duplicate withdraws: %u", peerAddr, (int)be32toh (StatsData->count32));
      break;
    case BMPSTATSTYPE_UPDATES_INVALIDATED_CLUSTER:
      writeLog (statisticsFile, prgName, soc4log, "Adj-RIB-in: %s : # of updates invalidated due to CLUSTER_LIST loop: %u", peerAddr, (int)be32toh (StatsData->count32));
      break;
    case BMPSTATSTYPE_UPDATES_INVALIDATED_AS_PATH:
      writeLog (statisticsFile, prgName, soc4log, "Adj-RIB-in: %s : # of updates invalidated due to AS_PATH loop: %u", peerAddr, (int)be32toh (StatsData->count32));
      break;
    case BMPSTATSTYPE_UPDATES_INVALIDATED_ORIGIN:
      writeLog (statisticsFile, prgName, soc4log, "Adj-RIB-in: %s : # of updates invalidated due to ORIGINATOR_ID: %u", peerAddr, (int)be32toh (StatsData->count32));
      break;
    case BMPSTATSTYPE_UPDATES_INVALIDATED_AS_CONF:
      writeLog (statisticsFile, prgName, soc4log, "Adj-RIB-in: %s : # of updates invalidated due to AS_CONFED loop: %u", peerAddr, (int)be32toh (StatsData->count32));
      break;
    case BMPSTATSTYPE_ROUTES_ADJ_RIBs_IN:
      writeLog (statisticsFile, prgName, soc4log, "Adj-RIB-in: %s : # of routes in Adj-RIBs-In: %lu", peerAddr, (int)be64toh (StatsData->count64));
      break;
    case BMPSTATSTYPE_ROUTES_LOC_RIB:
      writeLog (statisticsFile, prgName, soc4log, "Adj-RIB-in: %s : # of routes in Loc-RIB: %lu", peerAddr, (int)be64toh (StatsData->count64));
      break;
    }
  }
  BMPLogInfo->msgLen += rsize;

  return rsize;
}

/*******************************************************************************
 * BGP Peer Down **************************************************************/
static int BMPPeerDown (BMPLogInfoSet *BMPLogInfo, u_int8_t *rdata, ssize_t dlen, int version)
{
  int rsize;
  BMPMsgPeerDown  *BMPPeerDownHeader;
  u_int8_t        *msgData;
  BGPCommonHeader *BGPHeader;
  char            peerAddr[MAXPATHLEN];
  rsize = sizeof (BMPMsgPeerDown);
  if (dlen<rsize) {return 0;}
  /*=Init=====================================================================*/
  BMPPeerDownHeader = (BMPMsgPeerDown *)rdata;
  getBMPPeerAddr (BMPLogInfo, peerAddr, sizeof (peerAddr));
  if (BMPPeerDownHeader->reason==4) {
    writeLog (peerEventLog, prgName, soc4log, "BGP Peer Down: %s", peerAddr);
    BMPLogInfo->msgLen += rsize;
    return rsize;
  }
  BGPHeader = (BGPCommonHeader *)(BMPPeerDownHeader+1);
  rsize += (int)be16toh (BGPHeader->length);
  if      (dlen<rsize) {return 0;}
  else if (BGPHeader->type!=BGPMSGTYPE_NOTIFICATION) { /* sikp no target data */
    BMPLogInfo->msgLen += rsize;
    return rsize;
  }
  writeLog (peerEventLog, prgName, soc4log, "BGP Peer Down: %s", peerAddr);
  
  return rsize;
}

/*******************************************************************************
 * BGP Peer Up ****************************************************************/
static int BMPPeerUp (BMPLogInfoSet *BMPLogInfo, u_int8_t *rdata, ssize_t dlen, int version)
{
  int rsize;
  BGPCommonHeader *BGPHeader;
  BMPMsgPeerUp    *BMPPeerUpHeader;
  u_int8_t        *msgData;
  char            peerAddr[MAXPATHLEN];

  getBMPPeerAddr (BMPLogInfo, peerAddr, sizeof (peerAddr));
  if (BMPLogInfo->version==2) {
    rsize = sizeof (BGPCommonHeader);
    if (dlen<rsize) {return 0;}
    /*=Init===================================================================*/
    getBMPPeerAddr (BMPLogInfo, peerAddr, sizeof (peerAddr));
    BGPHeader = (BGPCommonHeader *)rdata;
    msgData = (u_int8_t *)(BGPHeader+1);
    if (dlen < be16toh(BGPHeader->length)) {return 0;}
    rsize = be16toh(BGPHeader->length);
    if (BGPHeader->type!=BGPMSGTYPE_OPEN) {return rsize;}
    writeLog (peerEventLog, prgName, soc4log, "BGP Peer Up: %s", peerAddr);
    BMPLogInfo->msgLen += rsize;
    return rsize;
  } else if (BMPLogInfo->version==3) {
#if 0
    printf ("rsize(BGP headder): %d\n", (int)rsize);
    printf ("dlen              : %d\n", (int)dlen);
    HEXoutput (rdata, dlen, 256);
#endif
    BMPPeerUpHeader = (BMPMsgPeerUp *)rdata;
    writeLog (peerEventLog, prgName, soc4log, "BGP Peer Up: %s", peerAddr);
    return dlen;
  }

  return -1;
}

/*******************************************************************************
 * BMP Initiation Messages ****************************************************/
static int BMPInitiation (u_int8_t *rdata, ssize_t dlen)
{
  int        psize;
  u_int8_t   *ptmp;
  u_int8_t   *pdata;
  int        pdlen;
  BMPMsgInit *BMPInitMessage;

  /*=Init=====================================================================*/
  ptmp = rdata;
  psize = dlen;
#if 0
  printf ("BMP Initiaion\n");
  HEXoutput (rdata, dlen, 256);
#endif

  /*=Main proc================================================================*/
  do {
    BMPInitMessage = (BMPMsgInit *)ptmp;
    pdlen = be16toh (BMPInitMessage->infoLen);
    ptmp += pdlen + sizeof (BMPMsgInit);;
    psize -= pdlen + sizeof (BMPMsgInit);;
    if ((pdata=malloc (pdlen+1))==NULL) {
      writeSystemLog (LOGLBL_STATS, pStrRemote, "Fault memory allocate for BMP initiaion.");
      return -1;
    }
#if 0
    printf ("type : %d\n", be16toh (BMPInitMessage->infoType));
#endif
    (void) memset (pdata, 0, pdlen+1);
    (void) memcpy (pdata, (BMPInitMessage+1), pdlen);
    switch (be16toh (BMPInitMessage->infoType)) {
    case BMPINITMSG_STRING:
      writeSystemLog (LOGLBL_STATS, pStrRemote, "BMP Initiaion: string\n    %s", pdata);
      break;
    case BMPINITMSG_SYSDESCR:
      writeSystemLog (LOGLBL_STATS, pStrRemote, "BMP Initiaion: sysDescr\n    %s", pdata);
      break;
    case BMPINITMSG_SYSNAME:
      writeSystemLog (LOGLBL_STATS, pStrRemote, "BMP Initiaion: sysName\n    %s", pdata);
      break;
    default:
      writeSystemLog (LOGLBL_STATS, pStrRemote, "BMP Initiaion: can't understand messages\n");
      free (pdata);
      return -1;
      break;
    }
    free (pdata);
  } while (psize>0);
  if (psize<0) {
    writeSystemLog (LOGLBL_STATS, pStrRemote, "BMP initiation messages length error.");
    return -1;
  }

  return dlen;
}

/*******************************************************************************
 * BMP Termination Messages ***************************************************/
static int BMPTermination (u_int8_t *rdata, ssize_t dlen)
{
  int              psize;
  u_int8_t         *ptmp;
  u_int8_t         *pdata;
  int              pdlen;
  BMPMsgTerminate  *BMPTermMessage;

  /*=Init=====================================================================*/
  ptmp = rdata;
  psize = dlen;
#if 0
  printf ("BMP Termination\n");
  HEXoutput (rdata, dlen, 256);
  printf ("psize: %d\n", psize);
#endif

  /*=Main proc================================================================*/
  do {
    BMPTermMessage = (BMPMsgTerminate *)ptmp;
    pdlen  = be16toh (BMPTermMessage->infoLen);
    ptmp  += pdlen + sizeof (BMPMsgTerminate);
    psize -= pdlen + sizeof (BMPMsgTerminate);
    printf ("pdlen: %d\n", pdlen);
    printf ("psize: %d\n\n", psize);
    if ((pdata=malloc (pdlen+1))==NULL) {
      writeSystemLog (LOGLBL_STATS, pStrRemote, "Fault memory allocate for BMP Termination.");
      return 0;
    }
#if 0
    printf ("type : %d\n", be16toh (BMPTermMessage->infoType));
#endif
    (void) memset (pdata, 0, pdlen+1);
    (void) memcpy (pdata, (BMPTermMessage+1), pdlen);
    switch (be16toh (BMPTermMessage->infoType)) {
    case BMPTERMMSG_STRING:
      writeSystemLog (LOGLBL_STATS, pStrRemote, "BMP Termination: string\n    %s", pdata);
      break;
    case BMPTERMMSG_REASON:
      writeSystemLog (LOGLBL_STATS, pStrRemote, "BMP Termination: reason: %d", be16toh (*((u_int16_t *)pdata)));
      break;
    }
    free (pdata);
  } while (psize>0);
  if (psize<0) {
    writeSystemLog (LOGLBL_STATS, pStrRemote, "BMP Term messages length error.");
    return 0;
  }

  return 1;
}


/*******************************************************************************
 * GET BMP peer address *******************************************************/
static int getBMPPeerAddr (BMPLogInfoSet *BMPLogInfo, char *pstr, const int size)
{
  struct in_addr  *sin_addr;
  struct in6_addr *sin6_addr;

  (void) memset (pstr, 0, size);
  sin_addr  = (struct in_addr  *)&BMPLogInfo->peerAddr[3];
  sin6_addr = (struct in6_addr *)&BMPLogInfo->peerAddr;
  if (FLGISSET(BMPLogInfo->peerFlags, BMPPEERFLG_VFLG_GET)) {
    inet_ntop (AF_INET6, sin6_addr, pstr, size);
  } else {
    inet_ntop (AF_INET, sin_addr, pstr, size);
  }

  return 1;
}

/*%''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''%**
 *: DEBUG CODE                                                               :**
 *%,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,%*/
/*******************************************************************************
 * BMP Information output *****************************************************/
static void debugBMPinfo (BMPLogInfoSet *BMPLogInfo)
{
  printf ("version  : 0x%02x\n"         , BMPLogInfo->version              );
  printf ("msgLen   : %d    \n"         , be32toh (BMPLogInfo->msgLen)     );
  printf ("msgType  : 0x%02x\n"         , BMPLogInfo->msgType              );
  printf ("peerType : 0x%02x\n"         , BMPLogInfo->peerType             );
  printf ("peerFlags: 0x%02x\n"         , BMPLogInfo->peerFlags            );
  printf ("peerDist :       \n"                                            );
  printf ("peerAddr :       \n"                                            );
  printf ("peerAS   : %d    \n"         , (int)be32toh (BMPLogInfo->peerAS));
  printf ("BGP ID   : %u    \n"         , be32toh (BMPLogInfo->peerBgpID)  );
  printf ("tv_sec   : %u    \n"         , be32toh (BMPLogInfo->rtr_tv_sec) );
  printf ("tv_usec  : %06u  \n"         , be32toh (BMPLogInfo->rtr_tv_usec));
}

/*******************************************************************************
 * HEX Data output ************************************************************/
static void HEXoutput (u_int8_t *rdata, int len, int max)
{
  int i;
  int outLen;
  int counter = -16;
  outLen = (len<max?len:max);

  printf ("len: %d, output: %d\n", len, outLen);
  for (i=0; i<outLen; i++) {
    if ((i%16)==0) {counter += 16; printf ("0x%04X:", counter);}
    if ((i%2)==0) {printf (" ");}
    printf ("%02x", rdata[i]);
    if ((i%16)==15) {printf ("\n");}
  }
  printf ("\n");
}
