/**************************************************************************
 Moenizer - Allow to set background image for OmegaT.
 
 Copyright (C) 2013-2014 Yu Tang
               Home page: http://sourceforge.jp/users/yu-tang/
               Support center: http://sourceforge.jp/users/yu-tang/pf/Moenizer/

 This file is part of plugin for OmegaT.
 http://www.omegat.org/

 License: GNU GPL version 3 or (at your option) any later version.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **************************************************************************/

package jp.sourceforge.users.yutang.omegat.plugin.moenizer;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.net.URISyntaxException;
import java.net.URL;
import java.security.CodeSource;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.omegat.util.LFileCopy;
import org.omegat.util.Log;
import org.omegat.util.OConsts;
import org.omegat.util.StaticUtils;

/**
 *
 * @author Yu-Tang
 */
public class MoeUtil {
    
    private static final Pattern RE_HAS_IMAGE_EXTENSION = Pattern.compile(".+\\.(?:bmp|png|jpg|jpeg|gif)\\z", Pattern.CASE_INSENSITIVE);
    private static final Pattern RE_HAS_URL_SHORTCUT_EXTENSION = Pattern.compile(".+\\.(?:bmp|png|jpg|jpeg|gif)\\z", Pattern.CASE_INSENSITIVE);
    private static final Pattern RE_URL_IN_SHORTCUT = Pattern.compile("\\[InternetShortcut\\]\\s+URL=(.+)\\b");
    private static final Pattern RE_URL_IN_WEBLOC = Pattern.compile("<key>URL</key>\\s+<string>(.+)</string>");
    private static final String SAFE_PLUGIN_NAME = "moenizer";
    private static File pluginJarFile;
    
    static {
        pluginJarFile = null;
    }
    
    public static File getPluginJarFile() throws URISyntaxException {
        if (pluginJarFile == null) {
            CodeSource codeSource = MoeUtil.class.getProtectionDomain().getCodeSource();
            pluginJarFile = new File(codeSource.getLocation().toURI().getPath());
        }
        return pluginJarFile;
    }
    
    public static File getPluginJarDir() throws URISyntaxException {
        File jarFile = getPluginJarFile();
        return new File(jarFile.getParentFile().getPath());
    }
    
    public static URL getURL(File file) {
        String fileName = file.getName().toLowerCase();
        try {
            if (fileName.endsWith(".url") || fileName.endsWith(".website")) {
                // .url is Internet ShortCut file for IE8 or earlier, Firefox
                // .website is Pinned Site ShortCut file for IE9 or later

                // Windows 日本語環境で作成すると Shift-JIS 文字列が含まれること
                // があるため、文字コードはその OS の既定文字コードを想定した方
                // が望ましいと思われる。
                String text = readTextFile(file, System.getProperty("sun.jnu.encoding"));
                Matcher matcher = RE_URL_IN_SHORTCUT.matcher(text);
                if (matcher.find()) {
                    return new URL(matcher.group(1));
                } else {
                    Log.log("Could not find valid URL in internet shortcut file '" + file.getCanonicalPath() + "'");
                }

            } else if (fileName.endsWith(".webloc")) {
                // .webloc is Mac OS X Website Location file for safari
                String text = readTextFile(file, OConsts.UTF8);
                Matcher matcher = RE_URL_IN_WEBLOC.matcher(text);
                if (matcher.find()) {
                    return new URL(matcher.group(1));
                } else {
                    Log.log("Could not find valid URL in website location file '" + file.getCanonicalPath() + "'");
                }

            } else {  // unknown
                // ignore
            }
        } catch (IOException ex) {
            Log.log(ex);
        }
        return null;
    }

    /**
     * Read file as platform dependent encoding text.
     */
    public static String readTextFile(File file, String encoding) throws IOException {
        BufferedReader rd = new BufferedReader(new InputStreamReader(new FileInputStream(file), encoding));
        try {
            StringWriter out = new StringWriter();
            LFileCopy.copy(rd, out);
            return out.toString();
        } finally {
            rd.close();
        }
    }

    public static boolean isURLShortcut(File file) {
        String name = file.getName();
        return (!name.startsWith(".") 
                && RE_HAS_URL_SHORTCUT_EXTENSION.matcher(name).matches());
    }

    public static boolean isImage(File file) {
        String name = file.getName();
        return (!name.startsWith(".") 
                && RE_HAS_IMAGE_EXTENSION.matcher(name).matches());
    }

    public static File getPrimaryMoeConfigDir() {
        return new File(StaticUtils.getConfigDir(), SAFE_PLUGIN_NAME);
    }

    public static File getPrimaryMoeConfigFile() {
        return new File(getPrimaryMoeConfigDir(), MoeConfig.FILE_NAME);
    }

    public static File getSecondaryMoeConfigDir() {
        return new File(StaticUtils.getConfigDir(),
                "plugins" + File.separator + SAFE_PLUGIN_NAME);
    }

    public static File getSecondaryMoeConfigFile() {
        return new File(getSecondaryMoeConfigDir(), MoeConfig.FILE_NAME);
    }

    /**
     * 設定ファイルを取得する
     * @return 候補パスのいずれかにファイルが実在すればそれを、どれも実在
     *          しなければ優先パスを返す
     */
    public static File getMoeConfigFile() {
        File primaryMoeConfigFile = getPrimaryMoeConfigFile();
        if (primaryMoeConfigFile.isFile()) {
            return primaryMoeConfigFile;
        }
        File secondaryMoeConfigFile = getSecondaryMoeConfigFile();
        if (secondaryMoeConfigFile.isFile()) {
            return secondaryMoeConfigFile;
        }
        return primaryMoeConfigFile;
    }

}
