﻿using CleanAuLait.Core.Converter;
using CleanAuLait.OuterEdge.Repository;
using NLog;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using TestNarou3.Domain.Boundary.Repository;
using TestNarou3.Domain.Model.API;

namespace TestNarou3.OuterEdge.Repository.API
{
    internal class NarouRepository : INarouRepository
    {
        private static readonly ILogger logger = LogManager.GetCurrentClassLogger();

        private const string LOGIN_POST_URL = "https://ssl.syosetu.com/login/login/";
        private const string LOGIN_REFFERER_URL = "https://ssl.syosetu.com/login/input/";

        private const string LOGOUT_POST_URL = "https://ssl.syosetu.com/login/logout/";
        private const string LOGOUT_REFFERER_URL = "https://syosetu.com/user/top/";

        private const string BOOKMARK_CATEGORY_GET_URL = "https://syosetu.com/favnovelmain/list/";
        private const string BOOKMARK_CATEGORY_REFFERER_URL = "https://syosetu.com/user/top/";

        private const string BOOKMARK_LIST_GET_URL = "https://syosetu.com/favnovelmain/list/?nowcategory={0}&order=new&p={1}";
        private const string BOOKMARK_LIST_REFFERER_URL = "https://syosetu.com/favnovelmain/list/";

        private const string NOVEL_API_URL = "https://api.syosetu.com/novelapi/api/?out=json&gzip=5&ncode={0}&lim={1}";

        private const string LOGIN_FORM_KEY_ID = "narouid";
        private const string LOGIN_FORM_KEY_PW = "pass";

        private const string LOGOUT_FORM_KEY_TOKEN = "token";

        private const string LOGIN_ERROR_MESSAGE = "エラーが発生しました";
        private const string LOGIN_ERROR_MESSAGE_REQUIRED_FIELD = "IDまたはメールアドレスが未入力です。";
        private const string LOGIN_ERROR_MESSAGE_INVALID_CREDENTIAL = "IDが存在しないかIDとパスワードが一致しません。";

        private const string DATA_TOKEN_REGEX = @"<a id=""logout"" class=""js-logout"" href=""https://syosetu.com/login/logout/"" data-token=""([0-9a-f]+)"">";

        private const string CATEGORY_LIST_REGEX = @"<ul class=""category_box"">(.+?)</ul>\s+<div class=""fav_box"">";
        private const string CATEGORY_ITEM_REGEX = @"<li.*><a href=""/favnovelmain/list/\?nowcategory=\d+&order=[_a-z]+"">(.+?)</a></li>";
        private const string CATEGORY_NAME_COUNT_REGEX = @"(.+?)\((\d+)\)";

        private const string BOOKMARK_ENTRY_REGEX = @"<table class=""favnovel"">(.+?)</table>";
        private const string BOOKMARK_ITEM_REGEX = @"<a class=""title"" href=""https://ncode.syosetu.com/(n[0-9a-z]+)/"">";

        private readonly HttpClientHelper httpClient = new();

        private bool disposedValue;

        #region IDisposable

        protected virtual void Dispose(bool disposing)
        {
            if (!disposedValue)
            {
                if (disposing)
                {
                    this.httpClient?.Dispose();
                    logger.Trace("NarouRepository disposed.");
                }

                disposedValue = true;
            }
        }

        public void Dispose()
        {
            Dispose(disposing: true);
            System.GC.SuppressFinalize(this);
        }

        #endregion

        public string Login(string id, string password)
        {
            var form = new Dictionary<string, string>()
            {
                { LOGIN_FORM_KEY_ID, id },
                { LOGIN_FORM_KEY_PW, password },
            };

            var response = this.httpClient.PostAsync(LOGIN_POST_URL, LOGIN_REFFERER_URL, form).Result;

            if (!response.IsSuccessStatusCode)
            {
                throw new RepositoryException($"ログイン応答が想定と異なります。{response.StatusCode}");
            }

            string content = HttpClientHelper.ReadContentAsync(response).Result;

            if (content.Contains(LOGIN_ERROR_MESSAGE))
            {
                if (content.Contains(LOGIN_ERROR_MESSAGE_REQUIRED_FIELD))
                {
                    throw new RepositoryException(LOGIN_ERROR_MESSAGE_REQUIRED_FIELD);
                }

                if (content.Contains(LOGIN_ERROR_MESSAGE_INVALID_CREDENTIAL))
                {
                    throw new RepositoryException(LOGIN_ERROR_MESSAGE_INVALID_CREDENTIAL);
                }

                logger.Trace(content);

                throw new RepositoryException("ログインエラーが発生しました。");
            }

            //<a id="logout" class="js-logout" href="https://syosetu.com/login/logout/" data-token="27098ab9dd0d56d017c7cd3ac650cdeb">

            Regex regex = new(DATA_TOKEN_REGEX);
            Match matcher = regex.Match(content);

            if (!matcher.Success)
            {
                throw new RepositoryException("ログイン後に DataToken が見つかりませんでした。");
            }

            string dataToken = matcher.Groups[1].Value;

            logger.Trace("DataToken {0}", dataToken);

            return dataToken;
        }

        public void Logout(string dataToken)
        {
            if (string.IsNullOrEmpty(dataToken))
            {
                return;
            }

            var form = new Dictionary<string, string>()
            {
                { LOGOUT_FORM_KEY_TOKEN, dataToken },
            };

            var response = this.httpClient.PostAsync(LOGOUT_POST_URL, LOGOUT_REFFERER_URL, form).Result;

#if false
            if (!response.IsSuccessStatusCode)
            {
                throw new RepositoryException($"ログアウト応答が想定と異なります。{response.StatusCode}");
            }
#endif
        }

        public IEnumerable<NarouCategoryInfo> GetNarouCategoryList()
        {
            var response = this.httpClient.GetAsync(
                    BOOKMARK_CATEGORY_GET_URL, BOOKMARK_CATEGORY_REFFERER_URL).Result;

            if (!response.IsSuccessStatusCode)
            {
                throw new RepositoryException($"応答が想定と異なります。{response.StatusCode}");
            }

            string content = HttpClientHelper.ReadContentAsync(response).Result;

            Regex regex = new(CATEGORY_LIST_REGEX, RegexOptions.Singleline);
            Regex regexItem = new(CATEGORY_ITEM_REGEX, RegexOptions.Multiline);
            Regex regexNameCount = new(CATEGORY_NAME_COUNT_REGEX);

            Match matcher = regex.Match(content);

            if (!matcher.Success)
            {
                logger.Trace(content);
                throw new RepositoryException("ブックマークカテゴリが見つかりません。");
            }

            string categoryList = matcher.Groups[1].Value;

            //logger.Trace("categoryList: {0}", categoryList);

            MatchCollection matcherItems = regexItem.Matches(categoryList);

            if (matcherItems.Count == 0)
            {
                logger.Trace(categoryList);
                throw new RepositoryException("ブックマークカテゴリの明細情報が見つかりません。");
            }

            IList<NarouCategoryInfo> infos = new List<NarouCategoryInfo>();

            foreach (Match matchItem in matcherItems)
            {
                string item = matchItem.Groups[1].Value;

                logger.Trace("categoryItem: {0}", item);

                Match matcherNameCount = regexNameCount.Match(item);

                if (!matcherNameCount.Success)
                {
                    logger.Trace(item);
                    throw new RepositoryException("ブックマークカテゴリ明細情報の形式が想定と異なります。");
                }

                string title = matcherNameCount.Groups[1].Value;
                int count = int.Parse(matcherNameCount.Groups[2].Value);
                //logger.Trace("categoryNameCount: {0} {1}", title, count);

                infos.Add(new(title, count));
            }

            return infos;
        }

        public IEnumerable<NarouNovelDetailInfo> GetNovelDetailInfoList(int categoryNo, int categoryItemsCount)
        {
            IList<string> ncodeList = new List<string>();

            for (int page = 1; page <= (categoryItemsCount - 1) / 50 + 1; page++)
            {
                var response = this.httpClient.GetAsync(
                        string.Format(BOOKMARK_LIST_GET_URL, categoryNo, page), BOOKMARK_LIST_REFFERER_URL).Result;

                if (!response.IsSuccessStatusCode)
                {
                    throw new RepositoryException($"応答が想定と異なります。{response.StatusCode}");
                }

                string content = HttpClientHelper.ReadContentAsync(response).Result;

                Regex regex = new(BOOKMARK_ENTRY_REGEX, RegexOptions.Singleline);
                Regex regexItem = new(BOOKMARK_ITEM_REGEX, RegexOptions.Singleline);

                MatchCollection matchers = regex.Matches(content);

                if (matchers.Count == 0)
                {
                    throw new RepositoryException("ブックマークの明細リストが見つかりません。");
                }

                foreach (Match matcher in matchers)
                {
                    string entry = matcher.Groups[1].Value;

                    Match matcherItem = regexItem.Match(entry);

                    if (!matcherItem.Success)
                    {
                        throw new RepositoryException("ブックマーク明細情報の形式が想定と異なります。");
                    }

                    string ncode = matcherItem.Groups[1].Value;

                    logger.Trace("bookmark: {0}", ncode);

                    ncodeList.Add(ncode);
                }
            }

            string narouJson = GetNarouJson(ncodeList);

            NarouNovelDetailInfo[] novelInfos = narouJson.DeserializeJson<NarouNovelDetailInfo[]>();

            if (novelInfos == null)
            {
                throw new RepositoryException("なろうAPIからの応答が想定と異なります");
            }

            return novelInfos;
        }

        private string GetNarouJson(IList<string> ncodeList)
        {
            string apiUrl = string.Format(NOVEL_API_URL, string.Join('-', ncodeList), ncodeList.Count);

            var response = this.httpClient.GetAsync(apiUrl, null).Result;

            if (!response.IsSuccessStatusCode)
            {
                throw new RepositoryException($"応答が想定と異なります。{response.StatusCode}");
            }

            string json = HttpClientHelper.ReadGzipContent(response);

            return json;
        }
    }
}
