#include <stdio.h>
#include <ctype.h>
#include <setjmp.h>
#include <math.h>
#include "hoc.h"
#include "y.tab.h"

extern int yyparse();
extern double Log();
extern double Log10();
extern double Exp();
extern double Sqrt();
extern double integer();

static struct {
	char *name;
	double cval;
} consts[] = {
	"PI",	 3.14159265358979323846,
	"E",	 2.71828182845904523536,
	"GAMMA", 0.57721566490153286060,
	"DEG",	57.29577951308232087680,
	"PHI",	 1.61803398874989484820,
	0,		 0
};
static struct {
	char *name;
	double (*func)();
} builtin[] = {
	"sin",	sin,
	"cos",	cos,
	"atan",	atan,
	"log",	Log,
	"log10", Log10,
	"exp",	Exp,
	"sqrt",	Sqrt,
	"int",	integer,
	"abs",	fabs,
	0,		0
};

static char *progname; 	/* for error message */
static int lineno = 1;
static jmp_buf begin;


int init() {
	int i;
	Symbol *sp;

	for (i = 0; consts[i].name; i++) {
		install(consts[i].name, VAR, consts[i].cval);
	}
	for (i = 0; builtin[i].name; i++) {
		sp = install(builtin[i].name, BLTIN, 0.0);
		sp->u.ptr = builtin[i].func;
	}
	return 0;
}

int main(int arc, char **argv) {
	progname = argv[0];
	init();
	setjmp(begin);

	initcode();
	while(yyparse()) {
		execute(prog);
		initcode();
	}

	return 0;
}

int warning(char *s, char* t) {
	fprintf(stderr, "%s: %s", progname, s);
	if(t)
		fprintf(stderr, " %s", t);
	fprintf(stderr, " near line %d\n", lineno);
	return 0;
}

int execerror(char *s, char *t) {
	warning(s, t);
	longjmp(begin, 0);
	return 0;
}

int yyerror(char *s) {
	warning(s, (char*)0);
	return 0;
}

int yylex() {
	int c;

	while ((c=getchar()) == ' ' || c == '\t')
		;

	if (c==EOF)
		return 0;

	if (c=='.' || isdigit(c)) { /*NUMBER*/
		double d;
		ungetc(c, stdin);
		scanf("%lf", &d);
		yylval.sym = install("", NUMBER, d);
		return NUMBER;
	}

	char var_buf[256], *pvar = var_buf;
	if (isalpha(c)) {
		do {
			*pvar++ = c;
			c = getchar();
		} while(isalnum(c));
		ungetc(c, stdin);
		*pvar = '\0';
		Symbol *psym = lookup(var_buf);
		if (psym==0) 
			psym = install(var_buf, UNDEF, 0);
		yylval.sym = psym;
		return yylval.sym->type==UNDEF? VAR: yylval.sym->type;
	}

	if (c=='\n')
		lineno++;

	return c;
}
