#include <stdio.h>
#include <string.h>
#include <math.h>
#include <stdbool.h>
#include "hoc.h"
#include "y.tab.h"

#define NSTACK 256
static Datum stack[NSTACK];
static Datum *stackp;

Datum push(Datum d) {
	if (stackp >= &stack[NSTACK])
			execerror("stack overflow", (char*)0);
	*stackp++ = d;
	return *stackp;
}

Datum pop() {
	if (stackp <= stack)
			execerror("stack underflow", (char*)0);
	return *--stackp;
}


#define NPROG 2000
Inst prog[NPROG]; /* the machine */ 
Inst *progp;
Inst *pc;

struct {
	char name[10];
	Inst inst;
} inst_name_table[] = {
	{"push", push},
	{"pop", pop},
	{"constpush", constpush},
	{"varpush", varpush},
	{"add", add},
	{"sub", sub},
	{"mul", mul},
	{"divid", divid},
	{"negate", negate},
	{"power", power},
	{"eval", eval},
	{"assign", assign},
	{"print", print},
};


const char *get_inst_name(Inst e) {
	static const char NOT_FOUND[]="N/A";
	const int INST_TBL_SIZE = sizeof(inst_name_table)/sizeof(inst_name_table[0]);
	for (int i = 0; i < INST_TBL_SIZE; i++) {
	   if (e==inst_name_table[i].inst) return inst_name_table[i].name;	   
	}
	
	return NOT_FOUND;
}

bool is_inst(Inst e){
	return (strcmp(get_inst_name(e), "N/A")!=0);
}

void PRINT_SYM_INFO(int p, Symbol *sp) {
	switch (sp->type) {
	case NUMBER: printf("$%d: NUMBER: %lf\n", p, sp->u.val); break;
	case VAR: 	 printf("$%d: VAR: %s\n", 	  p, sp->name); break;
	case UNDEF:  printf("$%d: UNDEF: %s\n",   p, sp->name); break;
	default: break;
	}
}

void PRINT_PROG_INFO(Inst *p) {
	if (is_inst(*p)) {
		printf("$%d: %s\n", (int)p, get_inst_name(*p));
	} else if (*p!=STOP) {
		PRINT_SYM_INFO( (int)p, (Symbol*)*p );
	}
}

void initcode() {
	stackp = stack;
	progp = prog;
}

Inst *code(Inst f) {
	Inst *oldp = progp;
	if (progp >= &prog[NPROG]) {
		execerror("out of program range", (char*)0);
	}
	*progp = f;
	PRINT_PROG_INFO(progp);
	progp++;
	return oldp;
}

void execute(Inst *p) {
	for (pc = p; *pc != STOP; pc++) {
		printf("exec:"); PRINT_PROG_INFO(pc);
		(*(*pc))();
	}
}

/*
 * Machine Instructions
 */
Datum constpush() {
	Datum d = {
		.val = ((Symbol*)*(pc+1))->u.val
	};
	push(d);
	pc++;
	return d;
}
Datum varpush() {
	Datum d = {
		.sym = (Symbol*)*(pc+1)
	};
	push(d);
	pc++;
	return d;
}
Datum add() {
	Datum d1, d2;
	d1 = pop();
	d2 = pop();
	d2.val += d1.val;
	push(d2);
	return d2; 
}
Datum sub() {
	Datum d1, d2;
	d1 = pop();
	d2 = pop();
	d2.val -= d1.val;
	push(d2);
	return d2; 
}
Datum mul() {
	Datum d1, d2;
	d1 = pop();
	d2 = pop();
	d2.val *= d1.val;
	push(d2);
	return d2; 
}
Datum divid() {
	Datum d1, d2;
	d1 = pop();
	d2 = pop();
	if (d1.val == 0)
		execerror("divid by zero.", (char*)0);
	d2.val /= d1.val;
	push(d2);
	return d2; 
}
Datum negate() {
	Datum d = pop();
	d.val = -d.val;
	push(d);
	return d;
}
Datum power() {
	Datum d1, d2;
	d1 = pop();
	d2 = pop();
	d2.val = pow(d2.val, d1.val);
	push(d2);
	return d2; 
}
Datum eval() {
	Datum d = pop();
	d.val = d.sym->u.val;
	push(d);
	return d;
}
Datum assign() {
	Datum d1, d2;
	d1 = pop();
	d2 = pop();
	if (d1.sym->type != VAR && d1.sym->type != UNDEF)
		execerror("not a variable.", (char*)0);
	d1.sym->u.val = d2.val;
	d1.sym->type = VAR;
	push(d1);
	return d1;
}
Datum print(){
	Datum d = pop();
	printf("\t%lf\n", d.val);
	return d;
}
Datum bltin(){
	pc++;
	return (Datum){};
}
