//============================================================================
// Name        : sentinel.cpp
// Author      : Yasuoki
// Version     : 0.1
// Copyright   : LGPL
// Description : sentinel server
//============================================================================

#include <stdlib.h>
#include <stdio.h>
#include <pthread.h>
#include <stdarg.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include "sentinel.h"
#include "scheduler.h"
#include "commands.h"
#include "queue.h"
#include "proc_command.h"
#include "proc_database.h"
#include "proc_listener.h"
#include "proc_requester.h"

namespace SST {

const char *codeToMessgae(ErrorCode code)
{
	const char *v = "Unknown error.";
	switch(code) {
	case ErrorOk:			v = "Ok";	break;
	case ErrorNoMemory:		v = "No memory error";	break;
	case ErrorRuntime:		v = "Runtime error";	break;
	case ErrorConnection:	v = "Connection error";	break;
	case ErrorDatabase:		v = "Database error";	break;
	case ErrorComm:			v = "TCP/IP communication error";	break;
	case ErrorMessage:		v = "Message error";	break;
	case ErrorCommand:		v = "Command error";	break;
	case ErrorNotLogin:		v = "Not login error";	break;
	case ErrorPermission:	v = "Permission error";	break;
	case ErrorNotFound:		v = "Not found error";	break;
	case ErrorDataFull:		v = "Data full error";	break;
	case ErrorDataLost:		v = "Data lost error";	break;
	case ErrorDataConflict:	v = "Data conflict error";	break;
	case ErrorParameter:	v = "Parameter error";	break;
	case ErrorHierarchy:	v = "Hierarchy error";	break;
	case ErrorDuplicate:	v = "Duplication error";	break;
	case ErrorAuth:			v = "Auth error";	break;
	case ErrorHTTP:			v = "HTTP comunication error";	break;
	case ErrorSystem:		v = "System call error";	break;
	case ErrorLocked:		v = "Object locked error";	break;
	case ErrorUser:			v = "User error";	break;
	case ErrorNoSpace:		v = "No spcae error";	break;
	case ErrorDataType:		v = "Data type error";	break;
	case ErrorData:			v = "Data value error";	break;
	case ErrorNotImpl:		v = "Not implement error";	break;
	case ErrorNotEmpty:		v = "Not empty error";	break;
	case ErrorRestriction:	v = "Restricted error";	break;
	case ErrorTimeout:		v = "Timeout error";	break;
	}
	return v;
}

Sentinel::Sentinel()
{
	mConf		= NULL;
	eventBase	= NULL;
	sigterm		= NULL;
	sighup		= NULL;
	sigint		= NULL;

	evthread_use_pthreads();

	pthread_mutex_init(&mLogMutex, NULL);

}

Sentinel::~Sentinel()
{
	if( mConf ) {
		freeConf(mConf);
	}
}

void Sentinel::log(int logLevel, const char *fmt, ...)
{
	va_list ap;
	va_start(ap, fmt);
	logv(logLevel, fmt, ap);
    va_end(ap);
}

void Sentinel::logv(int logLevel, const char *fmt, va_list ap)
{

	const char *l=" ";
	switch(logLevel) {
	case LOG_DEBUG: 	l="D";	break;
	case LOG_INFO: 		l="I";	break;
	case LOG_NOTICE:	l="N";	break;
	case LOG_WARNING:	l="W";	break;
	case LOG_ERR: 		l="E";	break;
	}
#ifdef _DEBUG
	pthread_mutex_lock(&mLogMutex);
	fprintf(stderr, "%8.8X: %s ", (unsigned int)pthread_self(), l);
	vfprintf(stderr, fmt, ap);
	fprintf(stderr, "\n");
	pthread_mutex_unlock(&mLogMutex);
#else
	char fmt_buff[512];
	char *fmt2 = fmt_buff;
	size_t len = strlen(fmt);
	if( len+10 > sizeof(fmt_buff)/sizeof(char) ) {
		fmt2 = (char*)malloc((len+10)*sizeof(char));
	}
	sprintf(fmt2, "%s %s", l, fmt);
    if( mConf ) {
    	if( logLevel == LOG_DEBUG )
    		if( mConf->logLevel > LogLevelDebug ) return;
    	if( logLevel == LOG_INFO )
    		if( mConf->logLevel > LogLevelVerbose ) return;
    	if( logLevel == LOG_NOTICE )
    		if( mConf->logLevel > LogLevelNotice ) return;
    	if( logLevel == LOG_WARNING )
    		if( mConf->logLevel > LogLevelWarning ) return;
    	if( mConf->logIdent ) {
            vsyslog(logLevel, fmt, ap);
    	} else {
        	pthread_mutex_lock(&mLogMutex);
        	vfprintf(stderr, fmt2, ap);
        	fprintf(stderr, "\n");
        	pthread_mutex_unlock(&mLogMutex);
        }
    } else {
    	if( logLevel >= LOG_WARNING ) {
        	pthread_mutex_lock(&mLogMutex);
			vfprintf(stderr, fmt2, ap);
			fprintf(stderr, "\n");
	    	pthread_mutex_unlock(&mLogMutex);
    	}
    }
    if( fmt2 != fmt_buff )
    	free(fmt2);
#endif
}

bool Sentinel::loadConfig(const char *file)
{
	log(LOG_DEBUG, "Loading configure file. %s", file);
	if( file == NULL ) {
		if( mConf == NULL ) {
			log(LOG_ERR, "Configure file is not specified.");
			return false;
		}
		file = mConf->confFile;
	}
	char wd[256];
	getcwd(wd, sizeof(wd));
	log(LOG_INFO, "loading %s from %s", file, wd);

	FILE *fp = fopen(file, "rt");
	if( fp == NULL ) {
		log(LOG_ERR, "%s can't open.", file);
		return false;
	}
	log(LOG_DEBUG, "open configure file %s ok.", file);

	Conf *new_conf = allocConf(file);
	if( new_conf == NULL ) {
		log(LOG_ERR, "Out of memory at loading %s.", file);
		return false;
	}
	log(LOG_DEBUG, "memory alloc ok.");

	bool hasError=false;
	int line=1;
	char buff[4096];
	char key[4096];
	char value[4096];
	while( !feof(fp) ) {
		if( fgets(buff, sizeof(buff), fp) == NULL ) break;
		char *p = buff;
		while( *p == ' ' || *p == '\t' ) p++;
		if( *p == 0 || *p == '\r' || *p == '\n' || *p == '#') {
			line++;
			continue;
		}
		char *pWord = p;
		while( *p != 0 && *p != '\r' && *p != '\n' && *p != '\t' && *p != ' ' && *p != '=') p++;
		if( size_t(p-pWord) > sizeof(key)-1 ) {
			log(LOG_ERR, "%s:%d keyword length is too long.", file, line);
			hasError = true;
			line++;
			continue;
		}
		strncpy(key, pWord, p-pWord);
		key[p-pWord] = 0;
		while( *p == ' ' || *p == '\t' ) p++;
		if( *p != '=' ) {
			log(LOG_ERR, "%s:%d syntax error.", file, line);
			hasError = true;
			line++;
			continue;
		}
		p++;
		while( *p == ' ' || *p == '\t' ) p++;
		char em = 0;
		if( *p == '"' ) {
			em = '"';
			p++;
		}
		pWord = p;
		while( *p != 0 ) {
			if( em ) {
				if( *p == em ) break;
			} else {
				if( *p == '\t' || *p == ' ' || *p == '\r' || *p == '\n' ) break;
			}
			p++;
		}
		if( size_t(p-pWord) > sizeof(value)-1 ) {
			log(LOG_ERR, "%s:%d value length is too long.", file, line);
			hasError = true;
			line++;
			continue;
		}
		strncpy(value, pWord, p-pWord);
		value[p-pWord] = 0;
		log(LOG_DEBUG, "Line %d Key=%s Value=%s", line, key, value);

		if( strcmp(key, "pidFile") == 0 ) {
			if( new_conf->pidFile ) free(new_conf->pidFile);
			new_conf->pidFile	= strdup(value);
		} else if( strcmp(key, "daemonize") == 0 ) {
			if( strcmp(value, "yes") == 0 )
				new_conf->daemonaize	= true;
			else if(strcmp(value, "no") == 0 )
				new_conf->daemonaize	= false;
			else {
				log(LOG_ERR, "%s:%d bad parameter. (yes or no)", file, line);
				hasError = true;
			}
		} else if( strcmp(key, "logLevel") == 0 ) {
			if( strcmp(value, "debug") == 0 )
				new_conf->logLevel	= LogLevelDebug;
			else if( strcmp(value, "verbose") == 0 )
				new_conf->logLevel	= LogLevelDebug;
			else if( strcmp(value, "notice") == 0 )
				new_conf->logLevel	= LogLevelDebug;
			else if( strcmp(value, "warning") == 0 )
				new_conf->logLevel	= LogLevelDebug;
			else if( strcmp(value, "error") == 0 )
				new_conf->logLevel	= LogLevelDebug;
			else {
				log(LOG_ERR, "%s:%d bad parameter. (debug, verbose, notice, warning or error)", file, line);
				hasError = true;
			}
		} else if( strcmp(key, "logIdent") == 0 ) {
			if( new_conf->logIdent ) free(new_conf->logIdent);
			new_conf->logIdent	= NULL;
			if( strcmp(value, SST_STDERRIDENT) != 0 )
				new_conf->logIdent	= strdup(value);
		} else if( strcmp(key, "numThreads") == 0 ) {
			new_conf->numThreads	= atoi(value);
			if( new_conf->numThreads != 0 && new_conf->numThreads < 6 ) {
				log(LOG_ERR, "%s:%d bad parameter. numThreads is 6 and more.", file, line);
				hasError = true;
			}

		} else if( strcmp(key, "maxDeviceThreads") == 0 ) {
			new_conf->maxDeviceProcess	= atoi(value);
			if( new_conf->maxDeviceProcess < 3 ) {
				log(LOG_ERR, "%s:%d bad parameter. maxDeviceThreads is 3 and more.", file, line);
				hasError = true;
			}
		} else if( strcmp(key, "minDeviceThreads") == 0 ) {
			new_conf->minDeviceProcess	= atoi(value);
			if( new_conf->minDeviceProcess < 1 ) {
				log(LOG_ERR, "%s:%d bad parameter. minDeviceThreads is 1 and more.", file, line);
				hasError = true;
			}
		} else if( strcmp(key, "maxDevices") == 0 ) {
			new_conf->maxDevices	= atoi(value);
			if( new_conf->maxDevices < 8 ) {
				log(LOG_ERR, "%s:%d bad parameter. maxDevices is 8 and more.", file, line);
				hasError = true;
			}
		} else if( strcmp(key, "maxConnections") == 0 ) {
			new_conf->maxConnections	= atoi(value);
			if( new_conf->maxConnections < 8 ) {
				log(LOG_ERR, "%s:%d bad parameter. maxConnections is 8 and more.", file, line);
				hasError = true;
			}
		} else if( strcmp(key, "serverName") == 0 ) {
			if( new_conf->serverName) free(new_conf->serverName);
			new_conf->serverName = strdup(value);
		} else if( strcmp(key, "port") == 0 ) {
			new_conf->port		= atoi(value);
		} else if( strcmp(key, "bind") == 0 ) {
			if( new_conf->bind ) free(new_conf->bind);
			new_conf->bind	= strdup(value);
		} else if( strcmp(key, "authPassword") == 0 ) {
			if( new_conf->authPasswd ) free(new_conf->authPasswd);
			new_conf->authPasswd	= strdup(value);
		} else if( strcmp(key, "authMethod") == 0 ) {
			if( new_conf->authMethod ) free(new_conf->authMethod);
			new_conf->authMethod	= strdup(value);
		} else if( strcmp(key, "timeout") == 0 ) {
			new_conf->timeout	= atoi(value);
		} else if( strcmp(key, "maxWaitTime") == 0 ) {
			new_conf->maxWaitTime	= atoi(value);
		} else if( strcmp(key, "syncQueueWaitTime") == 0 ) {
			new_conf->syncQueueWaitTime	= atoi(value);
		} else if( strcmp(key, "heartBeetInterval") == 0 ) {
			new_conf->heartBeetInterval	= atoi(value);

		} else if( strcmp(key, "maxProcessThreads") == 0 ) {
			new_conf->maxCommandProcess = atoi(value);
			if( new_conf->maxCommandProcess < 4 ) {
				log(LOG_ERR, "%s:%d bad parameter. maxProcessThreads is 4 and more.", file, line);
				hasError = true;
			}

		} else if( strcmp(key, "maxHTTPThreads") == 0 ) {
			new_conf->maxHTTPProcess	= atoi(value);
			if( new_conf->maxHTTPProcess < 1 ) {
				log(LOG_ERR, "%s:%d bad parameter. maxHTTPProcess is 1 and more.", file, line);
				hasError = true;
			}
		} else if( strcmp(key, "httpBufferSize") == 0 ) {
			new_conf->httpBufferSize	= atoi(value);
			if( new_conf->httpBufferSize < 1024 ) {
				log(LOG_ERR, "%s:%d bad parameter. httpBufferSize is 1024 and more.", file, line);
				hasError = true;
			}

		} else if( strcmp(key, "maxDbThreads") == 0 ) {
			new_conf->maxDBProcess	= atoi(value);
			if( new_conf->maxDBProcess < 1 ) {
				log(LOG_ERR, "%s:%d bad parameter. maxDbThreads is 1 and more.", file, line);
				hasError = true;
			}
		} else if( strcmp(key, "dbServer") == 0 ) {
			if( new_conf->dbServer ) free(new_conf->dbServer);
			new_conf->dbServer	= strdup(value);
		} else if( strcmp(key, "dbPort") == 0 ) {
			new_conf->dbPort	= atoi(value);
		} else if( strcmp(key, "dbAuth") == 0 ) {
			if( new_conf->dbAuth ) free(new_conf->dbAuth);
			new_conf->dbAuth	= strdup(value);
		} else if( strcmp(key, "adminId") == 0 ) {
			if( new_conf->adminId ) free(new_conf->adminId);
			new_conf->adminId	= strdup(value);
		} else if( strcmp(key, "adminPass") == 0 ) {
			if( new_conf->adminPass ) free(new_conf->adminPass);
			new_conf->adminPass	= strdup(value);

		} else if( strcmp(key, "masterServer") == 0 ) {
			if( new_conf->masterServer ) free(new_conf->masterServer);
			new_conf->masterServer	= strdup(value);
		} else if( strcmp(key, "masterPort") == 0 ) {
			new_conf->masterPort	= atoi(value);
		} else if( strcmp(key, "masterAuth") == 0 ) {
			if( new_conf->masterAuth ) free(new_conf->masterAuth);
			new_conf->masterAuth	= strdup(value);
		} else if( strcmp(key, "cloneMaster") == 0 ) {
		} else {
			log(LOG_ERR, "%s:%d unknown keyword error. (%s)", file, line, key);
			hasError = true;
			line++;
			continue;
		}
		line++;
	}
	fclose(fp);

	if( !hasError ) {
		if( new_conf->minDeviceProcess > new_conf->maxDeviceProcess ) {
			log(LOG_ERR, "maxDeviceThreads(%d) is %d and more.", new_conf->maxDeviceProcess, new_conf->minDeviceProcess);
			hasError = true;
		}
		if( new_conf->minCommandProcess > new_conf->maxCommandProcess ) {
			log(LOG_ERR, "maxProcessThreads(%d) is %d and more.", new_conf->maxCommandProcess, new_conf->minCommandProcess);
			hasError = true;
		}
		if( new_conf->minDBProcess	> new_conf->maxDBProcess ) {
			log(LOG_ERR, "maxDBThreads(%d) is %d and more.", new_conf->maxDBProcess, new_conf->minDBProcess);
			hasError = true;
		}
		if( new_conf->minHTTPProcess > new_conf->maxHTTPProcess ) {
			log(LOG_ERR, "maxHTTPThreads(%d) is %d and more.", new_conf->maxHTTPProcess, new_conf->minHTTPProcess);
			hasError = true;
		}

		int nThread = new_conf->minDeviceProcess + new_conf->maxCommandProcess + 4;
		if( new_conf->numThreads == 0 ) {
			int nMax = new_conf->maxDBProcess;
			if( nMax < new_conf->maxHTTPProcess ) nMax = new_conf->maxHTTPProcess;
			new_conf->numThreads = nThread + nMax;
		} else {
			if( new_conf->numThreads < nThread ) {
				log(LOG_DEBUG, "numThread is too small.");
				hasError	= true;
			}
		}
	}

	if( hasError ) {
		log(LOG_DEBUG, "configure file has error.");
		freeConf(new_conf);
		return false;
	}

	if( this->mConf ) freeConf(this->mConf);
	this->mConf = new_conf;
	log(LOG_DEBUG,"confFile: %s", mConf->confFile);
	log(LOG_DEBUG,"pidFile: %s", mConf->pidFile);
	log(LOG_DEBUG,"daemonaize: %s", mConf->daemonaize ? "yes" : "no");
	log(LOG_DEBUG,"logLevel: %d", mConf->logLevel);
	log(LOG_DEBUG,"logIdent: %s", mConf->logIdent);
	log(LOG_DEBUG,"numThreads: %d", mConf->numThreads);

	log(LOG_DEBUG,"maxDeviceThreads: %d", mConf->maxDeviceProcess);
	log(LOG_DEBUG,"minDeviceThreads: %d", mConf->minDeviceProcess);
	log(LOG_DEBUG,"maxDevices: %d", mConf->maxDevices);
	log(LOG_DEBUG,"maxConnections: %d", mConf->maxConnections);
	log(LOG_DEBUG,"port: %d", mConf->port);
	log(LOG_DEBUG,"bind: %s", mConf->bind);
	log(LOG_DEBUG,"authPasswd: %s", mConf->authPasswd ? mConf->authPasswd : "(empty)");
	log(LOG_DEBUG,"authMethod: %s", mConf->authMethod ? mConf->authMethod : "(empty)");
	log(LOG_DEBUG,"timeout: %d", mConf->timeout);
	log(LOG_DEBUG,"maxWaitTime: %d", mConf->maxWaitTime);
	log(LOG_DEBUG,"syncQueueWaitTime: %d", mConf->syncQueueWaitTime);
	log(LOG_DEBUG,"heartBeetInterval: %d", mConf->heartBeetInterval);

	log(LOG_DEBUG,"maxProcessThreads: %d", mConf->maxCommandProcess);
	log(LOG_DEBUG,"minProcessThreads: %d", mConf->minCommandProcess);

	log(LOG_DEBUG,"maxHTTPThreads: %d", mConf->maxHTTPProcess);
	log(LOG_DEBUG,"minHTTPThreads: %d", mConf->minHTTPProcess);
	log(LOG_DEBUG,"httpBufferSize: %d", mConf->httpBufferSize);

	log(LOG_DEBUG,"maxDBThreads: %d", mConf->maxDBProcess);
	log(LOG_DEBUG,"minDBThreads: %d", mConf->minDBProcess);
	log(LOG_DEBUG,"dbServer: %s", mConf->dbServer);
	log(LOG_DEBUG,"dbPort: %d", mConf->dbPort);
	log(LOG_DEBUG,"dbAuth: %s", mConf->dbAuth ? mConf->dbAuth : "(empty)");

	log(LOG_DEBUG,"masterServer: %s", mConf->masterServer ? mConf->masterServer : "(empty)");
	log(LOG_DEBUG,"masterPort: %d", mConf->masterPort);
	log(LOG_DEBUG,"masterAuth: %s", mConf->masterAuth ? mConf->masterAuth : "(empty)");
	log(LOG_DEBUG,"cloneMaster: %s", mConf->cloneMaster ? "yes" : "no");
	log(LOG_DEBUG, "Loading configure file complete.");
	return true;
}

Conf * Sentinel::getConf()
{
	return mConf;
}

bool Sentinel::configure()
{
	log(LOG_DEBUG, "Sentinel::configure");
	mScheduler.clear();
	mSessions.clear();
	mDevices.clear();
	mCommand.clear();
	mListener.clear();
	mDatabase.clear();
	mRequester.clear();

	log(LOG_DEBUG, "Sentinel::configure: configure thread pool");
	if( !mScheduler.configure(this, mConf) ) {
		return false;
	}
	log(LOG_DEBUG, "Sentinel::configure: configure database %x", &mDatabase);
	if( !mDatabase.configure(this, mConf) ) {
		return false;
	}
	log(LOG_DEBUG, "Sentinel::configure: configure resuqster %x", &mRequester);
	if( !mRequester.configure(this, mConf) ) {
		return false;
	}
	log(LOG_DEBUG, "Sentinel::configure: configure listener %x", &mListener);
	if( !mListener.configure(this, mConf) ) {
		return false;
	}
	log(LOG_DEBUG, "Sentinel::configure: configure command processor %x", &mCommand);
	if( !mCommand.configure(this, mConf) ) {
		return false;
	}
	log(LOG_DEBUG, "Sentinel::configure: configure device slot %x", &mDevices);
	if( !mDevices.configure(this, mConf) ) {
		return false;
	}
	log(LOG_DEBUG, "Sentinel::configure: configure session pool %x", &mSessions);
	if( !mSessions.configure(this, mConf) ) {
		return false;
	}
	/*
	log(LOG_DEBUG, "Sentinel::configure: configure services");
	if( !services.configure(this, conf) ) {
		return false;
	}
	*/
	log(LOG_DEBUG, "Sentinel::configure: configure taskManager");
	if( !mTaskManager.configure(this, mConf) ) {
		return false;
	}
	log(LOG_DEBUG, "Sentinel::configure: complete");
	return true;
}

void Sentinel::sigtermProc(int fd, short event, void *args)
{
	Sentinel *obj = (Sentinel*)args;
	obj->log(LOG_DEBUG, "accept SIGTERM");
	event_base_loopbreak(obj->eventBase);
	obj->waitThreads();
}

void Sentinel::sighupProc(int fd, short event, void *args)
{
	Sentinel *obj = (Sentinel*)args;
	obj->log(LOG_DEBUG, "accept SIGHUP");
	event_base_loopbreak(obj->eventBase);
	obj->waitThreads();
	if( !obj->loadConfig(NULL) ) {
		obj->log(LOG_ERR, "load configure error");
		return;
	}
	if( !obj->startThreads() ) {
		return;
	}
	obj->log(LOG_INFO, "restart");}

void Sentinel::sigintProc(int fd, short event, void *args)
{
	Sentinel *obj = (Sentinel*)args;
	obj->log(LOG_DEBUG, "accept SIGINT");
	event_base_loopbreak(obj->eventBase);
	obj->waitThreads();
}

bool Sentinel::run()
{
	log(LOG_INFO, "sentinel start");
	if( !configure() ) {
		return false;
	}
	log(LOG_DEBUG, "sentinel configure");

	eventBase = event_base_new();
	sigterm = evsignal_new(eventBase, SIGTERM, sigtermProc, this);
	evsignal_add(sigterm, NULL);
	sighup = evsignal_new(eventBase, SIGHUP, sighupProc, this);
	evsignal_add(sighup, NULL);
	sigint = evsignal_new(eventBase, SIGINT, sigintProc, this);
	evsignal_add(sigint, NULL);
/*
	sigset_t ss;
	sigemptyset(&ss);
    sigfillset(&ss);
    pthread_sigmask(SIG_BLOCK, &ss, NULL);
*/
	if( !startThreads() ) {
		return false;
	}
	event_base_dispatch(eventBase);
	/*
	int sn;
	struct timespec ts={5,0};
	while(1) {
		log(LOG_DEBUG, "watch signal");
		siginfo_t info;
		if( sigtimedwait(&ss, &info, &ts) > 0) {
			if( info.si_signo == SIGTERM ) {
				sig_term	= 1;
				log(LOG_DEBUG, "accept SIGTERM");
				waitThreads();
				sig_term	= 0;
				break;
			}
			if( info.si_signo == SIGINT ) {
				sig_int		= 1;
				log(LOG_DEBUG, "accept SIGINT");
				waitThreads();
				sig_int		= 0;
				break;
			}
			if( info.si_signo == SIGHUP ) {
				sig_hup		= 1;
				log(LOG_DEBUG, "accept SIGHUP");
				waitThreads();
				sig_hup		= 0;
				if( !loadConfig(NULL) ) {
					log(LOG_ERR, "load configure error");
					break;
				}
				if( !startThreads() ) {
					return false;
				}
				log(LOG_INFO, "restart");
			}
		}
	}
	*/

    evsignal_del(sigterm);
    evsignal_del(sighup);
    evsignal_del(sigint);
    event_free(sigterm);
    event_free(sighup);
    event_free(sigint);
    event_base_free(eventBase);
//    pthread_sigmask(SIG_UNBLOCK, &ss, NULL);
	log(LOG_INFO, "sentinel terminate");
	return true;
}

Conf * Sentinel::allocConf(const char *file)
{
	Conf *new_conf = (Conf*)malloc(sizeof(Conf));
	if( new_conf == NULL ) {
		return NULL;
	}

	if( file )
		new_conf->confFile	= strdup(file);
	else
		new_conf->confFile	= NULL;

	new_conf->pidFile		= strdup(SST_PIDFILE);
	new_conf->daemonaize	= false;
	new_conf->logLevel		= LogLevelError;
	new_conf->logIdent		= strdup(SST_SYSLOGIDENT);
	new_conf->numThreads	= 18;

	new_conf->maxDeviceProcess	= 6;
	new_conf->minDeviceProcess	= 1;
	new_conf->maxDevices		= 1000;
	new_conf->maxConnections	= 1000;
	new_conf->serverName		= NULL;
	new_conf->port				= SST_PORT;
	new_conf->bind				= NULL;
	new_conf->authPasswd		= NULL;
	new_conf->authMethod		= NULL;
	new_conf->timeout			= SST_TIMEOUT;
	new_conf->maxWaitTime		= SST_MAXWAITTIME;
	new_conf->syncQueueWaitTime	= SST_SYNCWAITTIME;
	new_conf->heartBeetInterval	= SST_HEARTBEETINTERVAL;

	new_conf->maxCommandProcess	= 4;
	new_conf->minCommandProcess	= 2;

	new_conf->maxHTTPProcess	= 4;
	new_conf->minHTTPProcess	= 2;
	new_conf->httpBufferSize	= 1024*16;

	new_conf->maxDBProcess		= 4;
	new_conf->minDBProcess		= 2;
	new_conf->dbServer			= strdup("localhost");
	new_conf->dbPort			= 6379;
	new_conf->dbAuth			= NULL;
	new_conf->adminId			= strdup("admin@system");
	new_conf->adminPass			= strdup("zz1119");

	new_conf->masterServer		= NULL;
	new_conf->masterPort		= 0;
	new_conf->masterAuth		= NULL;
	new_conf->cloneMaster		= false;

	return new_conf;
}

void Sentinel::freeConf(Conf *conf)
{
	log(LOG_DEBUG, "freeConf");
	if( conf ) {
		if( conf->confFile ) free(conf->confFile);
		if( conf->pidFile ) free(conf->pidFile);
		if( conf->serverName) free(conf->serverName);
		if( conf->bind ) free(conf->bind);
		if( conf->authPasswd ) free(conf->authPasswd);
		if( conf->authMethod ) free(conf->authMethod);
		if( conf->logIdent ) free(conf->logIdent);
		if( conf->dbServer ) free(conf->dbServer);
		if( conf->dbAuth ) free(conf->dbAuth);
		if( conf->masterServer ) free(conf->masterServer);
		if( conf->masterAuth ) free(conf->masterAuth);
		if( conf->adminId ) free(conf->adminId);
		if( conf->adminPass ) free(conf->adminPass);
		free(conf);
	}
}

SessionPool &Sentinel::getSessionPool()
{
	return mSessions;
}

int	Sentinel::getLoad() const
{
	size_t a = mSessions.getActiveSessions();
	size_t n = mSessions.getCapacity();
	return (a*100/n);
}

bool Sentinel::checkDatabase()
{
	return mDatabase.checkDatabase();
}

bool Sentinel::startThreads()
{
	log(LOG_DEBUG, "sentinel processor run");

	log(LOG_DEBUG, "Sentinel::configure: connect database");
	int nRetry=30;
	int i;
	for(i = 0; i < nRetry; i++ ) {
		if( mDatabase.connect() ) break;
		log(LOG_WARNING, "wait for database connection. %d/%d", i + 1, nRetry);
		::sleep(10);
	}
	if( i == nRetry ) {
		log(LOG_ERR, "can't connect to database.");
		return false;
	}
	if( !checkDatabase() ) {
		return false;
	}
	log(LOG_DEBUG, "Sentinel::configure: connect database ok.");

	log(LOG_DEBUG, "Sentinel::configure: create task slot.");
	for( i = 0; i < mConf->minDeviceProcess; i++ ) {
		Task *task = mTaskManager.createTaskFromSession(NULL);
		QueueNode *q = new QueueNode(task);
		q->setProcess(&mListener, CMDComListen, 0);
		task->startTask(q);
	}


	log(LOG_DEBUG, "Sentinel::configure: create task slot ok.");
	return true;
}

void Sentinel::waitThreads()
{
	// TODO: processor.stop();
}

}
