/*
 * database.cpp
 *
 *  Created on: 2012/06/08
 *      Author: yasuoki
 */

#include <stdlib.h>
#include "proc_database.h"
#include "redis_db.h"
#include "queue.h"
#include "sentinel.h"

namespace SST {

QnDBQuery::QnDBQuery(Task *task) : QueueNode(task)
{
	mType	= QueueTypeDbQuery;
	mProc	= &task->getSentinel()->getDatabase();
	mFunc	= (int)DBCMDNull;
	mQuery	= NULL;
}

QnDBQuery::~QnDBQuery()
{
	if( mQuery ) free(mQuery);
}
/////////////////////////////////////////////////////////////////////
QnDBRegister::QnDBRegister(Task *task) : QueueNode(task)
{
	mType	= QueueTypeDbRegister;
	mRegist	= NULL;
	mProc	= &(task->getSentinel()->getDatabase());
	mFunc	= (int)DBCMDNull;
}

QnDBRegister::~QnDBRegister() {
}

/////////////////////////////////////////////////////////////////////
QnDBResult::QnDBResult(Task *task) : QnResult(task)
{
	mType		= (int)QueueTypeDbResult;
	mResult		= NULL;
}

QnDBResult::~QnDBResult()
{
	if( mResult ) free(mResult);
}

/////////////////////////////////////////////////////////////////////

Database::Database()
{
	mThreads = 0;
}

Database::~Database()
{

}

bool Database::configure(Sentinel *obj, Conf *conf)
{
	if( !Process::configure(obj, conf) )
		return false;

	mMaxThread	= conf->maxDBProcess;
	mMinThread	= conf->minDBProcess;

	mThreads 	= conf->maxDBProcess;
	if( mThreads == 0 ) mThreads = 1;
	return true;
}

void Database::clear()
{
	mDBPool.clear();
}

bool Database::connect()
{
	int i;
	for( i = 0; i < mThreads; i++ ) {
		DBPort *port = new RedisDB();
		if( !port->configure(mSentinel, mSentinel->getConf()) ) {
			return false;
		}
		if( !port->connect() ) {
			return false;
		}
		mDBPool.addPort(port);
	}
	return true;
}

bool Database::disconnect()
{
	int i;
	for( i = 0; i < mThreads; i++ ) {
		DBPort *port = mDBPool.assignPort();
		if( port == NULL ) break;
		port->close();
		port->endProcess();
	}
	return true;
}

bool Database::checkDatabase()
{
	DBPort *port = mDBPool.assignPort();
	if( port == NULL ) {
		mSentinel->log(LOG_ERR, "Database: can't assign database connection.");
		return false;
	}

	QueryKeyType aref;
	aref.mId	= NULL;
	aref.mName	= mSentinel->getConf()->adminId;
	char aid[SST_ACCOUNTID_LEN+1];
	ErrorCode e = confirmAID(NULL, port, &aref, aid);
	if( e != ErrorOk ) {
		if( e != ErrorNotFound ) {
			mDBPool.freePort(port);
			mSentinel->log(LOG_ERR, "Database: %s. Confirm account failed. account=%s",
					codeToMessgae(e), aid);
			return false;
		}

		Account acc;
		memset(&acc, 0, sizeof(acc));
		acc.uid			= (char*)aref.mName;
		acc.name1		= (char*)"System Administrator";
		acc.name2		= (char*)"Builtin Account";
		acc.authType	= AuthPasswd;
		acc.authData	= (char*)mSentinel->getConf()->adminPass;
		acc.status		= AccountNormal;
		acc.level		= AccountSystemAdmin;
		e = port->setAccountInfo(&acc, aid);
		if( e != ErrorOk ) {
			mDBPool.freePort(port);
			mSentinel->log(LOG_ERR, "Database: %s. setAccount failed. account=%s",
					codeToMessgae(e), aref.mName);
			return false;
		}
		mSentinel->log(LOG_INFO, "Database: Create system account.");

		ServiceInfo svc;
		memset(&svc, 0, sizeof(svc));

		svc.name		= (char*)"SystemService";
		svc.title		= (char*)"System Service";
		svc.desc		= (char*)"Builtin System Service";
		svc.version		= (char*)"1.0";
		svc.adminId		= aid;
		svc.scope		= ScopePublic;
		svc.authType	= AuthUser; // AuthPasswd;
		svc.authData	= NULL; //(char*)mSentinel->getConf()->adminPass;
		svc.status		= ServiceOpen;
		char sid[SST_SERVICEID_LEN+1];
		e = port->setServiceInfo(&svc, sid);
		if( e != ErrorOk ) {
			mDBPool.freePort(port);
			mSentinel->log(LOG_ERR, "Database: %s. setService failed. service=%s",
					codeToMessgae(e), svc.name);
			return false;
		}

		mSentinel->log(LOG_INFO, "Database: Create system service.");
	}
	mDBPool.freePort(port);
	return true;
}

ErrorCode Database::confirmSID(QueueNode *q, DBPort *port, const QueryKeyType *key, char *sid)
{
	Task *task = q->getTask();

	char __sid[SST_SERVICEID_LEN+1];
	const char *_sid = NULL;
	const char *session_sid = task->getServiceId();
	if( key->mId == NULL && key->mName == NULL ) {
		_sid = session_sid;
	} else if( key->mId == NULL ) {
		ErrorCode e = port->getServiceId(key->mName, __sid);
		if( e != ErrorOk ) {
			return e;
		}
		_sid = __sid;
	} else {
		_sid = key->mId;
	}
	if( _sid == NULL ) {
		taskLog(q, LOG_DEBUG, "confirmSID: task=%x q=%x session_sid=%s sid is not specified", task, q, session_sid);
		return  ErrorParameter;
	}
	if( _sid == NULL )
		sid[0] = 0;
	else
		strcpy(sid, _sid);
	return ErrorOk;
}

ErrorCode Database::confirmRID(QueueNode *q, DBPort *port, const QueryKey1 *key, char *rid)
{
	char __rid[SST_RESOURCEID_LEN+1];
	const char *_rid = NULL;
	if( key->mKey1.mId == NULL && key->mKey1.mName == NULL ) {
		return ErrorParameter;
	} else if( key->mKey1.mId == NULL ) {
		char sid[SST_SERVICEID_LEN+1];
		ErrorCode e = confirmSID(q, port, &key->mSREF, sid);
		if( e != ErrorOk ) {
			return e;
		}
		if( sid[0] == 0 ) {
			return ErrorParameter;
		}
		e = port->getResourceId(sid, key->mKey1.mName, __rid);
		if( e != ErrorOk ) {
			return e;
		}
		_rid	= __rid;
	} else {
		_rid	= key->mKey1.mId;
	}
	if( _rid == NULL ) {
		taskLog(q, LOG_DEBUG, "confirmRID: q=%s rid is not specified", q);
		return  ErrorParameter;
	}
	strcpy(rid, _rid);
	return ErrorOk;
}

ErrorCode Database::confirmGID(QueueNode *q, DBPort *port, const QueryKey1 *key, char *gid)
{
	char __gid[SST_GROUPID_LEN+1];
	const char *_gid= NULL;
	if( key->mKey1.mId == NULL && key->mKey1.mName == NULL ) {
		taskLog(q, LOG_DEBUG, "confirmRID: q=%s rid is not specified", q);
		return ErrorParameter;
	} else if( key->mKey1.mId == NULL ) {
		char sid[SST_SERVICEID_LEN+1];
		ErrorCode e = confirmSID(q, port, &key->mSREF, sid);
		if( e != ErrorOk ) {
			return e;
		}
		if( sid[0] == 0 ) {
			return ErrorParameter;
		}

		e = port->getGroupId(sid, key->mKey1.mName, __gid);
		if( e != ErrorOk ) {
			return e;
		}
		_gid	= __gid;
	} else {
		_gid	= key->mKey1.mId;
	}
	if( _gid == NULL ) {
		taskLog(q, LOG_DEBUG, "confirmGID: q=%s gid is not specified", q);
		return  ErrorParameter;
	}
	strcpy(gid, _gid);
	return ErrorOk;
}

ErrorCode Database::confirmAID(QueueNode *q, DBPort *port, const QueryKeyType *key, char *aid)
{
	char __aid[SST_ACCOUNTID_LEN+1];
	const char *_aid = NULL;

	const char *	session_id = NULL;
	AccountLevel	session_lv = AccountSystemService;

	Task *task = NULL;
	if( q ) {
		task = q->getTask();
		if( task ) {
			session_id	= task->getAccountId();
			session_lv	= task->getAccountLevel();
		}
	}

	if( key->mId == NULL && key->mName == NULL ) {
		_aid = session_id;
	} else if( key->mId == NULL ) {
		ErrorCode e = port->getAccountId(key->mName, __aid);
		if( e != ErrorOk ) {
			taskLog(q, LOG_DEBUG, "confirmAID: aid not found. q=%s name=%s code=%d", q, key->mName, e);
			return e;
		}
		_aid = __aid;
	} else {
		_aid = key->mId;
	}

	if( _aid == NULL ) {
		taskLog(q, LOG_DEBUG, "confirmAID: q=%s aid is not specified", q);
		return ErrorParameter;
	}
	strcpy(aid, _aid);
	return ErrorOk;
}

ErrorCode Database::confirmDEV(QueueNode *q, DBPort *port, const QueryKeyType *key, char *did)
{
	char __did[SST_DEVICEID_LEN+1];
	const char *_did = NULL;

	Task *task = q->getTask();
	const char *session_did = NULL;
	session_did	= task->getDeviceId();

	if( key->mId == NULL && key->mName == NULL ) {
		_did = session_did;
	} else if( key->mId == NULL ) {
		ErrorCode e = port->getDeviceId(key->mName, __did);
		if( e != ErrorOk ) {
			return e;
		}
		_did = __did;
	} else {
		_did = key->mId;
	}

	if( _did == NULL ) {
		taskLog(q, LOG_DEBUG, "confirmDEV: q=%s did is not specified", q);
		return ErrorParameter;
	}
	strcpy(did, _did);
	return ErrorOk;
}

ErrorCode Database::confirmMID(QueueNode *q, DBPort *port, const QueryKeyType *key, char *mid)
{
	if( key->mId == NULL ) {
		taskLog(q, LOG_DEBUG, "confirmMID: q=%s mid is not specified", q);
		return ErrorParameter;
	}
	strcpy(mid, key->mId);
	return ErrorOk;
}
ErrorCode Database::confirmCID(QueueNode *q, DBPort *port, const QueryKeyType *key, char *cid)
{
	if( key->mId == NULL ) {
		taskLog(q, LOG_DEBUG, "confirmCID: q=%s cid is not specified", q);
		return ErrorParameter;
	}
	strcpy(cid, key->mId);
	return ErrorOk;
}

const char *Database::getFuncName(int id)
{
	switch(id) {
	case DBCMDService:			return "QueryService";
	case DBCMDServicePWList:	return "QueryServicePWList";
	case DBCMDServicePRList:	return "QueryServicePRList";
	case DBCMDServiceList:		return "QueryServiceList";
	case DBCMDServiceGroupList:	return "QueryServiceGroupList";
	case DBCMDAccount:			return "QueryAccount";
	case DBCMDAccountList:		return "QueryAccountList";
	case DBCMDAccountIsLock:	return "QueryAccountIsLock";
	case DBCMDDevice:			return "QueryDevice";
	case DBCMDDeviceCount:		return "QueryDeviceCount";
	case DBCMDDeviceList:		return "QueryDeviceList";
	case DBCMDDeviceShareCount:	return "QueryDeviceShareCount";
	case DBCMDDeviceShareList:	return "QueryDeviceShareList";
	case DBCMDDeviceShare:		return "QueryDeviceShare";
	case DBCMDGroup:			return "QueryGroup";
	case DBCMDGroupCount:		return "QueryGroupCount";
	case DBCMDGroupList:		return "QueryGroupList";
	case DBCMDIsGroupMember:	return "QueryIsGroupMember";
	case DBCMDResource:			return "QueryResource";
	case DBCMDResourceList:		return "QueryResourceList";
	case DBCMDShare:			return "QueryShare";
	case DBCMDShareList:		return "QueryShareList";
	case DBCMDSync:				return "QuerySync";
	case DBCMDSyncQueue:		return "QuerySyncQueue";
	case DBCMDNode:				return "QueryNode";
	case DBCMDNodeList:			return "QueryNodeList";
	case DBCMDRemAccount:		return "RemAccount";
	case DBCMDRemDevice:		return "RemDevice";
	case DBCMDRemGroupAccount:	return "RemGroupAccount";
	case DBCMDRemGroup:			return "RemGroup";
	case DBCMDRemResource:		return "RemResource";
	case DBCMDRemShare:			return "RemShare";
	case DBCMDRegService:		return "RegService";
	case DBCMDRegAccountLock:	return "RegAccountLock";
	case DBCMDRegAccountUnlock:	return "RegAccountUnlock";
	case DBCMDRegAccount:		return "RegAccount";
	case DBCMDRegDevice:		return "RegDevice";
	case DBCMDRegGroup:			return "RegGroup";
	case DBCMDRegResource:		return "RegResource";
	case DBCMDRegShare:			return "RegShare";
	case DBCMDRegDeviceShare:	return "RegDeviceShare";
	case DBCMDRegSync:			return "RegSync";
	case DBCMDRegSyncQueue:		return "RegSyncQueue";
	case DBCMDRegNode:			return "RegNode";
	case DBCMDRegGroupAccount:	return "RegGroupAccount";
	}
	return Process::getFuncName(id);
}

static int nThread=0;
bool Database::exec(QueueNode *q)
{
	Task *task = q->getTask();

	int nt;
	{
		WRITE_LOCK(mLock);
		nt = ++nThread;
	}
	DBPort *port = mDBPool.assignPort();
	if( port == NULL ) {
		taskLog(q, LOG_ERR, "%s::exec: can't assign database connection. task=%x q=%x qid=%d", getName(), task, q, q->mId);
		{
			WRITE_LOCK(mLock);
			nThread--;
		}
		return false;
	}

	bool r;
	switch(q->getType()) {
	case QueueTypeDbQuery:
		switch(q->getFunction()) {
		case DBCMDService:			r = cmdQueryService((QnDBQuery*)q, port);		break;
		case DBCMDServicePWList:	r = cmdQueryServicePWList((QnDBQuery*)q, port);	break;
		case DBCMDServicePRList:	r = cmdQueryServicePRList((QnDBQuery*)q, port);	break;
		case DBCMDServiceList:		r = cmdQueryServiceList((QnDBQuery*)q, port);	break;
		case DBCMDServiceGroupList:	r = cmdQueryServiceGroupList((QnDBQuery*)q, port);	break;
		case DBCMDAccount:			r = cmdQueryAccount((QnDBQuery*)q, port);		break;
		case DBCMDAccountList:		r = cmdQueryAccountList((QnDBQuery*)q, port);	break;
		case DBCMDAccountIsLock:	r = cmdQueryAccountIsLock((QnDBQuery*)q, port);	break;
		case DBCMDDevice:			r = cmdQueryDevice((QnDBQuery*)q, port);		break;
		case DBCMDDeviceCount:		r = cmdQueryDeviceCount((QnDBQuery*)q, port);	break;
		case DBCMDDeviceList:		r = cmdQueryDeviceList((QnDBQuery*)q, port);	break;
		case DBCMDDeviceShareCount:	r = cmdQueryDeviceShareCount((QnDBQuery*)q, port);	break;
		case DBCMDDeviceShareList:	r = cmdQueryDeviceShareList((QnDBQuery*)q, port);	break;
		case DBCMDDeviceShare:		r = cmdQueryDeviceShare((QnDBQuery*)q, port);	break;
		case DBCMDGroup:			r = cmdQueryGroup((QnDBQuery*)q, port);			break;
		case DBCMDGroupCount:		r = cmdQueryGroupCount((QnDBQuery*)q, port);	break;
		case DBCMDGroupList:		r = cmdQueryGroupList((QnDBQuery*)q, port);		break;
		case DBCMDIsGroupMember:	r = cmdQueryIsGroupMember((QnDBQuery*)q, port);	break;
		case DBCMDResource:			r = cmdQueryResource((QnDBQuery*)q, port);		break;
		case DBCMDResourceList:		r = cmdQueryResourceList((QnDBQuery*)q, port);	break;
//		case DBCMDGadget:			r = cmdQueryGadget((QnDBQuery*)q, port);		break;
//		case DBCMDGadgetList:		r = cmdQueryGadgetList((QnDBQuery*)q, port);	break;
		case DBCMDShare:			r = cmdQueryShare((QnDBQuery*)q, port);			break;
		case DBCMDShareList:		r = cmdQueryShareList((QnDBQuery*)q, port);		break;
		case DBCMDSync:				r = cmdQuerySync((QnDBQuery*)q, port);			break;
		case DBCMDSyncQueue:		r = cmdQuerySyncQueue((QnDBQuery*)q, port);		break;
//		case DBCMDSession:			r = cmdQuerySession((QnDBQuery*)q, port);		break;
//		case DBCMDSessionList:		r = cmdQuerySessionList((QnDBQuery*)q, port);	break;
		case DBCMDNode:				r = cmdQueryNode((QnDBQuery*)q, port);			break;
		case DBCMDNodeList:			r = cmdQueryNodeList((QnDBQuery*)q, port);		break;
		case DBCMDRemAccount:		r = cmdRemAccount((QnDBQuery*)q, port);			break;
		case DBCMDRemDevice:		r = cmdRemDevice((QnDBQuery*)q, port);			break;
		case DBCMDRemGroupAccount:	r = cmdRemGroupAccount((QnDBQuery*)q, port);	break;
		case DBCMDRemGroup:			r = cmdRemGroup((QnDBQuery*)q, port);			break;
		case DBCMDRemResource:		r = cmdRemResource((QnDBQuery*)q, port);		break;
		case DBCMDRemShare:			r = cmdRemShare((QnDBQuery*)q, port);			break;
		default:
			taskLog(q, LOG_ERR, "unknown query type. q=%x type=%d query=%d", q, q->getType(), q->getFunction());
			r = false;
			break;
		}
		break;
	case QueueTypeDbRegister:
		switch(q->getFunction()) {
		case DBCMDRegService:		r = cmdRegService((QnDBRegister*)q, port);				break;
		case DBCMDRegAccountLock:	r = cmdRegAccountLock((QnDBRegister*)q, port);			break;
		case DBCMDRegAccountUnlock:	r = cmdRegAccountUnlock((QnDBRegister*)q, port);		break;
		case DBCMDRegAccount:		r = cmdRegAccount((QnDBRegister*)q, port);				break;
		case DBCMDRegDevice:		r = cmdRegDevice((QnDBRegister*)q, port);				break;
		case DBCMDRegGroup:			r = cmdRegGroup((QnDBRegister*)q, port);				break;
		case DBCMDRegResource:		r = cmdRegResource((QnDBRegister*)q, port);				break;
		case DBCMDRegShare:			r = cmdRegShare((QnDBRegister*)q, port);				break;
		case DBCMDRegDeviceShare:	r = cmdRegDeviceShare((QnDBRegister*)q, port);			break;
		case DBCMDRegSync:			r = cmdRegSync((QnDBRegister*)q, port);					break;
		case DBCMDRegSyncQueue:		r = cmdRegSyncQueue((QnDBRegister*)q, port);			break;
		case DBCMDRegNode:			r = cmdRegNode((QnDBRegister*)q, port);					break;
		case DBCMDRegGroupAccount:	r = cmdRegGroupAccount((QnDBRegister*)q, port);			break;
		default:
			taskLog(q, LOG_ERR, "%s::exec: unknown register type. q=%x type=%d register=%d", q, q->getType(), q->getFunction());
			r = false;
			break;
		}
		break;
	default:
	 	taskLog(q, LOG_ERR, "unknown queue type. q=%x type=%d.", getName(), q, q->getType());
	 	r	= false;
	 	break;
	}
	mDBPool.freePort(port);
	{
		WRITE_LOCK(mLock);
		nThread--;
	}
	return r;
}

void Database::stop(QueueNode *q)
{
	// TODO: not implement stop
}

bool  Database::cmdQueryService(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	QueryBase *qp = (QueryBase*)q->mQuery;

	char sid[SST_SERVICEID_LEN+1];
	ErrorCode e = confirmSID(q, port, &qp->mSREF, sid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. serice=%s,%s.", codeToMessgae(e), qp->mSREF.mId, qp->mSREF.mName);
	}
	if( sid[0] == 0 ) {
		return taskResponse(q, ErrorParameter, "%s. serice=%s,%s.", codeToMessgae(ErrorParameter), qp->mSREF.mId, qp->mSREF.mName);
	}

	ResultService *res;
	e = port->getServiceInfo(sid, &res);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. serice=%s,%s.", codeToMessgae(e), qp->mSREF.mId, qp->mSREF.mName);
	}

	QnDBResult *r =  new QnDBResult(task);
	r->resultService(res);
	return task->endTask(r);
}

bool Database::cmdQueryServicePWList(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	QueryRange *qp = (QueryRange*)q->mQuery;

	char sid[SST_SERVICEID_LEN+1];
	ErrorCode e = confirmSID(q, port, &qp->mSREF, sid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. serice=%s,%s.", codeToMessgae(e), qp->mSREF.mId, qp->mSREF.mName);
	}

	ResultList *res;
	e = port->getServicePWList(sid, qp->mRange.mStart, qp->mRange.mCount, &res);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. serice=%s,%s.", codeToMessgae(e), qp->mSREF.mId, qp->mSREF.mName);
	}
	PROC_END("cmdQueryServicePWList", task, q);
	QnDBResult *r =  new QnDBResult(task);
	r->resultList(res);
	return task->endTask(r);
}

bool Database::cmdQueryServicePRList(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	QueryRange *qp = (QueryRange*)q->mQuery;

	char sid[SST_SERVICEID_LEN+1];
	ErrorCode e = confirmSID(q, port, &qp->mSREF, sid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. serice=%s,%s.", codeToMessgae(e), qp->mSREF.mId, qp->mSREF.mName);
	}

	ResultList *res;
	e = port->getServicePRList(sid, qp->mRange.mStart, qp->mRange.mCount, &res);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. serice=%s,%s.", codeToMessgae(e), qp->mSREF.mId, qp->mSREF.mName);
	}
	QnDBResult *r =  new QnDBResult(task);
	r->resultList(res);
	return task->endTask(r);
}

bool Database::cmdQueryServiceList(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	QueryRange *qp = (QueryRange*)q->mQuery;

	if( (task->getAccountLevel() != AccountSystemService && task->getAccountLevel() != AccountSystemAdmin)) {
		return taskResponse(q, ErrorPermission, "%s. serice=%s,%s.", codeToMessgae(ErrorPermission), qp->mSREF.mId, qp->mSREF.mName);
	}
	ResultList *res = NULL;
	ErrorCode e = port->getServiceList(qp->mRange.mStart, qp->mRange.mCount, &res);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. serice=%s,%s.", codeToMessgae(e), qp->mSREF.mId, qp->mSREF.mName);
	}

	QnDBResult *r =  new QnDBResult(task);
	r->resultList(res);
	return task->endTask(r);
}

bool Database::cmdQueryServiceGroupList(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	QueryRange *qp = (QueryRange*)q->mQuery;

	char _sid[SST_SERVICEID_LEN+1];
	ErrorCode e = confirmSID(q, port, &qp->mSREF, _sid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. serice=%s,%s.", codeToMessgae(e), qp->mSREF.mId, qp->mSREF.mName);
	}

	if( task->getAccountLevel() == AccountSystemService ) {
		return taskResponse(q, ErrorPermission, "%s. not login.", codeToMessgae(ErrorPermission));
	}
	ResultService *svc = NULL;
	if( task->getAccountLevel() == AccountUser ||
		task->getAccountLevel() == AccountServiceAdmin ) {
		const char *sessionSid = task->getServiceId();
		if( !sessionSid || strcmp(sessionSid, _sid) != 0 ) {
			e = port->getServiceInfo(_sid, &svc);
			if( e != ErrorOk ) {
				return taskResponse(q, e, "%s. service load error.", codeToMessgae(e));
			}
			if( svc->mService.scope == ScopePrivate ) {
				if( svc ) free(svc);
				return taskResponse(q, ErrorPermission, "%s.", codeToMessgae(ErrorPermission));
			}
		}
	}

	size_t from		= qp->mRange.mStart;
	size_t need		= qp->mRange.mCount;
	size_t count	= qp->mRange.mCount;
	if( task->getAccountLevel() == AccountUser ) {
		count = -1;
	}
	ResultList *res = NULL;
	e = port->getServiceGroupList(_sid, from, count, &res);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. serice=%s,%s.", codeToMessgae(e), qp->mSREF.mId, qp->mSREF.mName);
	}

	if( task->getAccountLevel() == AccountUser ) {
		if( svc == NULL ) {
			e = port->getServiceInfo(_sid, &svc);
			if( e != ErrorOk ) {
				return taskResponse(q, e, "%s. serice=%s,%s.", codeToMessgae(e), qp->mSREF.mId, qp->mSREF.mName);
			}
		}
		if( svc->mService.scope == ScopePrivate ) {
			const char *aid = task->getAccountId();
			size_t i;
			size_t n = res->mList.values;
			for( i = 0; i < n && (need==(size_t)-1 || i < need); i++ ) {
				bool ok=false;
				if( aid ) {
					e = port->isGroupAccount(res->mList.value[i], aid);
					if( e == ErrorOk ) {
						ok = true;
					}
				}
				if( !ok ) {
					size_t v;
					for( v = i; v < res->mList.values-1; v++ ) {
						res->mList.value[v] = res->mList.value[v+1];
					}
					n--;
					i--;
				}
			}
			for( ; i < res->mList.values; i++ ) {
				res->mList.value[i] = NULL;
			}
			res->mList.values = n;
		}
	}
	if( svc ) free(svc);
	QnDBResult *r =  new QnDBResult(task);
	r->resultList(res);
	return task->endTask(r);
}

bool  Database::cmdQueryAccount(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	QueryKey1 *qp = (QueryKey1*)q->mQuery;

	char _aid[SST_ACCOUNTID_LEN+1];
	const char *pAID = NULL;

	if( qp->mKey1.mId == NULL && qp->mKey1.mName == NULL ) {
		const char *session_aid = task->getAccountId();
		if( session_aid != NULL ) {
			pAID = session_aid;
		} else {
			return taskResponse(q, ErrorParameter, "%s. account is not specified.",
					codeToMessgae(ErrorParameter));
		}
	} else {
		if( qp->mKey1.mId == NULL ) {
			ErrorCode e = port->getAccountId(qp->mKey1.mName, _aid);
			if( e != ErrorOk ) {
				return taskResponse(q, e, "%s. confirm account id failed. account=%s.",
						codeToMessgae(e),
						qp->mKey1.mName);
			}
			pAID = _aid;
		} else {
			pAID = qp->mKey1.mId;
		}
	}

	ResultAccount *res;
	ErrorCode e = port->getAccountInfo(pAID, &res);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. account=%s,%s.", codeToMessgae(e), qp->mKey1.mId, qp->mKey1.mName);
	}

	QnDBResult *r =  new QnDBResult(task);
	r->resultAccount(res);
	return task->endTask(r);
}

bool  Database::cmdQueryAccountList(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdQueryAccountList",task,q);
	QueryKey1Range *key = (QueryKey1Range*)q->mQuery;


	char _gid[SST_GROUPID_LEN+1];
	QueryKey1 gkey;
	gkey.mSREF	= key->mSREF;
	gkey.mKey1	= key->mKey1;
	ErrorCode e = confirmGID(q, port, &gkey, _gid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. service=%s,%s group=%s,%s.",
				codeToMessgae(e),
				gkey.mSREF.mId, gkey.mSREF.mName,
				gkey.mKey1.mId, gkey.mKey1.mName);
	}

	ResultList *res = NULL;
	e = port->getGroupAccountList(_gid, key->mRange.mStart, key->mRange.mCount, &res);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. service=%s,%s group=%s,%s.",
				codeToMessgae(e),
				gkey.mSREF.mId, gkey.mSREF.mName,
				gkey.mKey1.mId, gkey.mKey1.mName);
	}

	QnDBResult *r =  new QnDBResult(task);
	r->resultList(res);
	return task->endTask(r);
}

bool Database::cmdQueryAccountIsLock(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdQueryAccountIsLock", task, q);
	QueryKey1 *qp = (QueryKey1*)q->mQuery;

	char _sid[SST_SERVICEID_LEN+1];
	ErrorCode e = confirmSID(q, port, &qp->mSREF, _sid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. service=%s,%s.",
				codeToMessgae(e),
				qp->mSREF.mId, qp->mSREF.mName);
	}

	char _aid[SST_ACCOUNTID_LEN+1];
	e = confirmAID(q, port, &qp->mKey1, _aid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. service=%s,%s account=%s,%s.",
				codeToMessgae(e),
				qp->mSREF.mId, qp->mSREF.mName,
				qp->mKey1.mId, qp->mKey1.mName);
	}

	e = port->isServiceRefuse(_sid, _aid);
	QnDBResult *r =  new QnDBResult(task);
	r->setDBResult(e, codeToMessgae(e));
	return task->endTask(r);
}

bool  Database::cmdQueryDevice(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	QueryKey1 *qp = (QueryKey1*)q->mQuery;

	char did[SST_DEVICEID_LEN+1];

	taskLog(q, LOG_DEBUG, "cmdQueryDevice: dev=%s,%s", qp->mKey1.mId, qp->mKey1.mName);

	if( qp->mKey1.mId == NULL && qp->mKey1.mName == NULL ) {
		return taskResponse(q, ErrorParameter, "queryDevice: %s. device key is not specified.",
				codeToMessgae(ErrorParameter));
	} else {
		ErrorCode e = confirmDEV(q, port, &qp->mKey1, did);
		if( e != ErrorOk ) {
			return taskResponse(q, e, "queryDevice: %s. device key error. key=%s,%s",
					codeToMessgae(e),
					qp->mKey1.mId, qp->mKey1.mName);
		}
	}

	ResultDevice *res;
	ErrorCode e = port->getDeviceInfo(did, &res);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "queryDevice: %s. device load error. key=%s",
				codeToMessgae(e), did);
	}
	taskLog(q, LOG_DEBUG, "cmdQueryDevice: load ok did=%s dev=%s,%s", did, res->mDevice.id, res->mDevice.udid);

	QnDBResult *r =  new QnDBResult(task);
	r->resultDevice(res);
	return task->endTask(r);
}

bool  Database::cmdQueryDeviceCount(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdQueryDeviceCount",task,q);
	QueryKey1 *key = (QueryKey1*)q->mQuery;

	char aid[SST_GROUPID_LEN+1];
	ErrorCode e = confirmAID(q, port, &key->mKey1, aid);

	const char * session_aid = task->getAccountId();
	AccountLevel session_lv = task->getAccountLevel();

	if( session_lv == AccountSystemService ) {
		return taskResponse(q, ErrorPermission, "%s: queryDeviceCount. not login.", codeToMessgae(ErrorPermission));
	}
	if( session_lv == AccountUser ) {
		if( !session_aid || strcmp(aid, session_aid) != 0 ) {
			return taskResponse(q, ErrorPermission, "%s: queryDeviceCount. user has not authority.", codeToMessgae(ErrorPermission));
		}
	}

	size_t nDev = 0;
	if( aid == NULL ) {
		e = port->getDeviceCount(nDev);
	} else {
		e = port->getAccountDeviceCount(aid, nDev);
	}
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. device count failed.", codeToMessgae(e));
	}
	PROC_END("cmdQueryDeviceCount",task,q);

	QnDBResult *r =  new QnDBResult(task);
	r->resultCount(DBCMDDeviceCount, nDev);
	return task->endTask(r);
}

bool  Database::cmdQueryDeviceList(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdQueryDeviceList",task,q);
	QueryKey1Range *key = (QueryKey1Range*)q->mQuery;

	char aid[SST_GROUPID_LEN+1];
	ErrorCode e = confirmAID(q, port, &key->mKey1, aid);

	const char * session_aid = task->getAccountId();
	AccountLevel session_lv = task->getAccountLevel();

	if( session_lv == AccountSystemService ) {
		return taskResponse(q, ErrorPermission, "%s: queryDeviceList. not login.", codeToMessgae(ErrorPermission));
	}
	if( session_lv == AccountUser ) {
		if( !session_aid || strcmp(aid, session_aid) != 0 ) {
			return taskResponse(q, ErrorPermission, "%s: queryDeviceList. user has not authority.", codeToMessgae(ErrorPermission));
		}
	}

	ResultList *res = NULL;
	if( aid == NULL ) {
		e = port->getDeviceList(key->mRange.mStart, key->mRange.mCount, &res);
	} else {
		e = port->getAccountDeviceList(aid, key->mRange.mStart, key->mRange.mCount, &res);
	}
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. listing failed.", codeToMessgae(e));
	}

	PROC_END("cmdQueryDeviceList",task,q);
	QnDBResult *r =  new QnDBResult(task);
	r->resultList(res);
	return task->endTask(r);
}

bool Database::cmdQueryDeviceShareCount(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdQueryDeviceShareCount",task,q);
	QueryKey1 *key = (QueryKey1*)q->mQuery;

	char did[SST_DEVICEID_LEN+1];
	ErrorCode e = confirmDEV(q, port, &key->mKey1, did);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. confirm device failed.", codeToMessgae(e));
	}
	if( did[0] == 0 ) {
		return taskResponse(q, ErrorParameter, "%s. confirm device failed.", codeToMessgae(ErrorParameter));
	}

	size_t nDev = 0;
	e = port->getDeviceShareCount(did, nDev);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. device share count failed.", codeToMessgae(e));
	}
	PROC_END("cmdQueryDeviceShareCount",task,q);
	QnDBResult *r =  new QnDBResult(task);
	r->resultCount(DBCMDDeviceCount, nDev);
	return task->endTask(r);
}

bool Database::cmdQueryDeviceShareList(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	QueryKey1Range *key = (QueryKey1Range*)q->mQuery;

	char did[SST_DEVICEID_LEN+1];
	ErrorCode e = confirmDEV(q, port, &key->mKey1, did);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. deviceId is not confirm", codeToMessgae(e));
	}
	if( did[0] == 0 ) {
		return taskResponse(q, ErrorParameter, "%s. deviceId is not confirm", codeToMessgae(ErrorParameter));
	}

	const char *sessionAid = task->getAccountId();
	if( task->getAccountLevel() == AccountSystemService ) {
		return taskResponse(q, ErrorPermission, "%s. not login", codeToMessgae(ErrorPermission));
	}

	if( task->getAccountLevel() == AccountUser ) {
		ResultDevice *dev;
		e = port->getDeviceInfo(did, &dev);
		if( e == ErrorOk ) {
			if( strcmp(dev->mDevice.accountId, sessionAid) != 0 ) {
				return taskResponse(q, ErrorPermission, "%s: can not load device share list", codeToMessgae(ErrorPermission));
			}
		}
	}

	ResultList *res = NULL;
	e = port->getDeviceShareList(did, key->mRange.mStart, key->mRange.mCount, &res);
	if( e != ErrorOk ) {
		taskLog(q, LOG_DEBUG, "cmdQueryDeviceShareList: redis getDeviceShareList. code=%d", e);
		return taskResponse(q, e, "%s. deviceShareList load failed.", codeToMessgae(e));
	}
/*
	if( task->getAccountLevel() == AccountServiceAdmin ) {
		taskLog(q, LOG_DEBUG, "cmdQueryDeviceShareList: admin permission check count=%d", res->mList.values);
		const char *serviceId = task->getServiceId();
		size_t i;
		for( i = 0; i < res->mList.values; i++ ) {
			bool ok = false;
			ResultResource *rsc;
			const char *shid = res->mList.value[i];
			StdBuffer rid;
			int index;
			DBUtil::keyShareRid(shid, rid, index);
			taskLog(q, LOG_DEBUG, "cmdQueryDeviceShareList: %s->%s %d getResourceInfo=%s", shid, rid.getPtr(), index, rid.getPtr());
			e = port->getResourceInfo(rid.getPtr(), &rsc);
			if( e == ErrorOk ) {
				if( strcmp(rsc->mResource.serviceId, serviceId) == 0 ) {
					taskLog(q, LOG_DEBUG, "cmdQueryDeviceShareList: admin permission %s = ok", rsc->mResource.id);
					ok = true;
				}
			}

			if( !ok ) {
				taskLog(q, LOG_DEBUG, "cmdQueryDeviceShareList: admin permission %s = ng", rsc->mResource.id);
				size_t n;
				for( n = i; n < res->mList.values-1; n++ ) {
					res->mList.value[n] = res->mList.value[n+1];
				}
				res->mList.values--;
			}
		}
	}
	*/
	taskLog(q, LOG_DEBUG, "cmdQueryDeviceShareList: return %d", res->mList.values);

	QnDBResult *r =  new QnDBResult(task);
	r->resultList(res);
	return task->endTask(r);
}


bool Database::cmdQueryDeviceShare(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	QueryKey1 *key = (QueryKey1*)q->mQuery;

	char cid[SST_DEVICESHAREID_LEN+1];
	ErrorCode e = confirmCID(q, port, &key->mKey1, cid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. confirm cid failed.", codeToMessgae(e));
	}

	// TODO: check permission

	ResultDeviceShare *res = NULL;
	e = port->getDeviceShareInfo(cid, &res);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. deviceShareinfo load failed.", codeToMessgae(e));
	}

	QnDBResult *r =  new QnDBResult(task);
	r->resultDeviceShare(res);
	return task->endTask(r);
}

bool Database::cmdQueryGroup(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	QueryKey1 *qp = (QueryKey1*)q->mQuery;

	char gid[SST_GROUPID_LEN+1];
	ErrorCode e = confirmGID(q, port, qp, gid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "queryGroup: Confirm group error. service=%s,%s group=%s,%s. %s.",
				qp->mSREF.mId, qp->mSREF.mName,
				qp->mKey1.mId, qp->mKey1.mName,
				codeToMessgae(e));
	}

	ResultGroup *res;
	e = port->getGroup(gid, &res);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "queryGroup: Group info load failed. service=%s,%s group=%s,%s. %s.",
				qp->mSREF.mId, qp->mSREF.mName,
				gid, qp->mKey1.mName,
				codeToMessgae(e));
	}

	QnDBResult *r =  new QnDBResult(task);
	r->resultGroup(res);
	return task->endTask(r);
}

bool  Database::cmdQueryGroupCount(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdQueryDeviceCount",task,q);
	QueryKey1 *qp = (QueryKey1*)q->mQuery;

	char _aid[SST_ACCOUNTID_LEN+1];
	ErrorCode e = confirmAID(q, port, &qp->mKey1, _aid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. confirm account failed.", codeToMessgae(e));
	}

	if( task->getAccountLevel() == AccountSystemService ) {
		return taskResponse(q, ErrorPermission, codeToMessgae(ErrorPermission));
	}
	const char *sessionAid = task->getAccountId();
	if( task->getAccountLevel() == AccountUser ) {
		if( !sessionAid || strcmp(sessionAid, _aid) != 0 ) {
			return taskResponse(q, ErrorPermission, codeToMessgae(ErrorPermission));
		}
	}

	size_t nRec = 0;

	if( task->getAccountLevel() == AccountServiceAdmin ) {
		ResultList *res;
		e = port->getAccountGroupList(_aid, 0, -1, &res);
		if( e != ErrorOk ) {
			return taskResponse(q, e, codeToMessgae(e));
		}
		size_t i;
		const char *sessionSid = task->getServiceId();
		for( i = 0; i < res->mList.values; i++ ) {
			ResultGroup *g;
			ErrorCode e = port->getGroup(res->mList.value[i], &g);
			if( e != ErrorOk ) {
				free(res);
				return taskResponse(q, e, codeToMessgae(e));
			}
			if( strcmp(g->mGroup.serviceId, sessionSid) == 0 ) {
				nRec++;
			}
			free(g);
		}
		free(res);
	} else {
		e = port->getAccountGroupCount(_aid, nRec);
		return taskResponse(q, e, "%s. account group count failed.", codeToMessgae(e));
	}
	QnDBResult *r =  new QnDBResult(task);
	r->resultCount(DBCMDGroupCount, nRec);
	return task->endTask(r);
}

bool  Database::cmdQueryGroupList(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	QueryKey1Range *qp = (QueryKey1Range*)q->mQuery;

	char _aid[SST_ACCOUNTID_LEN+1];

	ErrorCode e = confirmAID(q, port, &qp->mKey1, _aid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. confirm account failed.", codeToMessgae(e));
	}

	if( task->getAccountLevel() == AccountSystemService ) {
		return taskResponse(q, ErrorPermission, "%s. not login", codeToMessgae(ErrorPermission));
	}
	if( task->getAccountLevel() == AccountUser ) {
		const char *sessionAid = task->getAccountId();
		if( !sessionAid || strcmp(sessionAid, _aid) != 0 ) {
			return taskResponse(q, ErrorPermission, "%s.", codeToMessgae(ErrorPermission));
		}
	}

	ResultList *res;
	e = port->getAccountGroupList(_aid, qp->mRange.mStart, qp->mRange.mCount, &res);
	if( e != ErrorOk ) {
		return taskResponse(q, e, codeToMessgae(e));
	}

	if( task->getAccountLevel() == AccountServiceAdmin ) {
		size_t i;
		const char *sessionSid = task->getServiceId();
		for( i = 0; i < res->mList.values; i++ ) {
			ResultGroup *g;
			ErrorCode e = port->getGroup(res->mList.value[i], &g);
			if( e != ErrorOk ) {
				free(res);
				return taskResponse(q, e, codeToMessgae(ErrorPermission));
			}
			if( strcmp(g->mGroup.serviceId, sessionSid) != 0 ) {
				size_t n;
				for( n = i; n < res->mList.values-1; n++ ) {
					res->mList.value[n] = res->mList.value[n+1];
				}
				res->mList.values--;
				i--;
			}
			free(g);
		}
	}

	QnDBResult *r =  new QnDBResult(task);
	r->resultList(res);
	return task->endTask(r);
}

bool Database::cmdQueryIsGroupMember(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	QueryKey2 *qp = (QueryKey2*)q->mQuery;

	ErrorCode e;

	char aid[SST_ACCOUNTID_LEN+1];
	e = confirmAID(q, port, &qp->mKey2, aid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. confirm account failed.", codeToMessgae(e));
	}

	char gid[SST_GROUPID_LEN+1];
	gid[0]	= 0;
	size_t nGroupList = 0;
	ResultList *pGroupList = NULL;
	size_t gn=0;
	if( qp->mKey1.mId == NULL && qp->mKey1.mName == NULL ) {
		// all group
		char sid[SST_SERVICEID_LEN+1];
		e  = confirmSID(q, port, &qp->mSREF, sid);
		if( e != ErrorOk ) {
			return taskResponse(q, e, "%s. confirm service failed.", codeToMessgae(e));
		}
		e = port->getServiceGroupCount(sid, nGroupList);
		if( e != ErrorOk ) {
			return taskResponse(q, e, "%s. service group count failed.", codeToMessgae(e));
		}
		if( nGroupList > 0 ) {
			e = port->getServiceGroupList(sid, 0, nGroupList, &pGroupList);
			if( e != ErrorOk ) {
				return taskResponse(q, e, "%s. service group list failed.", codeToMessgae(e));
			}
			if( pGroupList->mList.values > 0 ) {
				strcpy(gid, pGroupList->mList.value[gn++]);
			}
		}
	} else {
		// group
		QueryKey1 gkey;
		gkey.mSREF	= qp->mSREF;
		gkey.mKey1	= qp->mKey1;
		e  = confirmGID(q, port, &gkey, gid);
		if( e != ErrorOk ) {
			return taskResponse(q, e, "%s. confirm group failed.", codeToMessgae(e));
		}
	}

	while( gid[0] ) {
		size_t nAcc;
		taskLog(q, LOG_DEBUG, "cmdQueryIsGroupMember: task=%x group=%s check=%s", task, gid, aid);
		e = port->getGroupAccountCount(gid, nAcc);
		if( e != ErrorOk ) {
			return taskResponse(q, e, "%s. group account count failed.", codeToMessgae(e));
		}
		if( nAcc > 0 ) {
			ResultList *pList = NULL;
			e = port->getGroupAccountList(gid, 0, nAcc, &pList);
			if( e != ErrorOk ) {
				return taskResponse(q, e, "%s. group account list failed.", codeToMessgae(e));
			}

			size_t i;
			for( i = 0; i < pList->mList.values; i++ ) {
				if( strcmp(pList->mList.value[i], aid) == 0 ) {
					free(pList);
					if( pGroupList ) free(pGroupList);
					QnDBResult *r =  new QnDBResult(task);
					r->setResult(DBCMDIsGroupMember, ErrorOk, codeToMessgae(ErrorOk));
					return task->endTask(r);
				}
			}
			free(pList);
		}
		gid[0]	= 0;
		if( pGroupList && gn < pGroupList->mList.values) {
			strcpy(gid, pGroupList->mList.value[gn++]);
		}
	}
	if( pGroupList ) free(pGroupList);
	return taskResponse(q, ErrorNotFound, "%s. group member.", codeToMessgae(ErrorNotFound));
}

bool Database::cmdQueryResource(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	QueryKey1 *qp = (QueryKey1*)q->mQuery;

	char rid[SST_SERVICEID_LEN+1];
	ErrorCode e = confirmRID(q, port, qp, rid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. confirm resource failed.", codeToMessgae(e));
	}

	ResultResource *res;
	e = port->getResourceInfo(rid, &res);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. Resource info load failed. rid=%s.", codeToMessgae(e), rid);
	}

	QnDBResult *r =  new QnDBResult(task);
	r->resultResource(res);
	return task->endTask(r);
}

bool  Database::cmdQueryResourceList(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdQueryResourceList", task, q);
	QueryRange *qp = (QueryRange*)q->mQuery;

	QnDBResult *r =  new QnDBResult(task);

	char sid[SST_SERVICEID_LEN+1];
	ErrorCode e = confirmSID(q, port, &qp->mSREF, sid);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}

	ResultList *res;
	if( sid[0] == 0 )
		e = port->getServiceList(qp->mRange.mStart, qp->mRange.mCount, &res);
	else
		e = port->getServiceResourceList(sid, qp->mRange.mStart, qp->mRange.mCount, &res);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	PROC_END("cmdQueryResourceList", task, q);
	r->resultList(res);
	return task->endTask(r);
}

/*
bool Database::cmdQueryGadget(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdQueryGadget", task, q);
	QueryKey2 *qp = (QueryKey2*)q->mQuery;

	QnDBResult *r =  new QnDBResult(task);
	char gid[SST_SERVICEID_LEN+1];
	ErrorCode e = confirmEID(q, port, qp, gid);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}

	ResultResource *res;
	e = port->getResourceInfo(gid, &res);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	PROC_END("cmdQueryGadget", task, q);
	r->resultResource(res);
	return task->endTask(r);
}

bool Database::cmdQueryGadgetList(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdQueryGadgetList", task, q);
	QueryKey1Range *qp = (QueryKey1Range*)q->mQuery;

	QnDBResult *r =  new QnDBResult(task);

	char rid[SST_SERVICEID_LEN+1];
	QueryKey1 rref;
	rref.mSREF	= qp->mSREF;
	rref.mKey1	= qp->mKey1;
	ErrorCode e = confirmRID(q, port, &rref, rid);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}

	ResultList *res;
	e = port->getResourceGadgetList(rid, qp->mRange.mStart, qp->mRange.mCount, &res);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	PROC_END("cmdQueryGadgetList", task, q);
	r->resultList(res);
	return task->endTask(r);
}
*/

bool Database::cmdQueryShare(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdQueryShare", task, q);
	QueryKey1 *qp = (QueryKey1*)q->mQuery;

	taskLog(q, LOG_DEBUG, "key sref=%s:%s mref=%s:%s", qp->mSREF.mId, qp->mSREF.mName, qp->mKey1.mId, qp->mKey1.mName);

	QnDBResult *r =  new QnDBResult(task);

	char _mid[SST_SHAREID_LEN+1];
	ErrorCode e = confirmMID(q, port, &qp->mKey1, _mid);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}

	ResultShare *res;
	e = port->getShareInfo(_mid, &res);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	PROC_END("cmdQueryShare", task, q);
	r->resultShare(res);
	return task->endTask(r);
}

bool  Database::cmdQueryShareList(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	QueryKey1Range *qp = (QueryKey1Range*)q->mQuery;

	QnDBResult *r =  new QnDBResult(task);
	char gid[SST_GROUPID_LEN+1];
	QueryKey1 key;
	key.mSREF	= qp->mSREF;
	key.mKey1	= qp->mKey1;
	ErrorCode e = confirmGID(q, port, &key, gid);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}

	if( task->getAccountLevel() == AccountSystemService ) {
		return taskResponse(q, ErrorPermission, "%s. not login", codeToMessgae(ErrorPermission));
	}

	ResultGroup *grp = NULL;
	ResultService *svc = NULL;
	if( task->getAccountLevel() == AccountUser ) {
		const char *aid = task->getAccountId();
		if( !aid || port->isGroupAccount(gid, aid) != ErrorOk ) {
			e = port->getGroup(gid, &grp);
			if( e != ErrorOk ) {
				return taskResponse(q, e, "%s. group info load error.", codeToMessgae(e));
			}
			e = port->getServiceInfo(grp->mGroup.serviceId, &svc);
			if( e != ErrorOk ) {
				if( grp ) free(grp);
				return taskResponse(q, e, "%s. service info load error.", codeToMessgae(e));
			}
			if( svc->mService.scope != ScopePublic ) {
				if( svc ) free(svc);
				if( grp ) free(grp);
				return taskResponse(q, ErrorPermission, "%s. ", codeToMessgae(ErrorPermission));
			}
		}
	}
	if( task->getAccountLevel() == AccountServiceAdmin ) {
		const char *sessionSid = task->getServiceId();
		e = port->getGroup(gid, &grp);
		if( e != ErrorOk ) {
			return taskResponse(q, e, "%s. group info load error.", codeToMessgae(e));
		}
		if( !sessionSid || strcmp(sessionSid, grp->mGroup.serviceId) != 0 ) {
			e = port->getServiceInfo(grp->mGroup.serviceId, &svc);
			if( e != ErrorOk ) {
				if( grp ) free(grp);
				return taskResponse(q, e, "%s. service info load error.", codeToMessgae(e));
			}
			if( svc->mService.scope == ScopePrivate ) {
				if( svc ) free(svc);
				if( grp ) free(grp);
				return taskResponse(q, ErrorPermission, "%s.", codeToMessgae(ErrorPermission));
			}
		}
	}
	if( svc ) free(svc);
	if( grp ) free(grp);

	ResultList *res;
	e = port->getGroupShareList(gid, qp->mRange.mStart, qp->mRange.mCount, &res);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	r->resultList(res);
	return task->endTask(r);
}

bool  Database::cmdQuerySync(QnDBQuery *q, DBPort *port)
{
	// TODO: not implement cmdQuerySync
	return taskResponse(q, ErrorNotImpl, "%s.", codeToMessgae(ErrorNotImpl));
}

bool  Database::cmdQuerySyncQueue(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	QueryBase *qp = (QueryBase*)q->mQuery;

	QnDBResult *r =  new QnDBResult(task);

	char sid[SST_SERVICEID_LEN+1];
	ErrorCode e = confirmSID(q, port, &qp->mSREF, sid);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	if( sid[0] == 0 ) {
		r->setResult(q->getFunction(), ErrorParameter, codeToMessgae(ErrorParameter));
		return task->endTask(r);
	}

	char id[256];
	e = port->popSyncQueue(sid, id);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	r->resultId(e, id);
	return task->endTask(r);
}

/*
bool  Database::cmdQuerySession(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	Session *con = task->getSession();
	PROC_BEGIN("cmdQuerySession", task, q);
	QueryKey1 *qp = (QueryKey1*)q->mQuery;

	QnDBResult *r =  new QnDBResult(task);
	char _cid[SST_ACCOUNTID_LEN+1];
	const char *cid = NULL;

	const char *session_id = NULL;
	if( con )
		session_id	= con->getSessionId();

	if( qp->mKey1.mId == NULL && qp->mKey1.mName == NULL ) {
		cid = session_id;
	} else {
		if( qp->mKey1.mId == NULL ) {
			DBError e = port->getSessionId(qp->mKey1.mName, _cid);
			if( e != DBErrorOk ) {
				r->setDBResult(e, codeToMessgae(e));
				return task->endTask(r);
			}
			cid = _cid;
		} else {
			cid = qp->mKey1.mId;
		}
	}
	if( cid == NULL ) {
		r->setResult(DBCMDSession, DBErrorParam, codeToMessgae(DBErrorParam));
		return task->endTask(r);
		return true;
	}
	bool difSession = false;
	if( session_id == NULL || strcmp(session_id, cid) != 0 ) {
		difSession = true;
	}
	if( difSession ) {
		if( con == NULL || (con->getAccountLevel() != AccountSystemService && con->getAccountLevel() != AccountServiceAdmin) ) {
			r->setResult(DBCMDSession, DBErrorPermission, codeToMessgae(DBErrorPermission));
			return task->endTask(r);
		}
	}

	ResultSession *res;
	DBError e = port->getSession(cid, &res);
	if( e != DBErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}

	if( difSession && con->getAccountLevel() == AccountServiceAdmin && strcmp(res->mSession.sessionId, con->getServiceId()) != 0 ) {
		free(res);
		r->setResult(DBCMDSession, DBErrorPermission, codeToMessgae(DBErrorPermission));
		return task->endTask(r);
		return true;
	}

	PROC_END("cmdQuerySession", task, q);
	r->resultSession(res);
	return task->endTask(r);
}

bool  Database::cmdQuerySessionList(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdQuerySessionList", task, q);
	QueryRange *key = (QueryRange*)q->mQuery;

	QnDBResult *r =  new QnDBResult(task);
	char sid[SST_SERVICEID_LEN+1];
	DBError e = confirmSID(q, port, &key->mSREF, sid);
	if( e != DBErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}

	ResultList *res;
	e = port->getServiceSessionList(sid, key->mRange.mStart, key->mRange.mCount, &res);
	if( e != DBErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}

	PROC_END("cmdQuerySessionList", task, q);
	r->resultList(res);
	return task->endTask(r);
}
*/

bool  Database::cmdQueryNode(QnDBQuery *q, DBPort *port)
{
	// TODO: not implement QueryNode
	return taskResponse(q, ErrorNotImpl, "%s.", codeToMessgae(ErrorNotImpl));
}

bool  Database::cmdQueryNodeList(QnDBQuery *q, DBPort *port)
{
	// TODO: not implement QueryNodeList
	return taskResponse(q, ErrorNotImpl, "%s.", codeToMessgae(ErrorNotImpl));
}

////////////////////////////////////////////////////////////////////////////////////////
// register command
bool  Database::cmdRegService(QnDBRegister *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdRegService", task, q);

	RegistService *cmd = (RegistService*)q->mRegist;

	ErrorCode e;
	if( cmd->mService.id == NULL ) {
		e = task->allowAddService(port, &cmd->mService);
	} else {
		e = task->allowSetService(port, &cmd->mService);
	}
	if( e != ErrorOk ) {
		return taskResponse(q, e, "regService: %s. service=%s.",
				codeToMessgae(e),
				cmd->mService.name);
	}

	char id_ret[SST_SERVICEID_LEN+1];
	e = port->setServiceInfo(&cmd->mService, id_ret);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "regService: %s. service info registration failed. service=%s.",
				codeToMessgae(e),
				cmd->mService.name);
	}
	QnDBResult *r =  new QnDBResult(task);
	r->resultId(e, id_ret);
	PROC_END("cmdRegService", task, q);
	return task->endTask(r);
}

bool Database::cmdRegAccountLock(QnDBRegister *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdRegAccountLock", task, q);

	QnDBResult *r =  new QnDBResult(task);
	QueryKey1 *key = (QueryKey1*)q->mRegist;

	char _sid[SST_SERVICEID_LEN+1];
	ErrorCode e = confirmSID(q, port, &key->mSREF, _sid);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	char _aid[SST_ACCOUNTID_LEN+1];
	e = confirmAID(q, port, &key->mKey1, _aid);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	e = port->setServiceRefuseList(_sid, _aid);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	r->setDBResult(ErrorOk, NULL);
	PROC_END("cmdRegAccountLock", task, q);
	return task->endTask(r);
}

bool Database::cmdRegAccountUnlock(QnDBRegister *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdRegAccountUnlock", task, q);

	QnDBResult *r =  new QnDBResult(task);
	QueryKey1 *key = (QueryKey1*)q->mRegist;

	char _sid[SST_SERVICEID_LEN+1];
	ErrorCode e = confirmSID(q, port, &key->mSREF, _sid);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	char _aid[SST_ACCOUNTID_LEN+1];
	e = confirmAID(q, port, &key->mKey1, _aid);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	e = port->remServiceRefuseList(_sid, _aid);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	r->setDBResult(ErrorOk, NULL);
	PROC_END("cmdRegAccountUnlock", task, q);
	return task->endTask(r);
}

bool Database::cmdRegAccount(QnDBRegister *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdRegAccount", task, q);

	RegistAccount *cmd = (RegistAccount*)q->mRegist;
	QnDBResult *r =  new QnDBResult(task);

	ErrorCode e;
	if( cmd->mAccount.id == NULL ) {
		e = task->allowAddAccount(port, &cmd->mAccount);
	} else {
		e = task->allowSetAccount(port, &cmd->mAccount);
	}
	if( e != ErrorOk ) {
		return taskResponse(q, e, "regAccount: %s. account=%s,%s.", codeToMessgae(e), cmd->mAccount.id,cmd->mAccount.uid);
	}

	char id_ret[256];
	e = port->setAccountInfo(&cmd->mAccount, id_ret);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	r->resultId(e, id_ret);
	PROC_END("cmdRegAccount", task, q);
	return task->endTask(r);
}

bool  Database::cmdRegDevice(QnDBRegister *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdRegDevice", task, q);
	RegistDevice *qp = (RegistDevice*)q->mRegist;

	QnDBResult *r =  new QnDBResult(task);

	char id_ret[256];
	ErrorCode e = port->setDeviceInfo(&qp->mDevice, id_ret);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	r->resultId(e, id_ret);
	PROC_END("cmdRegDevice", task, q);
	return task->endTask(r);
}

bool  Database::cmdRegGroup(QnDBRegister *q, DBPort *port)
{
	Task *task = q->getTask();
	RegistGroup *cmd = (RegistGroup*)q->mRegist;
	char id_ret[SST_GROUPID_LEN+1];
	ErrorCode e;

	char _sid[SST_SERVICEID_LEN+1];
	if( cmd->mGroup.serviceId == NULL ) {
		QueryKeyType sref;
		sref.mId	= NULL;
		sref.mName	= NULL;
		e = confirmSID(q, port, &sref, _sid);
		if( e != ErrorOk ) {
			return taskResponse(q, e, "regGroup: %s. confirmSID is failed. service=%s group=%s,%s.",
					codeToMessgae(e),
					cmd->mGroup.serviceId,
					cmd->mGroup.id, cmd->mGroup.name);
		}
		cmd->mGroup.serviceId = _sid;
	}

	ResultService *svc = NULL;
	e = port->getServiceInfo(cmd->mGroup.serviceId, &svc);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "regGroup: %s. service info load failed. service=%s group=%s,%s.",
				codeToMessgae(e),
				cmd->mGroup.serviceId,
				cmd->mGroup.id, cmd->mGroup.name);
	}
	if( cmd->mGroup.id == NULL ) {
		e = task->allowAddGroup(port, &svc->mService);
	} else {
		e = task->allowSetGroup(port, &svc->mService, &cmd->mGroup);
	}
	free(svc);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "regGroup: %s. service=%s group=%s,%s.",
				codeToMessgae(e),
				cmd->mGroup.serviceId,
				cmd->mGroup.id, cmd->mGroup.name);
	}

	e = port->setGroup(&cmd->mGroup, id_ret);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "regGroup: %s. setGroup is failed. service=%s group=%s,%s.",
				codeToMessgae(e),
				cmd->mGroup.serviceId,
				cmd->mGroup.id, cmd->mGroup.name);
	}
	QnDBResult *r =  new QnDBResult(task);
	r->resultId(e, id_ret);
	return task->endTask(r);
}

bool  Database::cmdRegResource(QnDBRegister *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdRegResource", task, q);

	RegistResource *cmd = (RegistResource*)q->mRegist;

	char id_ret[256];
	ErrorCode e = port->setResourceInfo(&cmd->mResource, id_ret);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "regResource: %s. setResourceInfo is failed. service=%s resource=%s,%s.",
				codeToMessgae(e),
				cmd->mResource.serviceId,
				cmd->mResource.id, cmd->mResource.name);
	}
	QnDBResult *r =  new QnDBResult(task);
	r->resultId(e, id_ret);
	PROC_END("cmdRegResource", task, q);
	return task->endTask(r);
}

bool  Database::cmdRegGroupAccount(QnDBRegister *q, DBPort *port)
{
	Task *task = q->getTask();
	RegistGroupAccount *cmd = (RegistGroupAccount*)q->mRegist;

	char _sid[SST_SERVICEID_LEN+1];
	ErrorCode e = confirmSID(q, port, &cmd->mSREF, _sid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "regGroupAccount: %s. confirmSID is failed. service=%s,%s.",
				codeToMessgae(e),
				cmd->mSREF.mId, cmd->mSREF.mName);
	}

	char _gid[SST_GROUPID_LEN+1];
	QueryKey1 gkey;
	gkey.mSREF.mId		= _sid;
	gkey.mSREF.mName	= NULL;
	gkey.mKey1			= cmd->mGroup;
	e = confirmGID(q, port, &gkey, _gid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "regGroupAccount: %s. confirmGID is failed. service=%s,%s group=%s,%s.",
				codeToMessgae(e),
				cmd->mSREF.mId, cmd->mSREF.mName,
				cmd->mGroup.mId, cmd->mGroup.mName);
	}

	char __aid[SST_ACCOUNTID_LEN+1];
	const char *_aid;

	const char *session_aid = task->getAccountId();

	if( cmd->mAccount.mId == NULL && cmd->mAccount.mName == NULL ) {
		_aid = session_aid;
	} else if( cmd->mAccount.mId == NULL ) {
		ErrorCode e = port->getAccountId(cmd->mAccount.mName, __aid);
		if( e != ErrorOk ) {
			return taskResponse(q, e, "regGroupAccount: %s. getAccountId is failed. service=%s,%s group=%s,%s account=%s.",
					codeToMessgae(e),
					cmd->mSREF.mId, cmd->mSREF.mName,
					cmd->mGroup.mId, cmd->mGroup.mName,
					cmd->mAccount.mName);
		}
		_aid = __aid;
	} else {
		_aid = cmd->mAccount.mId;
	}

	if( _aid == NULL ) {
		return taskResponse(q, ErrorParameter, "regGroupAccount: %s. account is not specified. service=%s,%s group=%s,%s.",
				codeToMessgae(ErrorParameter),
				cmd->mSREF.mId, cmd->mSREF.mName,
				cmd->mGroup.mId, cmd->mGroup.mName);
	}


	ResultGroup *group;
	e = port->getGroup(_gid, &group);
	if( e != ErrorOk ) {
		return taskResponse(q, ErrorParameter, "regGroupAccount: %s. group info load failed. service=%s,%s group=%s,%s.",
				codeToMessgae(ErrorParameter),
				_sid, cmd->mSREF.mName,
				_gid, cmd->mGroup.mName);
	}
	if( strcmp(_sid, group->mGroup.serviceId) != 0 ) {
		free(group);
		return taskResponse(q, ErrorDataConflict, "regGroupAccount: %s. service id is conflict. group->sid=%s service=%s,%s group=%s,%s.",
				codeToMessgae(ErrorDataConflict),
				group->mGroup.serviceId,
				_sid, cmd->mSREF.mName,
				_gid, cmd->mGroup.mName);
	}

	ResultService *serv;
	e = port->getServiceInfo(_sid, &serv);
	if( e != ErrorOk ) {
		free(group);
		return taskResponse(q, e, "regGroupAccount: %s. service info load failed. service=%s,%s group=%s,%s.",
				codeToMessgae(e),
				_sid, cmd->mSREF.mName,
				_gid, cmd->mGroup.mName);
	}

	ResultAccount *account;
	e = port->getAccountInfo(_aid, &account);
	if( e != ErrorOk ) {
		free(serv);
		free(group);
		return taskResponse(q, e, "regGroupAccount: %s. getAccountId is failed. service=%s,%s group=%s,%s account=%s,%s.",
				codeToMessgae(e),
				_sid, cmd->mSREF.mName,
				_gid, cmd->mGroup.mName,
				_aid, cmd->mAccount.mName);
	}

	e = task->allowAddGroupMember(port, &serv->mService, &group->mGroup, &account->mAccount);
	if( e != ErrorOk ) {
		free(account);
		free(serv);
		free(group);
		return taskResponse(q, e, "regGroupAccount: %s. service=%s,%s group=%s,%s account=%s,%s.",
				codeToMessgae(e),
				_sid, cmd->mSREF.mName,
				_gid, cmd->mGroup.mName,
				_aid, cmd->mAccount.mName);
	}
	free(account);
	free(serv);
	free(group);

	e = port->joinGroupAccount(_gid, _aid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "regGroupAccount: %s. joinGroupAccount failed. service=%s,%s group=%s,%s account=%s,%s.",
				codeToMessgae(e),
				_sid, cmd->mSREF.mName,
				_gid, cmd->mGroup.mName,
				_aid, cmd->mAccount.mName);
	}
	QnDBResult *r =  new QnDBResult(task);
	r->resultId(e, NULL);
	return task->endTask(r);
}

bool  Database::cmdRegShare(QnDBRegister *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdRegShare", task, q);

	RegistShare *cmd = (RegistShare*)q->mRegist;
	QnDBResult *r =  new QnDBResult(task);

	char id_ret[256];
	ErrorCode e = port->setShareInfo(&cmd->mShare, id_ret);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	r->resultId(e, id_ret);
	PROC_END("cmdRegResource", task, q);
	return task->endTask(r);
}

bool  Database::cmdRegDeviceShare(QnDBRegister *q, DBPort *port)
{
	RegistDeviceShare *cmd = (RegistDeviceShare*)q->mRegist;

	char mid[SST_SHAREID_LEN+1];
	ErrorCode e = confirmMID(q, port, &cmd->mMREF, mid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, codeToMessgae(e));
	}

	ResultShare *share;
	e = port->getShareInfo(mid, &share);
	if( e != ErrorOk ) {
		return taskResponse(q, e, codeToMessgae(e));
	}

	ResultResource *resource;
	e = port->getResourceInfo(share->mShare.resourceId, &resource);
	if( e != ErrorOk ) {
		free(share);
		return taskResponse(q, e, codeToMessgae(e));
	}

	char dev[SST_DEVICEID_LEN+1];
	e = confirmDEV(q, port, &cmd->mDREF, dev);
	if( e != ErrorOk ) {
		free(share);
		free(resource);
		return taskResponse(q, e, codeToMessgae(e));
	}

	ResultDevice *device;
	e = port->getDeviceInfo(dev, &device);
	if( e != ErrorOk ) {
		free(share);
		free(resource);
		return taskResponse(q, e, codeToMessgae(e));
	}

	// TODO: permission check !!!
//	if( device->mDevice.userAccount != NULL ) {
//		free(share);
//		free(device);
//		return taskResponse(q, ErrorPermission, codeToMessgae(ErrorPermission));
//	}

	DeviceShareInfo rec;
	memset(&rec, 0, sizeof(rec));
	rec.serviceId		= share->mShare.serviceId;
	rec.shareId			= mid;
	rec.deviceId		= device->mDevice.id;
	rec.accountId		= device->mDevice.accountId;
	rec.deviceTime		= resource->mResource.refTime;
	rec.deviceRevision	= resource->mResource.revision;
	rec.status			= DeviceShareProgress;
	rec.shareTime		= time(NULL);

	e = port->setDeviceShareInfo(device->mDevice.id, &rec);
	free(share);
	free(resource);
	free(device);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. setDeviceShare failed.", codeToMessgae(e));
	}
	return taskResponse(q, ErrorOk, NULL);
}

bool  Database::cmdRegSync(QnDBRegister *q, DBPort *port)
{
	// TODO: not implement cmdRegSync
	return taskResponse(q, ErrorNotImpl, "%s.", codeToMessgae(ErrorNotImpl));
}

bool  Database::cmdRegSyncQueue(QnDBRegister *q, DBPort *port)
{
	RegistSyncQueue *cmd = (RegistSyncQueue*)q->mRegist;

	char _sid[SST_SERVICEID_LEN+1];
	ErrorCode e = confirmSID(q, port, &cmd->mSREF, _sid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "regSyncQueue: Confirm service error. service=%s,%s. %s.",
				cmd->mSREF.mId, cmd->mSREF.mName, codeToMessgae(e));
	}

	if( cmd->mGREFCount == 0 ) {
		if( cmd->mGREFArr != NULL ) {
			pjson::builder jb;
			char _aid[SST_ACCOUNTID_LEN+1];
			e = confirmAID(q, port, &cmd->mAREF, _aid);
			if( e != ErrorOk ) {
				return taskResponse(q, e, "regSyncQueue: Confirm account error. account=%s,%s. %s.",
						cmd->mAREF.mId, cmd->mAREF.mName, codeToMessgae(e));
			}
			if( cmd->mGREFArr->mId && strcmp(cmd->mGREFArr->mId, "all") == 0 ) {
				if( !jb.init(512) ||
					!jb.beginObject() ||
					!jb.addObjectProp("cmd",3) ||
					!jb.valueInt(cmd->mCmd) ||
					!jb.addObjectProp("aid",3) ||
					!jb.valueString(_aid) ||
					!jb.endObject() ) {
					return taskResponse(q, ErrorRuntime, "regSyncQueue: Json build error(1). account=%s,%s group=%s,%s. %s.",
							cmd->mAREF.mId, cmd->mAREF.mName,
							cmd->mGREFArr->mId, cmd->mGREFArr->mName,
							codeToMessgae(ErrorRuntime));
				}

			} else {
				char _gid[SST_GROUPID_LEN+1];
				QueryKey1 gkey;
				gkey.mSREF.mId		= _sid;
				gkey.mSREF.mName	= NULL;
				gkey.mKey1			= *cmd->mGREFArr;
				e = confirmGID(q, port, &gkey, _gid);
				if( e != ErrorOk ) {
					return taskResponse(q, e, "regSyncQueue: Confirm group error. group=%s,%s. %s.",
							cmd->mGREFArr->mId, cmd->mGREFArr->mName, codeToMessgae(e));
				}
				if( !jb.init(512) ||
					!jb.beginObject() ||
					!jb.addObjectProp("cmd",3) ||
					!jb.valueInt(cmd->mCmd) ||
					!jb.addObjectProp("aid",3) ||
					!jb.valueString(_aid) ||
					!jb.addObjectProp("gid",3) ||
					!jb.valueString(_gid) ||
					!jb.endObject() ) {
					return taskResponse(q, ErrorRuntime, "regSyncQueue: Json build error(2). account=%s,%s group=%s,%s. %s.",
							cmd->mAREF.mId, cmd->mAREF.mName,
							cmd->mGREFArr->mId, cmd->mGREFArr->mName,
							codeToMessgae(ErrorRuntime));
				}
			}
			pjson::json *js = jb.detouch();
			StdBuffer syncdata;
			js->getText(&syncdata, false);
			delete js;
			e = port->pushSyncQueue(_sid, syncdata.getPtr());
			if( e != ErrorOk ) {
				taskLog(q, LOG_ERR, "regSyncQueue: PushSyncQueue error. account=%s,%s. %s",
						cmd->mAREF.mId, cmd->mAREF.mName, codeToMessgae(e));
			}

		} else {

			char _aid[SST_ACCOUNTID_LEN+1];
			e = confirmAID(q, port, &cmd->mAREF, _aid);
			if( e != ErrorOk ) {
				return taskResponse(q, e, "regSyncQueue: Confirm account error. account=%s,%s. %s.",
						cmd->mAREF.mId, cmd->mAREF.mName, codeToMessgae(e));
			}

			ResultList *group = NULL;
			int count=-1;
			e = port->getAccountGroupList(_aid, 0, count, &group);
			if( e != ErrorOk ) {
				if( e != ErrorNotFound ) {
					return taskResponse(q, e, "regSyncQueue: account group list load error. account=%s,%s. %s.",
							cmd->mAREF.mId, cmd->mAREF.mName, codeToMessgae(e));

				}
			} else {
				size_t i;
				for( i = 0; i < group->mList.values; i++ ) {
					pjson::builder jb;
					if( !jb.init(512) ||
						!jb.beginObject() ||
						!jb.addObjectProp("cmd",3) ||
						!jb.valueInt(cmd->mCmd) ||
						!jb.addObjectProp("aid",3) ||
						!jb.valueString(_aid) ||
						!jb.addObjectProp("gid",3) ||
						!jb.valueString(group->mList.value[i]) ||
						!jb.endObject() ) {
						free(group);
						return taskResponse(q, ErrorRuntime, "regSyncQueue: Json build error(3). account=%s,%s. %s.",
								cmd->mAREF.mId, cmd->mAREF.mName, codeToMessgae(ErrorRuntime));
					}
					pjson::json *js = jb.detouch();
					StdBuffer syncdata;
					js->getText(&syncdata, false);
					delete js;
					e = port->pushSyncQueue(_sid, syncdata.getPtr());
					if( e != ErrorOk ) {
						return taskResponse(q, ErrorRuntime, "regSyncQueue: Json build error(4). account=%s,%s. %s.",
								cmd->mAREF.mId, cmd->mAREF.mName, codeToMessgae(ErrorRuntime));
					}
				}
				if( group )
					free(group);
			}
		}
	} else {
		size_t i;
		for( i = 0; i < cmd->mGREFCount; i++ ) {
			char _gid[SST_GROUPID_LEN+1];
			QueryKey1 gkey;
			gkey.mSREF.mId		= _sid;
			gkey.mSREF.mName	= NULL;
			gkey.mKey1			= cmd->mGREFArr[i];
			e = confirmGID(q, port, &gkey, _gid);
			if( e != ErrorOk ) {
				taskLog(q, LOG_ERR, "regSyncQueue: Confirm group error. group=%s,%s. %s.",
						gkey.mKey1.mId, gkey.mKey1.mName, codeToMessgae(e));
				continue;
			}
			ResultList *member = NULL;
			int count=-1;
			e = port->getGroupAccountList(_gid, 0, count, &member);
			if( e != ErrorOk ) {
				if( e != ErrorNotFound ) {
					taskLog(q, LOG_ERR, "regSyncQueue: Group member list load error. group=%s,%s. %s.",
							gkey.mKey1.mId, gkey.mKey1.mName, codeToMessgae(e));
				}
				continue;
			}
			size_t i;
			for( i = 0; i < member->mList.values; i++ ) {
				pjson::builder jb;
				if( !jb.init(512) ||
					!jb.beginObject() ||
					!jb.addObjectProp("cmd",3) ||
					!jb.valueInt(cmd->mCmd) ||
					!jb.addObjectProp("aid",3) ||
					!jb.valueString(member->mList.value[i]) ||
					!jb.addObjectProp("gid",3) ||
					!jb.valueString(_gid) ||
					!jb.endObject() ) {
					free(member);
					return taskResponse(q, ErrorRuntime, "regSyncQueue: Json build error(5). account=%s,%s. %s.",
							cmd->mAREF.mId, cmd->mAREF.mName, codeToMessgae(ErrorRuntime));
				}
				pjson::json *js = jb.detouch();
				StdBuffer syncdata;
				js->getText(&syncdata, false);
				delete js;
				e = port->pushSyncQueue(_sid, syncdata.getPtr());
				if( e != ErrorOk ) {
					taskLog(q, LOG_ERR, "regSyncQueue: PushSyncQueue error. account=%s,%s. %s.",
							cmd->mAREF.mId, cmd->mAREF.mName, codeToMessgae(e));
				}
			}
			if( member ) free(member);
		}
	}
	return taskResponse(q, ErrorOk, "ok");
}

/*
bool  Database::cmdRegSession(QnDBRegister *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdRegSession", task, q);
	RegistSession *qp = (RegistSession*)q->mRegist;

	QnDBResult *r =  new QnDBResult(task);

	char id_ret[256];
	DBError e = port->setSession(&qp->mSession, id_ret);
	if( e != DBErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}
	r->resultId(e, id_ret);
	PROC_END("cmdRegSession", task, q);
	return task->endTask(r);
}
*/

bool Database::cmdRegNode(QnDBRegister *q, DBPort *port)
{
	// TODO: not implement cmdRegNode
	return taskResponse(q, ErrorNotImpl, "%s.", codeToMessgae(ErrorNotImpl));
}

bool Database::cmdRemAccount(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdRemAccount", task, q);
	QueryKey1 *qp = (QueryKey1*)q->mQuery;

	QnDBResult *r =  new QnDBResult(task);
	char _aid[SST_ACCOUNTID_LEN+1];
	ErrorCode e = confirmAID(q, port, &qp->mKey1, _aid);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}

	e = port->remAcocuntInfo(_aid);
	r->setDBResult(e, codeToMessgae(e));
	PROC_END("cmdRemAccount", task, q);
	return task->endTask(r);
}

bool Database::cmdRemDevice(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdRemDevice", task, q);
	QueryKey1 *qp = (QueryKey1*)q->mQuery;

	QnDBResult *r =  new QnDBResult(task);
	char _did[SST_DEVICEID_LEN+1];
	ErrorCode e = confirmDEV(q, port, &qp->mKey1, _did);
	if( e != ErrorOk ) {
		r->setDBResult(e, codeToMessgae(e));
		return task->endTask(r);
	}

	e = port->remDeviceInfo(_did);
	r->setDBResult(e, codeToMessgae(e));
	PROC_END("cmdRemDevice", task, q);
	return task->endTask(r);
}

bool Database::cmdRemGroupAccount(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdRemGroupAccount", task, q);
	QueryKey2 *cmd = (QueryKey2*)q->mQuery;

	char _gid[SST_GROUPID_LEN+1];
	QueryKey1 gkey;
	gkey.mSREF	= cmd->mSREF;
	gkey.mKey1	= cmd->mKey1;
	ErrorCode e = confirmGID(q, port, &gkey, _gid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "remGroupAccount: %s. confirmGID failed. service=%s,%s, group=%s,%s.",
				codeToMessgae(e),
				gkey.mSREF.mId, gkey.mSREF.mName,
				gkey.mKey1.mId, gkey.mKey1.mName);
	}

	char _aid[SST_ACCOUNTID_LEN+1];
	e = confirmAID(q, port, &cmd->mKey2, _aid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "remGroupAccount: %s. confirmAID failed. service=%s,%s group=%s,%s account=%s,%s.",
				codeToMessgae(e),
				gkey.mSREF.mId, gkey.mSREF.mName,
				_gid, gkey.mKey1.mName,
				cmd->mKey2.mId, cmd->mKey2.mName);
	}

	ResultGroup *group;
	e = port->getGroup(_gid, &group);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "remGroupAccount: %s. Group info load failed. service=%s,%s, group=%s,%s.",
				codeToMessgae(e),
				gkey.mSREF.mId, gkey.mSREF.mName,
				_gid, gkey.mKey1.mName);
	}

	ResultService *svc;
	e = port->getServiceInfo(group->mGroup.serviceId, &svc);
	if( e != ErrorOk ) {
		free(group);
		return taskResponse(q, e, "remGroupAccount: %s. Service info load failed. service=%s,%s, group=%s,%s.",
				codeToMessgae(e),
				gkey.mSREF.mId, gkey.mSREF.mName,
				_gid, gkey.mKey1.mName);
	}

	e = task->allowRemGroupMember(port, &svc->mService, &group->mGroup, _aid);
	if( e != ErrorOk ) {
		free(group);
		free(svc);
		return taskResponse(q, e, "remGroupAccount: %s. service=%s,%s, group=%s,%s account=%s,%s.",
				codeToMessgae(e),
				gkey.mSREF.mId, gkey.mSREF.mName,
				_gid, gkey.mKey1.mName,
				_aid, cmd->mKey2.mName);
	}
	free(group);
	free(svc);

	e = port->leaveGroupAccount(_gid, _aid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "remGroupAccount: %s. service=%s,%s, group=%s,%s account=%s,%s.",
				codeToMessgae(e),
				gkey.mSREF.mId, gkey.mSREF.mName,
				_gid, gkey.mKey1.mName,
				_aid, cmd->mKey2.mName);
	}

	QnDBResult *r =  new QnDBResult(task);
	r->setDBResult(e, codeToMessgae(e));
	PROC_END("cmdRemGroupAccount", task, q);
	return task->endTask(r);
}

bool Database::cmdRemGroup(QnDBQuery *q, DBPort *port)
{
	QueryKey1 *cmd = (QueryKey1*)q->mQuery;

	char _sid[SST_SERVICEID_LEN+1];
	ErrorCode e = confirmSID(q, port, &cmd->mSREF, _sid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "remGroup: %s. confirm service error. service=%s,%s",
				codeToMessgae(e),
				cmd->mSREF.mId, cmd->mSREF.mName);
	}

	char _gid[SST_GROUPID_LEN+1];
	QueryKey1 gkey;
	gkey.mSREF.mId		= _sid;
	gkey.mSREF.mName	= NULL;
	gkey.mKey1			= cmd->mKey1;
	e = confirmGID(q, port, &gkey, _gid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "remGroup: %s. confirm group error. service=%s,%s group=%s,%s",
				codeToMessgae(e),
				cmd->mSREF.mId, cmd->mSREF.mName,
				cmd->mKey1.mId, cmd->mKey1.mName);
	}

	e = port->remGroup(_gid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "remGroup: %s. remove group error. service=%s,%s group=%s,%s",
				codeToMessgae(e),
				cmd->mSREF.mId, cmd->mSREF.mName,
				cmd->mKey1.mId, cmd->mKey1.mName);
	}
	return taskResponse(q, ErrorOk, "ok");
}

bool Database::cmdRemResource(QnDBQuery *q, DBPort *port)
{
	QueryKey1 *cmd = (QueryKey1*)q->mQuery;

	char _rid[SST_RESOURCEID_LEN+1];
	ErrorCode e = confirmRID(q, port, cmd, _rid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. confirm resource error. service=%s,%s resource=%s,%s",
				codeToMessgae(e),
				cmd->mSREF.mId, cmd->mSREF.mName,
				cmd->mKey1.mId, cmd->mKey1.mName);
	}

	e = port->remResourceInfo(_rid);
	if( e != ErrorOk ) {
		return taskResponse(q, e, "%s. remove resourceInfo error. service=%s,%s resource=%s,%s",
				codeToMessgae(e),
				cmd->mSREF.mId, cmd->mSREF.mName,
				cmd->mKey1.mId, cmd->mKey1.mName);
	}

	return taskResponse(q, ErrorOk, "ok");
}

bool Database::cmdRemShare(QnDBQuery *q, DBPort *port)
{
	Task *task = q->getTask();
	PROC_BEGIN("cmdRemShare", task, q);
	QueryKey1 *qp = (QueryKey1*)q->mQuery;

	QnDBResult *r =  new QnDBResult(task);
	char mid[SST_SHAREID_LEN+1];
	ErrorCode e = confirmMID(q, port, &qp->mKey1, mid);
	if( e != ErrorOk ) {
		r->setDBResult(e, "%s: remShare", codeToMessgae(e));
		return task->endTask(r);
	}

	e = port->remShareInfo(mid);
	r->setDBResult(e, codeToMessgae(e));
	PROC_END("cmdRemShare", task, q);
	return task->endTask(r);
}


}
