/*
 * processor.cpp
 *
 *  Created on: 2012/07/04
 *      Author: Yasuoki
 */

#include "commands.h"
#include "proc_command.h"
#include "proc_database.h"
#include "proc_requester.h"
#include "task.h"
#include "sentinel.h"
#include "buffer.h"
#include "pjson.h"
#include <syslog.h>
#include <errno.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <curl/curl.h>

namespace SST {

CommandProc::CommandProc()
{
}

CommandProc::~CommandProc()
{
}

bool CommandProc::configure(Sentinel *obj, Conf *conf)
{
	if( !Process::configure(obj, conf) )
		return false;

	mMaxThread	= conf->maxCommandProcess;
	mMinThread	= conf->minCommandProcess;

	return true;
}

void CommandProc::clear()
{
	Process::clear();
}

void CommandProc::stop(QueueNode *q)
{
	return;
}

bool CommandProc::response(QueueNode *q, pjson::builder &jb)
{
	Task *task = q->getTask();
	QnResult *r = new QnResult(task);
	pjson::json *js = jb.detouch();
	r->setResult(q->getFunction(), ErrorOk, js);
	task->endTask(r);
	return true;
}

bool CommandProc::parseAccessMode(const char *src, ShareAccessMode &sam)
{
	const char *p = src;
	bool readset=false;
	sam = ShareAccessRead;
	while(*p) {
		bool valid=false;
		while(*p==' '||*p=='\t') p++;
		if( *p == 0 ) break;
		if( strncmp(p, "read", 4) == 0 ) {
			sam	= (ShareAccessMode)((int)sam | (int)ShareAccessRead);
			readset = true;
			valid	= true;
			p += 4;
		} else if( strncmp(p, "write",5) == 0 ) {
			sam	= (ShareAccessMode)((int)sam | (int)ShareAccessWrite);
			valid	= true;
			p += 5;
		} else if( strncmp(p, "gadget",6) == 0 ) {
			sam	= (ShareAccessMode)((int)sam | (int)ShareAccessGadget);
			valid	= true;
			p += 6;
		} else if( strncmp(p, "tag",3) == 0 ) {
			sam	= (ShareAccessMode)((int)sam | (int)ShareAccessTag);
			valid	= true;
			p += 3;
		} else if( strncmp(p, "create",6) == 0 ) {
			sam	= (ShareAccessMode)((int)sam | (int)ShareAccessCreate);
			valid	= true;
			p += 3;
		}
		if( !valid || (*p != 0 && *p != ' ' && *p != '\t' && *p != ',') ) {
			return false;
		}
		while(*p==' '||*p=='\t') p++;
		if( *p == 0 ) break;
		if( *p == ',' ) p++;
	}
	if( readset && sam != ShareAccessRead ) {
		return false;
	}

	return true;
}

bool CommandProc::stringAccessMode(ShareAccessMode mode, Buffer &text)
{
	text.clear();
	if( mode & ShareAccessWrite ) {
		text.add("write", -1);
	}
	if( mode & ShareAccessGadget ) {
		if( !text.empty() ) text.add(",",-1);
		text.add("gadget", -1);
	}
	if( mode & ShareAccessTag ) {
		if( !text.empty() ) text.add(",",-1);
		text.add("tag", -1);
	}
	if( mode & ShareAccessCreate ) {
		if( !text.empty() ) text.add(",",-1);
		text.add("create", -1);
	}
	if( text.empty() ) {
		text.add("read", -1);
	}
	return true;
}

const char * CommandProc::getFuncName(int id)
{
	switch(id) {
	case CMDExit:			return "Exit";

	case CMDLogin:			return "Login";
	case CMDRegAccount:		return "RegAccount";
	case CMDSetAccount:		return "SetAccount";
	case CMDRmAccount:		return "RmAccount";
	case CMDLsAccount:		return "LsAccount";
	case CMDShowAccount:	return "ShowAccount";
	case CMDLockAccount:	return "LockAccount";
	case CMDUnlockAccount:	return "UnlockAccount";

	case CMDRmDevice:		return "RmDevice";
	case CMDLsDevice:		return "LsDevice";
	case CMDShowDevice:		return "ShowDevice";

	case CMDRegService:		return "RegService";
	case CMDSetService:		return "SetService";
	case CMDRmService:		return "RmService";
	case CMDLsService:		return "LsService";
	case CMDShowService:	return "ShowService";
	case CMDLsServiceGroup:	return "LsServiceGroup";

	case CMDAddGroup:		return "AddGroup";
	case CMDSetGroup:		return "SetGroup";
	case CMDRmGroup:		return "RmGroup";
	case CMDLsGroup:		return "LsGroup";
	case CMDShowGroup:		return "ShowGroup";

	case CMDJoinGroup:		return "JoinGroup";
	case CMDLeaveGroup:		return "LeaveGroup";
	case CMDLsGroupUser:	return "LsGroupUser";

	case CMDRmResource:		return "RmResource";
	case CMDLsResource:		return "LsResource";
	case CMDShowResource:	return "ShowResource";

	case CMDShare:			return "Share";
	case CMDUnshare:		return "Unshare";
	case CMDShowShare:		return "ShowShare";
	case CMDLsGroupShare:	return "LsGroupShare";
	case CMDLsResourceShare:return "LsResourceShare";
	case CMDLsDeviceShare:	return "LsDeviceShare";

	case CMDUpdate:			return "Update";
	case CMDSync:			return "Sync";
	case CMDSyncQueue:		return "SyncQueue";
	case CMDSyncGroup:		return "SyncGroup";
	case CMDSyncAccount:	return "SyncAccount";
	case CMDSyncDevice:		return "SyncDevice";
	case CMDDownSync:		return "DownSync";
	case CMDUpSync:			return "UpSync";
	case CMDSyncStat:		return "SyncStat";

	case CMDDiagSession:	return "DiagSession";
	case CMDDiagTask:		return "DiagTask";
	case CMDDiagQueue:		return "DiagQueue";
	case CMDDiagProcess:	return "DiagProcess";
	case CMDDiagThread:		return "DiagThread";
	case CMDDiagRevKey:		return "DiagRevKey";
	case CMDDebugTest:		return "DebugTest";
	}
	return Process::getFuncName(id);
}

bool CommandProc::exec(QueueNode *q)
{
	switch(q->getFunction()) {
	case CMDExit:			return cmdExit(q);

	case CMDLogin:			return cmdLogin(q);
	case CMDRegAccount:		return cmdRegAccount(q);
	case CMDSetAccount:		return cmdSetAccount(q);
	case CMDRmAccount:		return cmdRmAccount(q);
	case CMDLsAccount:		return cmdLsAccount(q);
	case CMDShowAccount:	return cmdShowAccount(q);
	case CMDLockAccount:	return cmdLockAccount(q);
	case CMDUnlockAccount:	return cmdUnlockAccount(q);

	case CMDRmDevice:		return cmdRmDevice(q);
	case CMDLsDevice:		return cmdLsDevice(q);
	case CMDShowDevice:		return cmdShowDevice(q);

	case CMDRegService:		return cmdRegService(q);
	case CMDSetService:		return cmdSetService(q);
	case CMDRmService:		return cmdRmService(q);
	case CMDLsService:		return cmdLsService(q);
	case CMDShowService:	return cmdShowService(q);
	case CMDLsServiceGroup:	return cmdLsServiceGroup(q);

	case CMDAddGroup:		return cmdAddGroup(q);
	case CMDSetGroup:		return cmdSetGroup(q);
	case CMDRmGroup:		return cmdRmGroup(q);
	case CMDLsGroup:		return cmdLsGroup(q);
	case CMDShowGroup:		return cmdShowGroup(q);

	case CMDJoinGroup:		return cmdJoinGroup(q);
	case CMDLeaveGroup:		return cmdLeaveGroup(q);
	case CMDLsGroupUser:	return cmdLsGroupUser(q);

	case CMDRmResource:		return cmdRmResource(q);
	case CMDLsResource:		return cmdLsResource(q);
	case CMDShowResource:	return cmdShowResource(q);

	case CMDShare:			return cmdShare(q);
	case CMDUnshare:		return cmdUnshare(q);
	case CMDShowShare:		return cmdShowShare(q);
	case CMDLsGroupShare:	return cmdLsGroupShare(q);
	case CMDLsResourceShare:return cmdLsResourceShare(q);
	case CMDLsDeviceShare:	return cmdLsDeviceShare(q);

	case CMDUpdate:			return cmdUpdate(q);
	case CMDSync:			return cmdSync(q);
	case CMDSyncQueue:		return cmdSyncQueue(q);
	case CMDSyncGroup:		return cmdSyncGroup(q);
	case CMDSyncAccount:	return cmdSyncAccount(q);
	case CMDSyncDevice:		return cmdSyncDevice(q);
	case CMDDownSync:		return cmdDownSync(q);
	case CMDUpSync:			return cmdUpSync(q);
	case CMDSyncStat:		return cmdSyncStat(q);

	case CMDDiagSession:	return cmdDiagSession(q);
	case CMDDiagTask:		return cmdDiagTask(q);
	case CMDDiagQueue:		return cmdDiagQueue(q);
	case CMDDiagProcess:	return cmdDiagProcess(q);
	case CMDDiagThread:		return cmdDiagThread(q);
	case CMDDiagRevKey:		return cmdDiagRevKey(q);
	case CMDDebugTest:		return cmdDebugTest(q);
	}
	taskLog(q, LOG_ERR, "unknown queue type q=%x type=%d func=%d step=%d", q, q->getType(), q->getFunction(), q->getStep());
	return false;
}

bool CommandProc::cmdExit(QueueNode *q)
{
	return taskResponse(q, ErrorNotImpl, "%s", codeToMessgae(ErrorNotImpl));
}

bool CommandProc::cmdDiagSession(QueueNode *q)
{
	SessionPool &tm = mSentinel->getSessionPool();
	pjson::builder jb;
	jb.init(512);
	if( !tm.diag(jb) ) {
		return taskResponse(q, ErrorRuntime, "diag error");
	}
	if( !response(q, jb) )
		return false;
	return true;
}

bool CommandProc::cmdDiagTask(QueueNode *q)
{
	TaskManager &tm = mSentinel->getTaskManager();
	pjson::builder jb;
	jb.init(512);
	if( !tm.diag(jb) ) {
		return taskResponse(q, ErrorRuntime, "diag error");
	}
	if( !response(q, jb) )
		return false;
	return true;
}

// TODO: CommandProc::cmdDiagQueue 未実装
bool CommandProc::cmdDiagQueue(QueueNode *q)
{
	return taskResponse(q, ErrorNotImpl, "%s", codeToMessgae(ErrorNotImpl));
}

bool CommandProc::cmdDiagProcess(QueueNode *q)
{
	pjson::builder jb;
	jb.init(512);
	if( !jb.beginObject() ) {
		return taskResponse(q, ErrorRuntime, "%s::cmdDiagProcess json error 5. code=%d", getName(), jb.getError());
	}

	if( !mSentinel->getListener().diag(jb) ) {
		return taskResponse(q, ErrorRuntime, "diag error");
	}
	if( !mSentinel->getCommand().diag(jb) ) {
		return taskResponse(q, ErrorRuntime, "diag error");
	}
	if( !mSentinel->getDatabase().diag(jb) ) {
		return taskResponse(q, ErrorRuntime, "diag error");
	}
	if( !mSentinel->getRequester().diag(jb) ) {
		return taskResponse(q, ErrorRuntime, "diag error");
	}

	if( !jb.endObject() ) {
		return taskResponse(q, ErrorRuntime, "diag error: json error 5. code=%d", getName(), jb.getError());
	}
	if( !response(q, jb) )
		return false;
	return true;
}

bool CommandProc::cmdDiagThread(QueueNode *q)
{
	Scheduler &tm = mSentinel->getScheduler();
	pjson::builder jb;
	jb.init(512);
	if( !tm.diag(jb) ) {
		return taskResponse(q, ErrorRuntime, "diag error");
	}
	if( !response(q, jb) )
		return false;
	return true;
}

bool CommandProc::cmdDiagRevKey(QueueNode *q)
{
	return taskResponse(q, ErrorNotImpl, "%s", codeToMessgae(ErrorNotImpl));
}

bool  CommandProc::cmdDebugTest(QueueNode *q)
{
	Task *task = q->getTask();

	int step = q->getStep();
	switch(step) {
	case 0:
		{
			// syncronize account
			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("syncAccout", 10) ||
				!jb.beginObject() ||
				!jb.addObjectProp("aid", 3) ||
				!jb.valueString("A00000001") ||
				!jb.addObjectProp("sid", 3) ||
				!jb.valueString("S01") ||
				!jb.endObject() ||
				!jb.endObject() ) {
				taskLog(q, LOG_ERR, "%s::cmdUnunlockAccount: json builder failed 3. code=%d", getName(), jb.getError());
				return false;
			}

			QueueNode *qn = new QueueNode(task, this, CMDSyncAccount);
			qn->setJSONMessage(jb.detouch());
			return task->call(qn, step+1);
		}
		break;
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != 0 ) {
				taskLog(q, LOG_WARNING, "%s::cmdUnunlockAccount: update command error. code=%d.", getName(), code);
			}

			QnResult *r = new QnResult(task);
			r->setResult(q->getFunction(),code);
			return task->endTask(r);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step=%d.", step);
	return false;
}

}
