/*
 * cmd_group.cpp
 *
 *  Created on: 2012/12/28
 *      Author: yasuoki
 */

#include "commands.h"
#include "proc_command.h"
#include "proc_database.h"
#include "proc_requester.h"
#include "task.h"
#include "sentinel.h"
#include "buffer.h"
#include "pjson.h"
#include <syslog.h>
#include <errno.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

namespace SST {

class TBAddGroup : public TaskBuffer {
public:
	TBAddGroup() {
		mJsCommand	= NULL;
		mSRef.mId	= NULL;
		mSRef.mName	= NULL;
		mGRef.mId	= NULL;
		mGRef.mName	= NULL;
		mAccess		= ShareAccessCreate;
		mJsGName	= NULL;
		mJsDesc		= NULL;
		mJsAdmin	= NULL;
		mJsAccess	= NULL;
	}
	virtual ~TBAddGroup() {
		if( mJsCommand )	delete mJsCommand;
	}
	pjson::json *	mJsCommand;
	QueryKeyType	mSRef;
	QueryKeyType	mGRef;
	ShareAccessMode	mAccess;
	pjson::value *	mJsGName;
	pjson::value *	mJsDesc;
	pjson::value *	mJsAdmin;
	pjson::value *	mJsAccess;
};

bool CommandProc::cmdAddGroup(QueueNode *q)
{
	Task *task = q->getTask();
	int step = q->getStep();

	TBAddGroup *tb = (TBAddGroup *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBAddGroup();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = tb->mJsCommand->get("addGroup");
			if( !vCmd || vCmd->vt != pjson::vt_object ) {
				return taskResponse(q, ErrorParameter, "%s. \"addGroup\" is not object.", codeToMessgae(ErrorParameter));
			}
			pjson::value *vSID		= tb->mJsCommand->get("sid",  vCmd);
			pjson::value *vSName	= tb->mJsCommand->get("serviceName", vCmd);
			tb->mJsGName			= tb->mJsCommand->get("groupName",  vCmd);
			tb->mJsDesc				= tb->mJsCommand->get("description", vCmd);
			tb->mJsAdmin			= tb->mJsCommand->get("adminId", vCmd);
			tb->mJsAccess			= tb->mJsCommand->get("access", vCmd);

			if( vSID && vSID->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"sid\" type.", codeToMessgae(ErrorParameter));
			if( vSName && vSName->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"serviceName\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsGName && tb->mJsGName->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s, \"groupName\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsDesc && tb->mJsDesc->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"description\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsAdmin && tb->mJsAdmin->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"adminId\" type.", codeToMessgae(ErrorParameter));
			if( tb->mJsAccess && tb->mJsAccess->vt != pjson::vt_string )
				return taskResponse(q, ErrorParameter, "%s. \"access\" type.", codeToMessgae(ErrorParameter));
			if( !tb->mJsGName ) {
				return taskResponse(q, ErrorParameter, "%s. \"groupName\" is not specified.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsAccess ) {
				if( !parseAccessMode(tb->mJsAccess->vString, tb->mAccess) ) {
					return taskResponse(q, ErrorParameter, "%s. \"access\" value.", codeToMessgae(ErrorParameter));
				}
				if( tb->mAccess != ShareAccessCreate &&
					tb->mAccess != ShareAccessWrite &&
					tb->mAccess != ShareAccessRead) {
					return taskResponse(q, ErrorParameter, "%s. \"access\" value.", codeToMessgae(ErrorParameter));
				}
			}

			tb->mSRef.mId	= vSID   ? vSID->vString : NULL;
			tb->mSRef.mName	= vSName ? vSName->vString : NULL;
			tb->mGRef.mId	= NULL;
			tb->mGRef.mName	= NULL;

			if( tb->mJsAdmin ) {
				const char *session_id = task->getUserId();
				if( !session_id || strcmp(session_id, tb->mJsAdmin->vString) != 0 ) {
					QueryKeyType aref;
					aref.mId	= NULL;
					aref.mName	= tb->mJsAdmin->vString;
					QnDBQuery *qn = new QnDBQuery(task);
					qn->queryAccount(&aref);
					return task->call(qn, step+1);
				}
			}
		}
	case 1:
		{
			ResultAccount *acc = NULL;
			if( step == 1 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					return taskResponse(q, code, "Account info load failed. %s", res->getMessagePtr());
				}
				acc = (ResultAccount*)res->mResult;
			}

			step	= 2;

			GroupInfo g;
			memset(&g, 0, sizeof(g));
			g.id			= NULL;
			g.serviceId		= (char*)tb->mSRef.mId;
			g.name			= (char*)tb->mJsGName->vString;
			g.status		= GroupNormal;
			g.adminId		= acc ? acc->mAccount.id : NULL;
			g.description	= (char*)(tb->mJsDesc ? tb->mJsDesc->vString : NULL);
			g.accessMode	= tb->mAccess;

			QnDBRegister *qn = new QnDBRegister(task);
			qn->registGroup(&tb->mSRef, &g);
			return task->call(qn, step+1);
		}
		break;
	case 3:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Group info registration was failed. %s", res->getMessagePtr());
			}
			ResultId *rs = (ResultId*)(res->mResult);
			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("result", 6) ||
				!jb.valueInt(0) ||
				!jb.addObjectProp("sequence", 8) ||
				!jb.valueInt(task->getSequence()) ||
				!jb.addObjectProp("gid", 3) ||
				!jb.valueString(rs->mId) ||
				!jb.endObject() ) {
				taskLog(q, LOG_ERR, "json builder failed(1). code=%d.", jb.getError());
				return false;
			}
			return response(q, jb);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}

class TBSetGroup : public TaskBuffer {
public:
	TBSetGroup() {
		mJsCommand	= NULL;
		mSRef.mId	= NULL;
		mSRef.mName	= NULL;
		mGRef.mId	= NULL;
		mGRef.mName	= NULL;
		mAccess		= ShareAccessFull;
		mJsGName	= NULL;
		mJsDesc		= NULL;
		mJsAdmin	= NULL;
		mJsAccess	= NULL;
		mGroup		= NULL;

	}
	virtual ~TBSetGroup() {
		if( mJsCommand )	delete mJsCommand;
		if( mGroup )		free(mGroup);
	}
	pjson::json *	mJsCommand;
	QueryKeyType	mSRef;
	QueryKeyType	mGRef;
	ShareAccessMode	mAccess;
	pjson::value *	mJsGName;
	pjson::value *	mJsDesc;
	pjson::value *	mJsAdmin;
	pjson::value *	mJsAccess;
	ResultGroup *	mGroup;
};

bool CommandProc::cmdSetGroup(QueueNode *q)
{
	Task *task = q->getTask();
	int step = q->getStep();

	TBSetGroup *tb = (TBSetGroup *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBSetGroup();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vSID		= NULL;
			pjson::value *vSName	= NULL;
			pjson::value *vGID		= NULL;
			pjson::value *vGName	= NULL;
			pjson::value *vSet 		= NULL;
			pjson::value *vCmd = tb->mJsCommand->get("setGroup");
			if( vCmd ) {
				vSID	= tb->mJsCommand->get("sid",  vCmd);
				vSName	= tb->mJsCommand->get("serviceName", vCmd);
				vGID	= tb->mJsCommand->get("gid",  vCmd);
				vGName	= tb->mJsCommand->get("groupName", vCmd);
				vSet 	= tb->mJsCommand->get("set", vCmd);

				if( vSID && vSID->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"sid\" type.", codeToMessgae(ErrorParameter));
				if( vSName && vSName->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"serviceName\" type.", codeToMessgae(ErrorParameter));
				if( vGID && vGID->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"gid\" type.", codeToMessgae(ErrorParameter));
				if( vGName && vGName->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"groupName\" type.", codeToMessgae(ErrorParameter));

				if( !vSet ) {
					return taskResponse(q, ErrorParameter, "\"set\" is not specified.");
				}

				tb->mJsGName	= tb->mJsCommand->get("groupName", vSet);
				tb->mJsDesc		= tb->mJsCommand->get("description", vSet);
				tb->mJsAdmin	= tb->mJsCommand->get("adminId", vSet);
				tb->mJsAccess	= tb->mJsCommand->get("access", vSet);

				if( tb->mJsGName && tb->mJsGName->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"set.groupName\" type.", codeToMessgae(ErrorParameter));
				if( tb->mJsDesc && tb->mJsDesc->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"set.description\" type.", codeToMessgae(ErrorParameter));
				if( tb->mJsAdmin && tb->mJsAdmin->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"set.adminid\" type.", codeToMessgae(ErrorParameter));
				if( tb->mJsAccess && tb->mJsAccess->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"access\" type.", codeToMessgae(ErrorParameter));
				if( tb->mJsAccess ) {
					if( !parseAccessMode(tb->mJsAccess->vString, tb->mAccess) ) {
						return taskResponse(q, ErrorParameter, "%s. \"access\" value.", codeToMessgae(ErrorParameter));
					}
					if( tb->mAccess != ShareAccessCreate &&
						tb->mAccess != ShareAccessWrite &&
						tb->mAccess != ShareAccessRead ) {
						return taskResponse(q, ErrorParameter, "%s. \"access\" value.", codeToMessgae(ErrorParameter));
					}
				}
			}
			if( !vGID && !vGName ) {
				return taskResponse(q, ErrorParameter, "%s. Group is not specified.", codeToMessgae(ErrorParameter));
			}

			tb->mSRef.mId	= vSID   ? vSID->vString : NULL;
			tb->mSRef.mName	= vSName ? vSName->vString : NULL;
			tb->mGRef.mId	= vGID   ? vGID->vString : NULL;
			tb->mGRef.mName	= vGName ? vGName->vString : NULL;

			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryGroup(&tb->mSRef, &tb->mGRef);
			return task->call(qn, step+1);
		}
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Group info load failed. %s", res->getMessagePtr());
			}
			tb->mGroup		= (ResultGroup*)res->mResult;
			res->mResult	= NULL;
			tb->mSRef.mId	= tb->mGroup->mGroup.serviceId;

			if( tb->mJsAdmin ) {
				const char *session_id = task->getUserId();
				if( !session_id || strcmp(session_id, tb->mJsAdmin->vString) != 0 ) {
					QueryKeyType aref;
					aref.mId	= NULL;
					aref.mName	= tb->mJsAdmin->vString;
					QnDBQuery *qn = new QnDBQuery(task);
					qn->queryAccount(&aref);
					return task->call(qn, step+1);
				}
			}
		}
	case 2:
		{
			ResultAccount *acc = NULL;
			if( step == 2 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					return taskResponse(q, code, "Account info load failed. %s", res->getMessagePtr());
				}
				acc = (ResultAccount*)res->mResult;
			}

			step	= 2;

			GroupInfo g;
			g	= tb->mGroup->mGroup;
			if( tb->mJsGName )
				g.name			= (char*)tb->mJsGName->vString;
			if( tb->mJsDesc )
				g.description	= (char*)tb->mJsDesc->vString;
			if( acc )
				g.adminId		= acc->mAccount.id;
			if( tb->mJsAccess )
				g.accessMode	= tb->mAccess;
			QnDBRegister *qn = new QnDBRegister(task);
			qn->registGroup(&tb->mSRef, &g);
			return task->call(qn, step+1);
		}
		break;
	case 3:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Group info update failed. %s.", res->getMessagePtr());
			}
			ResultId *rs = (ResultId*)(res->mResult);
			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("result", 6) ||
				!jb.valueInt(0) ||
				!jb.addObjectProp("sequence", 8) ||
				!jb.valueInt(task->getSequence()) ||
				!jb.addObjectProp("gid", 3) ||
				!jb.valueString(rs->mId) ||
				!jb.endObject() ) {
				taskLog(q, LOG_ERR, "Json builder failed(1). code=%d.", jb.getError());
				return false;
			}
			return response(q, jb);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}

class TBRmGroup : public TaskBuffer {
public:
	TBRmGroup() {
		mJsCommand	= NULL;
		mSRef.mId	= NULL;
		mSRef.mName	= NULL;
		mGRef.mId	= NULL;
		mGRef.mName	= NULL;
		mGroup		= NULL;
		mMember		= NULL;
		mCount		= 0;
		mIndex		= 0;
	}
	virtual ~TBRmGroup() {
		if( mJsCommand )	delete mJsCommand;
		if( mGroup )		free(mGroup);
		if( mMember )		free(mMember);
	}
	pjson::json *	mJsCommand;
	QueryKeyType	mSRef;
	QueryKeyType	mGRef;
	ResultGroup *	mGroup;
	ResultList *	mMember;
	size_t			mCount;
	size_t			mIndex;
	pjson::builder	mJb;
};

bool CommandProc::cmdRmGroup(QueueNode *q)
{
	Task *task = q->getTask();
	int step = q->getStep();

	TBRmGroup *tb = (TBRmGroup *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBRmGroup();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vSID		= NULL;
			pjson::value *vSName	= NULL;
			pjson::value *vGID		= NULL;
			pjson::value *vGName	= NULL;

			pjson::value *vCmd = tb->mJsCommand->get("rmGroup");
			if( vCmd ) {
				vSID	= tb->mJsCommand->get("sid",  vCmd);
				vSName	= tb->mJsCommand->get("serviceName", vCmd);
				vGID	= tb->mJsCommand->get("gid",  vCmd);
				vGName	= tb->mJsCommand->get("groupName", vCmd);

				if( vSID && vSID->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"sid\" type.", codeToMessgae(ErrorParameter));
				if( vSName && vSName->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"serviceName\" type.", codeToMessgae(ErrorParameter));
				if( vGID && vGID->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"gid\" type.", codeToMessgae(ErrorParameter));
				if( vGName && vGName->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"groupName\" type.", codeToMessgae(ErrorParameter));
			}
			if( !vGID && !vGName ) {
				return taskResponse(q, ErrorParameter, "%s. Group is not specified.", codeToMessgae(ErrorParameter));
			}

			tb->mSRef.mId	= vSID   ? vSID->vString : NULL;
			tb->mSRef.mName	= vSName ? vSName->vString : NULL;
			tb->mGRef.mId	= vGID   ? vGID->vString : NULL;
			tb->mGRef.mName	= vGName ? vGName->vString : NULL;

			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryGroup(&tb->mSRef, &tb->mGRef);
			return task->call(qn, step+1);
		}
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Group info load failed. %s",
						res->getMessagePtr());
			}

			tb->mGroup		= (ResultGroup*)res->mResult;
			res->mResult	= NULL;
			tb->mGRef.mId	= tb->mGroup->mGroup.id;
			tb->mSRef.mId	= tb->mGroup->mGroup.serviceId;

			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryAccountList(&tb->mSRef, &tb->mGRef, NULL);
			return task->call(qn, step+1);
		}
	case 2:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Group member list load error. %s", res->getMessagePtr());
			}

			tb->mMember		= (ResultList*)res->mResult;
			res->mResult	= NULL;
			tb->mCount		= tb->mMember->mList.values;
		}
	case 3:
		{
			if( step == 3 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					return taskResponse(q, code, "leaveGroup error. %s", res->getMessagePtr());
				}
				tb->mIndex++;
			}

			step = 3;

			if( tb->mIndex < tb->mCount ) {
				pjson::builder jb;
				if( !jb.init(512) ||
					!jb.beginObject() ||
					!jb.addObjectProp("leaveGroup", 10),
					!jb.beginObject() ||
					!jb.addObjectProp("sid", 3) ||
					!jb.valueString(tb->mGroup->mGroup.serviceId) ||
					!jb.addObjectProp("gid", 3) ||
					!jb.valueString(tb->mGroup->mGroup.id) ||
					!jb.addObjectProp("aid", 3) ||
					!jb.valueString(tb->mMember->mList.value[tb->mIndex]) ||
					!jb.endObject() ||
					!jb.endObject() ) {
					taskLog(q, LOG_ERR, "json builder failed(1). code=%d.", jb.getError() );
					return false;
				}
				QueueNode *qn = new QueueNode(task, this, CMDLeaveGroup);
				qn->setJSONMessage(jb.detouch());
				return task->call(qn, step);
			}

			QnDBQuery *qn = new QnDBQuery(task);
			qn->removeGroup(&tb->mSRef, &tb->mGRef);
			return task->call(qn, step+1);
		}
		break;
	case 4:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Group info remove failed. %s", res->getMessagePtr());
			}
			return taskResponse(q, ErrorOk, "%d users leaved.", tb->mCount);
		}
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}

class TBLsGroup : public TaskBuffer {
public:
	TBLsGroup() {
		mJsCommand		= NULL;
		mSRef.mId		= NULL;
		mSRef.mName		= NULL;
		mARef.mId		= NULL;
		mARef.mName		= NULL;
		mRange.mStart	= 0;
		mRange.mCount	= -1;
		mGroupList		= NULL;
		mGroupCount		= 0;
		mGroupIndex		= 0;
	}
	virtual ~TBLsGroup() {
		if( mJsCommand )	delete mJsCommand;
		if( mGroupList )	free(mGroupList);
	}
	pjson::json *	mJsCommand;
	QueryKeyType	mSRef;
	QueryKeyType	mARef;
	QueryRangeType	mRange;
	ResultList *	mGroupList;
	size_t			mGroupCount;
	size_t			mGroupIndex;
	pjson::builder	mJb;
};

bool CommandProc::cmdLsGroup(QueueNode *q)
{
	Task *task = q->getTask();
	int step = q->getStep();

	TBLsGroup *tb = (TBLsGroup *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBLsGroup();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = tb->mJsCommand->get("lsGroup");
			if( vCmd ) {
				pjson::value *vSID		= tb->mJsCommand->get("sid",  vCmd);
				pjson::value *vSName	= tb->mJsCommand->get("serviceName", vCmd);
				pjson::value *vAID		= tb->mJsCommand->get("aid",  vCmd);
				pjson::value *vUID		= tb->mJsCommand->get("userId", vCmd);
				pjson::value *vFrom		= tb->mJsCommand->get("from", vCmd);
				pjson::value *vCount	= tb->mJsCommand->get("count", vCmd);

				if( vSID && vSID->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"sid\" type.", codeToMessgae(ErrorParameter));
				if( vSName && vSName->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"serviceName\" type.", codeToMessgae(ErrorParameter));
				if( vAID && vAID->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"aid\" type.", codeToMessgae(ErrorParameter));
				if( vUID && vUID->vt != pjson::vt_string )
					return taskResponse(q, ErrorParameter, "%s. \"userId\" type.", codeToMessgae(ErrorParameter));
				if( vFrom && vFrom->vt != pjson::vt_int )
					return taskResponse(q, ErrorParameter, "%s. \"from\" type.", codeToMessgae(ErrorParameter));
				if( vCount && vCount->vt != pjson::vt_int )
					return taskResponse(q, ErrorParameter, "%s. \"count\" type.", codeToMessgae(ErrorParameter));

				tb->mSRef.mId		= vSID   ? vSID->vString : NULL;
				tb->mSRef.mName		= vSName ? vSName->vString : NULL;
				tb->mARef.mId		= vAID   ? vAID->vString : NULL;
				tb->mARef.mName		= vUID   ? vUID->vString : NULL;
				tb->mRange.mStart	= vFrom  ? vFrom->vInt : 0;
				tb->mRange.mCount	= vCount ? vCount->vInt : -1;
			}

			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryGroupList(&tb->mARef, &tb->mRange);
			return task->call(qn, step+1);
		}
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Group list load failed. %s", res->getMessagePtr());
			}

			tb->mGroupList	= (ResultList*)res->mResult;
			res->mResult	= NULL;
			tb->mGroupCount	= tb->mGroupList->mList.values;

			if( !tb->mJb.init(256) ||
				!tb->mJb.beginObject() ||
				!tb->mJb.addObjectProp("result", 6) ||
				!tb->mJb.valueInt(0) ||
				!tb->mJb.addObjectProp("sequence", 8) ||
				!tb->mJb.valueInt(task->getSequence()) ||
				!tb->mJb.addObjectProp("from", 4) ||
				!tb->mJb.valueInt(tb->mRange.mStart) ) {
				taskLog(q, LOG_ERR, "Json builder failed(1). code=%d", tb->mJb.getError());
				return false;
			}
		}
	case 2:
		{
			if( step == 2) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					return taskResponse(q, code, "Group info load failed. %s", res->getMessagePtr());
				}
				ResultGroup *g = (ResultGroup*)res->mResult;
				if( tb->mGroupIndex == 0 ) {
					if( !tb->mJb.addObjectProp("groups", 6) ||
						!tb->mJb.beginArray() ) {
						taskLog(q, LOG_ERR, "Json builder failed(2). service=%s,%s account=%s,%s group=%s. code=%d. %s.",
								tb->mSRef.mId, tb->mSRef.mName,
								tb->mARef.mId, tb->mARef.mName,
								tb->mGroupList->mList.value[tb->mGroupIndex],
								tb->mJb.getError(),
								codeToMessgae(ErrorRuntime));
						return false;
					}
				}

				StdBuffer accessMode;
				stringAccessMode(g->mGroup.accessMode, accessMode);

				if( !tb->mJb.addArrayContent() ||
					!tb->mJb.beginObject() ||
					!tb->mJb.addObjectProp("gid", 3) ||
					!tb->mJb.valueString(g->mGroup.id) ||
					!tb->mJb.addObjectProp("groupName", 9) ||
					!tb->mJb.valueString(g->mGroup.name) ||
					!tb->mJb.addObjectProp("sid", 3) ||
					!tb->mJb.valueString(g->mGroup.serviceId) ||
					!tb->mJb.addObjectProp("access", 6) ||
					!tb->mJb.valueString(accessMode.getPtr()) ||
					!tb->mJb.endObject() ) {
					taskLog(q, LOG_ERR, "Json builder failed(3). service=%s,%s account=%s,%s group=%s. code=%d. %s.",
							tb->mSRef.mId, tb->mSRef.mName,
							tb->mARef.mId, tb->mARef.mName,
							tb->mGroupList->mList.value[tb->mGroupIndex],
							tb->mJb.getError(),
							codeToMessgae(ErrorRuntime));
					return false;
				}
				tb->mGroupIndex++;
			}

			step	= 2;
			if( tb->mGroupIndex < tb->mGroupCount ) {
				QnDBQuery *qn = new QnDBQuery(task);
				QueryKeyType gref;
				gref.mId	= tb->mGroupList->mList.value[tb->mGroupIndex];
				gref.mName	= NULL;
				qn->queryGroup(NULL, &gref);
				return task->call(qn, step);
			}

			if( tb->mGroupIndex > 0 ) {
				if( !tb->mJb.endArray() ) {
					taskLog(q, LOG_ERR, "Json builder failed(4). service=%s,%s account=%s,%s. code=%d. %s.",
							tb->mSRef.mId, tb->mSRef.mName,
							tb->mARef.mId, tb->mARef.mName,
							tb->mJb.getError(),
							codeToMessgae(ErrorRuntime));
					return false;
				}
			}
			if( !tb->mJb.endObject() ) {
				taskLog(q, LOG_ERR, "Json builder failed(5). service=%s,%s account=%s,%s. code=%d. %s.",
						tb->mSRef.mId, tb->mSRef.mName,
						tb->mARef.mId, tb->mARef.mName,
						tb->mJb.getError(),
						codeToMessgae(ErrorRuntime));
				return false;
			}
			return response(q, tb->mJb);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}

class TBShowGroup : public TaskBuffer {
public:
	TBShowGroup() {
		mCommand	= NULL;
		mSRef.mId	= NULL;
		mSRef.mName	= NULL;
		mGRef.mId	= NULL;
		mGRef.mName	= NULL;
		mGroup		= NULL;
		mAccount	= NULL;
		mGroupCount	= NULL;
		mDeviceCount	= NULL;
	}
	virtual ~TBShowGroup() {
		if( mCommand )		delete mCommand;
		if( mGroup )		free(mGroup);
		if( mAccount )		free(mAccount);
		if( mGroupCount )	free(mGroupCount);
		if( mDeviceCount )	free(mDeviceCount);
	}
	pjson::json *	mCommand;
	QueryKeyType	mSRef;
	QueryKeyType	mGRef;
	ResultGroup *	mGroup;
	ResultAccount *	mAccount;
	ResultCount *	mGroupCount;
	ResultCount *	mDeviceCount;
};

bool CommandProc::cmdShowGroup(QueueNode *q)
{
	Task *task = q->getTask();

	TBShowGroup *tb = (TBShowGroup *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBShowGroup();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	int step = q->getStep();
	switch(step) {
	case 0:
		{
			tb->mCommand	= q->detouchJSONMessage();
			pjson::value *vCmd = tb->mCommand->get("showGroup");
			if( vCmd ) {
				pjson::value * vSID		= tb->mCommand->get("sid",  vCmd);
				pjson::value * vSName	= tb->mCommand->get("serviceName",  vCmd);
				pjson::value * vGID		= tb->mCommand->get("gid",  vCmd);
				pjson::value * vGName	= tb->mCommand->get("groupName",  vCmd);
				if( vSID && vSID->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorParameter, "%s. \"sid\" type.", codeToMessgae(ErrorParameter));
				}
				if( vSName && vSName->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorParameter, "%s. \"serviceName\" type.", codeToMessgae(ErrorParameter));
				}
				if( vGID && vGID->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorParameter, "%s. \"rid\" type.", codeToMessgae(ErrorParameter));
				}
				if( vGName && vGName->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorParameter, "%s. \"resourceName\" type.", codeToMessgae(ErrorParameter));
				}
				tb->mSRef.mId	= (char*)(vSID ? vSID->vString : NULL);
				tb->mSRef.mName	= (char*)(vSName ? vSName->vString : NULL);
				tb->mGRef.mId	= (char*)(vGID   ? vGID->vString : NULL);
				tb->mGRef.mName	= (char*)(vGName ? vGName->vString : NULL);
			}
			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryGroup(&tb->mSRef, &tb->mGRef);
			return task->call(qn, step+1);
		}
		break;
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Group info load error. %s", res->getMessagePtr());
			}
			tb->mGroup 		= (ResultGroup*)res->mResult;
			res->mResult	= NULL;

			if( tb->mGroup->mGroup.adminId != NULL && *tb->mGroup->mGroup.adminId != 0 ) {
				QueryKeyType	aref;
				aref.mId	= tb->mGroup->mGroup.adminId;
				aref.mName	= NULL;
				QnDBQuery *qn = new QnDBQuery(task);
				qn->queryAccount(&aref);
				return task->call(qn, step+1);
			}
		}
	case 2:
		{
			const char *adminUid = NULL;
			StdBuffer _adminUid;
			if( step == 2 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					taskLog(q, LOG_ERR, "Administrator account load failed. %s", res->getMessagePtr());
					_adminUid.addfmt("*** unknwn account (aid=%s) ***", tb->mGroup->mGroup.adminId);
					adminUid	= _adminUid.getPtr();
				} else {
					adminUid	= ((ResultAccount*)res->mResult)->mAccount.uid;
				}
			}
			GroupInfo *group = &tb->mGroup->mGroup;

			char _createTime[256];
			const char *createTime = NULL;
			if( group->createTime != 0 ) {
				struct tm *tmp = gmtime(& group->createTime);
				strftime(_createTime, sizeof(_createTime), "%Y/%m/%d-%H:%M:%S", tmp);
				createTime = _createTime;
			}
			char _updateTime[256];
			const char *updateTime = NULL;
			if( group->updateTime != 0 ) {
				struct tm *tmp = gmtime(&group->updateTime);
				strftime(_updateTime, sizeof(_updateTime), "%Y/%m/%d-%H:%M:%S", tmp);
				updateTime = _updateTime;
			}
			const char *status = "unknown";
			switch(group->status) {
			case GroupNormal:	status = "normal";	break;
			case GroupStop:		status = "stop";	break;
			}

			StdBuffer accessMode;
			stringAccessMode(group->accessMode, accessMode);

			pjson::builder jb;
			if( !jb.init(512) ||
				!jb.beginObject() ||
				!jb.addObjectProp("result", 6) ||
				!jb.valueInt(0) ||
				!jb.addObjectProp("sequence", 8) ||
				!jb.valueInt(task->getSequence()) ||
				!jb.addObjectProp("gid", 3) ||
				!jb.valueString(group->id) ||
				!jb.addObjectProp("groupName", 9) ||
				!jb.valueString(group->name) ||
				!jb.addObjectProp("sid", 3) ||
				!jb.valueString(group->serviceId) ||
				!jb.addObjectProp("description", 11) ||
				!jb.valueString(group->description ? group->description : "") ||
				!jb.addObjectProp("access", 6) ||
				!jb.valueString(accessMode.getPtr()) ||
				!jb.addObjectProp("adminId", 7) ||
				!jb.valueString(adminUid ? adminUid : "") ||
				!jb.addObjectProp("status", 6) ||
				!jb.valueString(status) ||
				!jb.addObjectProp("createTime", 10) ||
				!jb.valueString(createTime) ||
				!jb.addObjectProp("updateTime", 10) ||
				!jb.valueString(updateTime) ||
				!jb.endObject() ) {
				taskLog(q, LOG_ERR, "%s. Json builder failed. code=%d service=%s,%s group=%s,%s.",
						codeToMessgae(ErrorRuntime), jb.getError(),
						tb->mSRef.mId, tb->mSRef.mName,
						tb->mGRef.mId, tb->mGRef.mName);
				return false;
			}
			return response(q,jb);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}

class TBJoinGroup : public TaskBuffer {
public:
	TBJoinGroup() {
		mJsCommand	= NULL;
		mJsUsers	= NULL;
		mSRef.mId	= NULL;
		mSRef.mName	= NULL;
		mGRef.mId	= NULL;
		mGRef.mName	= NULL;
		mARef.mId	= NULL;
		mARef.mName	= NULL;
		mUserSize	= 0;
		mUserIndex	= 0;
		mProc		= 0;
		mReq		= 0;
	}
	virtual ~TBJoinGroup() {
		if( mJsCommand )		delete mJsCommand;
	}
	pjson::json *	mJsCommand;
	pjson::value *	mJsUsers;
	QueryKeyType	mSRef;
	QueryKeyType	mGRef;
	QueryKeyType	mARef;
	size_t	mUserSize;
	size_t	mUserIndex;
	size_t	mReq;
	size_t	mProc;
};

bool CommandProc::cmdJoinGroup(QueueNode *q)
{
	Task *task = q->getTask();
	int step = q->getStep();

	TBJoinGroup *tb = (TBJoinGroup *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBJoinGroup();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = tb->mJsCommand->get("joinGroup");
			if( !vCmd ) {
				return taskResponse(q, ErrorCommand, "%s. \"joinGroup\" is not object.", codeToMessgae(ErrorCommand));
			}

			pjson::value *vSID		= tb->mJsCommand->get("sid", vCmd);
			pjson::value *vSName	= tb->mJsCommand->get("serviceName", vCmd);
			pjson::value *vGID		= tb->mJsCommand->get("gid", vCmd);
			pjson::value *vGName	= tb->mJsCommand->get("groupName", vCmd);
			tb->mJsUsers			= tb->mJsCommand->get("users", vCmd);

			if( !vGID && !vGName ) {
				return taskResponse(q, ErrorParameter, "%s. group is not specify.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsUsers ) {
				if( tb->mJsUsers->vt != pjson::vt_array ) {
					return taskResponse(q, ErrorParameter, "%s. \"users\" is not array.", codeToMessgae(ErrorParameter));
				}
				tb->mUserSize	= pjson::getArraySize(tb->mJsUsers);
			}

			tb->mSRef.mId	= vSID   ? vSID->vString   : NULL;
			tb->mSRef.mName	= vSName ? vSName->vString : NULL;
			tb->mGRef.mId	= vGID   ? vGID->vString   : NULL;
			tb->mGRef.mName	= vGName ? vGName->vString : NULL;
		}
	case 1:
		{
			if( step == 1 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code == ErrorOk ) {
					tb->mProc++;
				} else {
					taskLog(q, LOG_WARNING, "Group info update error. %s", res->getMessagePtr());
				}
				tb->mUserIndex++;
			}

			step	= 1;

			if( (tb->mJsUsers == NULL && tb->mUserIndex == 0) || (tb->mJsUsers && tb->mUserIndex < tb->mUserSize) ) {
				if( tb->mJsUsers == NULL ) {
					const char *session_id = task->getAccountId();
					tb->mARef.mId	= session_id;
					tb->mARef.mName	= NULL;
				} else {
					pjson::value *vArr	= pjson::getArrayContent(tb->mJsUsers, tb->mUserIndex);
					pjson::value *vAid	= tb->mJsCommand->get("aid", vArr);
					if( vAid ) {
						tb->mARef.mId	= vAid->vString;
						tb->mARef.mName	= NULL;
					} else {
						pjson::value *vUid = tb->mJsCommand->get("userId", vArr);
						if( vUid == NULL ) {
							return taskResponse(q, ErrorParameter, "%s. user account is not specifed. service=%s,%s group=%s,%s.",
									codeToMessgae(ErrorParameter),
									tb->mSRef.mId, tb->mSRef.mName,
									tb->mGRef.mId, tb->mGRef.mName);
						}
						tb->mARef.mId	= NULL;
						tb->mARef.mName	= vUid->vString;
					}
				}

				tb->mReq++;
				QnDBRegister *qn = new QnDBRegister(task);
				qn->registGroupAccount(&tb->mSRef, &tb->mGRef, &tb->mARef);
				return task->call(qn, step);
			}

			if( tb->mProc != 0 ) {
				QnDBRegister *qn = new QnDBRegister(task);
				qn->registSyncQueue(SyncCMDUpdate, &tb->mSRef, &tb->mGRef, 1);
				return task->call(qn, step+1);
			}
		}
	case 2:
		{
			if( step == 2 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					taskLog(q, LOG_ERR, "Sync queue add error. %s", res->getMessagePtr());

				} else {
					pjson::builder jb;
					if( tb->mSRef.mId || tb->mSRef.mName ) {
						if( !jb.init(256) ||
							!jb.beginObject() ||
							!jb.addObjectProp("syncQueue",9) ||
							!jb.beginObject() ) {
							taskLog(q, LOG_ERR, "%s. json builder failed(1). code=%d service=%s,%s group=%s,%s.",
									codeToMessgae(ErrorRuntime),
									jb.getError(),
									tb->mSRef.mId, tb->mSRef.mName,
									tb->mGRef.mId, tb->mGRef.mName);
							return false;
						}
						if( tb->mSRef.mId ) {
							if( !jb.addObjectProp("sid", 3) ||
								!jb.valueString(tb->mSRef.mId) ) {
								taskLog(q, LOG_ERR, "%s. json builder failed(2). code=%d service=%s,%s group=%s,%s.",
										codeToMessgae(ErrorRuntime),
										jb.getError(),
										tb->mSRef.mId, tb->mSRef.mName,
										tb->mGRef.mId, tb->mGRef.mName);
								return false;
							}
						}
						if( tb->mSRef.mName ) {
							if( !jb.addObjectProp("serviceName", 11) ||
								!jb.valueString(tb->mSRef.mName) ) {
								taskLog(q, LOG_ERR, "%s. json builder failed(3). code=%d service=%s,%s group=%s,%s.",
										codeToMessgae(ErrorRuntime),
										jb.getError(),
										tb->mSRef.mId, tb->mSRef.mName,
										tb->mGRef.mId, tb->mGRef.mName);
								return false;
							}
						}
						if( !jb.endObject() ||
							!jb.endObject() ) {
							taskLog(q, LOG_ERR, "%s. json builder failed(4). code=%d service=%s,%s group=%s,%s.",
									codeToMessgae(ErrorRuntime),
									jb.getError(),
									tb->mSRef.mId, tb->mSRef.mName,
									tb->mGRef.mId, tb->mGRef.mName);
							return false;
						}
					} else {
						if( !jb.init(256) ||
							!jb.valueString("syncQueue") ) {
							taskLog(q, LOG_ERR, "%s. json builder failed(5). code=%d service=%s,%s group=%s,%s.",
									codeToMessgae(ErrorRuntime),
									jb.getError(),
									tb->mSRef.mId, tb->mSRef.mName,
									tb->mGRef.mId, tb->mGRef.mName);
							return false;
						}
					}
					Task *st = mSentinel->getTaskManager().createTaskFromBaseTask(task);
					QueueNode *qn = new QueueNode(st, this, CMDSyncQueue);
					qn->setJSONMessage(jb.detouch());
					st->startTask(qn);
				}
			}

			step = 2;

			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("result", 6) ||
				!jb.valueInt(0) ||
				!jb.addObjectProp("sequence", 8) ||
				!jb.valueInt(task->getSequence()) ||
				!jb.addObjectProp("requested", 9) ||
				!jb.valueInt(tb->mReq) ||
				!jb.addObjectProp("processed", 9) ||
				!jb.valueInt(tb->mProc) ||
				!jb.endObject() ) {
				taskLog(q, LOG_ERR, "%s. json builder failed(6). code=%d service=%s,%s group=%s,%s.",
						codeToMessgae(ErrorRuntime),
						jb.getError(),
						tb->mSRef.mId, tb->mSRef.mName,
						tb->mGRef.mId, tb->mGRef.mName);
				return false;
			}
			return response(q,jb);
		}
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}



class TBLeaveGroup : public TaskBuffer {
public:
	TBLeaveGroup() {
		mJsCommand	= NULL;
		mJsUsers	= NULL;
		mSRef.mId	= NULL;
		mSRef.mName	= NULL;
		mGRef.mId	= NULL;
		mGRef.mName	= NULL;
		mARef.mId	= NULL;
		mARef.mName	= NULL;

		mUserSize	= 0;
		mUserIndex	= 0;
		mProc		= 0;
		mReq		= 0;

		mProcArray	= NULL;
	}
	virtual ~TBLeaveGroup() {
		if( mJsCommand )		delete mJsCommand;
		if( mProcArray )	free(mProcArray);
	}
	pjson::json *	mJsCommand;
	pjson::value *	mJsUsers;
	QueryKeyType	mSRef;
	QueryKeyType	mGRef;
	QueryKeyType	mARef;
	bool *	mProcArray;
	size_t	mUserSize;
	size_t	mUserIndex;
	size_t	mReq;
	size_t	mProc;
};

bool CommandProc::cmdLeaveGroup(QueueNode *q)
{
	Task *task = q->getTask();
	int step = q->getStep();

	TBLeaveGroup *tb = (TBLeaveGroup *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBLeaveGroup();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = tb->mJsCommand->get("leaveGroup");
			if( !vCmd ) {
				return taskResponse(q, ErrorCommand, "%s. \"leaveGroup\" is not object.", codeToMessgae(ErrorParameter));
			}

			pjson::value *vSID		= tb->mJsCommand->get("sid", vCmd);
			pjson::value *vSName	= tb->mJsCommand->get("serviceName", vCmd);
			pjson::value *vGID		= tb->mJsCommand->get("gid", vCmd);
			pjson::value *vGName	= tb->mJsCommand->get("groupName", vCmd);
			tb->mJsUsers			= tb->mJsCommand->get("users", vCmd);

			if( vSID && vSID->vt != pjson::vt_string ) {
				taskResponse(q, ErrorParameter, "%s. \"sid\" type.", codeToMessgae(ErrorParameter));
			}
			if( vSName && vSName->vt != pjson::vt_string ) {
				taskResponse(q, ErrorParameter, "%s. \"serviceName\" type.", codeToMessgae(ErrorParameter));
			}
			if( vGID && vGID->vt != pjson::vt_string ) {
				taskResponse(q, ErrorParameter, "%s. \"gid\" type.", codeToMessgae(ErrorParameter));
			}
			if( vGName && vGName->vt != pjson::vt_string ) {
				taskResponse(q, ErrorParameter, "%s. \"groupName\" type.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsUsers && tb->mJsUsers->vt != pjson::vt_array ) {
				taskResponse(q, ErrorParameter, "%s. \"users\" type.", codeToMessgae(ErrorParameter));
			}
			if( !vGID && !vGName ) {
				return taskResponse(q, ErrorParameter, "%s. group is not specified.", codeToMessgae(ErrorParameter));
			}

			tb->mSRef.mId	= vSID   ? vSID->vString   : NULL;
			tb->mSRef.mName	= vSName ? vSName->vString : NULL;
			tb->mGRef.mId	= vGID   ? vGID->vString   : NULL;
			tb->mGRef.mName	= vGName ? vGName->vString : NULL;
			if( tb->mJsUsers ) {
				tb->mUserSize	= pjson::getArraySize(tb->mJsUsers);
			}
			if( tb->mUserSize == 0 ) {
				tb->mProcArray = (bool*)malloc(sizeof(bool)*1);
				memset(tb->mProcArray, 0, sizeof(bool)*1);
			} else {
				tb->mProcArray = (bool*)malloc(sizeof(bool)*tb->mUserSize);
				memset(tb->mProcArray, 0, sizeof(bool)*tb->mUserSize);
			}
		}
	case 1:
		{
			if( step == 1 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code == ErrorOk ) {
					tb->mProcArray[tb->mUserIndex]	= true;
					tb->mProc++;
				} else {
					tb->mProcArray[tb->mUserIndex]	= false;
					taskLog(q, LOG_ERR, "group update faild. %s",
							res->getMessagePtr());
				}
				tb->mUserIndex++;
			}

			step	= 1;

			if( (tb->mJsUsers == NULL && tb->mUserIndex == 0) || (tb->mJsUsers && tb->mUserIndex < tb->mUserSize) ) {
				if( tb->mJsUsers == NULL ) {
					const char *aid = task->getAccountId();
					tb->mARef.mId	= aid;
					tb->mARef.mName	= NULL;
				} else {
					pjson::value *vArr = pjson::getArrayContent(tb->mJsUsers, tb->mUserIndex);
					pjson::value *aid = tb->mJsCommand->get("aid", vArr);
					if( aid ) {
						tb->mARef.mId	= aid->vString;
						tb->mARef.mName	= NULL;
					} else {
						pjson::value *uid = tb->mJsCommand->get("userId", vArr);
						if( uid == NULL ) {
							return taskResponse(q, ErrorParameter, "%s. user account is not specifed. service=%s,%s group=%s,%s.",
									codeToMessgae(ErrorParameter),
									tb->mSRef.mId, tb->mSRef.mName,
									tb->mGRef.mId, tb->mGRef.mName);
						}
						tb->mARef.mId	= NULL;
						tb->mARef.mName	= uid->vString;
					}
				}
				tb->mReq++;

				QnDBQuery *qn = new QnDBQuery(task);
				qn->removeGroupAccount(&tb->mSRef, &tb->mGRef, &tb->mARef);
				return task->call(qn, step);
			}
			tb->mUserIndex = 0;
		}
	case 2:
		{
			if( step == 2 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					taskLog(q, LOG_ERR, "Sync queue add failed. %s",
						res->getMessagePtr());
				}
				tb->mUserIndex++;
			}

			step	= 2;

			while( (tb->mJsUsers == NULL && tb->mUserIndex == 0) || (tb->mJsUsers && tb->mUserIndex < tb->mUserSize) ) {
				if( tb->mProcArray[tb->mUserIndex] ) {
					if( tb->mJsUsers == NULL ) {
						const char *aid = task->getAccountId();
						tb->mARef.mId	= aid;
						tb->mARef.mName	= NULL;
					} else {
						pjson::value *vArr = pjson::getArrayContent(tb->mJsUsers, tb->mUserIndex);
						pjson::value *aid = tb->mJsCommand->get("aid", vArr);
						pjson::value *uid = tb->mJsCommand->get("userId", vArr);
						tb->mARef.mId	= aid ? aid->vString : NULL;
						tb->mARef.mName	= uid ? uid->vString : NULL;
					}

					QnDBRegister *qn = new QnDBRegister(task);
					qn->registSyncQueue(SyncCMDUpdate, &tb->mSRef, &tb->mGRef, &tb->mARef);
					return task->call(qn, step);
				}
				tb->mUserIndex++;
			}

			if( tb->mProc != 0 ) {
				pjson::builder jb;
				if( tb->mSRef.mId || tb->mSRef.mName ) {
					if( !jb.init(256) ||
						!jb.beginObject() ||
						!jb.addObjectProp("syncQueue",9) ||
						!jb.beginObject() ) {
						taskLog(q, LOG_ERR, "%s. json builder failed(1). code=%d service=%s,%s group=%s,%s.",
								codeToMessgae(ErrorRuntime),
								jb.getError(),
								tb->mSRef.mId, tb->mSRef.mName,
								tb->mGRef.mId, tb->mGRef.mName);
						return false;
					}
					if( tb->mSRef.mId ) {
						if( !jb.addObjectProp("sid", 3) ||
							!jb.valueString(tb->mSRef.mId) ) {
							taskLog(q, LOG_ERR, "%s. json builder failed(2). code=%d service=%s,%s group=%s,%s.",
									codeToMessgae(ErrorRuntime),
									jb.getError(),
									tb->mSRef.mId, tb->mSRef.mName,
									tb->mGRef.mId, tb->mGRef.mName);
							return false;
						}
					}
					if( tb->mSRef.mName ) {
						if( !jb.addObjectProp("serviceName", 11) ||
							!jb.valueString(tb->mSRef.mName) ) {
							taskLog(q, LOG_ERR, "%s. json builder failed(3). code=%d service=%s,%s group=%s,%s.",
									codeToMessgae(ErrorRuntime),
									jb.getError(),
									tb->mSRef.mId, tb->mSRef.mName,
									tb->mGRef.mId, tb->mGRef.mName);
							return false;
						}
					}
					if( !jb.endObject() ||
						!jb.endObject() ) {
						taskLog(q, LOG_ERR, "%s. json builder failed(4). code=%d service=%s,%s group=%s,%s.",
								codeToMessgae(ErrorRuntime),
								jb.getError(),
								tb->mSRef.mId, tb->mSRef.mName,
								tb->mGRef.mId, tb->mGRef.mName);
						return false;
					}
				} else {
					if( !jb.init(256) ||
						!jb.valueString("syncQueue") ) {
						taskLog(q, LOG_ERR, "%s. json builder failed(5). code=%d service=%s,%s group=%s,%s.",
								codeToMessgae(ErrorRuntime),
								jb.getError(),
								tb->mSRef.mId, tb->mSRef.mName,
								tb->mGRef.mId, tb->mGRef.mName);
						return false;
					}
				}
				Task *st = mSentinel->getTaskManager().createTaskFromBaseTask(task);
				QueueNode *qn = new QueueNode(st, this, CMDSyncQueue);
				qn->setJSONMessage(jb.detouch());
				st->startTask(qn);
			}

			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("result", 6) ||
				!jb.valueInt(0) ||
				!jb.addObjectProp("sequence", 8) ||
				!jb.valueInt(task->getSequence()) ||
				!jb.addObjectProp("requested", 9) ||
				!jb.valueInt(tb->mReq) ||
				!jb.addObjectProp("processed", 9) ||
				!jb.valueInt(tb->mProc) ||
				!jb.endObject() ) {
				taskLog(q, LOG_ERR, "%s. json builder failed(6). code=%d service=%s,%s group=%s,%s.",
						codeToMessgae(ErrorRuntime),
						jb.getError(),
						tb->mSRef.mId, tb->mSRef.mName,
						tb->mGRef.mId, tb->mGRef.mName);
				return false;
			}
			return response(q,jb);
		}
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}

class TBLsGroupUser : public TaskBuffer {
public:
	TBLsGroupUser() {
		mJsCommand	= NULL;
		mGroup		= NULL;
		mUserList	= NULL;
		mUserCount	= 0;
		mUserIndex	= 0;
		mUserOut	= 0;
	}
	virtual ~TBLsGroupUser() {
		if( mJsCommand )	delete mJsCommand;
		if( mGroup )		free(mGroup);
		if( mUserList )		free(mUserList);
	}
	pjson::json *	mJsCommand;
	QueryKeyType	mSRef;
	QueryKeyType	mGRef;
	QueryRangeType	mRange;
	ResultGroup *	mGroup;
	ResultList *	mUserList;
	size_t			mUserCount;
	size_t			mUserIndex;
	size_t			mUserOut;
	const char *	mUserId;
	pjson::builder	mJb;
};

bool CommandProc::cmdLsGroupUser(QueueNode *q)
{
	Task *task = q->getTask();
	int step = q->getStep();

	TBLsGroupUser *tb = (TBLsGroupUser *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBLsGroupUser();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = tb->mJsCommand->get("lsGroupUser");
			if( !vCmd ) {
				return taskResponse(q, ErrorCommand, "%s. \"lsGroupUser\" is not object.", codeToMessgae(ErrorParameter));
			}
			pjson::value *vSID		= tb->mJsCommand->get("sid", 	vCmd);
			pjson::value *vSName	= tb->mJsCommand->get("serviceName", vCmd);
			pjson::value *vGID		= tb->mJsCommand->get("gid", vCmd);
			pjson::value *vGName	= tb->mJsCommand->get("groupName", vCmd);
			pjson::value *vFrom		= tb->mJsCommand->get("from", vCmd);
			pjson::value *vCount	= tb->mJsCommand->get("count", vCmd);

			if( vSID && vSID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"sid\" type.", codeToMessgae(ErrorParameter));
			}
			if( vSName && vSName->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"serviceName\" type.", codeToMessgae(ErrorParameter));
			}
			if( vGID && vGID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"gid\" type.", codeToMessgae(ErrorParameter));
			}
			if( vGName && vGName->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"groupName\" type.", codeToMessgae(ErrorParameter));
			}
			if( vFrom && vFrom->vt != pjson::vt_int ) {
				return taskResponse(q, ErrorParameter, "%s. \"from\" type.", codeToMessgae(ErrorParameter));
			}
			if( vCount && vCount->vt != pjson::vt_int ) {
				return taskResponse(q, ErrorParameter, "%s. \"count\" type.", codeToMessgae(ErrorParameter));
			}
			tb->mSRef.mId	= vSID   ? vSID->vString   : NULL;
			tb->mSRef.mName	= vSName ? vSName->vString : NULL;
			tb->mGRef.mId	= vGID   ? vGID->vString   : NULL;
			tb->mGRef.mName	= vGName ? vGName->vString : NULL;
			tb->mRange.mStart	= vFrom  ? vFrom->vInt : 0;
			tb->mRange.mCount	= vCount ? vCount->vInt : -1;

			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryGroup(&tb->mSRef, &tb->mGRef);
			return task->call(qn, step+1);
		}
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Group info load failed. %s", res->getMessagePtr());
			}
			tb->mGroup		= (ResultGroup*)res->mResult;
			res->mResult	= NULL;
			tb->mGRef.mId	= tb->mGroup->mGroup.id;

			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryAccountList(&tb->mSRef, &tb->mGRef, &tb->mRange);
			return task->call(qn, step+1);
		}
	case 2:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "User list load failed. %s", res->getMessagePtr());
			}
			tb->mUserList	= (ResultList*)res->mResult;
			res->mResult	= NULL;
			tb->mUserCount	= tb->mUserList->mList.values;
		}
	case 3:
		{
			if( step == 3 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					taskLog(q, LOG_ERR, "Account info load failed. %s", res->getMessagePtr());
				} else {
					if( tb->mUserIndex == 0 ) {
						if( !tb->mJb.init(512) ||
							!tb->mJb.beginObject() ||
							!tb->mJb.addObjectProp("result", 6) ||
							!tb->mJb.valueInt(0) ||
							!tb->mJb.addObjectProp("sequence", 8) ||
							!tb->mJb.valueInt(task->getSequence()) ||
							!tb->mJb.addObjectProp("from", 4) ||
							!tb->mJb.valueInt(tb->mRange.mStart) ||
							!tb->mJb.addObjectProp("gid", 3) ||
							!tb->mJb.valueString(tb->mGroup->mGroup.id) ||
							!tb->mJb.addObjectProp("groupName", 9) ||
							!tb->mJb.valueString(tb->mGroup->mGroup.name) ||
							!tb->mJb.addObjectProp("users", 5) ||
							!tb->mJb.beginArray() ) {
							taskLog(q, LOG_ERR, "%s. json builder failed(1). code=%d",
									codeToMessgae(ErrorRuntime), tb->mJb.getError());
							return false;
						}
					}
					ResultAccount *acc = (ResultAccount*)res->mResult;
					if( !tb->mJb.addArrayContent() ||
						!tb->mJb.beginObject() ||
						!tb->mJb.addObjectProp("aid", 3) ||
						!tb->mJb.valueString(acc->mAccount.id) ||
						!tb->mJb.addObjectProp("uid", 3) ||
						!tb->mJb.valueString(acc->mAccount.uid) ||
						!tb->mJb.endObject() ) {
						taskLog(q, LOG_ERR, "%s. json builder failed(2). code=%d",
								codeToMessgae(ErrorRuntime), tb->mJb.getError());
						return false;
					}
					tb->mUserOut++;
				}
				tb->mUserIndex++;
			}

			step	= 3;

			if( tb->mUserIndex < tb->mUserCount ) {
				tb->mUserId	= tb->mUserList->mList.value[tb->mUserIndex];
				QueryKeyType aref;
				aref.mId	= tb->mUserId;
				aref.mName	= NULL;
				QnDBQuery *qn = new QnDBQuery(task);
				qn->queryAccount(&aref);
				return task->call(qn, step);
			}

			if( tb->mUserOut == 0 ) {
				if( !tb->mJb.init(256) ||
					!tb->mJb.beginObject() ||
					!tb->mJb.addObjectProp("result", 6) ||
					!tb->mJb.valueInt(0) ||
					!tb->mJb.addObjectProp("sequence", 8) ||
					!tb->mJb.valueInt(task->getSequence()) ||
					!tb->mJb.addObjectProp("from", 4) ||
					!tb->mJb.valueInt(tb->mRange.mStart) ||
					!tb->mJb.addObjectProp("gid", 3) ||
					!tb->mJb.valueString(tb->mGroup->mGroup.id) ||
					!tb->mJb.addObjectProp("groupName", 9) ||
					!tb->mJb.valueString(tb->mGroup->mGroup.name) ||
					!tb->mJb.endObject() ) {
					taskLog(q, LOG_ERR, "%s. json builder failed(1). code=%d",
							codeToMessgae(ErrorRuntime), tb->mJb.getError());
					return false;
				}
			} else {
				if( !tb->mJb.endArray() ||
					!tb->mJb.endObject() ) {
					taskLog(q, LOG_ERR, "%s. json builder failed(1). code=%d",
							codeToMessgae(ErrorRuntime), tb->mJb.getError());
					return false;
				}
			}
			return response(q,tb->mJb);
		}
	}
	taskLog(q, LOG_ERR, "unknown step. step=%d.", step);
	return false;
}



}


