/*
 * pjson.cpp
 *
 *  Created on: 2012/07/05
 *      Author: tanaka
 */
#include "pjson.h"
#include "stdBuffer.h"
#include <assert.h>

namespace pjson {

json::json()
{
	pBuff	= NULL;
	pFree	= NULL;
	pTop	= NULL;
}

json::~json()
{
	buffer *p = pBuff;
	while( p ) {
		buffer *n = p->next;
		free(p);
		p = n;
	}
}

value *	json::getValue() const
{
	return pTop;
}

value * json::get(const char *path, value *from)
{
	value *v = from;
	if( !v ) v = getValue();
	return getFrom(v, path);
}

value *json::getFrom(pjson::value *from, const char *path)
{
	if( !from || from->vt != vt_object ) return NULL;
	const char *p = path;
	value *v = from;
	if( path ) {
		while(*p) {
			if(*p == '[' ) {
				if( v->vt != vt_array ) return NULL;
				size_t n=0;
				p++;
				while('0' <= *p && *p <= '9') {
					n = n * 10 + *p - '0';
					p++;
				}
				if( *p != ']' ) return NULL;
				array *a = v->vArray;
				while( a ) {
					if( a->n <= n ) {
						n -= a->n;
						a = a->next;
					} else {
						v = a->values[n];
						break;
					}
				}
				if( !a ) {
					return NULL;
				}
				p++;
			} else {
				if( v->vt != vt_object ) return NULL;
				const char *name = p;
				size_t len = 0;
				while( *p && *p != '.' && *p != '[' ) {
					p++; len++;
				}
				if( len == 0 ) {
					return NULL;
				}
				object *o = v->vObject;
				while( o ) {
					size_t i;
					for(i = 0; i < o->n; i++ ) {
						if( strncmp(o->props[i]->name,name,len) == 0 &&
							o->props[i]->name[len] == 0 ) break;
					}
					if( i >= o->n ) {
						o = o->next;
					} else {
						v = &o->props[i]->val;
						break;
					}
				}
				if( !o ) {
					return NULL;
				}
			}
			if( *p ) {
				if( *p == '.' ) {
					p++;
				} else if( *p != '[' ) {
					return NULL;
				}
			}
		}
	}
	return v;
}

bool json::dump_na(SST::Buffer *b, bool wf, const char *tab, pjson::value *v)
{
	return b->addfmt("!!na!!");
}

bool json::dump_null(SST::Buffer *b, bool wf, const char *tab, pjson::value *v)
{
	return b->addfmt("null");
}

bool json::dump_bool(SST::Buffer *b, bool wf, const char *tab, pjson::value *v)
{
	return b->addfmt("%s", v->vBool?"true":"false");
}
bool json::dump_str(SST::Buffer *b, bool wf, const char *tab, pjson::value *v)
{
	return b->addfmt("\"%s\"", v->vString);
}
bool json::dump_int(SST::Buffer *b, bool wf, const char *tab, pjson::value *v)
{
	return b->addfmt("%lld", v->vInt);
}
bool json::dump_num(SST::Buffer *b, bool wf, const char *tab, pjson::value *v)
{
	return b->addfmt("%g", *v->vNumber);
}
bool json::dump_obj(SST::Buffer *b, bool wf, const char *tab, pjson::value *v)
{
	if(!b->addfmt("%s{", tab) ) return false;
	char tabs[256];
	const char *delm = "\r\n";
	if( wf ) {
		strcpy(tabs, tab);
		strcat(tabs, "\t");
		delm = "\r\n";
	} else {
		strcpy(tabs, "");
		delm = "";
	}

	pjson::object *op = v->vObject;
	while(op) {
		size_t i;
		for(i = 0; i < op->n; i++) {
			if( !b->addfmt("%s%s\"%s\":", delm, tabs, op->props[i]->name) ) return false;
			if( wf ) {
				delm = ",\r\n";
			} else {
				delm = ",";
			}
			if( !dump_val(b, wf, tabs, &op->props[i]->val) ) return false;
		}
		op = op->next;
	}
	if( wf )
		if( !b->addfmt("\r\n%s", tab) ) return false;
	if( !b->addfmt("}") ) return false;
	return true;
}

bool json::dump_arr(SST::Buffer *b, bool wf, const char *tab, pjson::value *v)
{
	if( !b->add("[",1) ) return false;
	pjson::array *op = v->vArray;

	char tabs[256];
	const char *delm;
	if( wf ) {
		strcpy(tabs, tab);
		strcat(tabs, "\t");
		delm = "\r\n";
	} else {
		strcpy(tabs, "");
		delm = "";
	}
	while(op) {
		size_t i;
		for(i = 0; i < op->n; i++) {
			if( !b->addfmt("%s", delm) ) return false;
			if( wf ) {
				delm = ",\r\n";
			} else {
				delm = ",";
			}
			if( !dump_val(b, wf, tabs, op->values[i]) ) return false;
		}
		op = op->next;
	}
	if( wf )
		if( !b->addfmt("\r\n%s", tab) ) return false;
	if( !b->addfmt("]") ) return false;
	return true;
}

bool json::dump_val(SST::Buffer *b, bool wf, const char *tab, pjson::value *v)
{
	if( !v ) {
		dump_null(b, wf, tab, v);
		return true;
	}
	switch(v->vt) {
	case vt_na:		return dump_na(b, wf, tab, v);
	case vt_null:	return dump_null(b, wf, tab, v);
	case vt_bool:	return dump_bool(b, wf, tab, v);
	case vt_string:	return dump_str(b, wf, tab, v);
	case vt_int:	return dump_int(b, wf, tab, v);
	case vt_number:	return dump_num(b, wf, tab, v);
	case vt_object:	return dump_obj(b, wf, tab, v);
	case vt_array:	return dump_arr(b, wf, tab, v);
	}
	return false;
}

bool json::getText(SST::Buffer *b, bool wf) const
{
	b->clear();
	return dump_val(b, wf, "", getValue());
}

size_t getArraySize(value *arr)
{
	size_t n=0;
	pjson::array *op = arr->vArray;
	while(op) {
		n += op->n;
		op = op->next;
	}
	return n;
}

value *getArrayContent(value *arr, size_t index)
{
	size_t n=0;
	pjson::array *op = arr->vArray;
	while(op) {
		if( index < n + op->n ) {
			return op->values[index-n];
		}
		n += op->n;
		op = op->next;
	}
	return NULL;
}

void json::dumpjson(const char *tab, pjson::json *js)
{
	if( js == NULL ) {
		fprintf(stderr, "%svt:json is null\n", tab);
		return;
	}
	SST::StdBuffer b;
	dump_val(&b, true, tab, js->getValue());
	fprintf(stderr, "%s\n", b.getPtr());
}

buffer * json::allocBufferSeg(size_t size)
{
	size_t need = ((size/PJSON_MIN_BLOCKSIZE)+1)*PJSON_MIN_BLOCKSIZE;
	size_t safe = 0;
	if( pFree )
		safe = (size_t)((pFree->nAlloc*1.5/PJSON_MIN_BLOCKSIZE)+1)*PJSON_MIN_BLOCKSIZE;
	if( safe > PJSON_MAX_BLOCKSIZE && size < PJSON_MAX_BLOCKSIZE )
		safe = PJSON_MAX_BLOCKSIZE;
	if( need < safe ) need = safe;

	buffer *p	= (buffer*)malloc(sizeof(buffer)+need);
	if( !p ) return false;
	p->nAlloc	= need;
	p->nUse		= 0;
	p->next		= NULL;
	return p;
}

bool json::allocBuffer(size_t size)
{
	buffer *p = allocBufferSeg(size);
	if( !p ) return false;
	pBuff	= p;
	pFree	= p;
	return true;
}

void *json::getBuffer(size_t size)
{
	size_t nFree = 0;
	if( pFree )
		nFree = pFree->nAlloc - pFree->nUse;

	if( nFree <= size ) {
		buffer *pNext = allocBufferSeg(size);
		if( !pNext ) return false;
		pNext->next = pBuff;
		pBuff	= pNext;
		pFree	= pNext;
	}
	void *p = &pFree->data[pFree->nUse];
	pFree->nUse += size;
	return p;
}

/////////////////////////////////////////////
builder::builder()
{
	errMsg = NULL;
	err	= 0;
	bld	= NULL;
	js	= NULL;
	pc	= NULL;
	st	= NULL;
}

builder::~builder()
{
	clear();
}

void builder::setError(int code)
{
	err	= code;
	js	= NULL;
}

int builder::getError()
{
	return err;
}

void builder::setErrorMsg(const char *p)
{
	if( errMsg ) free(errMsg);
	errMsg = strdup(p);
}
const char *builder::getErrorMsg()
{
	return errMsg;
}


bool builder::init(size_t size)
{
	if( bld ) clear();
	bld = new json();
	if( !bld ) {
		setError(ERR_MEMORY);
		return false;
	}
	if( !bld->allocBuffer(size) ) {
		setError(ERR_MEMORY);
		return false;
	}
	bld->pTop = (value*)bld->getBuffer(sizeof(value));
	if( bld->pTop == NULL ) {
		setError(ERR_MEMORY);
		return false;
	}
	pc	= bld->pTop;
	memset(pc, 0, sizeof(value));
	js	= bld;
	return true;
}

void builder::clear()
{
	if( bld ) {
		delete bld;
		bld	= NULL;
		js	= NULL;
	}
	while( st ) {
		CHUNK *n = st->nStack;
		free(st);
		st = n;
	}
	if( errMsg ) free(errMsg);
	errMsg = NULL;
	err	= 0;
	pc	= NULL;
}

void * builder::getBuffer(size_t size)
{
	return bld->getBuffer(size);
}

json * builder::detouch()
{
	json *r = js;
	if( js ) {
		js	= NULL;
		bld	= NULL;
		while( st ) {
			CHUNK *n = st->nStack;
			free(st);
			st = n;
		}
		pc	= NULL;
	}
	return r;
}

bool builder::valueNull()
{
	if( !pc || pc->vt != vt_na ) {
		setError(ERR_HIERARCHY);
		return false;
	}
	pc->vt	= vt_null;
	return true;
}

bool builder::valueBool(bool v)
{
	if( !pc || pc->vt != vt_na ) {
		setError(ERR_HIERARCHY);
		return false;
	}
	pc->vt		= vt_bool;
	pc->vBool	= v;
	return true;
}

bool builder::valueInt(long long v)
{
	if( !pc || pc->vt != vt_na ) {
		setError(ERR_HIERARCHY);
		return false;
	}
	pc->vt		= vt_int;
	pc->vInt	= v;
	return true;
}

bool builder::valueNumber(double v)
{
	if( !pc || pc->vt != vt_na ) {
		setError(ERR_HIERARCHY);
		return false;
	}
	double *p = (double*)bld->getBuffer(sizeof(double));
	if( !p ) {
		setError(ERR_MEMORY);
		return false;
	}
	*p = v;
	pc->vt		= vt_number;
	pc->vNumber	= p;
	return true;
}

bool builder::valueString(const char *v)
{
	if( !pc || pc->vt != vt_na ) {
		setError(ERR_HIERARCHY);
		return false;
	}
	if( !v ) v = "";
	size_t escLen = escapeSize(v);
	size_t rawLen = strlen(v);
	char *p = (char*)bld->getBuffer(sizeof(char)*(escLen));
	if( !p ) {
		setError(ERR_MEMORY);
		return false;
	}
	if( escLen == rawLen+1 )
		strcpy(p, v);
	else
		escapeText(p, escLen, v);

	pc->vt		= vt_string;
	pc->vString	= p;
	return true;
}

bool builder::valueBufString(const char *v)
{
	if( !pc || pc->vt != vt_na ) {
		setError(ERR_HIERARCHY);
		return false;
	}
	if( !v ) v = "";
	size_t escLen = escapeSize(v);
	size_t rawLen = strlen(v);
	if( escLen != rawLen+1 ) {
		char *p = (char*)bld->getBuffer(sizeof(char)*(escLen));
		if( !p ) {
			setError(ERR_MEMORY);
			return false;
		}
		escapeText(p, escLen, v);
		v = p;
	}
	pc->vt		= vt_string;
	pc->vString	= v;
	return true;
}

size_t builder::escapeSize(const char *val)
{
	return escapeText(NULL, 0, val);
}

size_t builder::escapeText(char *dest, size_t dsize, const char *val)
{
	size_t len=0;
	while(*val) {
		if( *val == '"' || *val == '\b' || *val == '\r' || *val == '\t' || *val == '\n' || *val == '\f' || *val == '\\' ) {
			len++;
			if( dest && dsize >= len ) {
				*dest++ = '\\';
			}
			len++;
			if( dest && dsize >= len ) {
				switch(*val) {
				case '"':	*dest++	= '"';	break;
				case '\b':	*dest++	= 'b';	break;
				case '\r':	*dest++	= 'r';	break;
				case '\t':	*dest++	= 't';	break;
				case '\n':	*dest++	= 'n';	break;
				case '\f':	*dest++	= 'f';	break;
				default:	*dest++ = '\\';	break;
				}
			}
		} else {
			len++;
			if( dest && dsize >= len ) {
				*dest++ = *val;
			}
		}
		val++;
	}
	len++;
	if( dest && dsize >= len ) {
		*dest = 0;
	}
	return len;
}

bool builder::beginObject()
{
	if( !pc || pc->vt != vt_na ) {
		setError(ERR_HIERARCHY);
		return false;
	}
	CHUNK *c = (CHUNK*)malloc(sizeof(CHUNK));
	if( !c ) return false;
	pc->vt		= vt_object;
	pc->vObject	= NULL;
	c->nStack	= st;
	c->n		= 0;
	c->pc		= NULL;
	c->pObject	= NULL;
	c->pc		= pc;
	st	= c;
	return true;
}

bool builder::addObjectProp(const char *name, size_t len)
{
	if( !st || !pc || st->pc->vt != vt_object ) {
		setError(ERR_HIERARCHY);
		return false;
	}
	if( st->n == PJSON_CHUNK ) {
		object *pNewChunk	= (object*)getBuffer(sizeof(object)+sizeof(prop*)*(st->n-1));
		if( !pNewChunk ) {
			setError(ERR_MEMORY);
			return false;
		}
		if( !st->pc->vObject ) st->pc->vObject = pNewChunk;
		if( st->pObject) st->pObject->next = pNewChunk;
		st->pObject	= pNewChunk;

		pNewChunk->n	= st->n;
		pNewChunk->next	= NULL;
		for( size_t i = 0; i < st->n; i++ ) {
			pNewChunk->props[i]	= st->pchunk[i];
		}
		st->n	= 0;
	}
	if( len == 0 ) len = strlen(name);
	prop *pProp = (prop*)getBuffer(sizeof(prop));
	pProp->name	= (char*)getBuffer(len+1);
	strncpy((char*)pProp->name, name, len);
	((char*)pProp->name)[len] = 0;
	pProp->val.vt	= vt_na;
	st->pchunk[st->n++]	= pProp;
	pc 		= &pProp->val;
	return true;
}

bool builder::endObject()
{
	if( !st ) {
		setError(ERR_HIERARCHY);
		return false;
	}
	if( st->n ) {
		object *pNewChunk	= (object*)getBuffer(sizeof(object)+sizeof(prop*)*(st->n-1));
		if( !pNewChunk ) {
			setError(ERR_MEMORY);
			return false;
		}
		if( !st->pc->vObject ) st->pc->vObject = pNewChunk;
		if( st->pObject) st->pObject->next = pNewChunk;
		st->pObject	= pNewChunk;
		pNewChunk->n	= st->n;
		pNewChunk->next	= NULL;
		for( size_t i = 0; i < st->n; i++ ) {
			pNewChunk->props[i]	= st->pchunk[i];
		}
	}
	if( st->pc->vObject == NULL ) {
		setError(ERR_EMPTYOBJECT);
		return false;
	}
	CHUNK *c = st;
	pc	= c->pc;
	st	= c->nStack;
	free(c);
	return true;
}

bool builder::beginArray()
{
	if( !pc || pc->vt != vt_na ) {
		setError(ERR_HIERARCHY);
		return false;
	}
	CHUNK *c = (CHUNK*)malloc(sizeof(CHUNK));
	if( !c ) {
		setError(ERR_MEMORY);
		return false;
	}
	pc->vt		= vt_array;
	pc->vArray	= NULL;
	c->nStack	= st;
	c->n		= 0;
	c->pc		= pc;
	c->pArray	= NULL;
	st	= c;
	return true;
}

bool builder::addArrayContent()
{
	if( !st || !pc || st->pc->vt != vt_array ) {
		setError(ERR_HIERARCHY);
		return false;
	}
	if( st->n == PJSON_CHUNK ) {
		array *pNewChunk	= (array*)getBuffer(sizeof(array)+sizeof(value*)*(st->n-1));
		if( !pNewChunk ) {
			setError(ERR_MEMORY);
			return false;
		}
		if( !st->pc->vArray ) st->pc->vArray = pNewChunk;
		if( st->pArray ) st->pArray->next = pNewChunk;
		st->pArray	= pNewChunk;

		pNewChunk->n	= st->n;
		pNewChunk->next	= NULL;
		for( size_t i = 0; i < st->n; i++ ) {
			pNewChunk->values[i]	= st->vchunk[i];
		}
		st->n	= 0;
	}
	value *pValue = (value*)getBuffer(sizeof(value));
	pValue->vt	= vt_na;
	st->vchunk[st->n++]	= pValue;
	pc 		= pValue;
	return true;
}

bool builder::endArray()
{
	if( !st || !pc || st->pc->vt != vt_array ) {
		setError(ERR_HIERARCHY);
		return false;
	}
	if( st->n ) {
		array *pNewChunk	= (array*)getBuffer(sizeof(array)+sizeof(value*)*(st->n-1));
		if( !pNewChunk ) {
			setError(ERR_MEMORY);
			return false;
		}
		if( !st->pc->vArray ) st->pc->vArray = pNewChunk;
		if( st->pArray ) st->pArray->next = pNewChunk;
		st->pArray	= pNewChunk;

		pNewChunk->n	= st->n;
		pNewChunk->next	= NULL;
		for( size_t i = 0; i < st->n; i++ ) {
			pNewChunk->values[i]	= st->vchunk[i];
		}
	}
	/*
	if( st->pc->vArray == NULL ) {
		setError(ERR_EMPTYARRAY);
		return false;
	}
	*/
	CHUNK *c = st;
	pc	= c->pc;
	st	= c->nStack;
	free(c);
	return true;
}

/////////////////////////////////////////////
parser::parser()
{

}
parser::~parser()
{

}

json *parser::parse(const char *text, size_t textSize)
{
	src	= text;
	ptr	= src;
	bd.init((size_t)(textSize*1.5));
#ifdef _DEBUG1
	fprintf(stderr, "parse %s\n", ptr);
#endif
	if( !parseValue() || skipwsp() != 0 ) {
		return NULL;
	}
	return bd.detouch();
}

int parser::getErrorPos()
{
	return ptr - src;
}
int parser::getErrorCode()
{
	return bd.getError();
}
/*
const char *parser::getErrorMessage()
{
	return bd.getErrorMsg();
}
*/
bool parser::parseValue()
{
	char c = skipwsp();
	switch(c) {
	case '{':
		if( !parseObject()) return false;
		break;
	case '[':
		if( !parseArray()) return false;
		break;
	case '"':
		if( !parseString()) return false;
		break;
	case 't':
	case 'f':
		if( !parseBool()) return false;
		break;
	case 'n':
		if( !parseNull()) return false;
		break;
	default:
		if( '-' == c || '+' == c || ('0' <= c && c <= '9') ) {
			if( !parseNumber()) return false;
			break;
		}
		bd.setError(ERR_SYNTAX);
		return false;
	}
	return true;
}

bool parser::parseNull()
{
	if( !match("null") ) {
		bd.setError(ERR_NULL);
		return false;
	}
	return bd.valueNull();
}

bool parser::parseBool()
{
	if( match("true") )
		return bd.valueBool(true);
	else if( match("false") )
		return bd.valueBool(false);
	bd.setError(ERR_BOOL);
	return false;
}

bool parser::parseNumber()
{
//	char msg[8192];
//	char msg2[8192];

//	sprintf(msg2, "\nparseNumber\nptr=<%s>\n", ptr);
//	strcpy(msg, msg2);

	const char *sp = ptr;
	char c = getc();

//	sprintf(msg2, "getc=%c sp=<%s> ptr=<%s>\n", c, sp, ptr);
//	strcat(msg, msg2);

	if( c == '+' || c == '-' ) c = getc();
	while( '0' <= c && c <= '9' ) {
		c = getc();
	}
	if( c == '.' ) {
		c = getc();
		while( '0' <= c && c <= '9' ) {
			c = getc();
		}
	}
	if( c == 'e' || c == 'E' ) {
		c = getc();
		if( c == '+' || c == '-' ) c = getc();
		while( '0' <= c && c <= '9' ) {
			c = getc();
		}
	}

//	sprintf(msg2, "exit loop sp=<%s> ptr=<%s>\n", sp, ptr);
//	strcat(msg, msg2);

	ungetc();

//	sprintf(msg2, "ungetc sp=<%s> ptr=<%s>\n", sp, ptr);
//	strcat(msg, msg2);

	char *endptr;
	double d = strtod(sp, &endptr);
	if( endptr != ptr ) {
//		sprintf( msg2, "sp=%d <%s> ptr=%d <%s> end=%d <%s>\n", sp-src, sp, ptr-src, ptr, endptr-src, endptr);
//		strcat(msg, msg2);
//		bd.setErrorMsg(msg);
		bd.setError(ERR_NUMBER);
		return false;
	}
	long long id = (long long)d;
	double idd = id;
	if( d == idd ) {
		return bd.valueInt(id);
	} else {
		return bd.valueNumber(d);
	}
}

bool parser::parseString()
{
	int i;
	int hex;
	size_t	nch	= 0;
	char c = getc();
	if( c != '"' ) {
		bd.setError(ERR_STRING);
		return false;
	}
	const char *sp = ptr;
	char *op = NULL;
	char *str_val = NULL;
	c = getc();
	while( c != '"' ) {
		if( c == 0 ) {
			bd.setError(ERR_SYNTAX);
			return false;
		}
		if( c == '\\' ) {
			if( nch ) {
				op = (char*)bd.getBuffer(nch);
				if( !op ) {
					bd.setError(ERR_MEMORY);
					return false;
				}
				if( !str_val) str_val = op;
				memcpy(op, sp, nch);
			}
			c = getc();
			switch(c) {
			case '"':	c = '"'; 	break;
			case '\\':	c = '\\'; 	break;
			case '/':	c = '/'; 	break;
			case 'b':	c = '\b'; 	break;
			case 'r':	c = '\r'; 	break;
			case 't':	c = '\t'; 	break;
			case 'n':	c = '\n'; 	break;
			case 'f':	c = '\f'; 	break;
			case 'u':
				hex = 0;
				c = getc();
				for(i=0; i<4 && c != 0; i++) {
					if( '0' <= c && c <= '9' )
						hex = hex * 16 + c - '0';
					else if( 'a' <= c && c <= 'f' )
						hex = hex * 16 + 10 + c - 'a';
					else if( 'A' <= c && c <= 'F' )
						hex = hex * 16 + 10 + c - 'A';
					else {
						bd.setError(ERR_STRING);
						return false;
					}
				}
				if(i<4) {
					bd.setError(ERR_STRING);
					return false;
				}
				op = (char*)bd.getBuffer(2);
				if( !str_val ) str_val = op;
				*op++ = (hex >> 16) & 0xff;
				*op   = hex & 0xff;
				c = 0;
				break;
			default:
				bd.setError(ERR_STRING);
				return false;
			}
			if( c ) {
				op = (char*)bd.getBuffer(1);
				if( !op ) {
					bd.setError(ERR_MEMORY);
					return false;
				}
				if( !str_val ) str_val = op;
				*op	= c;
			}
			nch	= 0;
			sp	= ptr;
			continue;
		}
		c	= getc();
		nch++;
	}

	op = (char*)bd.getBuffer(nch+1);
	if( !op ) {
		bd.setError(ERR_MEMORY);
		return false;
	}
	if( !str_val ) str_val = op;
	memcpy(op, sp, nch);
	op[nch]	= 0;

	if( !bd.valueBufString(str_val) ) return false;
	return true;
}

bool parser::parseObject()
{
	if( getc() != '{' ) {
		bd.setError(ERR_OBJECT);
		return false;
	}
	if( !bd.beginObject() ) return false;
	char c;
	do {
		skipwsp();
		if( getc() != '"' ) {
			bd.setError(ERR_OBJECT);
			return false;
		}
		const char *name = ptr;
		c = getc();
		while( c != '"' ) {
			if( c == 0 ) {
				bd.setError(ERR_OBJECT);
				return false;
			}
			c = getc();
		}
		if( c != '"') {
			bd.setError(ERR_OBJECT);
			return false;
		}
		if( !bd.addObjectProp(name, ptr-name-1)) {
			return false;
		}
		skipwsp();
		if( getc() != ':' ) {
			bd.setError(ERR_OBJECT);
			return false;
		}
		if( !parseValue() ) {
			return false;
		}
		skipwsp();
		c = getc();
	} while( c == ',' );
	skipwsp();
	if( c != '}' ) {
		bd.setError(ERR_OBJECT);
		return false;
	}
	if( !bd.endObject() ) return false;
	return true;
}

bool parser::parseArray()
{
	if( getc() != '[' ) {
		bd.setError(ERR_ARRAY);
		return false;
	}
	if( !bd.beginArray() ) return false;
	char c;
	do {
		if( !bd.addArrayContent() ) return false;
		if( !parseValue() ) return false;
		skipwsp();
		c = getc();
	} while( c == ',' );
	if( c != ']' ) {
		bd.setError(ERR_ARRAY);
		return false;
	}
	if( !bd.endArray() ) return false;
	return true;
}

char parser::getc()
{
	if( *ptr ) return *ptr++;
	return 0;
}

void parser::ungetc()
{
	ptr--;
	while( src < ptr && (*(ptr-1) == ' ' || *(ptr-1) == '\r' || *(ptr-1) == '\n' || *(ptr-1) == '\t')) ptr--;
}

char parser::skipwsp()
{
	while( *ptr == ' ' || *ptr == '\r' || *ptr == '\n' || *ptr == '\t' ) ptr++;
	return *ptr;
}

bool parser::match(const char *pat)
{
	const char *p = ptr;
	while( *pat ) {
		if( *pat != *p ) break;
		pat++; p++;
	}
	if( *pat ) return false;
	ptr = p;
	return true;
}

}

#if 0
int _tmain(int argc, _TCHAR* argv[])
{
	_CrtSetDbgFlag(_CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF);
	pjson::parser ps;
	pjson::json * j = ps.parse("{\"user\":\"tanaka\", \"pass\":\"hello\"}", 0);
	pjson::json::dumpjson("",j);
	delete j;

	j = ps.parse("{\"ptr1\":null, \"ptr2\":null}", 0);
	pjson::json::dumpjson("",j);
	delete j;

	j = ps.parse("{\"bool1\":true, \"bool2\":false}", 0);
	pjson::json::dumpjson("",j);
	delete j;

	j = ps.parse("{\"str1\":\"test\\ttest\", \"str2\":\"test\\rtest\\ntest\\\\\"}", 0);
	pjson::json::dumpjson("",j);
	delete j;

	j = ps.parse("{\"int1\":100, \"int2\":-100}", 0);
	pjson::json::dumpjson("",j);
	delete j;

	j = ps.parse("{\"number1\":100.123, \"number2\":-100.123}", 0);
	pjson::json::dumpjson("",j);
	delete j;

	j = ps.parse("{\"obj1\":{\"v0\":\"test\"}, \"arr\":[\"a0\",\"a2\",\"a3\"]}", 0);
	pjson::json::dumpjson("",j);
	delete j;

	pjson::builder bx;
	bx.init(100);
	bx.beginObject();
	bx.addObjectProp("test");
	bx.valueString("hello");
	bx.valueString("hello2");
	bx.endObject();
	j = bx.detouch();
	pjson::json::dumpjson("",j);
	delete j;
	bx.clear();

	malloc(100);
	return 0;
}

#endif
