/*
 * cyclicBuffer.cpp
 *
 *  Created on: 2012/12/08
 *      Author: yasuoki
 */

#include "cyclicBuffer.h"

#include <stdio.h>
#include <stdarg.h>
#include <memory.h>
#include <malloc.h>

namespace SST {

bool CyclicBuffer::ptrTest(size_t a, size_t r, size_t w)
{
	return (a==0||nAlloc == a) && nWrite == w && nRead == r;
}

CyclicBuffer::CyclicBuffer()
{
	nAlloc	= 0;
	nWrite	= 0;
	nRead	= 0;
	ptr		= NULL;
}

CyclicBuffer::~CyclicBuffer()
{
	clear();
}

size_t CyclicBuffer::getFreeSpace(size_t &nLeft, size_t &nRight) const
{
	if( nAlloc == 0 ) {
		nLeft	= 0;
		nRight	= 0;
		return 0;
	}
	if( nRead == nWrite ) {
		// empty
		// Zooooooooooooooo nLeft = 0 nRight = 16
		// oooooooooZoooooo nLeft = 0 nRight = 16
		if( nRead == 0 ) {
			nLeft	= 0;
			nRight	= nAlloc - 1;
		} else {
			nLeft	= nRead - 1;
			nRight	= nAlloc - nWrite;
		}
	} else if( nRead < nWrite ) {
		// RxxxxxxxxWoooooo nLeft = 0 nRight = 7
		// ooooRxxxxWoooooo nLeft = 4 nRight = 7
		if( nRead == 0 ) {
			nLeft	= nRead;
			nRight	= nAlloc - nWrite - 1;
		} else {
			nLeft	= nRead - 1;
			nRight	= nAlloc - nWrite;
		}
	} else {
		// xxxxWoooRxxxxxxxx nLeft = 0 nRight = 3
		nLeft	= 0;
		nRight	= nRead - nWrite - 1;
	}
	return nLeft + nRight;
}

size_t CyclicBuffer::getUsingSpace(size_t &nLeft, size_t &nRight) const
{
	if( nRead == nWrite ) {
		// empty
		// Zooooooooooooooo nLeft = 0 nRight = 16
		// oooooooooZoooooo nLeft = 0 nRight = 16
		nLeft	= 0;
		nRight	= 0;
	} else if( nRead < nWrite ){
		nRight	= nRead - nWrite;
		// RxxxxxxxxWoooooo nLeft = 0 nRight = 7
		// ooooRxxxxWoooooo nLeft = 4 nRight = 7
		nLeft	= 0;
		nRight	= nWrite - nRead;
	} else {
		// xxxxWoooRxxxxxxx nLeft = 0 nRight = 4
		nLeft	= nWrite;
		nRight	= nAlloc - nRead;
	}
	return nLeft + nRight;
}

size_t CyclicBuffer::size() const
{
	size_t nRight;
	size_t nLeft;
	size_t nUse = getUsingSpace(nLeft, nRight);
	return nUse;
}

bool CyclicBuffer::add(const char *data, ssize_t dataSize)
{
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::add(%d) A=%d R=%d W=%d\n", dataSize, nAlloc, nRead, nWrite);
#endif
	if( dataSize == -1 ) {
		dataSize  = data ? strlen(data) : 0;
	}
	size_t size = (size_t)dataSize;
	size_t nRight;
	size_t nLeft;
	size_t nFree = getFreeSpace(nLeft, nRight);
	if( nFree < size+1 ) {
		size_t need	= size + 1 - nFree;
		size_t newAlloc = (((nAlloc + need)/SST_IOBUFFER_BLOCKSIZE)+1)*SST_IOBUFFER_BLOCKSIZE;
		char * newPtr = (char*)malloc(newAlloc);
		if( !newPtr ) return false;
		size_t nUse = 0;
		if( nAlloc ) {
			nUse = getUsingSpace(nLeft, nRight);
			memcpy(newPtr, &ptr[nRead], nRight);
			memcpy(&newPtr[nRight], ptr, nLeft);
			free(ptr);
		}
		nAlloc	= newAlloc;
		ptr		= newPtr;
		nRead	= 0;
		nWrite	= nUse;
		nLeft	= 0;
		nRight	= nAlloc - nUse - 1;
	}
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::add(%d) A=%d R=%d W=%d Left=%d Right=%d\n", size, nAlloc, nRead, nWrite, nLeft, nRight);
#endif

	size_t as=0;
	size_t cs;
	if( size && nRight ) {
		if( size > nRight )
			cs = nRight;
		else
			cs = size;
		memcpy(&ptr[nWrite], data, cs);
		size 	-= cs;
		nWrite	+= cs;
		as		+= cs;
		if( nWrite == nAlloc ) nWrite = 0;
	}
	if( size && nLeft ) {
		if( size > nLeft )
			cs = nLeft;
		else
			cs = size;
		memcpy(ptr, &data[as], cs);
		size	-= cs;
		nWrite	+= cs;
		if( nWrite == nAlloc ) nWrite = 0;
	}
	ptr[nWrite] = 0;
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::add(%d) A=%d R=%d W=%d ok\n", size, nAlloc, nRead, nWrite);
#endif
	return true;
}

size_t CyclicBuffer::get(char *buff, size_t buffSize)
{
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::get(%d) A=%d R=%d W=%d\n", buffSize, nAlloc, nRead, nWrite);
#endif
	size_t rs = 0;
	size_t cs;

	size_t nRight;
	size_t nLeft;
	size_t nUse = getUsingSpace(nLeft, nRight);

	if( nUse == 0 ) return 0;
	if( buffSize ) {
		if( nRight < buffSize )
			cs = nRight;
		else
			cs = buffSize;
		if( buff ) memcpy(buff, &ptr[nRead], cs);
		nRead += cs;
		if( nRead >= nAlloc ) nRead = 0;
		rs += cs;
	}
	if( rs < buffSize ) {
		if( nLeft < buffSize-rs )
			cs = nLeft;
		else
			cs = buffSize-rs;
		if( buff ) memcpy(&buff[rs], ptr, cs);
		nRead += cs;
		if( nRead >= nAlloc ) nRead = 0;
		rs += cs;
	}
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::get(%d) A=%d R=%d W=%d ok=%d\n", buffSize, nAlloc, nRead, nWrite, rs);
#endif
	return rs;
}

size_t CyclicBuffer::getLine(char *buff, size_t buffSize)
{
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::getLine(%d) A=%d R=%d W=%d\n", buffSize, nAlloc, nRead, nWrite);
#endif

	size_t bs = buffSize - 1;
	size_t rs = 0;
	size_t cs;

	size_t nRight;
	size_t nLeft;
	size_t nUse = getUsingSpace(nLeft, nRight);

	if( nUse == 0 ) return 0;
	if( bs ) {
		if( nRight < bs )
			cs = nRight;
		else
			cs = bs;
		size_t nc = 0;
		char *p0 = buff;
		char *p1 = &ptr[nRead];
		while(*p1 != '\r' && *p1 != '\n' && nc < cs ) {
			*p0++ = *p1++;
			nc++;
		}
		rs += nc;
		if( nc < cs ) {
			if( *p1 == '\r' || *p1 == '\n' ) {
				if( *p1 == '\r' ) { p1++; nc++; }
				if( *p1 == '\n' ) { p1++; nc++; }
			}
		}
		nRead += nc;
		if( nRead >= nAlloc ) nRead = 0;
	}
	if( rs < bs ) {
		if( nLeft < bs-rs )
			cs = nLeft;
		else
			cs = bs-rs;

		size_t nc = 0;
		char *p0 = &buff[rs];
		char *p1 = ptr;
		while(*p1 != '\r' && *p1 != '\n' && nc < cs ) {
			*p0++ = *p1++;
			nc++;
		}
		rs += nc;
		if( nc < cs ) {
			if( *p1 == '\r' || *p1 == '\n' ) {
				if( *p1 == '\r' ) { p1++; nc++; }
				if( *p1 == '\n' ) { p1++; nc++; }
			}
		}
		nRead += cs;
		if( nRead >= nAlloc ) nRead = 0;
		rs += cs;
	}
	buff[rs] = 0;
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::getLine(%d) A=%d R=%d W=%d ok=%d\n", buffSize, nAlloc, nRead, nWrite, rs);
#endif
	return rs;
}

size_t CyclicBuffer::getAt(char *buff, size_t start, size_t len) const
{
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::getAt(%d, %d) A=%d R=%d W=%d\n", start, len, nAlloc, nRead, nWrite);
#endif
	CyclicBuffer t;
	t.nAlloc	= nAlloc;
	t.nWrite	= nWrite;
	t.nRead		= nRead;
	t.ptr		= ptr;

	size_t ss = t.get(NULL, start);
	if( ss != start ) {
		t.ptr	= NULL;
		return 0;
	}
	size_t rs = t.get(buff, len-1);
	buff[rs] = 0;
	t.ptr	= NULL;
#ifdef _BUFF_DEBUG
	fprintf(stderr, "Buffer::getAt(%d, %d) A=%d R=%d W=%d ok=%d\n", start, len, nAlloc, nRead, nWrite, rs);
#endif
	return rs;
}

const char *CyclicBuffer::getPtr() const
{
	const char *r = NULL;
	size_t nRight;
	size_t nLeft;
	size_t nUse = getUsingSpace(nLeft, nRight);
	if( nUse == 0 ) return NULL;
	if( nLeft == 0 && nRead < nWrite ) {
		ptr[nWrite] = 0;
		r = &ptr[nRead];
	} else {
		char * tmpPtr = (char*)malloc(nUse+1);
		if( !tmpPtr ) return false;
		memcpy(tmpPtr, &ptr[nRead], nRight);
		memcpy(&tmpPtr[nRight], ptr, nLeft);
		memcpy(ptr, tmpPtr, nUse);
		ptr[nUse] = 0;
		free(tmpPtr);
		r = ptr;
	}
	(const_cast<CyclicBuffer*>(this))->nRead	= 0;
	(const_cast<CyclicBuffer*>(this))->nWrite	= 0;
	return r;
}

void CyclicBuffer::clear()
{
	if( ptr ) free(ptr);
	nAlloc	= 0;
	nWrite	= 0;
	nRead	= 0;
	ptr		= NULL;
}

}


