/*
 * cmd_sync.cpp
 *
 *  Created on: 2013/01/11
 *      Author: yasuoki
 */

#include "commands.h"
#include "proc_command.h"
#include "proc_database.h"
#include "proc_requester.h"
#include "proc_listener.h"
#include "task.h"
#include "sentinel.h"
#include "buffer.h"
#include "pjson.h"
#include <syslog.h>
#include <errno.h>
#include <netdb.h>
#include <assert.h>

namespace SST {


QnCommSync::QnCommSync(Task *task) : QueueNode(task)
{
	mType		= QueueCommUpdate;
	mProc		= &task->getSentinel()->getCommand();
	mData		= NULL;
}

QnCommSync::~QnCommSync()
{
	if( mData )		free(mData);
}
bool QnCommSync::setSync(ServiceInfo *svc, QueryKeyType *rref, const char *ref, QnCommSync::SyncMode mode)
{
	if( mData ) {
		free(mData);
		mData	= NULL;
	}
	mFunc	= (int)CMDSync;

	size_t size	= sizeof(SyncData)
				+ DBUtil::sizeofService(svc)
				+ DBUtil::sizeofKey(rref)
				+ (ref ? strlen(ref)+1 : 0);
	char *ptr = (char*)malloc(size);
	if( ptr == NULL )
		return false;
	mData	= (SyncData*)ptr;
	ptr	+= sizeof(SyncData);
	ptr	= DBUtil::writeService(&mData->mService, ptr, svc);
	ptr	= DBUtil::writeKey(&mData->mRRef, ptr, rref);
	ptr	= DBUtil::writeString(&mData->mRef, ptr, ref);
	mData->mMode	= mode;
	return true;
}

class TBUpdate: public TaskBuffer {
public:
	TBUpdate() {
		mJsCommand	= NULL;
		mJsResource	= NULL;
		mJsSID		= NULL;
		mJsSName	= NULL;
		mJsRID		= NULL;
		mJsRName	= NULL;
		mJsRef		= NULL;
		mJsDefMode	= NULL;
		mJsMode		= NULL;
		mService	= NULL;
		mList		= NULL;
		mAllUpdate	= false;
		mDefMode	= QnCommSync::SyncModeAdd;
		mReq		= 0;
		mProc		= 0;
		mUpdate		= 0;
	}
	virtual ~TBUpdate() {
		if( mJsCommand )	delete mJsCommand;
		if( mService ) 		free(mService);
		if( mList )			free(mList);
	}
	pjson::json *	mJsCommand;
	pjson::value *	mJsSID;
	pjson::value *	mJsSName;
	pjson::value *	mJsResource;
	pjson::value *	mJsRID;
	pjson::value *	mJsRName;
	pjson::value *	mJsRef;
	pjson::value *	mJsDefMode;
	pjson::value *	mJsMode;

	ResultService *	mService;
	bool			mAllUpdate;
	QnCommSync::SyncMode mDefMode;
	size_t 			mReq;
	size_t 			mProc;
	size_t 			mUpdate;

	ResultList *	mList;
};

bool CommandProc::cmdUpdate(QueueNode *q)
{
	Task *task = q->getTask();
	int step	= q->getStep();

	TBUpdate *tb = (TBUpdate *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBUpdate();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	switch(step) {
	case 0:
		{
			tb->mJsCommand = q->detouchJSONMessage();
			pjson::value *vUpdate = tb->mJsCommand->get("update");
			if( vUpdate ) {
				if( vUpdate->vt != pjson::vt_object ) {
					return taskResponse(q, ErrorParameter, "parameter error");
				}
				tb->mJsSID		= tb->mJsCommand->get("sid",  vUpdate);
				tb->mJsSName	= tb->mJsCommand->get("serviceName",  vUpdate);
				tb->mJsResource	= tb->mJsCommand->get("resource", vUpdate);
				tb->mJsDefMode	= tb->mJsCommand->get("mode", vUpdate);

				if( tb->mJsSID && tb->mJsSID->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorParameter, "%s. \"sid\" type.", codeToMessgae(ErrorParameter));
				}
				if( tb->mJsSName && tb->mJsSName->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorParameter, "%s. \"serviceName\" type.", codeToMessgae(ErrorParameter));
				}

				if( tb->mJsResource ) {
					if( tb->mJsResource->vt == pjson::vt_array ) {
						tb->mUpdate	= pjson::getArraySize(tb->mJsResource);
					} else if( tb->mJsResource->vt == pjson::vt_object ) {
						tb->mUpdate = 1;
					} else {
						return taskResponse(q, ErrorParameter, "%s. \"resource\" type.", codeToMessgae(ErrorParameter));
					}
				} else {
					tb->mAllUpdate = true;
				}
				if( tb->mJsDefMode ) {
					if( tb->mJsDefMode->vt != pjson::vt_string ) {
						return taskResponse(q, ErrorParameter, "%s. \"mode\" type.", codeToMessgae(ErrorParameter));
					}
					if( strcmp(tb->mJsDefMode->vString, "add") == 0 ) {
						if( tb->mAllUpdate ) {
							return taskResponse(q, ErrorParameter, "%s. \"mode\" value.", codeToMessgae(ErrorParameter));
						}
						tb->mDefMode = QnCommSync::SyncModeAdd;
					} else if( strcmp(tb->mJsDefMode->vString, "update") == 0 ) {
						tb->mDefMode = QnCommSync::SyncModeUpdate;
					} else if( strcmp(tb->mJsDefMode->vString, "remove") == 0 ) {
						tb->mDefMode = QnCommSync::SyncModeRemove;
					} else {
						return taskResponse(q, ErrorParameter, "%s. \"mode\" value.", codeToMessgae(ErrorParameter));
					}
				}

				size_t i;
				for( i = 0; i < tb->mUpdate; i++ ) {
					pjson::value *v;
					if( tb->mJsResource->vt == pjson::vt_object) {
						v	= tb->mJsResource;
					} else if( i < tb->mUpdate ){
						v	= pjson::getArrayContent(tb->mJsResource, i);
					} else {
						break;
					}
					tb->mJsRID		= tb->mJsCommand->get("rid",  v);
					tb->mJsRName	= tb->mJsCommand->get("resourceName",  v);
					tb->mJsRef		= tb->mJsCommand->get("ref",  v);
					tb->mJsMode		= tb->mJsCommand->get("mode",  v);

					if( tb->mJsRID && tb->mJsRID->vt != pjson::vt_string ) {
						return taskResponse(q, ErrorParameter, "%s. \"resource[%d].rid\" type.", codeToMessgae(ErrorParameter), i);
					}
					if( tb->mJsRName && tb->mJsRName->vt != pjson::vt_string ) {
						return taskResponse(q, ErrorParameter, "%s. \"resource[%d].resourceName\" type.", codeToMessgae(ErrorParameter), i);
					}
					if( tb->mJsRef && tb->mJsRef->vt != pjson::vt_string ) {
						return taskResponse(q, ErrorParameter, "%s. \"resource[%d].ref\" type.", codeToMessgae(ErrorParameter), i);
					}
					if( tb->mJsMode && tb->mJsMode->vt != pjson::vt_string ) {
						return taskResponse(q, ErrorParameter, "%s. \"resource[%d].mode\" type.", codeToMessgae(ErrorParameter), i);
					}
					QnCommSync::SyncMode	mode = tb->mDefMode;
					if( tb->mJsMode ) {
						if( strcmp(tb->mJsDefMode->vString, "add") == 0 ) {
							mode	= QnCommSync::SyncModeAdd;
						} else if( strcmp(tb->mJsDefMode->vString, "update") == 0 ) {
							mode	 = QnCommSync::SyncModeUpdate;
						} else if( strcmp(tb->mJsDefMode->vString, "remove") == 0 ) {
							mode	= QnCommSync::SyncModeRemove;
						} else {
							return taskResponse(q, ErrorParameter, "%s. \"resource[%d].mode\" value.", codeToMessgae(ErrorParameter), i);
						}
						if( mode == QnCommSync::SyncModeAdd ) {
							if( tb->mJsRID ) {
								return taskResponse(q, ErrorParameter, "%s. \"resource[%d].rid\" is specified against addition mode.", codeToMessgae(ErrorParameter), i);
							}
							if( tb->mJsRName ) {
								return taskResponse(q, ErrorParameter, "%s. \"resource[%d].resourceName\" is specified against addition mode.",codeToMessgae(ErrorParameter), i);
							}
							if( tb->mJsRef == NULL ) {
								return taskResponse(q, ErrorParameter, "%s. \"resource[%d].ref\" is not specified against addition mode.",codeToMessgae(ErrorParameter), i);
							}
						} else if( mode == QnCommSync::SyncModeUpdate ) {
							if( !tb->mJsRID && !tb->mJsRName ) {
								return taskResponse(q, ErrorParameter, "%s. \"resource[%d]\" resource is not specified against updating mode.",codeToMessgae(ErrorParameter), i);
							}
						} else if( mode == QnCommSync::SyncModeRemove ) {
							if( !tb->mJsRID && !tb->mJsRName ) {
								return taskResponse(q, ErrorParameter, "%s. \"resource[%d]\" resource is not specified against remove mode.",codeToMessgae(ErrorParameter), i);
							}
							if( tb->mJsRef ) {
								return taskResponse(q, ErrorParameter, "%s. \"resource[%d].ref\" is specified against remove mode.",codeToMessgae(ErrorParameter), i);
							}
						}
					}
					if( tb->mJsRID == NULL && tb->mJsRName == NULL && tb->mJsRef == NULL ) {
						return taskResponse(q, ErrorParameter, "%s. \"resource[%d]\" not specified", codeToMessgae(ErrorParameter), i);
					}
					tb->mReq++;
				}
			} else {
				tb->mAllUpdate	= true;
				tb->mDefMode	= QnCommSync::SyncModeUpdate;
			}

			QueryKeyType sref;
			sref.mId	= tb->mJsSID ? tb->mJsSID->vString : NULL;
			sref.mName	= tb->mJsSName ? tb->mJsSName->vString : NULL;
			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryService(&sref);
			return task->call(qn, step+1);
		}
		break;
	case 1:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Service info load failed. %s", res->getMessagePtr());
			}
			tb->mService	= (ResultService*)(res->mResult);
			res->mResult	= NULL;

			// TODO: permission check

			if( tb->mAllUpdate ) {
				QueryKeyType	sref;
				sref.mId	= tb->mJsSID ? tb->mJsSID->vString : NULL;
				sref.mName	= tb->mJsSName ? tb->mJsSName->vString : NULL;
				QnDBQuery *qn = new QnDBQuery(task);
				qn->queryResourceList(&sref, NULL);
				return task->call(qn, step+1);
			}
		}
	case 2:
		{
			if( step == 2 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					return taskResponse(q, code, "Resource list load failed. %s", res->getMessagePtr());
				}
				tb->mList		= (ResultList*)(res->mResult);
				res->mResult	= NULL;
				tb->mUpdate		= tb->mList->mList.values;
			}

			size_t i;
			for( i = 0; i < tb->mUpdate; i++ ) {
				const char * vRID	= NULL;
				const char * vRName	= NULL;
				const char * vRef	= NULL;
				QnCommSync::SyncMode	mode;
				if( tb->mAllUpdate ) {
					vRID	= tb->mList->mList.value[i];
					mode	= tb->mDefMode;
				} else {
					pjson::value *v;
					if( tb->mJsResource->vt == pjson::vt_object) {
						v	= tb->mJsResource;
					} else if( i < tb->mUpdate ){
						v	= pjson::getArrayContent(tb->mJsResource, i);
					} else {
						break;
					}
					tb->mJsRID		= tb->mJsCommand->get("rid",  v);
					tb->mJsRName	= tb->mJsCommand->get("resourceName",  v);
					tb->mJsRef		= tb->mJsCommand->get("ref",  v);
					tb->mJsMode		= tb->mJsCommand->get("mode",  v);
					vRID	= tb->mJsRID ? tb->mJsRID->vString : NULL;
					vRName	= tb->mJsRName ? tb->mJsRName->vString : NULL;
					vRef	= tb->mJsRef ? tb->mJsRef->vString : NULL;
					mode	= tb->mDefMode;
					const char *vMode	= tb->mJsMode ? tb->mJsMode->vString : NULL;
					if( vMode ) {
						if( strcmp(vMode, "add") == 0 )
							mode = QnCommSync::SyncModeAdd;
						else if( strcmp(vMode, "update") == 0 )
							mode = QnCommSync::SyncModeUpdate;
						else if( strcmp(vMode, "remove") == 0 )
							mode = QnCommSync::SyncModeRemove;
					}
				}

				Task *st = mSentinel->getTaskManager().createTaskFromBaseTask(task);
				QnCommSync *qn = new QnCommSync(st);
				QueryKeyType rref;
				rref.mId	= vRID;
				rref.mName	= vRName;
				qn->setSync(&tb->mService->mService, &rref, vRef, mode);
				taskLog(q, LOG_DEBUG, "new task task=%d q=%lld serviceId=%s rid=%s rname=%s URL=%s",
						st->getTaskId(), qn->getId(),
						st->getServiceId(),
						rref.mId ? rref.mId : "(not set)",
						rref.mName ? rref.mName : "(not set)",
						qn->mData->mRef);
				st->startTask(qn);
				tb->mProc++;
			}

			pjson::builder jb;
			if( !jb.init(256) ||
				!jb.beginObject() ||
				!jb.addObjectProp("result", 6) ||
				!jb.valueInt(0) ||
				!jb.addObjectProp("sequence", 8) ||
				!jb.valueInt(task->getSequence()) ||
				!jb.addObjectProp("requested", 9) ||
				!jb.valueInt(tb->mReq) ||
				!jb.addObjectProp("processed", 9) ||
				!jb.valueInt(tb->mProc) ||
				!jb.endObject() ) {
				taskLog(q, LOG_ERR, "%s::cmdUpdate: json builder failed code=%d", getName(), jb.getError());
				return false;
			}
			return response(q,jb);
		}
	}
	taskLog(q, LOG_ERR, "unknown step=%d.", step);
	return false;
}

class TBSyncSub: public TaskBuffer {
public:
	TBSyncSub() {
		mData			= NULL;
		mRefURL			= NULL;
		mRefGetOpt		= false;
		mResourceId		= NULL;
		mResource		= NULL;
		mRid			= NULL;

		mJsInfo			= NULL;
		mJsName			= NULL;
		mJsDate			= NULL;
		mJsTag			= NULL;
		mJsGadget		= NULL;
		mJsRefTime		= 0;

		mGadgetJsSize	= 0;
		mGadgetJsArr	= NULL;
		mGadgetJsSort	= NULL;
		mGadgetDbSize	= 0;
		mGadgetDbSort	= NULL;

		mShareDbSize	= 0;
		mShareDbIndex	= 0;

		mJsShareInfo	= NULL;
		mShareJsSize	= 0;
		mShareJsIndex	= 0;
		mShareJsArr		= NULL;

		mSyncGroupAlloc	= 100;
		mSyncGroupCount	= 0;
		mSyncGroupIndex	= 0;
		mSyncGroupArr	= (QueryKeyType*)malloc(sizeof(QueryKeyType)*100);
		mSyncGroupBuffAlloc	= 6000;
		mSyncGroupBuffUse	= 0;
		mSyncGroupBuffPtr	= (char*)malloc(6000);
	}
	virtual ~TBSyncSub() {
		if( mData )			free(mData);
		if( mResource )		free(mResource);
		if( mRid )			free(mRid);

		if( mJsInfo )		delete mJsInfo;
		if( mGadgetJsArr )	free(mGadgetJsArr);
		if( mGadgetJsSort )	free(mGadgetJsSort);
		if( mGadgetDbSort )	free(mGadgetDbSort);
		if( mJsShareInfo )	delete mJsShareInfo;
		if( mShareJsArr ) {
			size_t i;
			for( i = 0; i < mShareJsSize; i++) {
				if( mShareJsArr[i].groupArray ) {
					free(mShareJsArr[i].groupArray);
				}
			}
			free(mShareJsArr);
		}
		if( mSyncGroupArr )		free(mSyncGroupArr);
		if( mSyncGroupBuffPtr )	free(mSyncGroupBuffPtr);
	}
	SyncData *		mData;
	char *			mRefURL;
	bool			mRefGetOpt;
	char *			mResourceId;

	ResultResource*	mResource;
	ResultId *		mRid;

	pjson::json *	mJsInfo;
	pjson::value *	mJsName;
	pjson::value *	mJsDate;
	pjson::value *	mJsTag;
	pjson::value *	mJsGadget;
	time_t			mJsRefTime;

	GadgetInfo *	mGadgetJsArr;
	size_t			mGadgetJsSize;
	GadgetInfo **	mGadgetJsSort;
	size_t			mGadgetDbSize;
	GadgetInfo **	mGadgetDbSort;

	size_t			mShareDbSize;
	size_t			mShareDbIndex;

	pjson::json *	mJsShareInfo;
	size_t			mShareJsSize;
	size_t			mShareJsIndex;
	ShareInfo *		mShareJsArr;

	size_t			mSyncGroupAlloc;
	size_t			mSyncGroupCount;
	size_t			mSyncGroupIndex;
	QueryKeyType *	mSyncGroupArr;
	size_t			mSyncGroupBuffAlloc;
	size_t			mSyncGroupBuffUse;
	char *			mSyncGroupBuffPtr;
};

static int qsComparGadget(const void *a, const void *b)
{
	GadgetInfo *ap = *(GadgetInfo**)a;
	GadgetInfo *ab = *(GadgetInfo**)b;
	return strcmp(ap->name, ab->name);
}

bool CommandProc::syncParseInfo(bool &endTask, TaskBuffer *tbBase, QueueNode *q, int step)
{
	endTask	= true;
	QnHTTPResult *res	= (QnHTTPResult*)q;
	TBSyncSub *tb = (TBSyncSub*)tbBase;

	pjson::parser ps;
	pjson::json *js = ps.parse(res->mBody.getPtr(), res->mBody.size());
	if( js == NULL ) {
		taskLog(q, LOG_ERR, "json parse error. code=%d pos=%d rid=%s URL=%s",
				codeToMessgae(ErrorData), ps.getErrorCode(), ps.getErrorPos(),
				tb->mResource ? tb->mResource->mResource.id : "(not set)",
				tb->mRefURL);
		return false;
	}
	tb->mJsInfo = js;

	pjson::value * vRes		= tb->mJsInfo->get("resource");
	if( vRes == NULL ) {
		return taskResponse(q, ErrorData, "%s. get_info. \"resource\" not found. rid=%s URL=%s.",
				codeToMessgae(ErrorData),
				tb->mResource ? tb->mResource->mResource.id : "(not set)",
				tb->mRefURL);
	}
	tb->mJsName		= tb->mJsInfo->get("name", vRes);
	tb->mJsDate		= tb->mJsInfo->get("date", vRes);
	tb->mJsTag		= tb->mJsInfo->get("tag", vRes);
	tb->mJsGadget	= tb->mJsInfo->get("gadget", vRes);
	if( !tb->mJsName && tb->mJsName->vt != pjson::vt_string ) {
		return taskResponse(q, ErrorData, "%s. get_info. \"name\" value. rid=%s URL=%s.",
				codeToMessgae(ErrorData),
				tb->mResource ? tb->mResource->mResource.id : "(not set)",
				tb->mRefURL);
	}
	if( !tb->mJsDate && tb->mJsDate->vt != pjson::vt_string ) {
		return taskResponse(q, ErrorData, "%s. get_info. \"date\" value. rid=%s URL=%s.",
				codeToMessgae(ErrorData),
				tb->mResource ? tb->mResource->mResource.id : "(not set)",
				tb->mRefURL);
	}
	if( tb->mJsDate ) {
		if( tb->mJsDate ) {
			struct tm tm_date;
			char *r = strptime(tb->mJsDate->vString, "%Y/%m/%d-%H:%M:%S", &tm_date);
			if( r == NULL || *r != 0 ) {
				return taskResponse(q, ErrorData, "%s. get_info. \"date\" value. rid=%s URL=%s.",
						codeToMessgae(ErrorData),
						tb->mResource ? tb->mResource->mResource.id : "(not set)",
						tb->mRefURL);
			}
			tb->mJsRefTime = timegm(&tm_date);
		}
	}
	if( tb->mJsTag && tb->mJsTag->vt != pjson::vt_string ) {
		return taskResponse(q, ErrorData, "%s. get_info. \"tag\" type. rid=%s URL=%s.",
				codeToMessgae(ErrorData),
				tb->mResource ? tb->mResource->mResource.id : "(not set)",
				tb->mRefURL);
	}
	if( tb->mJsGadget && tb->mJsGadget->vt != pjson::vt_array ) {
		return taskResponse(q, ErrorData, "%s. get_info. \"gadget\" type. rid=%s URL=%s.",
				codeToMessgae(ErrorData),
				tb->mResource ? tb->mResource->mResource.id : "(not set)",
				tb->mRefURL);
	}

	if( tb->mJsGadget ) {
		tb->mGadgetJsSize	= pjson::getArraySize(tb->mJsGadget);
		tb->mGadgetJsArr	= (GadgetInfo*)malloc(sizeof(GadgetInfo)*tb->mGadgetJsSize);
		tb->mGadgetJsSort	= (GadgetInfo**)malloc(sizeof(GadgetInfo*)*tb->mGadgetJsSize);
		if( tb->mGadgetJsArr == NULL || tb->mGadgetJsSort == NULL ) {
			return taskResponse(q, ErrorNoMemory, "%s. get_info. gadget sort buffer. rid=%s URL=%s.",
					codeToMessgae(ErrorNoMemory),
					tb->mResource ? tb->mResource->mResource.id : "(not set)",
					tb->mRefURL);
		}
		size_t i;
		for( i = 0; i < tb->mGadgetJsSize; i++ ) {
			pjson::value * vArr = pjson::getArrayContent(tb->mJsGadget, i);
			if( !vArr || vArr->vt != pjson::vt_object ) {
				return taskResponse(q, ErrorData, "%s. get_info. gadget[%d] type. rid=%s URL=%s.",
						codeToMessgae(ErrorData), i,
						tb->mResource ? tb->mResource->mResource.id : "(not set)",
						tb->mRefURL);
			}
			pjson::value *vName	=  tb->mJsInfo->get("name", vArr);
			pjson::value *vTag	=  tb->mJsInfo->get("tag", vArr);
			pjson::value *vDate	=  tb->mJsInfo->get("date", vArr);
			if( !vName && vName->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorData, "%s. get_info. gadget[%d] \"name\" value or type. rid=%s URL=%s.",
						codeToMessgae(ErrorData), i,
						tb->mResource ? tb->mResource->mResource.id : "(not set)",
						tb->mRefURL);
			}
			if( vTag && vTag->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorData, "%s. get_info. gadget[%d] \"tag\" type. rid=%s URL=%s.",
						codeToMessgae(ErrorData), i,
						tb->mResource ? tb->mResource->mResource.id : "(not set)",
						tb->mRefURL);
			}
			time_t _refTime = 0;
			if( vDate ) {
				if( vDate->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorData, "%s. get_info. gadget[%d] \"date\" type. rid=%s URL=%s.",
							codeToMessgae(ErrorData), i,
							tb->mResource ? tb->mResource->mResource.id : "(not set)",
							tb->mRefURL);
				}
				struct tm tm_date;
				char *r = strptime(vDate->vString, "%Y/%m/%d-%H:%M:%S", &tm_date);
				if( r == NULL || *r != 0 ) {
					return taskResponse(q, ErrorData, "%s. get_info. gadget[%d] \"date\" value. rid=%s URL=%s.",
							codeToMessgae(ErrorData), i,
							tb->mResource ? tb->mResource->mResource.id : "(not set)",
							tb->mRefURL);
				}
				_refTime = timegm(&tm_date);
			}
			tb->mGadgetJsArr[i].id		= NULL;
			tb->mGadgetJsArr[i].name	= (char*)vName->vString;
			tb->mGadgetJsArr[i].tag		= (char*)(vTag ? vTag->vString : NULL);
			tb->mGadgetJsArr[i].refTime	= _refTime;
			tb->mGadgetJsArr[i].revision= 0;
			tb->mGadgetJsArr[i].status	= ResourceOpen;
			tb->mGadgetJsSort[i] = &tb->mGadgetJsArr[i];
		}
		qsort(tb->mGadgetJsSort, tb->mGadgetJsSize, sizeof(GadgetInfo*), qsComparGadget);
	}
	endTask	= false;
	return true;
}

bool CommandProc::syncParseShareInfo(bool &endTask, TaskBuffer *tbBase, QueueNode *q, int step)
{
	endTask	= true;
	QnHTTPResult *res	= (QnHTTPResult*)q;
	ErrorCode code	= res->getResultCode();
	if( code != ErrorOk || res->mStatusCode != 200) {
		return taskResponse(q, code, "%s. sync: getInfo error. status code = %d", res->getMessagePtr(), res->mStatusCode);
	}
	TBSyncSub *tb = (TBSyncSub*)tbBase;

	pjson::parser ps;
	pjson::json *js = ps.parse(res->mBody.getPtr(), res->mBody.size());
	if( js == NULL ) {
		return taskResponseLog(LOG_ERR, q, ErrorData, "%s. get_share_info json parse error. code=%d pos=%d", codeToMessgae(ErrorData), ps.getErrorCode(), ps.getErrorPos());
	}
	tb->mJsShareInfo = js;

	pjson::value *vCmd	= js->get("shareInfo");
	if( !vCmd || vCmd->vt != pjson::vt_array ) {
		return taskResponse(q, ErrorData, "%s. get_info. \"share_info\" not found. rid=%s URL=%s.",
				codeToMessgae(ErrorData),
				tb->mResource ? tb->mResource->mResource.id : "(not set)",
				tb->mRefURL);
	}
	tb->mShareJsSize	= pjson::getArraySize(vCmd);
	if( tb->mShareJsSize > 0 ) {
		size_t arr_size = sizeof(ShareInfo) * tb->mShareJsSize;
		tb->mShareJsArr	= (ShareInfo*)malloc(arr_size);
		if( tb->mShareJsArr	== NULL ) {
			return taskResponse(q, ErrorData, "%s. get_share_info. rid=%s URL=%s.",
					codeToMessgae(ErrorData),
					tb->mResource ? tb->mResource->mResource.id : "(not set)",
					tb->mRefURL);
		}
		memset(tb->mShareJsArr, 0, arr_size);

		size_t i;
		for( i = 0; i < tb->mShareJsSize; i++ ) {
			pjson::value *arr = pjson::getArrayContent(vCmd, i);
			pjson::value *vShareMode 	= js->get("shareMode", arr);
			pjson::value *vGroup		= js->get("groups", arr);
			pjson::value *vPerm			= js->get("permission", arr);

			tb->mShareJsArr[i].resourceId		= NULL;
			tb->mShareJsArr[i].index			= i;
			tb->mShareJsArr[i].serviceId		= tb->mData->mService.id;
			tb->mShareJsArr[i].dataMode			= ShareDataFull;
			tb->mShareJsArr[i].accessMode		= ShareAccessRead;

			if( vShareMode ) {
				ShareDataMode	sdm	= ShareDataFull;
				if( vShareMode->vt != pjson::vt_string ) {
					return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"share_mode\" type. rid=%s URL=%s.",
							codeToMessgae(ErrorData), i,
							tb->mResource ? tb->mResource->mResource.id : "(not set)",
							tb->mRefURL);
				}
				if( strcmp(vShareMode->vString, "index") == 0 ) {
					sdm	= ShareDataIndex;
				} else if( strcmp(vShareMode->vString, "full") == 0 ) {
					sdm	= ShareDataFull;
				} else {
					return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"share_mode\" value. rid=%s URL=%s.",
							codeToMessgae(ErrorData), i,
							tb->mResource ? tb->mResource->mResource.id : "(not set)",
							tb->mRefURL);
				}
				tb->mShareJsArr[i].dataMode	= sdm;
			}

			if( vGroup ) {
				if( vGroup->vt != pjson::vt_array ) {
					return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"groups\" type. rid=%s URL=%s.",
							codeToMessgae(ErrorData), i,
							tb->mResource ? tb->mResource->mResource.id : "(not set)",
							tb->mRefURL);
				}
				size_t nGroup = pjson::getArraySize(vGroup);
				if( nGroup > 0 ) {
					QueryKeyType *pGroup = (QueryKeyType*)malloc(sizeof(QueryKeyType)*nGroup);
					if( pGroup == NULL ) {
						return taskResponse(q, ErrorNoMemory, "group buffer alloc error. rid=%s URL=%s.",
									codeToMessgae(ErrorNoMemory),
									(tb->mResource ? tb->mResource->mResource.id : "(not set)"),
									tb->mRefURL);
					}
					tb->mShareJsArr[i].groupCount	= nGroup;
					tb->mShareJsArr[i].groupArray	= pGroup;
					size_t g = 0;
					for( g = 0; g < nGroup; g++) {
						pjson::value *arr = pjson::getArrayContent(vGroup, g);
						if( arr->vt != pjson::vt_object ) {
							return taskResponse(q, ErrorData, "%s. get_info. groups[%d] type. rid=%s URL=%s.",
									codeToMessgae(ErrorData), g,
									(tb->mResource ? tb->mResource->mResource.id : "(not set)"),
									tb->mRefURL);
						}
						pjson::value *vGID		= js->get("gid", arr);
						pjson::value *vGName	= js->get("groupName", arr);
						if( !vGID && !vGName ) {
							return taskResponse(q, ErrorData, "%s. get_info. groups[%d] group is not specified. rid=%s URL=%s.",
									codeToMessgae(ErrorData), g,
									tb->mResource ? tb->mResource->mResource.id : "(not set)",
									tb->mRefURL);
						}
						if( vGID && vGID->vt != pjson::vt_string ) {
							return taskResponse(q, ErrorData, "%s. get_info. groups[%d] \"gid\" type. rid=%s URL=%s.",
									codeToMessgae(ErrorData), g,
									tb->mResource ? tb->mResource->mResource.id : "(not set)",
									tb->mRefURL);
						}
						if( vGName && vGName->vt != pjson::vt_string ) {
							return taskResponse(q, ErrorData, "%s. get_info. groups[%d] \"groupName\" type. rid=%s URL=%s.",
									codeToMessgae(ErrorData), g,
									tb->mResource ? tb->mResource->mResource.id : "(not set)",
									tb->mRefURL);
						}
						pGroup[g].mId	= vGID ? vGID->vString : NULL;
						pGroup[g].mName	= vGName ? vGName->vString : NULL;
					}
				}
			}

			if( vPerm ) {
				if( vPerm->vt != pjson::vt_object ) {
					return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"permission\" type. rid=%s URL=%s.",
							codeToMessgae(ErrorData), i,
							tb->mResource ? tb->mResource->mResource.id : "(not set)",
							tb->mRefURL);
				}

				pjson::value *vAccess	= js->get("access", vPerm);
				pjson::value *vExpire	= js->get("expire", vPerm);
				pjson::value *vGeoLoc	= js->get("geoLocation", vPerm);
				pjson::value *vGpsGrc	= js->get("gpsGrace", vPerm);
				pjson::value *vOffGrc	= js->get("offlineGrace", vPerm);
				pjson::value *vTime		= js->get("time", vPerm);

				if( vAccess ) {
					if( vAccess->vt != pjson::vt_string) {
						return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"permission.access\" type. rid=%s URL=%s.",
								codeToMessgae(ErrorData), i,
								tb->mResource ? tb->mResource->mResource.id : "(not set)",
								tb->mRefURL);
					}
					const char *p = vAccess->vString;
					bool readset=false;
					ShareAccessMode	sam = ShareAccessRead;
					while(*p) {
						bool valid=false;
						while(*p==' '||*p=='\t') p++;
						if( *p == 0 ) break;
						if( strncmp(p, "read", 4) == 0 ) {
							sam	= (ShareAccessMode)((int)sam | (int)ShareAccessRead);
							readset = true;
							valid	= true;
							p += 4;
						} else if( strncmp(p, "write",5) == 0 ) {
							sam	= (ShareAccessMode)((int)sam | (int)ShareAccessWrite);
							valid	= true;
							p += 5;
						} else if( strncmp(p, "gadget",6) == 0 ) {
							sam	= (ShareAccessMode)((int)sam | (int)ShareAccessGadget);
							valid	= true;
							p += 6;
						} else if( strncmp(p, "tag",3) == 0 ) {
							sam	= (ShareAccessMode)((int)sam | (int)ShareAccessTag);
							valid	= true;
							p += 3;
						}
						if( !valid || (*p != 0 && *p != ' ' && *p != '\t' && *p != ',') ) {
							return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"permission.access\" value. rid=%s URL=%s.",
									codeToMessgae(ErrorData), i,
									tb->mResource ? tb->mResource->mResource.id : "(not set)",
									tb->mRefURL);
						}
						while(*p==' '||*p=='\t') p++;
						if( *p == 0 ) break;
						if( *p == ',' ) p++;
					}
					if( readset && sam != ShareAccessRead ) {
						return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"permission.access\" value. rid=%s URL=%s.",
								codeToMessgae(ErrorData), i,
								tb->mResource ? tb->mResource->mResource.id : "(not set)",
								tb->mRefURL);
					}
					tb->mShareJsArr[i].accessMode	= sam;
				}

				if( vExpire ) {
					if(vExpire->vt != pjson::vt_string) {
						return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"permission.expire\" type. rid=%s URL=%s.",
								codeToMessgae(ErrorData), i,
								tb->mResource ? tb->mResource->mResource.id : "(not set)",
								tb->mRefURL);
					}
					struct tm tm_date;
					char *r = strptime(vExpire->vString, "%Y/%m/%d-%H:%M:%S", &tm_date);
					if( r == NULL || *r != 0 ) {
						return taskResponse(q, ErrorData, "%s. get_info. gadget[%d] \"permission.expire\" value. rid=%s URL=%s.",
								codeToMessgae(ErrorData), i,
								tb->mResource ? tb->mResource->mResource.id : "(not set)",
								tb->mRefURL);
					}
					tb->mShareJsArr[i].expireTime = timegm(&tm_date);
				}

				if( vGeoLoc ) {
					if( vGeoLoc->vt != pjson::vt_array) {
						return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"permission.geo-location\" type. rid=%s URL=%s.",
								codeToMessgae(ErrorData), i,
								tb->mResource ? tb->mResource->mResource.id : "(not set)",
								tb->mRefURL);
					}

					size_t nGeoLoc = pjson::getArraySize(vGeoLoc);
					if( nGeoLoc > 2 ) {
						return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"permission.geo-location\" count. rid=%s URL=%s.",
								codeToMessgae(ErrorData), i,
								tb->mResource ? tb->mResource->mResource.id : "(not set)",
								tb->mRefURL);
					}
					size_t l;
					for( l = 0; l < nGeoLoc; l++) {
						pjson::value *arr = pjson::getArrayContent(vGeoLoc, l);
						if( arr->vt != pjson::vt_object ) {
							return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"permission.geo-location[%d]\" type. rid=%s URL=%s.",
									codeToMessgae(ErrorData), i, l,
									tb->mResource ? tb->mResource->mResource.id : "(not set)",
									tb->mRefURL);
						}
						pjson::value *vDis = js->get("dis", arr);
						pjson::value *vLat = js->get("lat", arr);
						pjson::value *vLon = js->get("lon", arr);
						if( !vDis || (vDis->vt != pjson::vt_int && vDis->vt != pjson::vt_number) ) {
							return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"permission.geo-location[%d].dis\" type. rid=%s URL=%s.",
									codeToMessgae(ErrorData), i, l,
									tb->mResource ? tb->mResource->mResource.id : "(not set)",
									tb->mRefURL);
						}
						if( !vLat || (vLat->vt != pjson::vt_int && vLat->vt != pjson::vt_number) ) {
							return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"permission.geo-location[%d].lat\" type. rid=%s URL=%s.",
									codeToMessgae(ErrorData), i, l,
									tb->mResource ? tb->mResource->mResource.id : "(not set)",
									tb->mRefURL);
						}
						if( !vLon || (vLon->vt != pjson::vt_int && vLon->vt != pjson::vt_number) ) {
							return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"permission.geo-location[%d].lon\" type. rid=%s URL=%s.",
									codeToMessgae(ErrorData), i, l,
									tb->mResource ? tb->mResource->mResource.id : "(not set)",
									tb->mRefURL);
						}
						tb->mShareJsArr[i].geoLocation[l].dis	= vDis->vt == pjson::vt_int ? (double)(vDis->vInt) : *vDis->vNumber;
						tb->mShareJsArr[i].geoLocation[l].lat	= vLat->vt == pjson::vt_int ? (double)(vLat->vInt) : *vLat->vNumber;
						tb->mShareJsArr[i].geoLocation[l].lon	= vLon->vt == pjson::vt_int ? (double)(vLon->vInt) : *vLon->vNumber;
					}
					tb->mShareJsArr[i].geoLocationCount	= nGeoLoc;
				}

				if( vGpsGrc && vGpsGrc->vt != pjson::vt_int) {
					return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"permission.gps-grace\" type. rid=%s URL=%s.",
							codeToMessgae(ErrorData), i,
							tb->mResource ? tb->mResource->mResource.id : "(not set)",
							tb->mRefURL);
				}
				tb->mShareJsArr[i].gpsGrace	= vGpsGrc ? vGpsGrc->vInt : (5*60);

				if( vOffGrc && vOffGrc->vt != pjson::vt_int) {
					return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"permission.offline-grace\" type. rid=%s URL=%s.",
							codeToMessgae(ErrorData), i,
							tb->mResource ? tb->mResource->mResource.id : "(not set)",
							tb->mRefURL);
				}
				tb->mShareJsArr[i].offlineGrace		= vOffGrc ? vOffGrc->vInt : (5*60);

				if( vTime ) {
					if( vTime->vt != pjson::vt_object) {
						return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"permission.time\" type. rid=%s URL=%s.",
								codeToMessgae(ErrorData), i,
								tb->mResource ? tb->mResource->mResource.id : "(not set)",
								tb->mRefURL);
					}
					pjson::value *vTimeFrom	= js->get("from", vTime);
					pjson::value *vTimeTo	= js->get("to", vTime);
					time_t _fromTime = 0;
					time_t _endTime = 0;
					if( vTimeFrom ) {
						if( vTimeFrom->vt != pjson::vt_string) {
							return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"permission.time.from\" type. rid=%s URL=%s.",
									codeToMessgae(ErrorData), i,
									tb->mResource ? tb->mResource->mResource.id : "(not set)",
									tb->mRefURL);
						}
						struct tm tm_date;
						char *r = strptime(vTimeFrom->vString, "%Y/%m/%d-%H:%M:%S", &tm_date);
						if( r == NULL || *r != 0 ) {
							return taskResponse(q, ErrorData, "%s. get_info. gadget[%d] \"permission.time.from\" value. rid=%s URL=%s.",
									codeToMessgae(ErrorData), i,
									tb->mResource ? tb->mResource->mResource.id : "(not set)",
									tb->mRefURL);
						}
						_fromTime = timegm(&tm_date);
					}
					if( vTimeTo ) {
						if( vTimeTo->vt != pjson::vt_string) {
							return taskResponse(q, ErrorData, "%s. get_info. share_info[%d] \"permission.time.to\" type. rid=%s URL=%s.",
									codeToMessgae(ErrorData), i,
									tb->mResource ? tb->mResource->mResource.id : "(not set)",
									tb->mRefURL);
						}
						struct tm tm_date;
						char *r = strptime(vTimeTo->vString, "%Y/%m/%d-%H:%M:%S", &tm_date);
						if( r == NULL || *r != 0 ) {
							return taskResponse(q, ErrorData, "%s. get_info. gadget[%d] \"permission.time.to\" value. rid=%s URL=%s.",
									codeToMessgae(ErrorData), i,
									tb->mResource ? tb->mResource->mResource.id : "(not set)",
									tb->mRefURL);
						}
						_endTime = timegm(&tm_date);
					}
					if( vTimeFrom && vTimeTo ) {
						if( _fromTime > _endTime ) {
							return taskResponse(q, ErrorData, "%s. get_info. gadget[%d] \"permission.time.to\" value. rid=%s URL=%s.",
									codeToMessgae(ErrorData), i,
									tb->mResource ? tb->mResource->mResource.id : "(not set)",
									tb->mRefURL);
						}
					}
					tb->mShareJsArr[i].fromTime	= _fromTime;
					tb->mShareJsArr[i].endTime	= _endTime;
				}
			}
		}
	}

	endTask	= false;
	return true;
}


bool CommandProc::cmdSync(QueueNode *q)
{
	Task *task = q->getTask();
	int step	= q->getStep();

	TBSyncSub *tb = (TBSyncSub *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBSyncSub();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	switch(step) {
	case 0:
		{
			// save task buffer and query resource
			QnCommSync *qp = (QnCommSync*)q;
			tb->mData	= qp->mData;
			qp->mData	= NULL;
/*
			taskLog(q, LOG_DEBUG, "mode=%d rid=%s rname=%s ref=%s sid=%s sname=%s",
				tb->mData->mMode,
				tb->mData->mRRef.mId,
				tb->mData->mRRef.mName,
				tb->mData->mRef,
				tb->mData->mService.id,
				tb->mData->mService.name);
*/
			if( tb->mData->mMode == QnCommSync::SyncModeUpdate ||
				tb->mData->mMode == QnCommSync::SyncModeRemove ) {
				QnDBQuery *qn = new QnDBQuery(task);
				QueryKeyType sref;
				sref.mId	= tb->mData->mService.id;
				sref.mName	= tb->mData->mService.name;
				qn->queryResource(&sref, &tb->mData->mRRef);
				return task->call(qn, step+1);
			}
		}
	case 1:
		{
			// resource response
			if( step == 1 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					return taskResponse(q, code, "%s Resource info load failed.", res->getMessagePtr());
				}
				tb->mResource	= (ResultResource*)(res->mResult);
				res->mResult	= NULL;
				tb->mResourceId = tb->mResource->mResource.id;

				// TODO: permission check

				// copy and sort gadget list
				tb->mGadgetDbSize	= tb->mResource->mResource.gadgetCount;
				if( tb->mGadgetDbSize > 0 ) {
					size_t sort_size = sizeof(GadgetInfo*) * tb->mGadgetDbSize;
					tb->mGadgetDbSort	= (GadgetInfo**)malloc(sort_size);
					if( tb->mGadgetDbSort == NULL ) {
						return taskResponseLog(LOG_ERR, q, ErrorNoMemory, "%s. gadget sort buffer. sref=%s:%s rref=%s:%s URL=%s.",
								codeToMessgae(ErrorNoMemory),
								tb->mData->mService.id, tb->mData->mService.name,
								tb->mData->mRRef.mId, tb->mData->mRRef.mName,
								tb->mRefURL);
					}
					size_t i;
					for( i = 0; i < tb->mGadgetDbSize; i++ ) {
						tb->mGadgetDbSort[i]	= &tb->mResource->mResource.gadgetArray[i];
					}
					qsort(tb->mGadgetDbSort, tb->mGadgetDbSize, sizeof(GadgetInfo*), qsComparGadget);
				}
				tb->mShareDbSize	= tb->mResource->mResource.shareCount;
			}
			step	= 1;

			if( tb->mData->mMode == QnCommSync::SyncModeUpdate ||
				tb->mData->mMode == QnCommSync::SyncModeRemove ) {
				const char *url1 = tb->mData->mRef;
				const char *url2 = tb->mResource ? tb->mResource->mResource.refURL : NULL;
				const char *url = url1;
				if( url1 )
					url = url1;
				else if( url2 )
					url	= url2;
				tb->mRefURL	= (char*)url;
			} else if( tb->mData->mMode == QnCommSync::SyncModeAdd ) {
				tb->mRefURL	= tb->mData->mRef;
			}
			if( !tb->mRefURL ) {
				return taskResponse(q, ErrorDataConflict, "%s. reference URL is not set. service=%s,%s resource=%s,%s mode=%d refURL=%s.",
							codeToMessgae(ErrorDataConflict),
							tb->mData->mService.id, tb->mData->mService.name,
							tb->mData->mRRef.mId, tb->mData->mRRef.mName,
							tb->mData->mMode, tb->mData->mRef);
			}
		}
	case 2:
		{
			if( step == 2 ) {
				// response query share loop
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					taskLog(q, LOG_WARNING, "%s. shareInfo load errorr. code=%d sref=%s:%s rref=%s:%s share:%d/%d URL=%s.",
							codeToMessgae(code),
							code,
							tb->mData->mService.id, tb->mData->mService.name,
							tb->mData->mRRef.mId, tb->mData->mRRef.mName,
							tb->mShareDbIndex+1, tb->mShareDbSize,
							tb->mRefURL);
				} else {
					ResultShare* share = (ResultShare*)res->mResult;
					size_t needSize = tb->mSyncGroupCount + share->mShare.groupCount;
					if( needSize >= tb->mSyncGroupAlloc ) {
						size_t alloc = sizeof(QueryKeyType) * needSize;
						QueryKeyType *arr = (QueryKeyType*)realloc(tb->mSyncGroupArr, alloc);
						if( arr == NULL ) {
							return taskResponseLog(LOG_ERR, q, ErrorNoMemory, "%s. share group buffer error. sref=%s:%s rref=%s:%s URL=%s.",
									codeToMessgae(ErrorNoMemory),
									tb->mData->mService.id, tb->mData->mService.name,
									tb->mData->mRRef.mId, tb->mData->mRRef.mName,
									tb->mRefURL);
						}
						tb->mSyncGroupArr	= arr;
						tb->mSyncGroupAlloc	= needSize;
					}
					size_t i;
					size_t needBuff = 0;
					for( i = 0; i < share->mShare.groupCount; i++ ) {
						size_t size = DBUtil::sizeofKey(&share->mShare.groupArray[i]);
						needBuff += size;
					}

					size_t freeBuff = tb->mSyncGroupBuffAlloc - tb->mSyncGroupBuffUse;
					if( freeBuff < needBuff ) {
						size_t sizeBuff = tb->mSyncGroupBuffUse + needBuff;
						char *buffPtr = (char*)realloc(tb->mSyncGroupBuffPtr, sizeBuff);
						if( buffPtr == NULL ) {
							return taskResponseLog(LOG_ERR, q, ErrorNoMemory, "%s. share group buffer error. sref=%s:%s rref=%s:%s URL=%s.",
									codeToMessgae(ErrorNoMemory),
									tb->mData->mService.id, tb->mData->mService.name,
									tb->mData->mRRef.mId, tb->mData->mRRef.mName,
									tb->mRefURL);
						}
						int addr_diff = buffPtr - tb->mSyncGroupBuffPtr;
						tb->mSyncGroupBuffPtr	= buffPtr;
						tb->mSyncGroupBuffAlloc	= sizeBuff;
						size_t m;
						for( m = 0; m < tb->mSyncGroupCount; m++ ) {
							if( tb->mSyncGroupArr[m].mId )
								tb->mSyncGroupArr[m].mId	+= addr_diff;
							if( tb->mSyncGroupArr[m].mName )
								tb->mSyncGroupArr[m].mName	+= addr_diff;
						}
					}

					for( i = 0; i < share->mShare.groupCount; i++ ) {
						char *ptr = &tb->mSyncGroupBuffPtr[tb->mSyncGroupBuffUse];
						char *ext = DBUtil::writeKey(&tb->mSyncGroupArr[tb->mSyncGroupCount], ptr, &share->mShare.groupArray[i]);
						tb->mSyncGroupBuffUse += ext-ptr;
						tb->mSyncGroupCount++;
					}
				}
				tb->mShareDbIndex++;
			}
			step	= 2;

			if( tb->mData->mMode == QnCommSync::SyncModeUpdate ||
				tb->mData->mMode == QnCommSync::SyncModeRemove ) {
				if( tb->mShareDbIndex < tb->mShareDbSize ) {
					StdBuffer keyShare;
					DBUtil::keyShare(tb->mResourceId, tb->mShareDbIndex, keyShare);
					QnDBQuery *qn = new QnDBQuery(task);
					QueryKeyType sref;
					QueryKeyType mref;
					sref.mId	= tb->mData->mService.id;
					sref.mName	= tb->mData->mService.name;
					mref.mId	= keyShare.getPtr();
					mref.mName	= NULL;
					qn->queryShare(&sref, &mref);
					return task->call(qn, step); // loop step
				}
			}

			// request resource info
			if( tb->mData->mMode == QnCommSync::SyncModeAdd ||
				tb->mData->mMode == QnCommSync::SyncModeUpdate ) {

				const char *p = tb->mRefURL;
				while( *p ) {
					if( *p == '?' ) break;
					p++;
				}
				if( *p == '?' ) {
					tb->mRefGetOpt = true;
				} else {
					tb->mRefGetOpt = false;
				}
				QnHTTPRequest *qn = new QnHTTPRequest(task);
				StdBuffer optURL;
				if( tb->mRefGetOpt ) {
					optURL.addfmt("%s&opt=info",  tb->mRefURL);
				} else {
					optURL.addfmt("%sinfo",  tb->mRefURL);
				}
				qn->setGetRequest(optURL.getPtr());
				return task->call(qn, step+1);
			}
		}
	case 3:
		{
			// resource info response
			if( step == 3 ) {
				QnHTTPResult *res	= (QnHTTPResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk || res->mStatusCode != 200) {
					if( code == 0 ) {
						code = ErrorHTTP;
					}
					return taskResponse(q, code, "getInfo request failed. status code = %d URL=%s rid=%s resourceName=%s. %s",
								res->mStatusCode,
								tb->mRefURL,
								tb->mResourceId ? tb->mResourceId : "(not set)",
								tb->mJsName ? tb->mJsName->vString : "(not set)",
								res->getMessagePtr());
				}
				bool endTask=false;
				if( !syncParseInfo(endTask, tb, q, step) ) {
					return false;
				}
				if( endTask ) {
					taskLog(q, LOG_DEBUG, "syncParseInfo endTask=true");
					return true;
				}
			}

			step = 3;

			// request share info
			if( tb->mData->mMode == QnCommSync::SyncModeUpdate ||
				tb->mData->mMode == QnCommSync::SyncModeAdd ) {
				QnHTTPRequest *qn = new QnHTTPRequest(task);
				StdBuffer optURL;
				if( tb->mRefGetOpt ) {
					optURL.addfmt("%s&opt=share_info",  tb->mRefURL);
				} else {
					optURL.addfmt("%sshare_info",  tb->mRefURL);
				}
				qn->setGetRequest(optURL.getPtr());
				return task->call(qn, step+1);
			}
		}

	case 4:
		{
			// share_info response
			if( step == 4 ) {
				QnHTTPResult *res	= (QnHTTPResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk || res->mStatusCode != 200) {
					if( code == 0 ) {
						code = ErrorHTTP;
					}
					return taskResponse(q, code, "get_share_info resuest failed. status code = %d URL=%s rid=%s resourceName=%s. %s",
								res->mStatusCode,
								tb->mRefURL,
								tb->mResourceId ? tb->mResourceId : "(not set)",
								tb->mJsName ? tb->mJsName->vString : "(not set)",
								res->getMessagePtr());
				}

				bool endTask;
				if( !syncParseShareInfo(endTask, tb, q, step) )
					return false;
				if( endTask )
					return true;

				size_t s;
				for( s = 0; s < tb->mShareJsSize; s++ ) {
					ShareInfo *share = &tb->mShareJsArr[s];
					size_t needSize = tb->mSyncGroupCount + share->groupCount;
					if( needSize >= tb->mSyncGroupAlloc ) {
						size_t alloc = sizeof(QueryKeyType) * needSize;
						QueryKeyType *arr = (QueryKeyType*)realloc(tb->mSyncGroupArr, alloc);
						if( arr == NULL ) {
							return taskResponseLog(LOG_ERR, q, ErrorNoMemory, "%s. share group buffer error. sref=%s:%s rref=%s:%s URL=%s.",
									codeToMessgae(ErrorNoMemory),
									tb->mData->mService.id, tb->mData->mService.name,
									tb->mData->mRRef.mId, tb->mData->mRRef.mName,
									tb->mRefURL);
						}
						tb->mSyncGroupArr	= arr;
						tb->mSyncGroupAlloc	= needSize;
					}
					size_t i;
					size_t needBuff = 0;
					for( i = 0; i < share->groupCount; i++ ) {
						size_t size = DBUtil::sizeofKey(&share->groupArray[i]);
						needBuff += size;
					}
					size_t freeBuff = tb->mSyncGroupBuffAlloc - tb->mSyncGroupBuffUse;
					if( freeBuff < needBuff ) {
						size_t sizeBuff = tb->mSyncGroupBuffUse + needBuff;
						assert(tb->mSyncGroupBuffAlloc<sizeBuff);
						char *buffPtr = (char*)realloc(tb->mSyncGroupBuffPtr, sizeBuff);
						if( buffPtr == NULL ) {
							return taskResponseLog(LOG_ERR, q, ErrorNoMemory, "%s. share group buffer error. sref=%s:%s rref=%s:%s URL=%s.",
									codeToMessgae(ErrorNoMemory),
									tb->mData->mService.id, tb->mData->mService.name,
									tb->mData->mRRef.mId, tb->mData->mRRef.mName,
									tb->mRefURL);
						}
						int addr_diff = buffPtr - tb->mSyncGroupBuffPtr;
						tb->mSyncGroupBuffPtr	= buffPtr;
						tb->mSyncGroupBuffAlloc	= sizeBuff;
						size_t m;
						for( m = 0; m < tb->mSyncGroupCount; m++ ) {
							if( tb->mSyncGroupArr[m].mId )
								tb->mSyncGroupArr[m].mId	+= addr_diff;
							if( tb->mSyncGroupArr[m].mName )
								tb->mSyncGroupArr[m].mName	+= addr_diff;
						}
					}
					size_t top = tb->mSyncGroupCount;
					for( i = 0; i < share->groupCount; i++ ) {
						size_t n;
						for( n = top; n < tb->mSyncGroupCount; n++ ) {
							if( (share->groupArray[i].mId != NULL && tb->mSyncGroupArr[n].mId != NULL &&
								 strcmp(share->groupArray[i].mId, tb->mSyncGroupArr[n].mId) == 0) ||
								(share->groupArray[i].mName != NULL && tb->mSyncGroupArr[n].mName != NULL &&
								 strcmp(share->groupArray[i].mName, tb->mSyncGroupArr[n].mName) == 0)) {
								return taskResponse(q, ErrorDuplicate, "%s. share group error. sref=%s:%s rref=%s:%s URL=%s.",
										codeToMessgae(ErrorDuplicate),
										tb->mData->mService.id, tb->mData->mService.name,
										tb->mData->mRRef.mId, tb->mData->mRRef.mName,
										tb->mRefURL);
							}
						}
						char *ptr = &tb->mSyncGroupBuffPtr[tb->mSyncGroupBuffUse];
						char *ext = DBUtil::writeKey(&tb->mSyncGroupArr[tb->mSyncGroupCount], ptr, &share->groupArray[i]);
						tb->mSyncGroupBuffUse += ext - ptr;
						tb->mSyncGroupCount++;
					}
				}
			}

			step = 4;

			ResourceInfo res;
			bool needReg = false;

			if( tb->mData->mMode == QnCommSync::SyncModeAdd ) {
				needReg	= true;

				size_t jsi = 0;
				for( jsi = 0; jsi < tb->mGadgetJsSize; jsi++  ) {
					tb->mGadgetJsSort[jsi]->id			= 0;
					tb->mGadgetJsSort[jsi]->revision	= 1;
				}

				res.id			= NULL;
				res.name		= const_cast<char*>(tb->mJsName->vString);
				res.tag			= const_cast<char*>(tb->mJsTag ? tb->mJsTag->vString : NULL);
				res.serviceId	= tb->mData->mService.id;
				res.refURL		= tb->mRefURL;
				res.refTime		= tb->mJsRefTime;
				res.revision	= 1;
				res.status		= ResourceOpen;
				res.createTime	= 0;
				res.updateTime	= 0;
				res.shareCount	= tb->mShareJsSize;
				res.gadgetCount	= 0;
				res.gadgetArray	= NULL;
			}

			// compare update
			if( tb->mData->mMode == QnCommSync::SyncModeUpdate ) {
				size_t jsi = 0;
				size_t dbi = 0;
				bool gadgetUpdate = false;
				while( jsi < tb->mGadgetJsSize && dbi < tb->mGadgetDbSize ) {
					GadgetInfo *pJs = tb->mGadgetJsSort[jsi];
					GadgetInfo *pDb = tb->mGadgetDbSort[jsi];
					int cond = strcmp(pJs->name, pDb->name);
					if( cond == 0 ) {
						if( (pJs->refTime != pDb->refTime) ||
							(pJs->tag == NULL &&  pDb->tag != NULL) ||
							(pJs->tag != NULL &&  pDb->tag == NULL) ||
							(pJs->tag && strcmp(pJs->tag, pDb->tag) != 0) ) {
							tb->mGadgetJsSort[jsi]->id			= tb->mGadgetDbSort[dbi]->id;
							tb->mGadgetJsSort[jsi]->revision	= tb->mGadgetDbSort[dbi]->revision + 1;
							gadgetUpdate = true;
						}
						jsi++;
						dbi++;
					} else if( cond < 0 ) {
						gadgetUpdate = true;
						tb->mGadgetJsSort[jsi]->id			= 0;
						tb->mGadgetJsSort[jsi]->revision	= 1;
						jsi++;
					} else {
						gadgetUpdate = true;
						dbi++;
					}
				}
				if( jsi < tb->mGadgetJsSize ) {
					gadgetUpdate = true;
					while( jsi < tb->mGadgetJsSize ) {
						tb->mGadgetJsSort[jsi]->id			= 0;
						tb->mGadgetJsSort[jsi]->revision	= 1;
						jsi++;
					}
				}

				bool resourceUpdate = false;
				if( (tb->mJsRefTime != tb->mResource->mResource.refTime) ||
					(tb->mShareJsSize != tb->mResource->mResource.shareCount) ||
					(strcmp(tb->mRefURL, tb->mResource->mResource.refURL) != 0) ||
					((tb->mJsTag == NULL || tb->mJsTag->vString == NULL) &&  tb->mResource->mResource.tag != NULL) ||
					(tb->mJsTag && tb->mResource->mResource.tag == NULL) ||
					(tb->mJsTag && strcmp(tb->mJsTag->vString, tb->mResource->mResource.tag) != 0) ) {
					resourceUpdate = true;
				}

				if( resourceUpdate || gadgetUpdate ) {
					needReg = true;

					res.id			= tb->mResource->mResource.id;
					res.name		= const_cast<char*>(tb->mJsName->vString);
					res.tag			= const_cast<char*>(tb->mJsTag ? tb->mJsTag->vString : NULL);
					res.serviceId	= tb->mData->mService.id;
					res.refURL		= tb->mRefURL;
					res.refTime		= tb->mJsRefTime;
					if( resourceUpdate )
						res.revision	= tb->mResource->mResource.revision + 1;
					else
						res.revision	= tb->mResource->mResource.revision;
					res.status		= ResourceOpen;
					res.shareCount	= tb->mShareJsSize;
					res.gadgetCount	= 0;
					res.gadgetArray	= NULL;
					res.createTime	= tb->mResource->mResource.createTime;
					res.updateTime	= tb->mResource->mResource.updateTime;
				}
			}

			if( needReg ) {
				QueryKeyType sref;
				sref.mId	= tb->mData->mService.id;
				sref.mName	= NULL;
				QnDBRegister *qn = new QnDBRegister(task);
				qn->registResource(&sref, &res, tb->mGadgetJsArr, tb->mGadgetJsSize);
				return task->call(qn, step+1);
			}

			if( tb->mData->mMode == QnCommSync::SyncModeRemove ) {
				QueryKeyType sref;
				QueryKeyType rref;
				sref.mId	= tb->mData->mService.id;
				sref.mName	= NULL;
				rref.mId	= tb->mResource->mResource.id;
				rref.mName	= NULL;
				QnDBQuery *qn = new QnDBQuery(task);
				qn->removeResource(&sref, &rref);
				return task->call(qn, step+1);
			}
		}

	case 5:
		{
			// resource update
			if( step == 5 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					if( tb->mData->mMode == QnCommSync::SyncModeAdd ||
						tb->mData->mMode == QnCommSync::SyncModeUpdate ) {
						return taskResponse(q, code, "%s. sync: update resource error. rid=%s resourceName=%s.",
								res->getMessagePtr(),
								tb->mResourceId ? tb->mResourceId : "(not set)",
								tb->mJsName ? tb->mJsName->vString : "(not set)");
					}
					if( tb->mData->mMode == QnCommSync::SyncModeRemove ) {
						return taskResponse(q, code, "%s. sync: remove resource error. rid=%s resourceName=%s.",
								res->getMessagePtr(),
								tb->mResourceId ? tb->mResourceId : "(not set)",
								tb->mJsName ? tb->mJsName->vString : "(not set)");
					}
				}
				if( tb->mData->mMode == QnCommSync::SyncModeAdd ) {
					tb->mRid		= (ResultId*)res->mResult;
					res->mResult	= NULL;
					tb->mResourceId	= tb->mRid->mId;
				}
			}

			step	= 5;

			tb->mShareDbIndex	= 0;
		}
	case 6:
		{
			// remove shareInfo
			if( step == 6 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					taskLog(q, LOG_ERR, "%s. shareInfo remove error. rid=%s resourceName=%s index=%d.",
								res->getMessagePtr(),
								tb->mResourceId ? tb->mResourceId : "(not set)",
								tb->mJsName ? tb->mJsName->vString : "(not set)",
								tb->mShareDbIndex);
				}
				tb->mShareDbIndex++;
			}
			step	= 6;

			if( tb->mData->mMode == QnCommSync::SyncModeUpdate ||
				tb->mData->mMode == QnCommSync::SyncModeRemove ) {
				if( tb->mShareDbIndex < tb->mShareDbSize ) {
					// remove shareList
					StdBuffer keyShare;
					DBUtil::keyShare(tb->mResourceId, tb->mShareDbIndex, keyShare);
					QueryKeyType sref;
					QueryKeyType mref;
					sref.mId	= tb->mData->mService.id;
					sref.mName	= NULL;
					mref.mId	= keyShare.getPtr();
					mref.mName	= NULL;
					QnDBQuery *qn = new QnDBQuery(task);
					qn->removeShare(&sref, &mref);
					return task->call(qn, step);
				}
			}
		}

	case 7:
		{
			// add shareList
			if( step == 7 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					taskLog(q, LOG_ERR, "ShareInfo add failed. %s", res->getMessagePtr());
				}
				tb->mShareJsIndex++;
			}
			step	= 7;
			if( tb->mData->mMode == QnCommSync::SyncModeUpdate ||
				tb->mData->mMode == QnCommSync::SyncModeAdd ) {
				if( tb->mShareJsIndex < tb->mShareJsSize ) {
					size_t i = tb->mShareJsIndex;
					tb->mShareJsArr[i].resourceId	= tb->mResourceId;
					tb->mShareJsArr[i].index		= i;
					// register share info
					QueryKeyType sref;
					QueryKeyType rref;
					sref.mId	= tb->mData->mService.id;
					sref.mName	= NULL;
					rref.mId	= tb->mResourceId;
					rref.mName	= NULL;
					QnDBRegister *qn	= new QnDBRegister(task);
					taskLog(q, LOG_DEBUG, "share[%d] groupCount=%d groupArray[0]=%s,%s",
							i,
							tb->mShareJsArr[i].groupCount,
							tb->mShareJsArr[i].groupCount > 0 ? tb->mShareJsArr[i].groupArray[0].mId : "",
							tb->mShareJsArr[i].groupCount > 0 ? tb->mShareJsArr[i].groupArray[0].mName : "");
					qn->registShare(&sref, &rref, &tb->mShareJsArr[i]);
					return task->call(qn, step);
				}
			}

			if( tb->mSyncGroupCount != 0 ) {
				QueryKeyType sref;
				sref.mId	= tb->mData->mService.id;
				sref.mName	= NULL;
				QnDBRegister *qn = new QnDBRegister(task);
				qn->registSyncQueue(SyncCMDUpdate, &sref, tb->mSyncGroupArr, tb->mSyncGroupCount);
				return task->call(qn, step+1);
			}
		}
	case 8:
		{
			if( step == 8 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					taskLog(q, LOG_ERR, "ShareInfo sync queue add error. %s", res->getMessagePtr());
				}

				pjson::builder jb;

				if( !jb.init(256) ||
					!jb.beginObject() ||
					!jb.addObjectProp("syncQueue",9) ||
					!jb.beginObject() ||
					!jb.addObjectProp("sid", 3) ||
					!jb.valueString(tb->mData->mService.id) ||
					!jb.endObject() ||
					!jb.endObject() ) {
					taskLog(q, LOG_ERR, "json builder failed(1). code=%d", jb.getError());
					return false;
				}

				Task *st = mSentinel->getTaskManager().createTaskFromBaseTask(task);
				QueueNode *qn = new QueueNode(st, this, CMDSyncQueue);
				qn->setJSONMessage(jb.detouch());
				st->startTask(qn);
			}

			QnResult *r =  new QnResult(task);
			r->setResult(CMDSync, ErrorOk);
			taskLog(q, LOG_INFO, "sync process end. rid=%s resourceName=%s URL=%s",
					getName(), tb->mResourceId, tb->mJsName ? tb->mJsName->vString : "(not set)", tb->mRefURL);
			return task->endTask(r);
		}
	}
	return false;
}


class TBSyncQueue: public TaskBuffer {
public:
	TBSyncQueue() {
		mJsCommand		= NULL;
		mCmd			= SyncCMDUpdate;
		mSRef.mId		= NULL;
		mSRef.mName		= NULL;
		mResId			= NULL;
		mJsQueue		= NULL;
		mGRef.mId		= NULL;
		mGRef.mName		= NULL;
		mARef.mId		= NULL;
		mARef.mName		= NULL;
		mShareList		= NULL;
		mShareIndex		= 0;
		mDeviceList		= NULL;
		mDeviceCount	= 0;
	}
	virtual ~TBSyncQueue() {
		if( mJsCommand )	delete mJsCommand;
		if( mResId )		free(mResId);
		if( mJsQueue )		delete mJsQueue;
		if( mShareList )	free(mShareList);
		if( mDeviceList )	free(mDeviceList);
	}
	pjson::json *	mJsCommand;
	SyncQueueCmd	mCmd;
	QueryKeyType	mSRef;
	ResultId *		mResId;
	pjson::json *	mJsQueue;
	QueryKeyType	mGRef;
	QueryKeyType	mARef;
	ResultList *	mShareList;
	size_t			mShareIndex;
	ResultList *	mDeviceList;
	size_t			mDeviceCount;
};

bool  CommandProc::cmdSyncQueue(QueueNode *q)
{
	Task *task = q->getTask();
	int step	= q->getStep();

	TBSyncQueue *tb = (TBSyncQueue *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBSyncQueue();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();
			pjson::value *vCmd = tb->mJsCommand->get("syncQueue");
			if( !vCmd || vCmd->vt != pjson::vt_object ) {
				return taskResponse(q, ErrorParameter, "%s. \"syncQueue\" is not object.", codeToMessgae(ErrorParameter));
			}
			pjson::value * vSID		= tb->mJsCommand->get("sid",  vCmd);
			pjson::value * vSName	= tb->mJsCommand->get("serviceName",  vCmd);
			if( vSID && vSID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"sid\" type.", codeToMessgae(ErrorParameter));
			}
			if( vSName && vSName->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"serviceName\" type", codeToMessgae(ErrorParameter));
			}
			tb->mSRef.mId	= vSID ? vSID->vString : NULL;
			tb->mSRef.mName	= vSName ? vSName->vString : NULL;

			QueueNode *qn = new QueueNode(task, this, q->getFunction(), step+1);
			mSentinel->getScheduler().setQueueTimer(qn, mSentinel->getConf()->syncQueueWaitTime);
			return true;
		}
	case 1:
		{
			taskLog(q, LOG_DEBUG, "*** sync queue timeup q=%x ***", q);
			QnDBQuery *qn = new QnDBQuery(task);
			qn->querySyncQueue(&tb->mSRef);
			return task->call(qn, step+1);
		}
	case 2:
		{
			bool endQueue = false;

			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				if( code != ErrorNotFound ) {
					return taskResponse(q, code, "Sync queue pop failed. %s", res->getMessagePtr());
				}
				endQueue = true;
			}
			if( endQueue ) {
				return taskResponse(q, ErrorOk, "queue end");
			}

			if( tb->mResId ) {
				free(tb->mResId);
				tb->mResId	= NULL;
			}
			tb->mResId 		= (ResultId*)res->mResult;
			res->mResult	= NULL;

			pjson::parser ps;
			pjson::json *js = ps.parse(tb->mResId->mId, strlen(tb->mResId->mId));
			if( js == NULL ) {
				taskLog(q, LOG_DEBUG, "*** sync queue error=%s ***", tb->mResId->mId);
				return taskResponse(q, ErrorData, "%s. json parse error. service=%s,%s code=%d pos=%d src=%s.",
						codeToMessgae(ErrorData),
						tb->mSRef.mId, tb->mSRef.mName,
						ps.getErrorCode(), ps.getErrorPos(),
						tb->mResId->mId);
			}

			if( tb->mJsQueue ) delete tb->mJsQueue;
			tb->mJsQueue	= js;

			pjson::value * vCmd		= tb->mJsQueue->getValue();
			pjson::value * vCMD		= tb->mJsQueue->get("cmd", vCmd);
			pjson::value * vGID		= tb->mJsQueue->get("gid", vCmd);
			pjson::value * vAID		= tb->mJsQueue->get("aid", vCmd);

			if( vCMD && vCMD->vt != pjson::vt_int ) {
				return taskResponse(q, ErrorParameter, "%s. \"cmd\" type. service=%s,%s",
						codeToMessgae(ErrorData),
						tb->mSRef.mId, tb->mSRef.mName);
			}
			if( vGID && vGID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"gid\" type. service=%s,%s",
						codeToMessgae(ErrorData),
						tb->mSRef.mId, tb->mSRef.mName);
			}
			if( vAID && vAID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"aid\" type. service=%s,%s",
						codeToMessgae(ErrorData),
						tb->mSRef.mId, tb->mSRef.mName);
			}
			if( !vAID ) {
				return taskResponse(q, ErrorParameter, "%s. \"aid\" is not specified. service=%s,%s",
						codeToMessgae(ErrorData),
						tb->mSRef.mId, tb->mSRef.mName);
			}
			tb->mCmd		= (SyncQueueCmd)(vCMD ? vCMD->vInt : SyncCMDUpdate);
			tb->mGRef.mId	= vGID   ? vGID->vString : NULL;
			tb->mGRef.mName	= NULL;
			tb->mARef.mId	= vAID   ? vAID->vString : NULL;
			tb->mARef.mName	= NULL;

			/***************************/
			taskLog(q, LOG_DEBUG, "%s::cmdSyncQueue: *** sync gid=%s aid=%s ***", getName(),
					tb->mGRef.mId, tb->mARef.mId );
			/***************************/

			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryDeviceList(&tb->mARef, NULL);
			return task->call(qn, step+1);
		}
	case 3:
		{
			if( tb->mDeviceList ) {
				free(tb->mDeviceList);
			}
			tb->mDeviceList 	= NULL;
			tb->mDeviceCount	= 0;
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				if( code != ErrorNotFound ) {
					return taskResponse(q, code, "Device list load failed. %s", res->getMessagePtr());
				}
				/***************************/
				taskLog(q, LOG_DEBUG, "*** dev list empty. gid=%s aid=%s ***",
						tb->mGRef.mId, tb->mARef.mId);
				/***************************/
			} else {
				tb->mDeviceList		= (ResultList*)res->mResult;
				tb->mDeviceCount 	= tb->mDeviceList->mList.values;
				res->mResult		= NULL;
				/***************************/
				taskLog(q, LOG_DEBUG, "*** sync gid=%s aid=%s device count=%d ***",
						tb->mGRef.mId, tb->mARef.mId, tb->mDeviceList->mList.values );
				/***************************/
			}

			SessionPool &pool = mSentinel->getSessionPool();
			size_t i;
			for( i = 0; i < tb->mDeviceCount; i++ ) {
				/***************************/
				taskLog(q, LOG_DEBUG, "*** dev gid=%s aid=%s did=%s ***",
						tb->mGRef.mId, tb->mARef.mId, tb->mDeviceList->mList.value[i] );
				/***************************/

				Session *con = pool.findDeviceSession(tb->mDeviceList->mList.value[i]);
				if( con ) {
					const char *aid = con->getAccountId();
					if( aid && strcmp(aid, tb->mARef.mId) == 0 ) {
						const char *mode = NULL;
						if( tb->mCmd == SyncCMDRemove )
							mode = "remove";
						else
							mode = "update";
						pjson::builder jb;
						if( !jb.init(256) ||
							!jb.beginObject() ||
							!jb.addObjectProp("doSync", 6) ||
							!jb.beginObject() ||
							!jb.addObjectProp("mode", 4) ||
							!jb.valueString(mode)) {
							taskLog(q, LOG_ERR, "json builder failed(1). code=%d", jb.getError());
							return false;
						}
						if( tb->mSRef.mId ) {
							if( !jb.addObjectProp("sid", 3) ||
								!jb.valueString(tb->mSRef.mId) ) {
								taskLog(q, LOG_ERR, "json builder failed(2). code=%d", jb.getError());
								return false;
							}
						}
						if( tb->mSRef.mName ) {
							if( !jb.addObjectProp("serviceName", 11) ||
								!jb.valueString(tb->mSRef.mName) ) {
								taskLog(q, LOG_ERR, "json builder failed(3). code=%d", jb.getError());
								return false;
							}
						}
						if( tb->mGRef.mId ) {
							if(	!jb.addObjectProp("gid", 3) ||
								!jb.valueString(tb->mGRef.mId) ) {
								taskLog(q, LOG_ERR, "json builder failed(4). code=%d", jb.getError());
								return false;
							}
						}
						if(	!jb.endObject() ||
							!jb.endObject() ) {
							taskLog(q, LOG_ERR, "json builder failed(5). code=%d", jb.getError());
							return false;
						}
						Task *st = mSentinel->getTaskManager().createTaskFromBaseTask(task);
						QueueCommSend *qn = new QueueCommSend(st, con);
						qn->setJSONMessage(jb.detouch());
						st->startTask(qn);
					}
				}
			}

			// next loop
			QnDBQuery *qn = new QnDBQuery(task);
			qn->querySyncQueue(&tb->mSRef);
			return task->call(qn, 2);
		}
	}
	taskLog(q, LOG_ERR, "unknown step=%d.", step);
	return false;
}

bool CommandProc::cmdSyncStat(QueueNode *q)
{
	return taskResponse(q, ErrorNotImpl, "syncStat");
}

bool CommandProc::cmdSyncGroup(QueueNode *q)
{
	return taskResponse(q, ErrorNotImpl, "syncGroup");
}

class TBSyncAccount: public TaskBuffer {
public:
	TBSyncAccount() {
		mJsCommand	= NULL;
		mJsSID		= NULL;
		mJsSName	= NULL;
		mJsAID		= NULL;
		mJsUID		= NULL;
		mSID		= NULL;
		mAID		= NULL;
		mService	= NULL;
		mAccount	= NULL;
		mDeviceIndex	= 0;
		mDeviceList	= NULL;
	}
	virtual ~TBSyncAccount() {
		if( mJsCommand )	delete mJsCommand;
		if( mService )		free(mService);
		if( mAccount )		free(mAccount);
		if( mDeviceList )	free(mDeviceList);
	}
	pjson::json *	mJsCommand;
	pjson::value *	mJsSID;
	pjson::value *	mJsSName;
	pjson::value *	mJsAID;
	pjson::value *	mJsUID;

	const char *	mSID;
	const char *	mAID;

	ResultService *	mService;
	ResultAccount *	mAccount;
	size_t			mDeviceIndex;
	ResultList *	mDeviceList;
};


bool CommandProc::cmdSyncAccount(QueueNode *q)
{
	Task *task = q->getTask();
	int step = q->getStep();

	TBSyncAccount *tb = (TBSyncAccount *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBSyncAccount();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();
			pjson::value *vCmd = tb->mJsCommand->get("syncAccount");
			if( !vCmd || vCmd->vt != pjson::vt_object ) {
				return taskResponse(q, ErrorParameter, "%s. \"syncAccount\" is not object.", codeToMessgae(ErrorParameter));
			}
			tb->mJsSID		= tb->mJsCommand->get("sid",  vCmd);
			tb->mJsSName	= tb->mJsCommand->get("serviceName", vCmd);
			tb->mJsAID		= tb->mJsCommand->get("aid", vCmd);
			tb->mJsUID		= tb->mJsCommand->get("userId", vCmd);

			if( tb->mJsSID && tb->mJsSID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"sid\" type.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsSName && tb->mJsSName->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"serviceName\" type.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsAID && tb->mJsAID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"aid\" type.", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsUID && tb->mJsUID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s. \"userId\" type.", codeToMessgae(ErrorParameter));
			}

			if( tb->mJsSID ) {
				tb->mSID = tb->mJsSID->vString;
			} else if( tb->mJsSName == NULL ) {
				tb->mSID = task->getServiceId();
				if( tb->mSID == NULL ) {
					return taskResponse(q, ErrorParameter, "%s. service is not specified.", codeToMessgae(ErrorParameter));
				}
			} else {
				QueryKeyType sref;
				sref.mId	= NULL;
				sref.mName	= tb->mJsSName->vString;
				QnDBQuery *qn = new QnDBQuery(task);
				qn->queryService(&sref);
				return task->call(qn, step+1);
			}
		}
	case 1:
		{
			if( step == 1 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					return taskResponse(q, code, "Service info load failed. %s", res->getMessagePtr());
				}
				tb->mService	= (ResultService*)(res->mResult);
				res->mResult	= NULL;
				tb->mSID		= tb->mService->mService.id;
			}
			step = 1;

			if( tb->mJsAID ) {
				tb->mAID = tb->mJsAID->vString;
			} else if( tb->mJsUID == NULL ) {
				tb->mAID = task->getAccountId();
				if( tb->mAID == NULL ) {
					return taskResponse(q, ErrorParameter, "%s. account is not specified.", codeToMessgae(ErrorParameter));
				}
			} else {
				QueryKeyType aref;
				aref.mId	= NULL;
				aref.mName	= tb->mJsUID->vString;
				QnDBQuery *qn = new QnDBQuery(task);
				qn->queryAccount(&aref);
				return task->call(qn, step+1);
			}
		}
	case 2:
		{
			if( step == 2 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					return taskResponse(q, code, "Account info load failed. %s", res->getMessagePtr());
				}
				tb->mAccount	= (ResultAccount*)(res->mResult);
				res->mResult	= NULL;
				tb->mAID		= tb->mAccount->mAccount.id;
			}
			step = 2;

			QueryKeyType aref;
			aref.mId	= tb->mAID;
			aref.mName	= NULL;
			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryDeviceList(&aref, NULL);
			return task->call(qn, step+1);
		}
		break;
	case 3:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, ErrorDatabase, "Device list loaf failed. %s", res->getMessagePtr());
			}
			tb->mDeviceList = (ResultList*)(res->mResult);
			res->mResult	= NULL;
		}
	case 4:
		{
			if( step == 4 ) {
				QnResult *res	= (QnResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					taskLog(q, LOG_WARNING, "syncDevice command failed.", res->getMessagePtr());
				}
			}
			step	= 4;

			if( tb->mDeviceIndex < tb->mDeviceList->mList.values ) {
				pjson::builder jb;
				if( !jb.init(256) ||
					!jb.beginObject() ||
					!jb.addObjectProp("syncDevice", 10) ||
					!jb.beginObject() ||
					!jb.addObjectProp("sid", 3) ||
					!jb.valueString(tb->mSID) ||
					!jb.addObjectProp("did", 3) ||
					!jb.valueString(tb->mDeviceList->mList.value[tb->mDeviceIndex++]) ||
					!jb.endObject() ) {
					taskLog(q, LOG_ERR, "json builder failed(1). code=%d", jb.getError());
					return false;
				}
				QueueNode *cmd = new QueueNode(task, this, CMDSyncDevice);
				return task->call(cmd, step);
			}

			return taskResponse(q, ErrorOk, "account synced");
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step=%d.", step);
	return false;
}

class TBSyncDevice: public TaskBuffer {
public:
	TBSyncDevice() {
		mJsCommand	= NULL;
		mJsSID		= NULL;
		mJsSName	= NULL;
		mJsDID		= NULL;
		mJsUDID		= NULL;
		mSID		= NULL;
		mDID		= NULL;
		mService	= NULL;
		mDevice	= NULL;
		mDeviceIndex	= 0;
		mDeviceList	= NULL;
	}
	virtual ~TBSyncDevice() {
		if( mJsCommand )	delete mJsCommand;
		if( mService )		free(mService);
		if( mDevice )		free(mDevice);
		if( mDeviceList )	free(mDeviceList);
	}
	pjson::json *	mJsCommand;
	pjson::value *	mJsSID;
	pjson::value *	mJsSName;
	pjson::value *	mJsDID;
	pjson::value *	mJsUDID;

	const char *	mSID;
	const char *	mDID;

	ResultService *	mService;
	ResultDevice *	mDevice;
	size_t			mDeviceIndex;
	ResultList *	mDeviceList;
};


bool CommandProc::cmdSyncDevice(QueueNode *q)
{
	Task *task = q->getTask();

	TBSyncDevice *tb = (TBSyncDevice *)task->getTaskBuffer();
	if( tb == NULL ) {
		tb = new TBSyncDevice();
		if( tb == NULL ) {
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
		if( !task->setTaskBuffer(tb) ) {
			delete tb;
			taskLog(q, LOG_ERR, "%s.", codeToMessgae(ErrorNoMemory));
			return false;
		}
 	}

	int step = q->getStep();
	switch(step) {
	case 0:
		{
			tb->mJsCommand	= q->detouchJSONMessage();

			pjson::value *vCmd = tb->mJsCommand->get("syncDevice");
			if( !vCmd || vCmd->vt != pjson::vt_object ) {
				return taskResponse(q, ErrorParameter, "%s. \"syncDevice\" is not object.", codeToMessgae(ErrorParameter));
			}
			tb->mJsSID		= tb->mJsCommand->get("sid",  vCmd);
			tb->mJsSName	= tb->mJsCommand->get("serviceName", vCmd);
			tb->mJsDID		= tb->mJsCommand->get("did", vCmd);
			tb->mJsUDID		= tb->mJsCommand->get("udid", vCmd);

			if( tb->mJsSID && tb->mJsSID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s \"sid\" type", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsSName && tb->mJsSName->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s \"serviceName\" type", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsDID && tb->mJsDID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s \"did\" type", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsUDID && tb->mJsUDID->vt != pjson::vt_string ) {
				return taskResponse(q, ErrorParameter, "%s \"udid\" type", codeToMessgae(ErrorParameter));
			}
			if( tb->mJsSID ) {
				tb->mSID = tb->mJsSID->vString;
			} else if( tb->mJsSName == NULL ) {
				tb->mSID = task->getServiceId();
				if( tb->mSID == NULL ) {
					return taskResponse(q, ErrorParameter, "%s. Service is not specified.", codeToMessgae(ErrorParameter));
				}
			} else {
				QueryKeyType sref;
				sref.mId	= NULL;
				sref.mName	= tb->mJsSName->vString;
				QnDBQuery *qn = new QnDBQuery(task);
				qn->queryService(&sref);
				return task->call(qn, step+1);
			}
		}
	case 1:
		{
			if( step == 1 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					return taskResponse(q, code, "%s. Service info load failed. %s", res->getMessagePtr());
				}
				tb->mService	= (ResultService*)(res->mResult);
				res->mResult	= NULL;
				tb->mSID		= tb->mService->mService.id;
			}
			step = 1;

			if( tb->mJsDID ) {
				tb->mDID = tb->mJsDID->vString;
			} else if( tb->mJsUDID == NULL ) {
				tb->mDID = task->getDeviceId();
				if( tb->mDID == NULL ) {
					return taskResponse(q, ErrorParameter, "%s. device is not specified.", codeToMessgae(ErrorParameter));
				}
			} else {
				QueryKeyType dref;
				dref.mId	= NULL;
				dref.mName	= tb->mJsUDID->vString;
				QnDBQuery *qn = new QnDBQuery(task);
				qn->queryDevice(&dref);
				return task->call(qn, step+1);
			}
		}
	case 2:
		{
			if( step == 2 ) {
				QnDBResult *res	= (QnDBResult*)q;
				ErrorCode code	= res->getResultCode();
				if( code != ErrorOk ) {
					return taskResponse(q, code, "Device info load failed. %s", res->getMessagePtr());
				}
				tb->mDevice		= (ResultDevice*)(res->mResult);
				res->mResult	= NULL;
				tb->mDID		= tb->mDevice->mDevice.id;
			}
			step = 2;

			QueryKeyType sref;
			QueryKeyType dref;
			sref.mId	= tb->mSID;
			sref.mName	= NULL;
			dref.mId	= tb->mDID;
			dref.mName	= NULL;
			QnDBQuery *qn = new QnDBQuery(task);
			qn->queryDeviceShareCount(&dref);
			return task->call(qn, step+1);
		}
		break;
	case 3:
		{
			QnDBResult *res	= (QnDBResult*)q;
			ErrorCode code	= res->getResultCode();
			if( code != ErrorOk ) {
				return taskResponse(q, code, "Device share count load failed. %s", res->getMessagePtr());
			}
			ResultCount *count = (ResultCount*)(res->mResult);
			if( count->mCount > 0 ) {
				SessionPool &pool = mSentinel->getSessionPool();
				Session *session = pool.findDeviceSession(tb->mDID);

				pjson::builder jb;
				if( !jb.init(256) ||
					!jb.beginObject() ||
					!jb.addObjectProp("doSync", 6) ||
					!jb.beginObject() ||
					!jb.addObjectProp("sid", 3) ||
					!jb.valueString(tb->mSID)  ||
					!jb.endObject() ||
					!jb.endObject() ) {
					taskLog(q, LOG_ERR, "json builder failed(1). code=%d", jb.getError());
					return false;
				}

				QueueCommSend *sq = new QueueCommSend(task, session);
				sq->setJSONMessage(jb.detouch());
				return task->call(sq, step+1);
			}
		}
	case 4:
		{
			ErrorCode code = ErrorOk;
			if( step == 4 ) {
				QnResult *res	= (QnResult*)q;
				code	= res->getResultCode();
				if( code != 0 ) {
					taskLog(q, LOG_WARNING, "doSync failed. %s", res->getMessagePtr());
				}
			}
			step	= 4;
			QnResult *r = new QnResult(task);
			r->setResult(q->getFunction(),code);
			return task->endTask(r);
		}
		break;
	}
	taskLog(q, LOG_ERR, "unknown step=%d.", step);
	return false;
}

bool CommandProc::cmdUpSync(QueueNode *q)
{
	return taskResponse(q, ErrorNotImpl, "%s", codeToMessgae(ErrorNotImpl));
}

}

