/*
 * task.h
 *
 *  Created on: 2012/12/08
 *      Author: yasuoki
 */

#ifndef TASK_H_
#define TASK_H_

#include "../include/sst_types.h"
#include "cyclicBuffer.h"
#include "process.h"
#include "commands.h"
#include "lock.h"
#include "pjson.h"
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <netdb.h>

namespace SST {

class TaskManager;

class TaskBuffer {
public:
	TaskBuffer() {
	}
	virtual ~TaskBuffer() {
	}
};

class TaskStack : public LinkListNode {
public:
	TaskStack(Process *proc, int func) {
		mBaseProc	= proc;
		mBaseFunc	= func;
		mBaseStep	= 0;
		mTaskBuffer	= NULL;
	}
	~TaskStack() {
		delete mTaskBuffer;
	}

	Process *		mBaseProc;
	int				mBaseFunc;
	int				mBaseStep;
	TaskBuffer *	mTaskBuffer;
};

class Task {
public:
	friend class TaskManager;

	Task(Sentinel *obj);
	~Task();
	void		initTask(Session *conn, Task *baseTask, Process *baseProc, int baseFunc, int baseStep);
	int			getTaskId() { return mId; }
	TaskStatus	getTaskStatus() const { return mStatus; }
	bool		call(QueueNode *q, int retStep=0, int retFunc=0, Process *retProc=NULL);
	bool		startTask(QueueNode *q);
	bool		startReciveTask(QueueNode *q);
	void		beginProcess(QueueNode *q);
	void		endProcess(QueueNode *q);
	bool		endTask(QueueNode *retq);

	bool		putQueue(QueueNode *q);

	int			getWaitingQueue() const { return mWaitQueue; }

	int			getSequence() const { return mSequence; }
	void		setSequence(int seq) { mSequence = seq; }

	size_t		getTaskStackSize() {
		return mTaskStack.getSize();
	}

	Session *	getSession() { return mConn; }
	Process *	getExecProcess() { return mCurStack ? mCurStack->mBaseProc : NULL; }
	QueueNode *	getExecQueue() { return mExecQueue; }
	size_t		getSubTasks() {
		READ_LOCK(mLock);
		return mSubTasks;
	}

	const char *	getDeviceId() const;
	const char *	getServiceId() const;
	const char *	getAccountId() const;
	AccountLevel	getAccountLevel() const;
	const char *	getUserId() const;
	const char *	getPasswd() const;

	bool		setTaskBuffer(TaskBuffer *tb);
	TaskBuffer *getTaskBuffer();
	Sentinel *	getSentinel() { return mSentinel; }

	// security
	ErrorCode allowShowAccount(const Account *acc);
	ErrorCode allowAddAccount(DBPort *port, const Account *acc);
	ErrorCode allowSetAccount(DBPort *port, const Account *acc);
	ErrorCode allowRemAccount(DBPort *port, const Account *acc);
	ErrorCode allowShowDevice(DBPort *port, const DeviceInfo *dev);
	ErrorCode allowRemDevice(DBPort *port, const DeviceInfo *dev);
	ErrorCode allowShowService(const ServiceInfo *svc);
	ErrorCode allowAddService(DBPort *port, const ServiceInfo *svc);
	ErrorCode allowSetService(DBPort *port, const ServiceInfo *svc);
	ErrorCode allowRemService(DBPort *port, const ServiceInfo *svc);
	ErrorCode allowAddGroup(DBPort *port, const ServiceInfo *svc);
	ErrorCode allowSetGroup(DBPort *port, const ServiceInfo *svc, const GroupInfo *group);
	ErrorCode allowRemGroup(DBPort *port, const ServiceInfo *svc, const GroupInfo *group);
	ErrorCode allowAddGroupMember(DBPort *port, const ServiceInfo *svc, const GroupInfo *group, const Account *acc);
	ErrorCode allowRemGroupMember(DBPort *port, const ServiceInfo *svc, const GroupInfo *group, const char *acc);

protected:
	static int		mTaskIdSeq;
	int				mId;
	Sentinel *		mSentinel;
	LockObject		mLock;
	TaskStatus		mStatus;
	Session	*		mConn;
	char			mDeviceId[SST_DEVICEID_LEN+1];
	char			mServiceId[SST_SERVICEID_LEN+1];
	char			mAccountId[SST_ACCOUNTID_LEN+1];
	char			mUserId[SST_USERID_LEN+1];
	char			mPasswd[SST_USERID_LEN+1];
	AccountLevel	mAccountLevel;
	int				mSequence;

	QueueNode *		mExecQueue;
	size_t			mWaitQueue;

	TaskStack *		mCurStack;
	LinkList		mTaskStack;

	size_t			mSubTasks;
	LinkList		mTaskBuffer;
};

class TaskManager {
public:

	TaskManager();
	~TaskManager();

	bool	configure(Sentinel *obj, Conf *conf);
	Task *	createTaskFromSession(Session *conn);
	Task *	createTaskFromBaseTask(Task *task);
	bool	removeTask(Task *t);
	bool	diag(pjson::builder &jb);

protected:
	Task *	get(size_t pos);
	size_t	size() const;
	bool	empty() const;

	class TaskList {
	public:
		TaskList();
		~TaskList();
		void	clear();
		bool	add(Task *t);
		bool	remove(Task *t);
		Task *	get(size_t pos);
		size_t	size() const;
		bool	empty() const;

		size_t		mSize;
		size_t		mBlock;
		Task **		mPtr;
	};

private:
	TaskManager(const TaskManager &c);
	TaskManager &operator=(const TaskManager &c);

protected:
	Sentinel *	mSentinel;
	LockObject	mLock;
	TaskList	mTaskList;
	TaskList	mPoolList;
};

}

#endif /* TASK_H_ */
