//! @file		pf_led.c
//! @brief		プラットフォーム(LED)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "pf_gpio.h"
#include "pf_led.h"

//! @brief		LED→GPIO IDテーブル
static const PF_GPIO_ID pf_led_to_gpio[PF_LED_ID_MAX] =
{
	PF_GPIO_ID_MAQUEEN_LED_L,				//!< PF_LED_ID_L
	PF_GPIO_ID_MAQUEEN_LED_R,				//!< PF_LED_ID_R
};

//! @brief		LED情報テーブル
static BOOL pf_led_info[PF_LED_ID_MAX] =
{
	FALSE,
	FALSE,
};

//! @brief		LED初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
void pf_led_init(void)
{
	PF_LED_ID id;
	PF_GPIO_ID gpio;

	// オート変数初期化
	id = 0;
	gpio = PF_GPIO_ID_MAQUEEN_LED_L;

	// すべてのIDをループ
	for (id = 0; id < PF_LED_ID_MAX; id++)
	{
		// LED消灯(GPIOに対し'H'レベルで点灯、'L'レベルで消灯)
		gpio = pf_led_to_gpio[id];
		pf_gpio_output(gpio, FALSE);

		// LED情報を更新
		pf_led_info[id] = FALSE;
	}
}

//! @brief		LED定期タスク
//! @remarks	プラットフォーム定期タスク(出力系)処理から呼び出すこと
void pf_led_task(void)
{
	PF_LED_ID id;
	PF_GPIO_ID gpio;

	// オート変数初期化
	id = 0;
	gpio = PF_GPIO_ID_MAQUEEN_LED_L;

	// すべてのIDをループ
	for (id = 0; id < PF_LED_ID_MAX; id++)
	{
		// GPIOへ反映
		gpio = pf_led_to_gpio[id];
		pf_gpio_output(gpio, pf_led_info[id]);
	}
}

//! @brief		LED制御
//! @param		[in] id			LEDのID
//! @param		[in] ctrl		LED制御情報(TRUE=LED点灯/FALSE=LED消灯)
void pf_led_ctrl(PF_LED_ID id, BOOL ctrl)
{
	// パラメータチェック
	if (id < PF_LED_ID_MAX)
	{
		// LED情報を更新
		pf_led_info[id] = ctrl;
	}
}
