/**
 * @file
 * @brief readconf() 使用サンプル。
 * @author tsntsumi
 * @since v1.0 2013
 */
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include "readconf.h"

static bool validatePortnoRange(char *name, RCValueType type, RCConfValue value);
static bool convertFigureToNumber(const char *name,
				  RCValueType type,
				  const char *string,
				  RCConfValue *value);
static bool convertFileSizeWithUnit(const char *name,
				    RCValueType type,
				    const char *string,
				    RCConfValue *value);

RCConfItem SampleConfItems[] =
  {
    { "StringConf", RCValueTypeString, "default string" },
    { "NumberConf", RCValueTypeInteger, "14" },
    { "HexNumberConf", RCValueTypeInteger, "0x0000ffff" },
    { "OctalNumberConf", RCValueTypeInteger, "017" },
    { "MinusConf",  RCValueTypeInteger, "-10" },
    { "PlusConf",   RCValueTypeInteger, "+12" },
    { "NoTrail",    RCValueTypeInteger, "0" },
    { "RealNumberConf", RCValueTypeReal, "-.987e-1" },
    { "ValidateConf", RCValueTypeInteger, "50000", validatePortnoRange },
    { "BoolConf",     RCValueTypeBool,    "false" },
    { "ConstantConf", RCValueTypeInteger, "Three", NULL, convertFigureToNumber },
    { "WithUnitConf", RCValueTypeInteger, "8MB",   NULL, convertFileSizeWithUnit },
  };

enum SampleConfItemNo
  {
    StringConf = 0,
    NumberConf,
    HexNumberConf,
    OctalNumberConf,
    RealNumberConf,
    ValidateConf,
    ConstantConf,
    WithUnitConf,
    NumSampleConfItem
  };

int main(int argc, char *argv[])
{
  int i, j, k;

  if (argc < 1)
    {
      printf("sample CONFFILE...\n");
      exit(EXIT_SUCCESS);
    }
  for (i = 1; i < argc; i ++)
    {
      readconf(SampleConfItems, ELEMENTSOF(SampleConfItems), argv[i]);
      for (j = 0; j < RCNumRaisedErrors; j ++)
	{
	  printf("%s:%lu: %s\n", 
		 argv[i], RCRaisedErrors[j].lineNo, RCRaisedErrors[j].reason);
	}
      for (k = 0; k < ELEMENTSOF(SampleConfItems); k ++)
	{
	  if (SampleConfItems[k].type == RCValueTypeString)
	    {
	      printf("%s: %s = \"%s\" (default:%s)\n",
		     argv[0],
		     SampleConfItems[k].name,
		     SampleConfItems[k].value.string,
		     SampleConfItems[k].defaultAsString);
	    }
	  else if (SampleConfItems[k].type == RCValueTypeInteger)
	    {
	      printf("%s: %s = %ld (0x%lx, %03lo) (default:%s)\n",
		     argv[0],
		     SampleConfItems[k].name,
		     SampleConfItems[k].value.integer,
		     SampleConfItems[k].value.integer,
		     SampleConfItems[k].value.integer,
		     SampleConfItems[k].defaultAsString);
	    }
	  else if (SampleConfItems[k].type == RCValueTypeReal)
	    {
	      printf("%s: %s = %g (default:%s)\n",
		     argv[0],
		     SampleConfItems[k].name,
		     SampleConfItems[k].value.real,
		     SampleConfItems[k].defaultAsString);
	    }
	  else if (SampleConfItems[k].type == RCValueTypeBool)
	    {
	      printf("%s: %s = %d (default:%s)\n",
		     argv[0],
		     SampleConfItems[k].name,
		     SampleConfItems[k].value.boolean,
		     SampleConfItems[k].defaultAsString);
	    }
	}
      RCFreeStringValueInItems(SampleConfItems, ELEMENTSOF(SampleConfItems));
      RCFreeRaisedErrors();
    }
  return EXIT_SUCCESS;
}

static bool validatePortnoRange(char *name, RCValueType type, RCConfValue value)
{
  if (value.integer < 40960)
    {
      return false;
    }
  if (value.integer > 65535)
    {
      return false;
    }
  return true;
}

static bool convertFigureToNumber(const char *name,
				  RCValueType type,
				  const char *string,
				  RCConfValue *value)
{
  const char *const figures[] = 
    {
      "Zero", "One",   "Two",   "Three", "Four", "Five",
      "Six",  "Seven", "Eight", "Nine",  "Ten",
    };
  long i;

  for (i = 0; i < sizeof(figures)/sizeof(figures[0]); i ++)
    {
      if (strcmp(string, figures[i]) == 0)
	{
	  value->integer = i;
	  return true;
	}
    }
  return false;
}

static bool convertFileSizeWithUnit(const char *name,
				    RCValueType type,
				    const char *string,
				    RCConfValue *value)
{
  static const struct {
    const char *name;
    long unit;
  } units[] =
    {
      { "B", 1 },
      { "KB", 1024L },
      { "MB", 1024L * 1024L },
      { "GB", 1024L * 1024L * 1024L },
    };
  long size;
  char *unitptr = NULL;

  size = strtol(string, &unitptr, 10);
  if (unitptr != NULL && *unitptr != '\0')
    {
      int i;

      for (i = 0; i < ELEMENTSOF(units); i ++)
	{
	  if (strcmp(unitptr, units[i].name) == 0)
	    {
	      size *= units[i].unit;
	      value->integer = size;
	      return true;
	    }
	}
      return false;
    }
  return true;
}
