/**
 * @file
 * @brief 設定ファイル読み込み関数用構造体等定義ファイル。
 * @author tsntsumi
 * @version 1.3
 * @since v1.0 2013/12
 */
#ifndef READCONF_H
#define READCONF_H

#include <stdbool.h>

/**
 * @defgroup ReadConf 設定ファイル読み込みモジュール
 */

/**
 * @brief 設定項目の型ID。
 * @ingroup ReadConf
 */
typedef enum RCValueType
{
  /** @brief 文字列型。 */
  RCValueTypeString  = 'S',
  /** @brief 整数型(long)。 */
  RCValueTypeInteger = 'I',
  /** @brief 実数型(double)。 */
  RCValueTypeReal    = 'D'
} RCValueType;

/**
 * @brief 設定項目値の共用体。
 * @ingroup ReadConf
 */
typedef union RCConfValue
{
  char *string;
  long integer;
  double real;
} RCConfValue;

/**
 * @brief 設定項目値を検証する関数。
 */
typedef bool (RCValidator)(char *name, RCValueType type, RCConfValue value);

/**
 * @brief 設定項目の構造体。
 * @details
 * この構造体は、 readconf() が読み込むべき設定項目の名前と
 * デフォルト値を指定するのに、アプリケーションによって使用されます。
 * また、読み込んだ設定項目の値を格納する場所としても
 * readconf() によって使用されます。
 * @ingroup ReadConf
 */
typedef struct RCConfItem
{
  /** @brief 項目名。 */
  char *name;
  /** @brief 項目の型。 */
  RCValueType type;
  /** @brief デフォルト値の文字列表現。 */
  char *defaultAsString;
  /** @brief 読み込んだ値を検証する関数。 */
  RCValidator *validator;
  /** @brief デフォルト値。 */
  RCConfValue defaultValue;
  /** @brief 読み込んだ値。 */
  RCConfValue value;
  /** @brief 読み込んだ時の行番号。 */
  size_t lineNo;
} RCConfItem;

/**
 * @defgroup ErrorInfo エラー情報
 * @{
 */

/**
 * @brief エラー番号。
 */
typedef enum RCErrorNo
{
  /** @brief エラー無し。 */
  RCErrorNone = 0,
  /** @brief 読み込み先の設定項目構造体が指定されませんでした。 */
  RCErrorNoItem,
  /** @brief 読み込み先の設定項目構造体がNULLでした。 */
  RCErrorNullItem,
  /** @brief 設定ファイルをオープンできませんでした。 */
  RCErrorOpen,
  /** @brief メモリを確保できませんでした。 */
  RCErrorNoMemory,
  /** @brief 読み込もうとした文字列が長すぎました。 */
  RCErrorStringTooLong,
  /** @brief 読み込もうとした名前が長すぎました。 */
  RCErrorNameTooLong,
  /** @brief 設定ファイルの構文誤りです。 */
  RCErrorSyntax,
  /** @brief 設定項目の型が合っていません。 */
  RCErrorTypeMismatch,
  /** @brief 予期しない設定項目名が現れました。 */
  RCErrorUnexpectedName,
  /** @brief 設定項目が再定義されました。 */
  RCErrorRedefinition,
  /** @brief 数値の形式が間違っています。 */
  RCErrorNumberFormat,
  /** @brief 検証に失敗しました。 */
  RCErrorValidationFailed
} RCErrorNo;

/**
 * @brief エラー情報格納構造体。
 */
typedef struct RCError
{
  /** @brief エラー番号。 */
  RCErrorNo errorNo;
  /** @brief 発生した行番号。 */
  size_t lineNo;
  /** @brief 発生した理由。 */
  char *reason;
} RCError;

/**
 * @brief エラー情報格納配列。
 */
extern RCError *RCRaisedErrors;

/**
 * @brief エラー情報数。
 */
extern size_t RCNumRaisedErrors;

/**
 * @}
 */

/**
 * @defgroup OptionParams オプションパラメータグローバル変数
 * @{
 */

/**
 * @brief 設定ファイルからシンボル(項目名など)を読み込むの最大の長さ。
 */
extern size_t RCMaxSymbolLength;

/**
 * @brief 設定ファイルから文字列を読み込む際の最大の長さ。
 */
extern size_t RCMaxStringLength;

/**
 * @}
 */

/**
 * @brief 指定の設定ファイルから設定を読み込み、指定された構造体配列に格納します。
 * @details
 * 設定ファイルから読み込みたい項目名と、
 * 項目を読み込めなかった際のデフォルト値を指定して本関数を呼び出します。
 * 本関数は設定ファイルを読み込んで、指定された項目に対応する値を設定します。
 * @param[inout] items 読み込むべき設定項目の名前とデフォルト値を設定した
 *                     構造体配列へのポインタ。
 *                     読み込んだ設定項目の値を格納するのにも使用されます。
 * @param[in] numItems 構造体配列 items の項目数。
 * @param[in] confPath 読み込む設定ファイルへのパス。
 * @return 読み込みに成功したら true、読み込み時にエラーがあったら false。
 * @ingroup ReadConf
 */
bool readconf(RCConfItem *items, size_t numItems, const char *confPath);

/**
 * @brief 指定された設定項目構造体配列の文字列を解放します。
 * @details 設定項目構造体が表す型が文字列の場合に、文字列を解放します。
 * 解放した後は NULL を設定します。
 * @param[inout] items 解放すべき文字列が格納された設定項目構造体配列。
 * @param[in] numItems 構造体配列 items の項目数。
 * @ingroup ReadConf
 */
void RCFreeStringValueInItems(RCConfItem *items, size_t numItems);

/**
 * @brief readconf() による読み込み処理で発生したエラー情報構造体配列を解放します。
 * @ingroup ReadConf
 */
void RCFreeRaisedErrors(void);

/**
 * @brief 指定された文字列を複製します。
 * @param str 複製する文字列。
 * @return 複製した文字列。
 */
char *RCDuplicateString(const char *str);

#ifndef ELEMENTSOF
#define ELEMENTSOF(ARRAY) (sizeof(ARRAY)/sizeof(ARRAY[0]))
#endif

#endif /* READCONF_H */

/*
 * Local variables: ***
 * coding: utf-8-unix ***
 * mode: C ***
 * c-file-style: "gnu" ***
 * tab-width: 8 ***
 * indent-tabs-mode: nil ***
 * End: ***
 */
